//Source file: D:/Projects/Isolde/parser/Parser.java

package FSParser;

import java.util.Vector;
import java.util.StringTokenizer;
import java.util.Hashtable;
import domainModel.*;

/**
The parser has to know about the guards on the transitions - these can't be declarative.
We'll assume noun/verb/preposition for now.

The parser needs an accessor to return the constituents table numbers to the GUI interface
so it can set the heads and constituent boundaries properly.
*/
public class Parser
{
    private Grammar _grammar;          // the finite-state grammar we are currently using
    public DmManager _dmManager;       // pointer to the domain model
    private Vector _constituents;      // list of constituents found in the grammar
    private Hashtable name2Index;      // maps from constituent name to constituent table index
    private String _cnlString;         // the string being parsed

    public Parser(DmManager inDmManager)
    {
	// Set up the grammar - parse() will initialize the grammar.
	_grammar = new Grammar();
        _dmManager = inDmManager;

	// Set up the constituent table and map hash - parse() will initialize the indexes.
	_constituents = new Vector();
	name2Index = new Hashtable();
	Vector constituentNames = new Vector();
	constituentNames = _grammar.constituents();
	for (int i = 0; i < constituentNames.size(); i++)
	    {
		Constituent tempConstituent =
		    new Constituent(((String)(constituentNames.elementAt(i))));
		_constituents.add(((Constituent)(tempConstituent)));
		name2Index.put(constituentNames.elementAt(i), new Integer(i));
	    }

    }

    /**
       Takes a string, parses it, and creates DM concepts/instances as appropriate.
       It returns a boolean indicating if the parse worked or not.

       It will include its own view (i.e., the parser interface).

       For now, it will "reparse" by clearing the previous parse and starting from scratch.
       It will set all the constituent starts to length(parseString).
       initialize the grammar
       tokenize the input string
       set currentPosition and workingBufferStart = 0
       gets list of guards, picks one, transitions the grammar - if none works,
       do an unknown transition.
       gets list of required operations and does them.
       @roseuid 395A007A02AB
    */
    public boolean parse(String cnlString)
    {
	_cnlString = cnlString;

	System.out.println("parsing: " + cnlString);

	// Some working variables for the parser
	int currentPosition;         // a cnlString index of the current parser position
	int workingBufferStart;      // an index to the beginning of the current constituent
	StringTokenizer tokenizedInput; // tokenized list of input words to parse
	Vector possibleTransitions;  // a vector of outgoing links from the current state
	String currentToken;         // the current word of the input
	String currentTransition;    // the chosen transition
	String nounReference;        // the name of the DM object instance found by isNoun
	String verbReference;        // the name of the DM concept instance found by isVerb
	Vector currentOperations;    // operations to perform as specified by the grammar
	String headReference;        // name of the DM thing referenced by the head
	String previousState;        // name of the previous state in the parse
	int headLength;              // number of words of the constituent head

	// Initialize the grammar, the constituent table, the context variables
	//     and the tokenized input.
	_grammar.initialize();
	for (int i = 0; i < _constituents.size(); i++)
	    ((Constituent)(_constituents.elementAt(i))).initialize();
	currentPosition = 0;
	workingBufferStart = 0;
        if (cnlString.equals(""))
            return _grammar.inTerminalState();
        tokenizedInput = new StringTokenizer(cnlString);

	// Loop through the words of the input to drive the grammar.
	while (tokenizedInput.hasMoreTokens()) {
	    currentToken = tokenizedInput.nextToken();
	    possibleTransitions = _grammar.possibleTransitions();
	    currentTransition = "unknown";   // default
	    headReference = "";              // default
	    headLength = 0;                  // default
	    	    System.out.println("   current token: |" + currentToken + "|");

	    // determine the POS of the input (there could be more than one of them I suppose)
	    nounReference = isNoun(currentToken, tokenizedInput);
	    verbReference = isVerb(currentToken);

	    // pick a transition that matches the POS and take it
	    for (int i = 0; i < possibleTransitions.size(); i++) {
		if ((possibleTransitions.elementAt(i) == "noun") &&
		    (nounReference != "")) {
		    currentTransition = "noun";
		    headReference = nounReference;
		    headLength = nounReference.length();
		    break;
		}
		else if ((possibleTransitions.elementAt(i) == "verb") &&
			 (verbReference != "")) {
		    currentTransition = "verb";
		    headReference = verbReference;
		    headLength = verbReference.length();
		    break;
		}
		else if ((possibleTransitions.elementAt(i) != "unknown") &&
			 (literalTransitionMatch(currentToken,
						 ((String)(possibleTransitions.elementAt(i)))))) {
		    currentTransition = ((String)(possibleTransitions.elementAt(i)));
		    break;
		}

	    }
	    	    System.out.println("      perform transition: " + currentTransition);
	    previousState = _grammar.getCurrentState();  // save this for one of the operations
	    currentOperations = _grammar.transition(currentTransition);

	    // perform the operations specified by the grammar
	    for (int i = 0; i < currentOperations.size(); i++)
		switch (((Integer)(currentOperations.elementAt(i))).intValue())
		    {
		    case Transition.INCREMENT_CURRENT:
			currentPosition += currentToken.length() + 1;
			break;
		    case Transition.START_EQ_CURRENT:
			((Constituent)(_constituents.elementAt(((Integer)(name2Index.get(_grammar.getCurrentState()))).intValue()))).setStartPosition(currentPosition);
			break;
		    case Transition.START_EQ_WORKING:
			((Constituent)(_constituents.elementAt(((Integer)(name2Index.get(_grammar.getCurrentState()))).intValue()))).setStartPosition(workingBufferStart);
			break;
		    case Transition.WORKING_EQ_START:
			workingBufferStart = ((Constituent)(_constituents.elementAt(((Integer)(name2Index.get(_grammar.getCurrentState()))).intValue()))).startPosition();
			break;
		    case Transition.END_EQ_CURRENT_MINUS_1:
 			((Constituent)(_constituents.elementAt(((Integer)(name2Index.get(previousState))).intValue()))).setEndPosition(currentPosition - 1);
			break;
		    case Transition.HEAD_EQ_CURRENT:
			((Constituent)(_constituents.elementAt(((Integer)(name2Index.get(_grammar.getCurrentState()))).intValue()))).setHeadPosition(currentPosition);
			((Constituent)(_constituents.elementAt(((Integer)(name2Index.get(_grammar.getCurrentState()))).intValue()))).setHeadLength(headLength);
			((Constituent)(_constituents.elementAt(((Integer)(name2Index.get(_grammar.getCurrentState()))).intValue()))).setDomainObjectName(headReference);
			break;
		    }
	}

	System.out.println("out of the loop in state: " + _grammar.getCurrentState());

	// fix one special case: the last sentence constituent will have a
	// startPosition and a -1 endPosition
	((Constituent)(_constituents.elementAt(((Integer)(name2Index.get(_grammar.getCurrentState()))).intValue()))).setEndPosition(cnlString.length());

	printConstituentTable();

	// return the appropriate status
	return _grammar.inTerminalState();
    }

    /**
       Checks the list of known GUI and domain objects

       returns the name of the DM object that was recognized

       It should match the longest possible DM object (e.g., "task model" rather than
       just "task"). and it should return the length of that object name (using the
       length of the internal name is not good enough).
       @roseuid 395A021B0097
    */
    public String isNoun(String token, StringTokenizer st)
    {
	if (token.equals("user") || token.equals("button") ||
	    token.equals("option") || token.equals("model") ||
	    token.equals("mouse"))
	    return token;
	else
	    return "";
    }

    private String isNoun(String token)
    {
    	if (token.equals("user") || token.equals("button") ||
	    token.equals("option") || token.equals("model") ||
	    token.equals("mouse"))
	    return token;
	else
	    return "";
     }


    /**
       checks the list of predefined GUI verbs.  We may also want to use a list of
       "general" verbs from wordnet or something like that.

       returns the length of the recognized string and a pointer to the concept for
       that process (if it exists).
       @roseuid 395A0243035C
    */
    public String isVerb(String token)
    {
	if (token.equals("press") || token.equals("choose") ||
	    token.equals("save") || token.equals("click"))
	    return token;
	else
	    return "";
    }

    /**
       returns true if the current token matches a literal transition (surrounded by quotes
       in the grammar)
       @roseuid 395A024902EC
    */
    public boolean literalTransitionMatch(String token, String transition)
    {
	return (transition.startsWith("\"") &&
		token.equals(transition.substring(1,transition.length() - 1)));
    }

    // a utility for debugging.
    public void printConstituentTable()
    {
	for (int i = 0; i < _constituents.size(); i++)
	    ((Constituent)(_constituents.elementAt(i))).print();
    }

    /**
       goes through the constituent start positions, creating DM concepts and instances
       as required.  It is careful not to "recreate" entities when they already exist in the DM.
       It is hard-coded for the constituents we know to be in the grammar.
       @roseuid 395A07B50266
    */
    public void createDM()
    {
	DmConcept dc;   // a utility variable for creating DmConcept objects

	// grab the process constituent
	Constituent c = ((Constituent)(_constituents.elementAt(((Integer)(name2Index.get("process"))).intValue())));

	// create a raw process instance - there has to be one no matter what
	System.out.println("Create new action instance with:");

	if (c.startPosition() == -1)
	    // there is no process, so it must all be canned
	    System.out.println("   canned string: \"" + _cnlString + "\"");

	else if (c.headPosition() == -1)
	    // there is a process, but no head
	    System.out.println("   canned string: \"" +
			       _cnlString.substring(c.startPosition(),
						   c.endPosition()) +
			       "\"");

	else {
	    // make sure there is a domain concept for the process
	    if (c.domainObjectName() == "") {
		System.out.println("   create new concept: \"" +
				   _cnlString.substring(c.headPosition(),
						       c.headPosition() + c.headLength() + 1) +
				   "\"");
		c.setDomainObjectName(_cnlString.substring(c.headPosition(),
							  c.headPosition() + c.headLength() + 1));
	    }

	    // link the process action to its action concept
	    System.out.println("   concept parent: \"" + c.domainObjectName() + "\"");
	    dc = new DmConcept(c.domainObjectName(), "action type");
	    _dmManager.addConcept(dc);

	    // link the pre and post modifiers if there are any
	    if (c.headPosition() != c.startPosition())
		System.out.println("   premod: \"" +
				   _cnlString.substring(c.startPosition(),
						       c.headPosition()-1) +
				   "\"");
	    if ((c.headPosition() + c.headLength()) != c.endPosition())
		System.out.println("   postmod: \"" +
				   _cnlString.substring(c.headPosition() + c.headLength() + 1,
						       c.endPosition()) +
				   "\"");

	    // deal with the objects (agent, patient, location, etc.)
	    for (int i = 0; i < _constituents.size(); i++) {
		if (((Constituent)(_constituents.elementAt(i))).name() != "process") {
		    createDMObject(((Constituent)(_constituents.elementAt(i))).name());
		// we will have to explicitly link the new role filler to the process here
		}
	    }
	}
	return;
    }

    // this will have to return the name of the new object or a pointer to it
    private void createDMObject(String constituentName)
    {
	// first grab the constituent so that it's easy to refer to
	Constituent c = ((Constituent)(_constituents.elementAt(((Integer)(name2Index.get(constituentName))).intValue())));

	// skip non-existent constituents
	if (c.startPosition() != -1) {

	    System.out.println("   " + constituentName + ":");

	    if (c.domainObjectName() == "")
		// create a brand new object when none already exists

                ;
		//System.out.println("      create canned object instance: \"" +
		//		   _cnlString.substring(c.startPosition(), c.endPosition()) +
		//		   "\"");
	    else {
                if (isNoun(c.domainObjectName()) == "")
		   // create new object specified by user
		   System.out.println("      create new user specified object instance: \"" +
			   	      c.domainObjectName() + "\"");

                if ((c.headPosition() != c.startPosition()) ||
		     (c.headPosition() + c.headLength()) != c.endPosition()) {
		  // create a new object that is a copy of the head object instance
		  System.out.println("      create object instance: \"" +
		  		     _cnlString.substring(c.startPosition(), c.endPosition()) +
				      "\" with same features as \"" +
				    c.domainObjectName() + "\"");
		// add the pre and post modifiers if they exist
		if (c.headPosition() != -1) {
		    if (c.headPosition() != c.startPosition())
			System.out.println("         add object premod: \"" +
					   _cnlString.substring(c.startPosition(),
							       c.headPosition()-1) +
					   "\"");
		    if ((c.headPosition() + c.headLength()) != c.endPosition())
			System.out.println("         add object postmod: \"" +
					   _cnlString.substring(c.headPosition() +
							       c.headLength() + 1,
							       c.endPosition()) +
					   "\"");
		}
	    }
	    else
                   // just use/return the existing DM object
		   System.out.println("      use existing object instance: \"" +
			   	      c.domainObjectName() + "\"");
             }
             		}
    }


    /**
       This function will assume that it will never be asked to move a start forward or
       backward beyond the next start in the default consituent order.

       The constituent will either be agent/process/patient/prepositional phrase.
       @roseuid 395B42DC03C8
    */
    public boolean moveConstituent(String constituentName, Integer newStartPosition, Integer newEndPosition)
    {
      ((Constituent)(_constituents.elementAt(((Integer)(name2Index.get(constituentName))).intValue()))).setStartPosition(newStartPosition.intValue());
      ((Constituent)(_constituents.elementAt(((Integer)(name2Index.get(constituentName))).intValue()))).setEndPosition(newEndPosition.intValue());
      return false;
    }

    /**
       Changes the head of the passed constituent.
       @roseuid 395B49E3004A
    */
    public boolean changeHead(String constituentName, Integer newHeadStart, Integer newHeadEnd)
    {
      ((Constituent)(_constituents.elementAt(((Integer)(name2Index.get(constituentName))).intValue()))).setHeadPosition(newHeadStart.intValue());
      ((Constituent)(_constituents.elementAt(((Integer)(name2Index.get(constituentName))).intValue()))).setHeadLength(newHeadEnd.intValue() - newHeadStart.intValue());
      ((Constituent)(_constituents.elementAt(((Integer)(name2Index.get(constituentName))).intValue()))).setDomainObjectName(_cnlString.substring(newHeadStart.intValue(),
                                                                                                                                                 newHeadEnd.intValue()));
      return false;
    }

    public String cnlString()
    {
	return _cnlString;
    }

    public Vector constituents()
    {
	return _constituents;
    }

    public Vector possibleTransitions()
    {
      return _grammar.possibleTransitions();
    }
}
