/*
   Tamot --- Task Modelling Editor
   Copyright (C) 2000 Commonwealth Scientific and Industrial Research Organisation
   (CSIRO), Australia. All rights reserved. 

   The Software, owned by CSIRO Australia, has been developed by the CSIRO 
   Intelligent Interactive Technology group. 
   For more information, please see http://www.cmis.csiro.au/iit/.

   The Software is a beta-test version for internal research and evaluation purposes 
   by the Licensee. The Software is still at a development stage. 
   It has not undergone complete testing and may have inherent errors,
   bugs or deficiencies that can affect the operation of the Software. 

   We encourage your feedback and suggestions. Any bugs / suggestions found please email to
   Shijian.Lu@cmis.csiro.au and Thomas.Lo@cmis.csiro.au. 

   After you have signed the Beta-Test Software License Agreement with the CSIRO,
   CSIRO grants you a non-exclusive, non-transferable license to use it only for your 
   personal, non-commercial and lawful end use. Implied licenses are negated. 

   Warranty Disclaimer : 

   CSIRO supplies the Software on an "as is" basis and makes no warranties that use of
   the Software by the Licensee will not infringe any third partys 
   Intellectual Property Rights nor that the Software will be of merchantable quality,
   or suitable for a particular purpose. CSIRO excludes all terms, conditions and
   warranties implied by custom, the general law or statute in relation to the
   Software.
*/

package domainModel;

import java.util.Vector;

/**
 * This class implements the Singleton pattern 
 * so that only one Concept-Type object can be created.
 * Concept-types created on the fly is instently visible.
 * It manages the Concept-types. 
 */
public class ConceptType extends Object {
	private final static String _typeSuffix= " type"; 

	private final static String defaultEntityType= "object"+_typeSuffix; 
	private final static String defaultActionType= "material"+_typeSuffix;
	private final static String defaultUserType= "user"+_typeSuffix; 
	private final static String defaultSystemType= "system"+_typeSuffix;
	private final static String defaultInterfaceType= "interface"+_typeSuffix;
	//18-10-2001
	private final static String D3InterfaceType= "3d interface"+_typeSuffix;

	//2001-8-14
	private final static String d12LocationType= "1-2d location"+_typeSuffix; 
	private final static String d3LocationType= "3d location"+_typeSuffix; 
	private final static String motionType= "motion"+_typeSuffix; 
	
	private final static String predefinedTypes[] 
		= {defaultEntityType, 		//entity-type, such as "button", "document"
			defaultActionType, 		//action-type, such as "select", "press"
			defaultUserType,		// user-type, such as, "editor", "reader"
			defaultSystemType, 		// system-type, such as , Tamot, Word
			defaultInterfaceType,
			//d12LocationType,
			//d3LocationType,
			//motionType,	
			//D3InterfaceType,	//18-10-2001
			"quality"+_typeSuffix};		// quality-type
			
				// note in theory, "user" and "system" are sub-type type of
				// "entity" type, but for convenience, they are listed in the
				// same order
	//2001-8-14
	private final static String predefinedObjectTypes[] 
		= {defaultEntityType, 		//entity-type, such as "button", "document"
			d12LocationType, 	
			d3LocationType};		
	//2001-8-14
	private final static String predefinedActionTypes[] 
		= {defaultActionType, 		
			motionType};		
	//2001-10-18
	private final static String predefinedInterfaceTypes[] 
		= {defaultInterfaceType,		
			D3InterfaceType};		
	//2001-10-19
	private final static String predefinedUserTypes[] 
		= {defaultUserType 	
		};		
			
	//2001-10-19
	private final static String predefinedSystemTypes[] 
		= {defaultSystemType 		
		};		
			
	private static Vector allTypes; // types collection: storing all known concept types
   private static ConceptType cTInstance
      = new ConceptType();
    
   //static initialisation
   static {
    	allTypes = new Vector(5);	
    	for (int i = 0; i < predefinedTypes.length; i++) {	// add predefined types to the 
    		allTypes.add(predefinedTypes[i]);				// collection
       	}    

   }

    /**
     * This private constructor is defined so the compiler won't
     * generate a default public constructor.
     */
    private ConceptType() { 
    }
    
    /**
     * This private method preprocess a input type for 
     * comparison or add-in to the collection. it makes sure that
     * 	(1) type is in lower-case, and
     *	(2) type has the suffix of "concept"
     * @param type, the string to be processed
     */
    private String getWellFormedType(String type){
    	String temp = type.toLowerCase().trim();
    	//String Suffix = "T";
    	if (temp.endsWith(_typeSuffix)) return temp;
    	else return (temp + _typeSuffix);
    } //getWellFormedType
    
    /**
     * Return a reference to the only instance of this class.
     */
    public static ConceptType getInstance() {
        return cTInstance;
    } // getInstance()

    /**
     * Check if the given string is a defined concept-type 
     * it will return true whether or not there is "type"
     * in type, eg, "entity" or "entity type". Note that 
     * lower-case is used for comparison.
     * @param type, the string to be checked
     */
    public boolean isConceptType(String type) {
        if (allTypes.contains(getWellFormedType(type) )){
        	return true;
        }else return false;
    } // isConceptType()

    /**
     * Add a new concept type to the collection. 
     * 
     * @param type, the new concept-type to be added. 
     * If type is already in the collection, a message is print out.
     */
    public void addConceptType(String type) {
    	String temp = getWellFormedType(type);
    	if (isConceptType(temp)) {
    		System.out.println ("conept-type: "+ temp + 
    			" is already in type-collection!");
    	} else allTypes.add(temp);
    } // addConceptType(String type)

    /**
     * get a valid concept type from the collection. 
     * 
     * @param type, it may be not-well-formed. 
	 * @return String, well-formed type
     */
    public String getConceptType(String type) {
    	//String temp = 
    	return getWellFormedType(type);
    //	if (isConceptType(temp)) 
    //		return temp;
    //	else return null;
    } // getConceptType(String type)

    /**
     * Delete a concept type from the collection. 
     * 
     * @param type, the concept-type to be deleted. 
     * If type is not in the collection, a message is print out.
     */
    public void deleteConceptType(String type) {
    	String temp = getWellFormedType(type);
    	boolean aPredefinedType =false;
    	for (int i = 0; i < predefinedTypes.length; i++) {	// check if type is a predefined type 
    		if (temp.equals(predefinedTypes[i])){
    			aPredefinedType = true;	
    			break;										// collection
    		}
       	}    

    	if (!(isConceptType(temp))) {
    		System.out.println ("conept-type: "+ temp + 
    			" is not in type-collection!");
    	} else if (aPredefinedType) {
    		System.out.println ("conept-type: "+ temp + 
    			" is a predefined type, so it cannot be deleted from the collection!");
    	} else
    	allTypes.removeElement(temp);
    } // deleteConceptType(String type)

    /**
     * retrun all known concept-types 
     */
    public String[] getAllTypes() {
    	String types[] = new String[allTypes.size()];
        allTypes.copyInto(types);
        return types;
    } // getAllTypes()
    
    /**
     * retrun number of all known concept-types 
     */
    public int getCount() {
    	return allTypes.size();
    } // getCount()
    
    /**
     * retrun default entity concept-type 
     */
    public static String getDefaultEntityType() {
    	return defaultEntityType;
    } // getDefaultEntityType() 
    /**
     * retrun default action concept-type 
     */
    public static String getDefaultActionType() {
    	return defaultActionType;
    } // getDefaultActionType()
    
    /**
     * retrun default user concept-type 
     */
    public static String getDefaultUserType() {
    	return defaultUserType;
    } // getDefaultUserType()()
        
    /**
     * retrun default system concept-type 
     */
    public static String getDefaultSystemType() {
    	return defaultSystemType;
    } // getDefaultSystemType()()
    
    /**
     * retrun default interface concept-type 
     */
    public static String getInterfaceType() {
    	return defaultInterfaceType;
    } // getDefaultUserType()()

    /**
     * retrun default user concept-type 
     */
    public static String[] getDefinedTypes() {
    	return predefinedTypes;
    } // getDefaultUserType()()
    
    /**
     * retrun predefined action -type 
     */
    public static String[] getDefinedActionTypes() {
    	return predefinedActionTypes;
    } 
    
    /**
     * retrun predefined object -type 
     */
    public static String[] getDefinedObjectTypes() {
    	return predefinedObjectTypes;
    } 

   //2001-8-15 
   	public static boolean isActionType(String type) {
		String d3types[]=getDefinedActionTypes();
		for (int i = 0; i < d3types.length; i++) {
    		if(type.equals(d3types[i]))	
    			return true;
       	}   
        return false;
    } // isActionType()
    
    //2001-8-15
   	public static boolean isObjectType(String type) {
		String d3types[]=getDefinedObjectTypes();
		for (int i = 0; i < d3types.length; i++) {
    		if(type.equals(d3types[i]))	
    			return true;
       	}   
        return false;
    } // isObjectType()

//2001-8=21
    /**
     * retrun motion action concept-type 
     */
    public static String getMotionActionType() {
    	return motionType;
    } // getMotionActionType()

//2001-8=21
    /**
     * retrun 1 or 2 D concept-type 
     */
    public static String getD12Type() {
    	return d12LocationType;
    } // getD12Type()

//2001-8=21
    /**
     * retrun 3 D concept-type 
     */
    public static String getD3Type() {
    	return d3LocationType;
    } // getD3Type()
    
    //18-10-2001
    /**
     * retrun default 3d concept-type 
     */
    public static String getD3InterfaceType() {
    	return D3InterfaceType;
    } // 
    
    /**
     * retrun predefined interface -types
     */
    public static String[] getDefinedInterfaceTypes() {
    	return predefinedInterfaceTypes;
    } 

   	public static boolean isInterfaceType(String type) {
		String d3types[]=getDefinedInterfaceTypes();
		for (int i = 0; i < d3types.length; i++) {
    		if(type.equals(d3types[i]))	
    			return true;
       	}   
        return false;
    } // isInterfaceType() 

    //19-10-2001
   	public static boolean isUserType(String type) {
		String d3types[]=getDefinedUserTypes();
		for (int i = 0; i < d3types.length; i++) {
    		if(type.equals(d3types[i]))	
    			return true;
       	}   
        return false;
    } // isUserType() 
   	public static boolean isSystemType(String type) {
		String d3types[]=getDefinedSystemTypes();
		for (int i = 0; i < d3types.length; i++) {
    		if(type.equals(d3types[i]))	
    			return true;
       	}   
        return false;
    } // isUserType() 
    /**
     * retrun predefined user -types
     */
    public static String[] getDefinedUserTypes() {
    	return predefinedUserTypes;
    } 

    /**
     * retrun predefined system -types
     */
    public static String[] getDefinedSystemTypes() {
    	return predefinedSystemTypes;
    } 

    /**
	 * @return void
	* test the class.
	*
	* 
	*/
	public static void main(String args[]) {
    	
		ConceptType d1 = ConceptType.getInstance();		
		System.out.println("nO of concept-types: " + d1.getCount());
		d1.addConceptType("something");
		d1.addConceptType("new concept");
		d1.addConceptType("entitY");
		String ty="action";
		if (!(d1.isConceptType(ty))) d1.addConceptType("entitY");
		
		ConceptType d2 = ConceptType.getInstance();
		d2.addConceptType("d2_new"); 
		d2.deleteConceptType("entity");
		d2.deleteConceptType("something");
		System.out.println("nO of concept-types: " + d1.getCount());

		String types[] = d1.getAllTypes();
		for (int i = 0; i < types.length; i++) {
    		System.out.println("concept-types: " + types[i]);				
       	}  
       	
       	System.out.println();
		String d2types[] = d2.getDefinedObjectTypes();
		for (int i = 0; i < d2types.length; i++) {
    		System.out.println("object concept-types: " + d2types[i]);				
       	}   
       	
       	System.out.println();
		String d3types[] = d2.getDefinedActionTypes();
		for (int i = 0; i < d3types.length; i++) {
    		System.out.println("action concept-types: " + d3types[i]);				
       	}   
       	
       	System.out.println();
		String d4types[] = d2.getDefinedInterfaceTypes();
		for (int i = 0; i < d4types.length; i++) {
    		System.out.println("interface concept-types: " + d4types[i]);				
       	} 
       	
       	String inter= " 3D interface type";
       	if (isInterfaceType(inter))
       		System.out.println("yes, it is inter");
       	else 
       		System.out.println("no, it is not inter");
				
    } //main()


} // class ConceptType