<?xml version="1.0" encoding="UTF-8"?>
<xsl:stylesheet version="1.0" xmlns:xsl="http://www.w3.org/1999/XSL/Transform">

<!-- Stylesheet to generate a HTML file with:
		   * a Table of contents
		   * the Details of Tasks 
		   * an image map for each task
   from a Tamot XML output file.
   The XML file is compliant with the Task_Model DTD.
   
   In the detail of a task, we list the attributes which are requested by the command line
   with Saxon:
   Ex.: saxon fic.xml fic.xsl name=y terminal=y <==> we write the name of the task and if 
   the task is terminal or not.
   If all the attributes, write :
   saxon fic.xml fix.xsl "title=Generation HTML depuis XML" "composite=y" "feedback=y" 
                         "mandatory=y" "cardinality=y" "precondition=y" "postcondition=y" 
                         "semantics=y" "style=y" "terminal=y"
-->

<!-- this version has a better layout -->
     
<!-- Written by Jean-Claude Tarby, 17th July 2000 
	 corrected on 18th July (Bug in table for contexts) 
	 improved  on 19th July to hide table for Parents
	 improved  on 29th July to add image map to the .png files
	    (some pbs to display labels in Netscape, but not with MIE) (due to a bug of Netscape only !)
	    (some pbs for the links in MIE, but not with Netscape)
	    (an improvement to do : image map ONLY on tasks and not on boolean connectors too)
	 improved on 31th July to eliminate unuseful commas by keeping only tasks for main task(s)
	 updated  on  1st August to manage the HTML comment generated by Tamot
     Modified by Thomas Lo, 8th Nov 2000 : 
	 corrected for semantics output
	 corrected for comment output
	 
	 Modified by Shijian Lu, 4-10-2001: 
	 add the following attributes for tasks
	 	- Background
	 	- Tip
	 	- Warning
	 	- Annotation	
-->



<!-- the result is a HTML file -->
<xsl:output method = "html" />


<!-- Definition of the parameters used by Saxon -->
<xsl:param name="title"/>		<!-- Web page title -->
<xsl:param name="name"/>    		<!-- (core) write the task's name --> 
					<!-- name is not used here because obviously used -->
<xsl:param name="composite"/>		<!-- (core) write if the task is composite or elementary-->
<xsl:param name="semantics"/>		<!-- (core) write the task's semantics -->
<xsl:param name="style"/>		<!-- (core) write the task's style-->
<xsl:param name="background"/>		<!-- (core) write the task's background-->
<xsl:param name="comment"/>		<!-- (core) write the task's comment-->
<xsl:param name="tip"/>		<!-- (core) write the task's tip-->
<xsl:param name="warning"/>		<!-- (core) write the task's warning-->
<xsl:param name="annotation"/>		<!-- (core) write the task's annotation-->
<xsl:param name="feedback"/>		<!-- (context) write the task's feedback -->
<xsl:param name="mandatory"/>		<!-- (context) write if the task is mandatory or optional -->
<xsl:param name="cardinality"/>		<!-- (context) write the minCard and the maxCard of the task -->
<xsl:param name="precondition"/>	<!-- (context) write the task's precondition -->
<xsl:param name="postcondition"/>	<!-- (context) write the task's postcondition -->
<xsl:param name="terminal"/>		<!-- (context) write if the task is terminal or not -->
<xsl:param name="dirty"/>		<!-- (not used) write if the task is dirty or not -->



<!-- Definition of XSLT 'keys' to facilitate the search of values (id) -->

<!-- index the Expansions by Component id -->
<xsl:key name="expansion-component_id" match="Expansion" use="Component/@id" />

<!-- index the Expansions by Task_Name -->
<xsl:key name="expansion-name" match="Expansion" use="@Task_Name" />

<!-- index the Tasks by task core name  -->
<xsl:key name="task-task_core_name" match="Task" use="Task_Core/@Name" />

<!-- index the Tasks by task context id -->
<xsl:key name="task-task_context_id" match="Task" use="Task_Context/@id" />


<!-- Template for the output HTML file -->
<xsl:template match="/">
  <html>
    <head>
      <title><xsl:value-of select="$title"/></title>
    </head>
    <body bgcolor="#FFFFFF">  <!-- White background -->
         <a name="top"></a>	<!-- to go to the top later in the document -->
         <!-- A title for the content of the page -->
         <h1 align="center"><xsl:value-of select="$title"/></h1>
       	 <!-- Main Menu which appears at the top of the HTML file -->
         <p align="center"><a href="#TOC">Table of Contents</a> | 
         <a href="#Detail">Detail</a></p>

    	 <!-- Table of Contents 
    	      (task names sorted by alphabetical order)
    	 -->
    	 <!-- write "Table of contents" in a color frame -->
    	 <table width="100%" border="0" bgcolor="#CCFFCC">
    	   <tr>
    	     <td>
    	       <h2><a name="TOC">Table of Contents</a></h2>
    	     </td>
    	   </tr>
    	 </table>
         <xsl:apply-templates select="Tamot/Task_Model_Components" mode="table-of-contents"/>
         
         <hr></hr>

    	 <!-- Task Details -->       
    	 <!-- write "Detail" in a color frame -->
    	 <table width="100%" border="0" bgcolor="#CCFFCC">
    	   <tr>
    	     <td>
               <h2><a name="Detail">Detail</a></h2>
    	     </td>
    	   </tr>
    	 </table>
         <xsl:apply-templates select="Tamot/Task_Model_Components" mode="details"/>
          
    </body>
  </html>
</xsl:template>

<!-- ************ Task_Model  mode="table-of-contents"  ******************  -->

<xsl:template match="Task_Model_Components" mode="table-of-contents">
  <div id="contents">
    <!-- First, write  the main task of the Task Model
         The ROOT task must be called 'TM_ROOT_TM' (or whatever, but change the name below)
    -->
      <p><i><b><xsl:text>Main Task : </xsl:text></b></i>
      <xsl:variable name="expansion-of-main-task" select="/Tamot/Task_Model_Decomposition/Expansion[@Task_Name='TM_ROOT_TM']"/>
      <!-- expansion-of-main-task contains the Expansion element of the root (main) task-->
      <xsl:for-each select="$expansion-of-main-task/Component">
      <!-- for each Component element of this Expansion element, we write the Task Name
           with a hyperlink to the Detail of the Task -->
          <!-- main_task contains the Task element correspondant to the current Component -->
	      <xsl:variable name="main_task" select="key('task-task_context_id',@id)"/>
          <!-- We keep only tasks, i.e if main_task exists -->
          <!-- So, if the component is not  a task, we do not write a comma -->
	      <xsl:if test="$main_task">
		      <!-- we write a <a href =...> tag for the Task Name-->
		      <a href = "#{generate-id($main_task)}">
		         <xsl:value-of select="$main_task/Task_Core/@Name" />
		      </a>
	          <!-- if this is not the last main task, we put a comma -->
			  <xsl:if test="position() != last()"> , </xsl:if>
		  </xsl:if>
      </xsl:for-each> 
      </p>
      <!-- because there is main task(s), there is a root picture -->
      <!-- the name must be Root.png - If not change below -->
      <!-- like all task details, the Root has an image map -->
      <p><img src="Root.png" border="0" usemap="#Root"/></p>
            <map name="Root">
				<xsl:for-each select="key('expansion-name','TM_ROOT_TM')/Component">
				   <area shape="rect" coords="{@PNGx1},{@PNGy1},{@PNGx2},{@PNGy2}" href="#{generate-id(key('task-task_context_id',@id))}" alt="{key('task-task_context_id',@id)/Task_Core/@Name}"/>
				</xsl:for-each>
		    </map>                  

    <!-- Then, write the list of all task, including main task -->
    <i><b><xsl:text>All tasks : </xsl:text></b></i>
    <xsl:for-each select="Task/Task_Core">
      <xsl:sort select="@Name" order="ascending" />
      <br>
      <!-- generate-id() is a XSLT function which generate an unique ID for a node.
           Here, the node is the Task element which has the name @Name. So we generate
           a hyperlink to the Task element which has the name @Name.
           This implies that the name of a task is UNIQUE in the Task Model.
      -->
      <a href = "#{generate-id(key('task-task_core_name',@Name))}">
         <xsl:value-of select="@Name" />
      </a>
      </br>
    </xsl:for-each>
    <p><a href="#top"><xsl:text>Go to the Top</xsl:text></a></p>
  </div>
</xsl:template>


<!-- ************ Task_Model mode="details" *****************************  -->

<xsl:template match="Task_Model_Components" mode="details">

  <!-- goes through the Tasks in the order they appear in the
       input - and sort this alphabetically 
       
       1. Find all parents
       2. Find all sub-tasks ordered by decomposition
  -->
  
  <xsl:for-each select="Task">	
  	<!-- define a variable to use easily the name of the current task -->								
	<xsl:variable name="taskName" select="Task_Core/@Name"/>
	<!-- Remarks: 
	     1. because it is impossible to use the variable taskName here to sort the task by 
	        alphabetical order, so I use the definition of the variable instead the variable  
	     2. some parsers may not recognize xsl:sort if it is not the first line after xsl:for-each.
	        So, if the list is not alphabetically ordered, move the xsl:sort just after the xsl:for-each
	-->
	<!-- the task names will be sorted by alphabetical order -->
	<xsl:sort select="Task_Core/@Name"/>
		
	<!-- Wite the name of the Task in a coloured frame -->
	<table width="100%" border="0" bgcolor="#CCFFFF">
	  <tr>
	    <td>
			<!-- define the HTML anchor for the Task -->
			<a name="{generate-id()}"></a>
			<h3> <xsl:value-of select="$taskName" /> </h3> 
			<!-- <h2> is used to read easily in HTML and to produce automatically 
			     a table on index in Word by importing the HTML file into Word -->
	    </td>
	  </tr>
	</table>
    
    <p></p>
    <!-- write the requested attributes of the task's core -->
    <!-- if no attribute requested, write nothing-->
    <xsl:if test="$composite or $semantics or $style or $background or $comment or $tip or $warning or $annotation">
    <b><xsl:text>Core's attributes</xsl:text></b>
    <ul>
		<!-- (core) write if the task is composite or elementary-->
	    <xsl:if test="$composite='y'">
		    <li>
		    <xsl:value-of select="Task_Core/@Composite"/>
		    </li>
	    </xsl:if>
	
		<!-- (core) write the task's semantics -->
	    <xsl:if test="$semantics='y'">
		    <li>semantics : 
		    <xsl:value-of select="Task_Core/Semantics"/>
		    </li>
	    </xsl:if>
	
		<!-- (core) write the task's style-->
	    <xsl:if test="$style='y'">
		    <li>
		    <xsl:value-of select="Task_Core/@Style"/>
		    </li>
	    </xsl:if>

		<!-- (core) write the task's background-->
	    <xsl:if test="$background='y'">
		    <li> <b><i>Background: </i></b>
		    <xsl:apply-templates select="Task_Core/Background_HTML"/>
		    </li>
	    </xsl:if>

		<!-- (core) write the task's comment-->
	    <xsl:if test="$comment='y'">
		    <li> <b><i>Comment: </i></b>
		  <!--  <xsl:value-of select="Task_Core/Comment_HTML/html/body"/> -->
		    <xsl:apply-templates select="Task_Core/Comment_HTML"/>
		    </li>
	    </xsl:if>

		<!-- (core) write the task's tip-->
	    <xsl:if test="$tip='y'">
		    <li> <b><i>Tip: </i></b>
		    <xsl:apply-templates select="Task_Core/Tip_HTML"/>
		    </li>
	    </xsl:if>

		<!-- (core) write the task's Warning-->
	    <xsl:if test="$warning='y'">
		    <li> <b><i>Warning: </i></b>
		    <xsl:apply-templates select="Task_Core/Warning_HTML"/>
		    </li>
	    </xsl:if>

		<!-- (core) write the task's Annotation-->
	    <xsl:if test="$annotation='y'">
		    <li> <b><i>Annotation: </i></b>
		    <xsl:apply-templates select="Task_Core/Annotation_HTML"/>
		    </li>
	    </xsl:if>
	  	  	    
    </ul>
    </xsl:if>
    
    <p></p>
    
    <!--  ***** 1. Find all parent Tasks which use the current task ****** -->
    <!-- write a table with the parent on left and the associated context's attributes
         on the right
    -->
    <table width="100%" border="0">
    <!-- border="0" because too hard to read when printed. For the same reason, I added
         a blank line after each context -->
      <tr>
        <td colspan="2"><b>Parent(s)</b></td>
      </tr>
	      <xsl:choose>
	      <xsl:when test="Task_Context"> <!-- if there is at least a context-->
	        <xsl:for-each select="Task_Context">
	        <!-- the Parent task(s) will be alphabetically ordered -->
	        <xsl:sort select="key('expansion-component_id', @id)/@Task_Name"/>
               <tr align="left" valign="top">
		           <!-- we generate a hyperlink to the Task element which has the name Task_Core/@Name -->
		           <td>
		           <a href = "#{generate-id(key('task-task_core_name',key('expansion-component_id', @id)/@Task_Name))}">
		           <xsl:value-of select="key('expansion-component_id', @id)/@Task_Name" />
		           </a>
		           </td>
		           <!-- write the requested attributes of the task's current context -->
				   <!-- if no attribute requested, write nothing-->
	               <xsl:if test="$feedback or $mandatory or $cardinality or $precondition or $postcondition or $terminal">
	               <td><b><p><xsl:text>Context's attributes</xsl:text></p></b>
		           <ul>
			          <!-- (context) write the task's feedback -->
		              <xsl:if test="$feedback='y'">
			             <li>feedback : 
			             <xsl:value-of select="Feedback"/>
			             </li>
		              </xsl:if>
	  	            
		              <!-- (context) write if the task is mandatory or optional -->
		              <xsl:if test="$mandatory='y'">
			             <li>
			             <xsl:value-of select="@Mandatory"/>
			             </li>
		              </xsl:if>
	  	
		              <!-- (context) write the minCard and the maxCard of the task -->
		              <xsl:if test="$cardinality='y'">
			             <li>
			             MinCard = <xsl:value-of select="@MinCardinal"/> , MaxCard = <xsl:value-of select="@MaxCardinal"/>
			             </li>
		              </xsl:if>
		
		              <!-- (context) write the task's precondition -->
		              <xsl:if test="$precondition='y'">
			             <li>precondition : 
			             <xsl:value-of select="Precondition"/>
			             </li>
		              </xsl:if>
		
		              <!-- (context) write the task's postcondition -->
		              <xsl:if test="$postcondition='y'">
			             <li>postcondition : 
			             <xsl:value-of select="Postcondition"/>
			             </li>
		              </xsl:if>
		
		              <!-- (context) write if the task is terminal or not -->
		              <xsl:if test="$terminal='y'">
			             <li>terminal : 
			             <xsl:value-of select="@Terminal"/>
			             </li>
		              </xsl:if>
		           </ul>	
		           <br></br> <!-- blank line added for legibility -->
		           </td>
		           </xsl:if>
		        <br></br>
               </tr>
	        </xsl:for-each>	<!-- end for-each Task_Context-->
	      </xsl:when>
	      <!-- if there is no Parent, output 'None' -->
	      <xsl:otherwise>None</xsl:otherwise>
	      </xsl:choose>
    </table>
   
    <p></p>
   												
    <!-- 2. *** Find all the sub-tasks ordered by decomposition ***** -->
    
    <!-- Remark : theorically, it should have ONE expansion for ONE task, but this
         stylesheet runs as if there is MANY expansions for ONE task. This allows to
         control that the task model is correct.
         If we want to simplify, we can write for example :
             <xsl:variable name="expansions" select="Expansion[@Task_Name=Task_Core/@Name]"/>
             but some parser refuse this, and they oblige to use a variable to represent
             Task_Core/@Name, that means Expansion[@Task_Name=variable]

    -->
    
    <!-- define a variable expansions to contain all the Expansions for the current task -->
    <xsl:variable name="expansions" select="key('expansion-name', Task_Core/@Name)" />
    <b>SubTasks :</b>
    <xsl:choose>
      <!-- if there are any expansion... -->
      <xsl:when test="$expansions">
      <ul>
      <!-- go through the expansion one by one (document order) -->
      <xsl:for-each select="$expansions">			
        <li>    
           <xsl:for-each select="Component[key('task-task_context_id',@id)]">		
             <!-- we process all Component elements which correpond to Tasks 
                  (we don't want links or boolean connectors). To do that, we select
                  only Tasks by using the Component id and by matching it with
                  the Task_Context ids.   
                  The result is sorted by task names (alphabetical order).
             -->
             <xsl:sort select="key('task-task_context_id',@id)/Task_Core/@Name"/>
             <!-- we generate a hyperlink to the Task element which has the name Task_Core/@Name -->
             <a href = "#{generate-id(key('task-task_context_id',@id))}">
             <xsl:value-of select="key('task-task_context_id',@id)/Task_Core/@Name" />
             </a>
              <!-- if this is not the last sub-task, we put a comma -->
			  <xsl:if test="position() != last()">, </xsl:if>
           </xsl:for-each>   <!-- end for-each Component-->
        </li>
      </xsl:for-each>		<!-- end for-each $expansions-->
        
      <!-- Display the image because there is a decomposition -->
      <!-- The picture must have the same name as the task -->
      
      <!-- create an image map (includes tasks AND boolean connectors) -->
      <p> 
           <!-- because XSL cannot create right names for HTML with
                <img src="{$taskName}.png" border="0" usemap="#{$taskName}"/>  
                we create an element which represents this <img> -->
            <xsl:element name="img">
               <xsl:attribute name="src"><xsl:value-of select="$taskName"/>.png</xsl:attribute>
               <xsl:attribute name="border">0</xsl:attribute>
               <xsl:attribute name="usemap" saxon:disable-output-escaping="yes" xmlns:saxon="http://icl.com/saxon">#<xsl:value-of select="$taskName"/></xsl:attribute>
            </xsl:element>
  
			<map name="{$taskName}">
			<xsl:for-each select="key('expansion-name',$taskName)/Component">
			   <area shape="rect" coords="{@PNGx1},{@PNGy1},{@PNGx2},{@PNGy2}" href="#{generate-id(key('task-task_context_id',@id))}" alt="{key('task-task_context_id',@id)/Task_Core/@Name}"/>
			</xsl:for-each>
			</map>  
      </p>				
       
      </ul>
      </xsl:when>
      <!-- if there aren't, output 'None' -->
      <xsl:otherwise>None</xsl:otherwise>
    </xsl:choose>
    
    <!-- Insert a link to the top. Useful for big task models -->
    <p><a href="#top"><xsl:text>Go to the Top of the page</xsl:text></a></p>
    

    <hr></hr>
  </xsl:for-each>			<!-- end for-each Task -->
</xsl:template>     

<!-- Templates to write the (HTML) comments of the task core -->
<xsl:template match="Comment_HTML">
  <xsl:apply-templates select="p"/>
</xsl:template>

		<!-- to manage the P tag -->
		<xsl:template match="p">
		  <p><xsl:apply-templates/></p>
		</xsl:template>
		
		<!-- to manage the B tag -->
		<xsl:template match="b">
		  <b><xsl:apply-templates/></b>
		</xsl:template>
		
		<!-- to manage the I tag -->
		<xsl:template match="i">
		  <i><xsl:apply-templates/></i>
		</xsl:template>
		
		<!-- to manage the U tag -->
		<xsl:template match="u">
		  <u><xsl:apply-templates/></u>
		</xsl:template>

  
</xsl:stylesheet>
