/*****************************
File:       RemoteTimePacket.CPP
Language:   C++ (header)
Project:    H3DNetworkingUtils
The contents of this file are subject to the Mozilla Public License
Version 1.1 (the "License"); you may not use this file except in
compliance with the License. You may obtain a copy of the License at
http://www.mozilla.org/MPL/

Software distributed under the License is distributed on an "AS IS"
basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
License for the specific language governing rights and limitations
under the License.

The Original Code is H3DNetworkingUtils v1.0.

The Initial Developer of the Original Code is CSIRO.
Portions created by the Initial Developer are Copyright (C) 1009 CSIRO. All Rights Reserved.
Contributor(s):
    Chris Gunn  <Chris.Gunn@csiro.au> <ChrisJGunn@gmail.com>
***************************/
#include "H3DNetworkingUtils/RemoteTimePacket.h"
#include "H3DNetworkingUtils/RemoteSync.h"
#include "H3DNetworkingUtils/RemoteConnection.h"
#include <iomanip>

#ifdef _WIN32
#include "RemoteSF.cpp"
#endif

using namespace H3D;
using namespace H3DNetworkingUtils;

H3DNodeDatabase RemoteTimePacket::database(   
  "RemoteTimePacket", 
  &(newInstance<RemoteTimePacket>),
  typeid(RemoteTimePacket),
  &RemoteSF< SFTimePacket>::database);

//toBeSent and received are really fields of the parent class but because of
// a bug in Win32 template instantiation they must be included here instead.
namespace RemoteTimePacketInternals {   
  FIELDDB_ELEMENT( RemoteTimePacket, toBeSent, INPUT_ONLY );
  FIELDDB_ELEMENT( RemoteTimePacket, received, OUTPUT_ONLY );
}

// ----------------------------------------------------------------------------
RemoteTimePacket::RemoteTimePacket() :
RemoteSF< SFTimePacket>(H3D::X3DTypes::SFTIME)  {

  type_name = "RemoteTimePacket";
  database.initFields( this );

  received->initialiseValue(TimePacket());
}

// ----------------------------------------------------------------------------
void RemoteTimePacket::writeValue( TimePacket const &val ) {
  writeBool( val.state );
  writeDouble( val.send_time );
  writeDouble( val.remote_time );
  writeDouble( val.receive_time );
  //  cout << "Time packet sent: "
  //       << val.state << " "
  //       << setprecision(16) << val.send_time << " "
  //       << setprecision(16) << val.remote_time << " "
  //       << setprecision(16) << val.receive_time << endl;
}

// ----------------------------------------------------------------------------
void RemoteTimePacket::readValue() {
  ////  readTimestamp();
  TimePacket val;
  val.state = readBool();
  if ( !conP->closeRequested() ) {
    val.send_time = readDouble();
  }
  if ( !conP->closeRequested() ) {
    val.remote_time = readDouble();
  }
  if ( !conP->closeRequested() ) {
    val.receive_time = readDouble();
  }

  //  cout << "Time packet received: "
  //    << val.state << " "
  //   << setprecision(16) << val.send_time << " "
  //   << setprecision(16) << val.remote_time << " "
  //   << setprecision(16) << val.receive_time << endl;

  H3DTime now = TimeStamp::now();
  if ( !val.state ) {
    // We need to send a reply time stamp.
    // However, this is the receiving thread.
    // We therefore lock the writing thread to avoid a clash with the
    // standard sending thread.
    conP->lockWriteThread();
    val.state = true;
    val.remote_time = now;
    val.receive_time = 0.0;
    conP->writeFieldHeader(fieldId->getValue(), timestamps->getValue());
    writeValue(val);
    conP->sendPacket();
    conP->unlockWriteThread();
  } else {
    val.receive_time = now;
    if ( !conP->closeRequested() ) {
      received->setValue( val, id );
    }
  }
}

