/*****************************
File:      VecFilter.h
Language:   C++ (header)
Project:    H3DNetworkingUtils
The contents of this file are subject to the Mozilla Public License
Version 1.1 (the "License"); you may not use this file except in
compliance with the License. You may obtain a copy of the License at
http://www.mozilla.org/MPL/

Software distributed under the License is distributed on an "AS IS"
basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
License for the specific language governing rights and limitations
under the License.

The Original Code is H3DNetworkingUtils v1.0.

The Initial Developer of the Original Code is CSIRO.
Portions created by the Initial Developer are Copyright (C) 1009 CSIRO. All Rights Reserved.
Contributor(s):
    Chris Gunn  <Chris.Gunn@csiro.au> <ChrisJGunn@gmail.com>
***************************/

/// \file VecFilter.h
/// \brief Header file for VecFilter, filters a Vec3f.

#ifndef VecFilter_H
#define VecFilter_H

#include "H3DNetworkingUtils/Config.h"
#include <deque>
#include <H3DUtil/H3DMath.h>
#include <H3DUtil/Vec3f.h>


namespace H3DNetworkingUtils {
   
/// \class VecFilter
/// The VecFilter class is a weighted averaging filter in the time domain.
/// It takes a moving Vec3f, (represented by a number of Vec3f / time pairs),
/// and filters it with a sin(t)/t function. The number of samples averaged
/// is the number of samples in the array passed to filter.

class H3D_NETWORKING_UTILS_DLL_SPEC VecFilter {
public:
  /// Constructor = the sample_sz is the number of past samples to average over.
  VecFilter(int sample_sz);

  /// Returns the filtered average of the values in raw_vals   \n
  /// The parameters to filter() are deque's to enable a sliding window of
  /// raw_values to be used, by popping a value off the start and adding one
  /// on to the end before each call.
  H3DUtil::ArithmeticTypes::Vec3f filter(H3DUtil::ArithmeticTypes::Vec3f const & raw_value, double time);

  /// Reset all historical values to start sampling a fresh, with a new sample size
  virtual void resetValue(int sample_sz);

  /// Returns the sample size
  int sampleSize() const { return sample_size;}

private:
  std::deque<H3DUtil::ArithmeticTypes::Vec3f> sample_vals;
  std::deque<double> sample_times;
  int sample_size; // This number of values to average over.
  int num_samples;
  bool filling_samples;
};

}

#endif
