/*****************************
File:      RemoteSFTime.h
Language:   C++ (header)
Project:    H3DNetworkingUtils
The contents of this file are subject to the Mozilla Public License
Version 1.1 (the "License"); you may not use this file except in
compliance with the License. You may obtain a copy of the License at
http://www.mozilla.org/MPL/

Software distributed under the License is distributed on an "AS IS"
basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
License for the specific language governing rights and limitations
under the License.

The Original Code is H3DNetworkingUtils v1.0.

The Initial Developer of the Original Code is CSIRO.
Portions created by the Initial Developer are Copyright (C) 1009 CSIRO. All Rights Reserved.
Contributor(s):
    Chris Gunn  <Chris.Gunn@csiro.au> <ChrisJGunn@gmail.com>
***************************/

/// \file RemoteSFTime.h
/// \brief Header file for RemoteSFTime, sends and receives a time value.

#ifndef RemoteSFTime_H
#define RemoteSFTime_H

// The RemoteSFTime class provides a  remote Time field.  It inherits most of its funtionality from its parent
// class.
// This class adds a VRML interface and the details of how to read and write
// the field data.
//
// Fields:
// <INPUT_ONLY  "toBeSent" "SFTime" ""  >
// <OUTPUT_ONLY "received"   "SFTime" "0" >
//
#include "H3DNetworkingUtils/RemoteSF.h"
#include <H3D/SFTime.h>

namespace H3DNetworkingUtils {

/// \class RemoteSFTime
/// The RemoteSFTime class provides a remote single time field.  It inherits most of its funtionality from its parent
/// class.
/// This class adds a X3D database and the details of how to read and write
/// the field data. \n
/// Use the 'toBeSent' and 'received' fields of the base class.
/// The structure of this class is different to that of other RemoteSF* classes in this library.
/// This is because and H3D::SFTime is implemented as holding a double, but can only be routed to and from an SFTime.
/// This class creates a send_t and received_t which respectively link to the normal, toBeSent and received fields.
/// This satisfies the type checking when you are routing in and out.
/// From X3D, vrml & python, the routing is the same, (ie to 'toBeSent' and from 'received') as the mapping to string field names 
/// hides the send-t and received-t. From C++ route to send_t and received_t.  \n
/// <b>Examples:</b>
///   - <a href="../../examples/RemoteSFTestServer.x3d">RemoteMFTestServer.x3d</a>
///   - <a href="../../examples/RemoteSFTestClient.x3d">RemoteMFTestClient.x3d</a>

class H3D_NETWORKING_UTILS_DLL_SPEC RemoteSFTime : public RemoteSF<H3D::SFDouble> {

public:

   struct SFTimeTransfer : public H3D::TypedField<H3D::SFTime, void, H3D::SField<H3DDouble> >  {
     virtual void update();
   };
  
   /// Constructor
   RemoteSFTime(H3D::Inst<H3D::SFTime> _sent_t = 0,
                 H3D::Inst<SFTimeTransfer> _received_t = 0);

  /// X3D interface
   static H3D::H3DNodeDatabase database;

  /// Use this from C++ only, not from X3D (see detailed description) \n
  /// access type: inputOnly \n
  /// basic type: SFTime \n
  /// default value: 0
  auto_ptr< H3D::SFTime> send_t; 

  /// Use this from C++ only, not from X3D (see detailed description) \n
  /// access type: outputOnly \n
  /// basic type: SFTime \n
  /// default value: 0
  auto_ptr< SFTimeTransfer> received_t; 

protected:
  /// Read the parts of a fields value into the received field.
  virtual void readValue();

  /// Write the correct parts of the value.
  virtual void writeValue(H3DUtil::H3DTime const & val) {writeDouble(val);}

};

}

#endif
