/*****************************
File:      Filter.h
Language:   C++ (header)
Project:    H3DNetworkingUtils
The contents of this file are subject to the Mozilla Public License
Version 1.1 (the "License"); you may not use this file except in
compliance with the License. You may obtain a copy of the License at
http://www.mozilla.org/MPL/

Software distributed under the License is distributed on an "AS IS"
basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
License for the specific language governing rights and limitations
under the License.

The Original Code is H3DNetworkingUtils v1.0.

The Initial Developer of the Original Code is CSIRO.
Portions created by the Initial Developer are Copyright (C) 1009 CSIRO. All Rights Reserved.
Contributor(s):
    Chris Gunn  <Chris.Gunn@csiro.au> <ChrisJGunn@gmail.com>
***************************/

/// \file Filter.h
/// \brief Header file for Filter, filters a Vec3f stream depending on past history.

#ifndef Filter_H
#define Filter_H

#include "H3DNetworkingUtils/Config.h"
#include <H3D/X3DChildNode.h>
#include <H3D/FieldTemplates.h>
#include <H3D/SFBool.h>
#include "H3DNetworkingUtils/FilteredSFVec3f.h"


// This next line is needed because H3DAPI.dll instantiates this template but this class does not see the header with 
// the instantiation in it. So without the line, this class would instantiate an identical one and there would
// be a multiply defined symbol in the link stage.
template struct __declspec(dllimport) H3D::AutoUpdate <H3D::SFBool>;

namespace H3DNetworkingUtils {

/// \class Filter
/// The Filter class is a node that can filter raw (Vec3f) values into a value depending on past
/// history. This differs from the FilteredSFVec3f class in that this is a Node, whereas the latter
/// is a specialised field. They both have the ability to perform the same function.

class H3D_NETWORKING_UTILS_DLL_SPEC Filter : public H3D::X3DChildNode {
public:

  class EnableSwitch;

  /// Constructor
  Filter(H3D::Inst<H3D::SFVec3f        > _rawValue = 0,
         H3D::Inst<FilteredSFVec3f     > _filteredValue = 0,
         H3D::Inst<H3D::SFInt32        > _samp_size = 0,
         H3D::Inst<EnableSwitch        > _enabled   = 0);

  static H3D::H3DNodeDatabase database;

  class EnableSwitch : public H3D::AutoUpdate< H3D::SFBool >  {
  public:
    virtual void update() {
      H3D::SFBool * on = static_cast<H3D::SFBool *>(event.ptr);
      if (on->getValue() == value) {
        return;
      }
      value = on->getValue();
      if (value) {
        filtP->rawValue->route(filtP->filteredValue);
        filtP->sampleSize->route(filtP->filteredValue);
      }
      else {
        filtP->rawValue->unroute(filtP->filteredValue);
        filtP->sampleSize->unroute(filtP->filteredValue);
      }
    }
    Filter * filtP;
  };

  /// The raw unfiltered values \n
  /// access type: inputOnly \n
  /// basic type: SFVec3f \n
  /// default value: 
  auto_ptr<H3D::SFVec3f    > rawValue;

  /// The filtered value \n
  /// access type: outputOnly \n
  /// basic type: SFVec3f \n
  /// default value: 
  auto_ptr<FilteredSFVec3f > filteredValue;

  /// The sampel size to filter over \n
  /// access type: outputOnly \n
  /// basic type: SFInt32 \n
  /// default value: 8
  auto_ptr<H3D::SFInt32    > sampleSize;

  /// Switches the filtering on \n
  /// access type: outputOnly \n
  /// basic type: MFVec3f \n
  /// default value: 
  auto_ptr<EnableSwitch    > enabled;
};

}

#endif

