/*****************************
File:      CollisionGeometry.h
Language:   C++ (header)
Project:    H3DNetworkingUtils
The contents of this file are subject to the Mozilla Public License
Version 1.1 (the "License"); you may not use this file except in
compliance with the License. You may obtain a copy of the License at
http://www.mozilla.org/MPL/

Software distributed under the License is distributed on an "AS IS"
basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
License for the specific language governing rights and limitations
under the License.

The Original Code is H3DNetworkingUtils v1.0.

The Initial Developer of the Original Code is CSIRO.
Portions created by the Initial Developer are Copyright (C) 1009 CSIRO. All Rights Reserved.
Contributor(s):
    Chris Gunn  <Chris.Gunn@csiro.au> <ChrisJGunn@gmail.com>
***************************/

/// \file CollisionGeometry.h
/// \brief Header file for CollisionGeometry, an abstract base class for inter-object collision shapes.

#ifndef CollisionGeometry_H
#define CollisionGeometry_H


#include "H3DNetworkingUtils/Config.h"
#include <H3D/Transform.h>
#include <H3D/TransformInfo.h>
#include <H3DUtil/AutoRef.h>

namespace H3DNetworkingUtils {

/// \class CollisionGeometry
/// The CollisionGeometry class is an abstract base class for a piece of simple geometry used to represent a complex geometry for the
/// purposes of inter object collision - not rendered.
/// The transform of this node provides the coordinate system of the object.
/// Within that the points of the object it surrounds may not be centered on this, in that case the 
/// local_offset is displaced from the origin.

class CollidableDynamic;

class H3D_NETWORKING_UTILS_DLL_SPEC CollisionGeometry : public H3D::Transform {
public:

   /// Constructor
   CollisionGeometry();

   /// Datbase
   static H3D::H3DNodeDatabase database;

   /// Initialize
   virtual void initialize();
   
   /// Set the Dynamic that contai9ns this CollisionGeometry
   virtual void setDyn( CollidableDynamic *ptr );  

   /// Get the Dynamic that contai9ns this CollisionGeometry
   CollidableDynamic * getDyn() {return collidable_dynamicP;}
    
   /// Return a matrfix to convert local to global coords
   H3D::Matrix4f getLocalToGlobal() {
      return child_local_info->accForwardMatrix->getValue();
   }

   /// Return a matrfix to convert global to local coords
   H3D::Matrix4f getGlobalToLocal() {
      return child_local_info->accInverseMatrix->getValue();
   }
  
   /// Get the local center position
   H3D::Vec3f getLocalPosition() {
      return translation->getValue();
   }
   
   /// Get the global center position
   H3D::Vec3f getGlobalPosition() {
      return getLocalToGlobal() * H3D::Vec3f(0,0,0);
   }
  
   /// Create a CollisionGeometry (can be overridden)
   virtual CollisionGeometry * clone() {return new CollisionGeometry;}

    /// Copy contents into newP.
   virtual void copy(CollisionGeometry * newP);

   /// Set the hierarchy layer of this geometry
   void setLayer(u_int l) {layer = l;}

   /// Set the offset of the bounding sphere within the coordin system of the object.
   /// This may be different if the points are not distributed around the origin.
   void setLocalOffset(H3D::Vec3f offs) {local_offset->translation->setValue(offs);}


protected:
   H3D::AutoRef<H3D::Transform> local_offset;
   H3D::AutoRef<H3D::TransformInfo> child_local_info;
   CollidableDynamic * collidable_dynamicP;  // Can't use a AutoRef here as it causews an unwanted ref()

   u_int layer;
   H3D::Vec3f trans;
   bool initialized;
};



}

#endif 
