/*********************************************************************
 *
 * CSIRO Automation
 * Queensland Centre for Advanced Technologies
 * PO Box 883, Kenmore, QLD 4069, Australia
 * www.cat.csiro.au/cmst
 *
 * Copyright (c) CSIRO Manufacturing Science & Technology
 *
 *********************************************************************/

static char rcsid[] = "$Header$";

/** \file
 * \brief XID image Utility 
 * \author Elliot Duff Elliot.Duff@csiro.au
 * 
 * This program display both PGM and PPM images on an X11 display.
 * It is also able to handle jpeg and gif images with the help of 
 * extern filters. 
 * 
 * For PGM image (ie. monochrome) there are an number of image
 * manipulation filters that can be applied (from PIP). Type \c h for help. 
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "pip.h"
#include "xid.h"

#define XID_DrawPIP(w, i)     XID_DrawImage(w, i->data, i->nc, i->nr)
#define XID_DrawPIPOver(w, i) XID_DrawImageOver(w, i->data, i->mask, i->nc, i->nr)
#define XID_DrawRGB(w, i)     XID_DrawImageRGB(w, i[0]->data, i[1]->data, i[2]->data, i[0]->nc, i[0]->nr)

void usage(char *s)
    {
    fprintf(stderr, "\nUsage : %s {-flag options} [filename]\n\n", s);
    fprintf(stderr, "\t[-i infile] \tInput pgm file (default stdin)\n");
    fprintf(stderr, "\t[-o outfile]\tOutput pgm file (default stdout)\n");
    fprintf(stderr, "\t[-c cmap] \tInput Colourmap file (default grey)\n");
    fprintf(stderr, "\t[-d name] \tDisplay name\n");
    fprintf(stderr, "\t[-t title] \tDisplay title\n");
    fprintf(stderr, "\t[-z zoom] \tZoom (integer)\n");
    fprintf(stderr, "\t[-W win]  \tDraw on host window of given name\n");
    fprintf(stderr, "\t[-P] \t\tPIP Debug mode\n");
    fprintf(stderr, "\t[-v] \t\tPIP Verbose output\n");
    fprintf(stderr, "\t[-x] \t\tXID Verbose output\n");
    fprintf(stderr, "\t[-m] \t\tMonochrome\n");
    fprintf(stderr, "\t[-D] \t\tDon't Destroy \n");
    fprintf(stderr, "\t[-s] \t\tSilent mode\n");
    fprintf(stderr, "\t[-V] \t\tPrint Version Number\n");
    fprintf(stderr, "\t[-U] \t\tPrint Usage\n\n");

    fprintf(stderr, "Type h for a list of interactive commands\n\n");
    fprintf(stderr, "By default, display a RGB image in colour\n");
    fprintf(stderr, "Unless specified by the monochrome mode\n\n");

    strtok(strdup(rcsid), " "); /* Needed for some reason */
    fprintf(stderr, "Source  : %s\n", strtok(NULL," "));
    fprintf(stderr, "Version : %s ",  strtok(NULL, " "));
    fprintf(stderr, "%s ",  strtok(NULL, " "));
    fprintf(stderr, "%s ",  strtok(NULL, " "));
    fprintf(stderr, "%s\n", strtok(NULL, " "));
    fprintf(stderr, "Build   : %s %s\n\n", __TIME__,__DATE__);
    exit(1);
    }

void help() 
    {
    printf("XID commands\n");
    printf("? - info \n");
    printf("q - quit \n");
    printf("a - annotate \n");
    printf("s - save as gif\n");
    printf("j - save as jpeg\n");
    printf("i - invert \n");
    printf("m - median \n");
    printf("d - deinterlace \n");
    printf("c - crop (based on ROI)\n");
    printf("u - unary operation\n");
    printf("= - spin clockwise \n");
    printf("- - spin anti-clockwise \n");
    printf("f - flip horizontal \n");
    printf("v - flip vertical\n");
    printf("< - zoom out by 2\n");
    printf("> - zoom in by 2\n");
    printf("Mouse Press   - define ROI lowerbound\n");
    printf("Mouse Release - define ROI upperbound\n");
    }

/**
* Display PGM & PPM file - and wait for commands 
*/

int main(int argc, char *argv[])
    {
    int  i;
    char *in = NULL;
    char *out = NULL;
    char *cmap = "grey";

    Pimage *map;
    Pimage *ip;			/* Generic working image pointer */
    Pimage *rgb[3];		/* Pointer to original RGB image */
    Phist  hist;

    XIDwin *w;			/* The one and only Window */
    XIDwin *wh;			/* Bivariant Histogram Window */

    char op;
    int  value;
    char event[40];		/* Buffer for event string */
    char comm[40];		/* Mouse Command String */
    char buff[40];		/* General Buffer */

    int  b, x, y;		/* Button index and position */
    int  zoom = 0;		/* Flag to zoom in */
    char *win = NULL;		/* name of Window */
    char *name = NULL;		/* name of Display null = current */
    char *title = NULL;		/* Title of Display */
    int monochrome = 0;		/* Display RGB by default */
    int redraw;

    pip_silent  = 1;
    xid_verbose = 0;
    pip_verbose = 0;

    for(i=1; i<argc; i++)
      {
      if( argv[i][0] != '-') in = argv[i++]; 
      else switch( argv[i][1] )
        {
	case 'V': PIP_version(); exit(0);
        case 'U': usage(argv[0]); exit(0);
        case 'd': name = argv[++i]; break;
        case 'P': pip_debug   = 1; break;
        case 'v': pip_verbose = 1; 
	case 's': pip_silent  = 0; break;
        case 'x': xid_verbose = 1; break;
        case 'm': monochrome = 1; break;
        case 'i': in   = argv[++i]; break;
        case 't': title= argv[++i]; break;
        case 'o': out  = argv[++i]; break;
        case 'c': cmap = argv[++i]; break;
        case 'W': win  = argv[++i]; break;
        case 'z': zoom = atoi(argv[++i]); break;
	default : usage(argv[0]); 
	}
      }

    /* Make Connection to X server */

    if( XID_Open(name) ) exit(-1); 

    /* 
     * Load Images
     */

    map    = PIP_new();
    rgb[0] = PIP_new();
    rgb[1] = PIP_new();
    rgb[2] = PIP_new();

    if( PIP_loadRGB(rgb, in) ) exit(-1); 			
    if( rgb[1]->data == NULL || rgb[2]->data == NULL ) monochrome = 1;
    if( title != NULL ) strcpy(rgb[0]->name, title);

    /* 
     * Display Colour Image
     */

    if( !monochrome ) {

      w = XID_WindowCreate(rgb[0]->name, rgb[0]->nc, rgb[0]->nr, 0, 0, 0, 0);
      XID_DrawRGB(w, rgb);  

      while( XID_GetEvent(event, 40) ) {
	switch( event[0] ) {

	case 'i': 
	  PIP_info(rgb[0]); 
	  PIP_info(rgb[1]); 
	  PIP_info(rgb[2]); 
	  break;

        case 'q':
	  XID_Destroy(w);
	  XID_Close(); 
	  exit(0);

        case 'h': 
	  PIP_RGBtoHLS(rgb);  
	  XID_LoadCmap(w, "hue", 0); 
	  XID_DrawPIP(w, rgb[0]);  
	  break;
	  }
        }
      }

    /* 
     * Otherwise it is a PGM image 
     */

    ip = rgb[0];
    if( zoom ) PIP_zoom(ip, zoom); 

    w = XID_WindowCreate(ip->name, ip->nc, ip->nr, 0, 0, 0, 0);
    XID_LoadCmap(w, cmap, 0);
    XID_DrawPIP(w, ip);

    while ( (XID_GetEvent(event, 40) ) ) {

      redraw = 0;		/* By Default we dont need to redraw screen */
      
      switch( event[0] ) {

	case 'q': XID_Close(); exit(0);
	case '?': XID_Info(w); break;
        case 'I': PIP_info(ip); break;

  	case 'B': /* Mouse Event */
	  sscanf(event, "%s%d%d%d", comm, &b, &x, &y);
	  if( !strcmp(comm, "BP") ) { ip->c1 = x; ip->r1 = y; }
	  else { if( !strcmp(comm, "BR") ) { ip->c2 = x; ip->r2 = y; }
	  else { fprintf(stderr, "Unknown Mouse Command %s", comm); }}
	  XID_DrawMark(w,x,y,5);
	  break;

        case 'h': 
	  PIP_segment_simple(ip, &hist ); 
	  for(i=0; i<256; i++) printf("%d %d\n", i, hist.data[i]);
	  break;

	case 'H': 
	  PIP_roi(rgb[1], ip->r1, ip->c1, ip->r2, ip->c2); PIP_roi_mask(rgb[1]);
	  PIP_roi(rgb[2], ip->r1, ip->c1, ip->r2, ip->c2); PIP_roi_mask(rgb[2]);
	  PIP_bivariant(map, rgb[1], rgb[2]);
	  PIP_unary(map, '>', 0);
	  wh = XID_WindowCreate("Bivariant Histogram", 256, 256, 0, 0, 0, 0);
	  XID_DrawPIP(wh, map);
	  break;

	case 'a': 
	  printf("Enter String :"); scanf("%40s", buff); 
	  XID_DrawText(w, buff, ip->c1, ip->r1); 
	  break;

        case 'l': 
	  XID_DrawLine(w, ip->c1, ip->r1, ip->c2, ip->r2);
	  break; 

        case 'b': 
	  XID_DrawBox(w, ip->c1, ip->r1, ip->c2-ip->c1, ip->r2-ip->r1, 0);
	  break; 

	case 's': /* Save to File */
	  if( strcmp(cmap, "grey") == 0 ) 
	    PIP_save(ip, "xid.gif"); 
	  else {
	    fprintf(stderr, "Colourize image and Save xid.gif\n");
	    PIP_loadcmap(cmap);
	    PIP_colourize(rgb, ip);
	    PIP_saveRGB(rgb, "xid.gif");
	    }
	  break;

	case 'u': 
	  printf("Enter Unary Operator and Value:");
	  scanf("%c%d", &op, &value);
	  PIP_unary(ip, op, value);
	  redraw = 1; 
	  break;

	case 'C': 
	  printf("Enter Colour Name :" ); scanf("%s", buff);
	  XID_SetColour(w, XID_GetColour(w, buff)); 
	  break;

	case 'n': redraw = 1; XID_Rename(w, "Test New Name"); break;
	case 'i': redraw = 1; PIP_unary(ip, 'i', 0); break;
	case 'm': redraw = 1; PIP_rank(ip,"med",9,9); break;
	case 'd': redraw = 1; PIP_deinterlace(ip); break;
	case 'f': redraw = 1; PIP_flip(ip, 'h'); break;
	case 'v': redraw = 1; PIP_flip(ip, 'v'); break;

	case 'c': redraw = 2; PIP_crop(ip); break;
	case '=': redraw = 2; PIP_spin(ip, 'c'); break;
	case '-': redraw = 2; PIP_spin(ip, 'c'); break;
	case '>': redraw = 2; PIP_zoom(ip,  2);  break;
	case '<': redraw = 2; PIP_zoom(ip, -2);  break;

	case 'j':
	  PIP_loadcmap(cmap);
	  PIP_colourize(rgb, ip);
	  PIP_saveRGB(rgb, "xid.jpg");
	  break;

	case 'S': /* Ignore Shift */ break;
        default : help();
	}

      if( redraw > 2 ) XID_Destroy(w); 
      if( redraw > 1 ) w = XID_WindowCreate(ip->name, ip->nc, ip->nr, 0, 0, 0, 0); 
      if( redraw > 0 ) XID_DrawPIP(w, ip); 
      }

    printf("Trace: you should not be here \n");
    exit(0);
    }
