/** \file
 * @brief X Image Drawing routines
 */

#define _XID_EXTERN
#include "xid.h"
#include <stdio.h>
#include <errno.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>

#ifndef  DOXYGEN_SHOULD_SKIP_THIS
#define  FUNCPROTO 1
#define  XIDerr(A) fprintf(stderr, "XID Error : %s\n", A); 

static int TranslateKeyCode(XEvent *, char *, int);

/* Global Display Variables (that should not need to change) */

static Display  *display;
static Screen   *displayScreen;
static Visual 	*displayVisual;
static int	displayNumber;			
static int      displayDepth;	
static int 	displayWidth;
static int 	displayHeight; 

static char *visual_classes[] =
    { "StaticGray", "GrayScale", "StaticColor",
      "PseudoColor", "TrueColor", "DirectColor", };

/**
 * These global variables are needed for Image Colourmap support 
 */ 

#define COLOR_COMPOSE(r,g,b) (redLook[r] | grnLook[g] | bluLook[b])
static int redShift, grnShift, bluShift;  /* Amount to shift color */
static int redPrec,  grnPrec,  bluPrec;   /* Loss of precision from 8bits */
static unsigned long redLook[256], grnLook[256], bluLook[256];

/*
 * Some local functions for generating lookup 
 */

static void fill_lookup_array (unsigned long *array, int shift, int prec)
    { int i; for (i = 0; i < 256; i++) array [i] = ((i >> prec) << shift); }

static void get_color_mask (unsigned long mask, int *shift, int *prec)
    { *shift = 0; *prec = 8;
    while (!(mask & 0x1)) { (*shift)++; mask >>= 1; }
    while (mask & 0x1) { (*prec)--; mask >>= 1; } }

static int endian() {
    unsigned int i = 0x01020304; unsigned char *p = (unsigned char *) &i;
    if( p[0] == 0x01 ) return MSBFirst; else return LSBFirst; }

/* 
 * Some local function for debugging 
 */

typedef struct { long code; char *name; } binding;
static char _lookup_buffer[100];

char *LookupL(long code, binding *table) { 
    char *name; name = _lookup_buffer;
    sprintf(_lookup_buffer, "unknown (code = %ld. = 0x%lx)", code, code);
    while (table->name) { 
      if (table->code == code) { name = table->name; break; } table++;
      }
    return(name);
    }

char *Lookup(int code, binding *table) { return LookupL((long)code, table); }

static binding event_names[] = { 
      { KeyPress, "KeyPress"}, 
      { KeyRelease, "KeyRelease"},  
      { ButtonPress, "ButtonPress"},  
      { ButtonRelease, "ButtonRelease"},  
      { MotionNotify, "MotionNotify"}, 
      { EnterNotify, "EnterNotify" } ,
      { 0, 0 } };

#endif

/**
 * Function to Open Display 
 * @param name display name
 * @return -1 on Failure 
 * @warning This function uses the DISPLAY environment variable if NULL name is  * specified
 */

int 
XID_Open(char *name)
    {
    if( name == NULL ) {
      if((name = getenv("DISPLAY")) == NULL)
        { fprintf(stderr, "Cannot get DISPLAY\n"); return -1; }
      }

    if((display = XOpenDisplay(name)) == NULL) 
      { fprintf(stderr, "Cannot Open %s\n", name); return -1; }

    displayNumber  = DefaultScreen(display);
    displayScreen  = ScreenOfDisplay(display,displayNumber);
    displayDepth   = DefaultDepth(display,displayNumber);
    displayVisual  = DefaultVisual(display,displayNumber);
    displayWidth   = DisplayWidth(display,displayNumber);
    displayHeight  = DisplayHeight(display,displayNumber);

    if (displayVisual->class != TrueColor) { XIDerr("Visual is not TRUE"); return 0;}
    get_color_mask (displayVisual->red_mask,   &redShift, &redPrec); 
    get_color_mask (displayVisual->green_mask, &grnShift, &grnPrec);
    get_color_mask (displayVisual->blue_mask,  &bluShift, &bluPrec);

    if( xid_verbose ) {
      printf("Width %d Height %d Depth %d\n", displayWidth, displayHeight, displayDepth);
      printf("Visual %s ",      visual_classes[displayVisual->class]);
      printf("BitsPerRGB %d\n", displayVisual->bits_per_rgb); 
      }

    return 0;
    }

/**
 * Function to Close Display 
 */

void XID_Close() { XCloseDisplay(display); }

/**
 * Function to destroy the current Window 
 * @param win   Pointer to Window
 */

void XID_Destroy(XIDwin *win) { 
  XDestroyWindow(display, win->id); 
  free(win->name);
  free(win);
  }

/**
 * Function to raise the current Window 
 * @param win   Pointer to Window
 * @warning Not tested
 */

void XID_Raise(XIDwin *win) { XRaiseWindow(display, win->id); }

/**
 * Function to lower the current Window 
 * @param win   Pointer to Window
 * @warning Not tested
 */

void XID_Lower(XIDwin *win) { XLowerWindow(display, win->id); }

/**
 * Function to change windows name
 * @param win   Pointer to Window
 * @param name 	Name of window (displayed at top)
 */

void XID_Rename(XIDwin *win, char *name) { 
    XChangeProperty(display, win->id, XA_WM_NAME,XA_STRING, 8,
        PropModeReplace, (unsigned char *) name, strlen(name));
    }

/**
 * This function creates a simple window. 
 * The backing store is used to assist with redrawing the window.
 * It should not be used for animation.
 * @param name 	Name of window (displayed at top)
 * @param w    	Width of window in pixels 
 * @param h    	Height of window in pixels 
 * @param x    	Horizontal Position of window in pixels 
 * @param y    	Vertical Position of window in pixels 
 * @param b    	Border width of window in pixels 
 * @param bs   	Whether Backing store is used 
 * @return 	Pointer to XID window structure 
 * @warning 	The Position of the window is not respected by all window managers
 * @todo 	Fix up window position so that it works for most window managers
 */

XIDwin * 
XID_WindowCreate( char *name, int w, int h, int x, int y, int b, int bs)		
    {
    int       window;
    XEvent    event;
    XSetWindowAttributes sattr;

    if( w >= displayWidth  ) { XIDerr("Window too wide"); return NULL; }
    if( h >= displayHeight ) { XIDerr("Window too high"); return NULL; }
    if( xid_verbose ) printf("Window (%s) %d %d\n", name, w, h);

    sattr.border_pixel     = WhitePixelOfScreen(displayScreen);
    sattr.background_pixel = BlackPixelOfScreen(displayScreen);
    sattr.event_mask       = ExposureMask|ButtonPressMask|ButtonReleaseMask|
		             StructureNotifyMask|PointerMotionMask|KeyPressMask;

    if( bs ) {
        sattr.backing_store    = DoesBackingStore(displayScreen);
        window = XCreateWindow(display, RootWindow(display, displayNumber),
	  x, y, w, h, b, CopyFromParent, InputOutput, CopyFromParent, 	
          CWBackingStore|CWEventMask|CWBorderPixel|CWBackPixel, &sattr);
      }
    else {
        window = XCreateWindow(display, RootWindow(display, displayNumber),
	  x, y, w, h, b, CopyFromParent, InputOutput, CopyFromParent, 	
          CWEventMask|CWBorderPixel|CWBackPixel, &sattr);
      }

    XChangeProperty(display, window, XA_WM_NAME,XA_STRING, 8,
        PropModeReplace, (unsigned char *) name, strlen(name));

    XMapWindow(display, window);
    do { XNextEvent(display, &event); }
    while( event.type != MapNotify );

    return XID_WindowWithId(window, 0);
    }

/**
 * This function creates a Window Handler
 * It is normally used as an internal function
 * @param window Window ID (used by X)
 * @param hostwindnow Boolean whether the window has a host window or not
 * @return Pointer to Window
 * @also XID_WindowCreate()
 */

XIDwin *
XID_WindowWithId(Window window, int hostWindow)
    {
    int i;
    XIDwin    *win;		
    XGCValues xgcv;
    XSetWindowAttributes   sattr;
    XWindowAttributes 	   wattr;
    char *name;

    if( window == 0 ) { XIDerr("Zero Window"); return NULL; }

    if( (win = ((XIDwin *) malloc( sizeof(XIDwin) ))) == NULL ) 
      { XIDerr("Cannot allocate memory window attributes"); return NULL; }

    if( !XFetchName(display, window, &name) ) 
      { XIDerr("Cannot get window name"); return NULL; }

    if( !XGetWindowAttributes(display, window, &wattr) )
      { XIDerr("Cannot get window attributes"); return NULL; }

    win->id     = window;
    win->name   = name;
    win->w 	= wattr.width;
    win->h 	= wattr.height;
    win->sx 	= 0;
    win->sy 	= 0;
    win->dx 	= 0;
    win->dy 	= 0;

    if( xid_verbose ) printf("Window %ld Name %s Width %d Height %d\n", 
      (unsigned long)win->id, win->name, win->w, win->h);

    /* 
     * I don't know why - but this allows one to look at events 
     * This only works for windows which have enabled the appropriate Events 
     * (ie. GTK drawing surface, but not TCL frame)
     * if( wattr.all_event_masks & ButtonPressMask ) sattr.event_mask &= ~ButtonPressMask; 
     */

    if( hostWindow ) {
      sattr.event_mask = ExposureMask|ButtonPressMask|ButtonReleaseMask|
	                 StructureNotifyMask|PointerMotionMask|KeyPressMask;
      sattr.event_mask &= ~SubstructureRedirectMask;
      XSelectInput(display, win->id, sattr.event_mask);
      }

    /* 
     * Create the Graphics Context 
     */

    xgcv.foreground = 255;
    xgcv.font = XLoadFont(display,(char *)"9x15");
    win->gc = XCreateGC(display, win->id, GCForeground|GCFont, &xgcv);

    /*
     * Create the Image
     */

    win->xi = XGetImage(display, win->id, 0, 0, win->w, win->h, AllPlanes, ZPixmap);

    if( xid_verbose ) {
      printf("Image  Depth %d BitsPerPixel %d ByteOrder %d Offset %d\n", 
      win->xi->depth, win->xi->bits_per_pixel, win->xi->byte_order, win->xi->xoffset);
      printf("Bitmap Unit %d BitOrder %d Pad %d BytesPerLine %d\n", 
      win->xi->bitmap_unit,win->xi->bitmap_bit_order,win->xi->bitmap_pad,win->xi->bytes_per_line);
      }

    if( displayDepth == 24 ) win->l = win->xi->bytes_per_line/4;
    else win->l = win->xi->bytes_per_line/2;

    if( xid_verbose ) {
        printf("RedMask %08x Shift %d Prec %d \n", (unsigned int) displayVisual->red_mask,   redShift, redPrec); 
        printf("GrnMask %08x Shift %d Prec %d \n", (unsigned int) displayVisual->green_mask, grnShift, grnPrec); 
        printf("RedMask %08x Shift %d Prec %d \n", (unsigned int) displayVisual->blue_mask,  bluShift, bluPrec); 
	}

    /* Invert Shift of Opposite ARCH */

    if( endian() != win->xi->byte_order ) {
	if( xid_verbose ) printf("Invert Shift\n");
        redShift = 24 - redShift;
        grnShift = 24 - grnShift;
        bluShift = 24 - bluShift;
      }

    fill_lookup_array (redLook, redShift, redPrec);
    fill_lookup_array (grnLook, grnShift, grnPrec);
    fill_lookup_array (bluLook, bluShift, bluPrec);

    /* Generate Grey Scale Colourmap & Red Overlay */

    for(i=0; i<256; i++) { win->redCmap[i] = i; win->grnCmap[i] = i; win->bluCmap[i] = i; }
    for(i=0; i<256; i++) { win->redOver[i] = 255; win->grnOver[i] = 0; win->bluOver[i] = 0; }

    return win;
    }

/* 
 * This function is just too hard at the moment 

void
SetDrawMode(int mode)
  {
  if (mode == SANE_XOR)
    {
    cur_di->mask = cur_di->foreground ^ cur_di->background;
    XSetForeground(display, win->gc, 0xffffffff);
    XSetBackground(display, win->gc, attr.background_pixel)
    XSetFunction(display, win->gc, GXxor);
    XSetPlaneMask(display, win->gc, cur_di->mask);
    }
 else
    {
    XSetForeground(display, win->gc, cur_di->foreground);
    XSetBackground(display, win->gc, cur_di->background);
    XSetFunction(display, win->gc, mode);
    XSetPlaneMask(display, win->gc, 0xffffffff);
    cur_di->mask = 0xffffffff;
    } 
  }
*/

/**
 * Wait for next event. The nature of the event is recorded in the 
 * string arguement. If this arguement is NULL, no event is recorded.
 * This can be used if you are not particularly interested in the
 * nature of the event (i.e. when you want to block a program)
 * - A Mouse Press event is BP [b] [x] [y]
 * - A Mouse Release event is BR [b] [x] [y]
 * - Otherwise it is the raw keystroke
 *
 * @param win Pointer to XID structure 
 * @param s   String to hold result 
 * @param n   Length of string 
 * @return The window that reported the event
 * @todo  Should improve keystroke translation (ie Uppercase)
 * @note Events are a union of differnt structures, the elements can 
 * either be cast with a pointer, or directly referenced through
 * such mechanisms as event.xkey.window, or event.xbutton.window
 */

int XID_GetEvent(char *s, int n)	
  {
  static int x, y;
  static int buttonDown = 0;
  XEvent event;
  XButtonEvent *be;
  XMotionEvent *me;

  XFlush(display); 

  while(1)
    {
    XNextEvent(display, &event);
    if( xid_verbose ) printf("Event %d %s\n", event.type, Lookup(event.type, event_names));

    switch( event.type )
      {
      case KeyPress:
	if( s == NULL ) return 0;
	/* XLookupKeysym((XKeyEvent *) &event,0); */
	TranslateKeyCode(&event, s, n); 
	return event.xkey.window;

      case ButtonPress:
	if( s == NULL ) break;
  	be = (XButtonEvent *) &event; buttonDown = 1;
	sprintf(s, "BP %d %d %d", be->button, be->x, be->y); 
	x = be->x; y = be->y;
	return event.xbutton.window;

      case ButtonRelease:
	if( s == NULL ) return 0;
  	be = (XButtonEvent *) &event; buttonDown = 0;
	sprintf(s, "BR %d %d %d", be->button, be->x, be->y); 
	return event.xbutton.window;

      case MotionNotify:
	if( s == NULL ) break;
	me = (XMotionEvent *) &event; 
	sprintf(s, "ME %d %d", me->x, me->y); 
	return event.xmotion.window;	

	/* don't know how to handle this yet
	i = me->x + me->y*nc;
	if( b2 == NULL ) sprintf(s, "(%3d %3d) %3d", me->x,me->y,b1[i]);
	else if( b3 == NULL ) sprintf(s, "(%3d %3d) %3d %3d ", me->x,me->y,b1[i],b2[i]);
	else sprintf(s,"(%3d %3d) %3d:%3d:%3d",me->x,me->y,b1[i],b2[i],b3[i]);
	if( buttonDown ) XID_DrawBox(win, x, y, me->x-x, me->y-y, 0); 
	XID_DrawText(win, s, 10, 10); break; 
	*/
        }
    }
  }

/**
 * This function returns the Next Event 
 * @return Index of window the event occured in
 * @todo should have translate in here
 * @param s String to hold result 
 * @param n Length of string 
 */

int XID_Event(
	char *s, 	
	int n) 		
  {
  XEvent event;
  XButtonEvent *be;
  XMotionEvent *me;

  XFlush(display); 
  XNextEvent(display, &event);

  if( xid_verbose ) printf("Event %d %s\n", event.type, 
      Lookup(event.type, event_names));

  if( event.type == KeyPress ) {
	TranslateKeyCode(&event, s, n); 
	return event.xkey.window;
	}

  if( event.type == ButtonRelease ) {
  	be = (XButtonEvent *) &event; 
        sprintf(s, "B%d %d %d", be->button, be->x, be->y); 
	return event.xbutton.window;
	}

  if( event.type == ButtonPress ) {
  	be = (XButtonEvent *) &event; 
        sprintf(s, "b%d %d %d", be->button, be->x, be->y); 
	return event.xbutton.window;
	}
  
  if( event.type == MotionNotify ) {
  	me = (XMotionEvent *) &event; 
        sprintf(s, "ME %d %d", me->x, me->y); 
	return event.xmotion.window;
	}

  return 0;
  }

/**
 * This function checks the event queue for pending events.
 * If there are any they are converted into a string and passed back
 * @param s String to hole result 
 * @param n Length of string 
 * @return Index of window the event occured in
 * @todo should have translate in here
 */

int XID_IfEvent(char *s, int n)
  {
  XEvent event;
  XButtonEvent *be;
  XMotionEvent *me;

  while( XPending(display) )
    {
    XNextEvent(display, &event);
    if( xid_verbose ) printf("Event %d %s\n", event.type, 
      Lookup(event.type, event_names));

    if( event.type == KeyPress ) {
	TranslateKeyCode(&event, s, n); 
	return event.xkey.window;
	}

    if( event.type == ButtonRelease ) {
  	be = (XButtonEvent *) &event; 
        sprintf(s, "B%d %d %d", be->button, be->x, be->y); 
	return event.xbutton.window;
	}

	if( event.type == ButtonPress ) {
  	be = (XButtonEvent *) &event; 
        sprintf(s, "b%d %d %d", be->button, be->x, be->y); 
	return event.xbutton.window;
	}

	if( event.type == MotionNotify ) {
  	me = (XMotionEvent *) &event; 
        sprintf(s, "ME %d %d", me->x, me->y); 
	return event.xmotion.window;
	}

    }

  return 0;
  }

/**
 * Set Foreground colour 
 * @param xid Pointer to Window
 * @param colour colour index - 
 */

void XID_SetColour(XIDwin *win, int color) { XSetForeground(display, win->gc, color); }

/**
 * Set Background colour 
 * @param win Pointer to Window
 * @param colour colour index - 
 */

void XID_SetBGColour(XIDwin *win, int color) { XSetBackground(display, win->gc, color); }

/**
 * Clear Drawing Area 
 * @param win Pointer to Window
 */

void XID_DrawClear(XIDwin *win) { XClearWindow(display, win->id); }

/**
 * Drawing pixel
 * @param win Pointer to Window
 */

void XID_DrawPixel(XIDwin *win, int x, int y) { XDrawPoint(display,win->id,win->gc,x,y); }

/**
 * Draw Mark 
 * @param win Pointer to Window
 * @param w width of mark in pixels 
 */

void XID_DrawMark(XIDwin *win, int x, int y, int w)
  { 
  int i, j;
  for(i=-w; i<w; i++) for(j=-w; j<w; j++) 
    XDrawPoint(display,win->id,win->gc,x+i,y+j); 
  }

/** 
 * Draw Line 
 * @param win Pointer to Window
 */

void XID_DrawLine(XIDwin *win, int x1, int y1, int x2, int y2)
  { XDrawLine(display,win->id,win->gc,x1,y1,x2,y2); }

/** 
 * Draw Text 
 * @param win Pointer to Window
 */

void XID_DrawText(XIDwin *win, char *string, int x, int y)
  { XDrawImageString(display,win->id,win->gc,x,y,string,strlen(string)); }

/** 
 * Draw Empty Polygon  
 * @param win Pointer to Window
 */

void XID_DrawPolyline(XIDwin *win, XPoint *points, int n)
  { XDrawLines(display,win->id,win->gc,points,n,CoordModeOrigin); }

/** 
 *  Draw Filled Polygon 
 * @param win Pointer to Window
 */

void XID_DrawPolygon(XIDwin *win, XPoint *points, int n)
  { XFillPolygon(display,win->id,win->gc,points,n,Complex,CoordModeOrigin); }

/** 
 * Draw Box 
 * @param win Pointer to Window 
 */

void XID_DrawBox(XIDwin *win, int x,int y,int w,int h,int fill) {
  if(w<0) { w *= -1; x -= w; } if(h<0) { h *= -1; y -= h; }
  if( fill ) XFillRectangle(display, win->id, win->gc, x, y, w, h); 
  else XDrawRectangle(display, win->id, win->gc, x, y, w, h); }

/** 
 * Draw Arc  
 * @param win Pointer to Window 
 */

void XID_DrawArc(XIDwin *win, int x, int y, int w, int h, int a, int b, int fill) {
  a *= 64; b *= 64; if (w<0) { w *= -1; x -= w; } if(h<0) { h *= -1; y -= h; }
  if( fill ) XFillArc (display, win->id, win->gc, x, y, w, h, a, b);
  else XDrawArc (display, win->id, win->gc, x, y, w, h, a, b); }

/**
 * This function draws a normal 8bit image 
 * @param win  Pointer to Window
 * @param ip   Pointer to Image 
 * @param nc   Number of Columns 
 * @param nr   Number of Rows 
 * @note This function is affected by the colourmap
 * @todo Have a colourmap for the mask
 */

void XID_DrawImage( XIDwin *win, unsigned char *ip, int nc, int nr)	
  {
  int i, j, l;

  if( ip == NULL ) { XIDerr("There is no Image"); return; }
  if( win == NULL ) { XIDerr("There is no Window"); return; }
  if( win->xi  == NULL ) { XIDerr("There is no Ximage"); return; }

  if( displayDepth == 24 ) {
    	unsigned long *op; op = (unsigned long *) ( win->xi->data );
    for(i=0; i<nr; i++) {
      for(j=0; j<nc; j++) { l = (int) ip[i*nc+j]; 
	op[j] = COLOR_COMPOSE (win->redCmap[l], win->grnCmap[l], win->bluCmap[l]); }
      op += win->l;
      }
    }

  else {
    unsigned short *op; op = (unsigned short *) ( win->xi->data );
    for(i=0; i<nr; i++) { 
      for(j=0; j<nc; j++) { l = (int) ip[i*nc+j]; 
	op[j] = COLOR_COMPOSE (win->redCmap[l], win->grnCmap[l], win->bluCmap[l]); }
      op += win->l;
      }
    }

  XPutImage(display, win->id, win->gc, win->xi, 
     win->sx, win->sy, win->dx, win->dy, win->w, win->h);
  }

/**
 * This function draws a normal 8bit image with a mask overlay
 * @param win   Pointer to Window
 * @param data  Pointer to data image (Original)
 * @param mask  Pointer to mask image (Overlay)
 * @param nc    Number of Columns 
 * @param nr    Number of Rows 
 * @note 	This function is affected by the data and overlay colourmap
 * @warning 	The size of the images must match
 */

void XID_DrawImageOver( XIDwin *win, unsigned char *data, unsigned char *mask, int nc, int nr) 
  {
  int i, j, l, k;

  if( data == NULL ) { XIDerr("There is no Data"); return; }
  if( mask == NULL ) { XIDerr("There is no Mask"); return; }
  if( win  == NULL ) { XIDerr("There is no Window"); return; }
  if( win->xi == NULL ) { XIDerr("There is no Ximage"); return; }

  if( displayDepth == 24 ) {
    unsigned long *op; 
    op = (unsigned long *) ( win->xi->data );
    for(i=0; i<nr; i++) {
      for(j=0; j<nc; j++) { k = i*nc+j;
	if( mask[k] ) {  l = (int) mask[k]; op[j] = COLOR_COMPOSE (win->redOver[l], win->grnOver[l], win->bluOver[l]); }
	else { l = (int) data[k]; op[j] = COLOR_COMPOSE (win->redCmap[l], win->grnCmap[l], win->bluCmap[l]); }
	}
      op += win->l;
      }
    }

  else {
    unsigned short *op; 
    op = (unsigned short *) ( win->xi->data );
    for(i=0; i<nr; i++) { 
      for(j=0; j<nc; j++) { k = i*nc+j;
	if( mask[k] ) {  l = (int) mask[k]; op[j] = COLOR_COMPOSE (win->redOver[l], win->grnOver[l], win->bluOver[l]); }
	else { l = (int) data[k]; op[j] = COLOR_COMPOSE (win->redCmap[l], win->grnCmap[l], win->bluCmap[l]); }
	}
      op += win->l;
      }
    }

  XPutImage(display, win->id, win->gc, win->xi, 
     win->sx, win->sy, win->dx, win->dy, win->w, win->h);
  }

/**
 * This function display a 3 Plane RGB image
 * @param win Pointer to Window
 * @param r   Pointer to red image 
 * @param g   Pointer to green image 
 * @param b   Pointer to blue image 
 * @param nc  Number of Columns 
 * @param nr  Number of Rows 
 * @note      This function is unaffected by the colourmap
 */

void XID_DrawImageRGB( XIDwin *win, unsigned char *r, unsigned char *g, unsigned char *b, int nc, int nr)
  { 
  int i; 
  
  if( win  == NULL ) { XIDerr("There is no Window"); return; }
  if( win->xi  == NULL ) { XIDerr("There is no Ximage"); return; }
  if( r == NULL ) { XIDerr("There is no Red Image"); return; }
  if( g == NULL ) { XIDerr("There is no Green Image"); return; }
  if( b == NULL ) { XIDerr("There is no Blue Image"); return; }

  if( displayDepth == 24 ) {
    unsigned long *op; op = (unsigned long *) ( win->xi->data );
    for(i=0; i<nc*nr; i++) *op++ = COLOR_COMPOSE (*r++, *g++, *b++); 
    }
  else {
    unsigned short *op; op = (unsigned short *) ( win->xi->data );
    for(i=0; i<nc*nr; i++) *op++ = COLOR_COMPOSE (*r++, *g++, *b++); 
    }

  XPutImage(display, win->id, win->gc, win->xi, 
     win->sx, win->sy, win->dx, win->dy, win->w, win->h);
  }

/**
 * This function display an RGB Interleaved image ie. RGBRGBRGB etc
 * @param win Pointer to window
 * @param ip  Pointer to interleaved image 
 * @param nc  Number of Columns 
 * @param nr  Number of Rows 
 * @note This function is unaffected by the colourmap
 */

void XID_DrawImageRGBI(XIDwin *win, unsigned char *ip, int nc, int nr)	
  {
  int i; 
  unsigned char r, b, g; 

  if( win->xi == NULL ) { XIDerr("There is no Ximage"); return; }
  if( ip == NULL ) { XIDerr("There is no Image"); return; }

  if( displayDepth == 24 ) {
    unsigned long *op; op = (unsigned long *) ( win->xi->data );
    for(i=0; i<nc*nr; i++) { b = *ip++; g = *ip++; r = *ip++; *op++ = COLOR_COMPOSE (r, g, b); }
    }
  else {
    unsigned short *op; op = (unsigned short *) ( win->xi->data );
    for(i=0; i<nc*nr; i++) { b = *ip++; g = *ip++; r = *ip++; *op++ = COLOR_COMPOSE (r, g, b); }
    }

  XPutImage(display, win->id, win->gc, win->xi, 
     win->sx, win->sy, win->dx, win->dy, win->w, win->h);
  }

/**
 * This function display an RGB Interleaved image
 * @param win Pointer to window
 * @param ip  Pointer to interleaved image 
 * @param nc  Number of Columns 
 * @param nr  Number of Rows 
 * @note This function is unaffected by the colourmap
 * @warning Image and display format much match
 */

void XID_DrawImageRGBI_raw( XIDwin *win, unsigned char *ip, int nc, int nr)		
  {
  if( win->xi == NULL ) { XIDerr("There is no Ximage"); return; }
  if( ip == NULL ) { XIDerr("There is no Image"); return; }
  memcpy(win->xi->data, ip, nc*nr);

  XPutImage(display, win->id, win->gc, win->xi, 
     win->sx, win->sy, win->dx, win->dy, win->w, win->h);
  }

/**
 * This function displays Four RGB Interleaved image, 
 * @param win Pointer to window
 * @note it is unaffected by any colourmap
 */

void XID_DrawImageRGBI4(
    XIDwin *win, 	
    unsigned char *ip1, 
    unsigned char *ip2, 
    unsigned char *ip3, 
    unsigned char *ip4,
    int nc, int nr)
  {
  int i, j;
  unsigned char r, b, g;

  if( win->xi  == NULL ) { XIDerr("There is no Ximage"); return; }
  if( ip1 == NULL ) { XIDerr("There is no Image"); return; }
  if( ip2 == NULL ) { XIDerr("There is no Image"); return; }
  if( ip3 == NULL ) { XIDerr("There is no Image"); return; }
  if( ip4 == NULL ) { XIDerr("There is no Image"); return; }

  if( displayDepth == 24 ) {
    unsigned long *op; op = (unsigned long *) ( win->xi->data );
    for(i=0; i<nr; i++) {
      for(j=0; j<nc; j++) { b = *ip1++; g = *ip1++; r = *ip1++; *op++ = COLOR_COMPOSE (r, g, b); }
      for(j=0; j<nc; j++) { b = *ip2++; g = *ip2++; r = *ip2++; *op++ = COLOR_COMPOSE (r, g, b); }
      }
    for(i=0; i<nr; i++) {
      for(j=0; j<nc; j++) { b = *ip3++; g = *ip3++; r = *ip3++; *op++ = COLOR_COMPOSE (r, g, b); }
      for(j=0; j<nc; j++) { b = *ip4++; g = *ip4++; r = *ip4++; *op++ = COLOR_COMPOSE (r, g, b); }
      }
    }
  else {
    unsigned short *op; op = (unsigned short *) ( win->xi->data );
    for(i=0; i<nr; i++) {
      for(i=0; i<nc; i++) { b = *ip1++; g = *ip1++; r = *ip1++; *op++ = COLOR_COMPOSE (r, g, b); }
      for(i=0; i<nc; i++) { b = *ip2++; g = *ip2++; r = *ip2++; *op++ = COLOR_COMPOSE (r, g, b); }
      }
    for(i=0; i<nr; i++) {
      for(i=0; i<nc; i++) { b = *ip3++; g = *ip3++; r = *ip3++; *op++ = COLOR_COMPOSE (r, g, b); }
      for(i=0; i<nc; i++) { b = *ip4++; g = *ip4++; r = *ip4++; *op++ = COLOR_COMPOSE (r, g, b); }
      }
    }

  XPutImage(display, win->id, win->gc, win->xi, 
     win->sx, win->sy, win->dx, win->dy, win->w, win->h);
  }

/**
 * Get Named Colour
 * @param win Pointer to window
 * @param name Name of colour, specified in X11 rgb database
 * @return index to closest colour
 */

int XID_GetColour( XIDwin *win, char *name)
  {
  XColor exact, close;
  XWindowAttributes wattr;
  if (!XGetWindowAttributes(display, win->id, &wattr))
    { printf("Can't get window attributes."); exit(-1); }
  if(XAllocNamedColor(display, wattr.colormap, name, &exact, &close))
    return close.pixel;
  else return -1;
  }

/**
 * Get RGB Colour
 * @param win Pointer to window
 * @param r Red index of colour  (0 to 256) 
 * @param g Green index of colour  (0 to 256) 
 * @param b Blue index of colour  (0 to 256) 
 */

int XID_GetRGBColour(XIDwin *win, int r, int g, int b)
  {
  XColor exact;
  XWindowAttributes wattr;

  exact.flags = DoRed | DoGreen | DoBlue;
  exact.red   = (unsigned short) ((r * 65535) / 256);
  exact.green = (unsigned short) ((g * 65535) / 256);
  exact.blue  = (unsigned short) ((b * 65535) / 256);

  if (!XGetWindowAttributes(display, win->id, &wattr))
    { printf("Can't get window attributes."); exit(-1); }

  if(XAllocColor(display, wattr.colormap, &exact)) return exact.pixel;
  else return -1;
  }

/**
 * Read Named Colour Map file. This file should consist of 
 * 256 colour triplets, ranging from 0 to 256.  If the file cannot be
 * found in the current dierctory it will look in the directory 
 * specified by the environment variable XID. 
 * @param win Pointer to window
 * @param filename Filename of Colour Map file
 * @param overlay  Load overlay Cmap rather than the default 
 * @warning This colourmap is only used for the direct image drawing functions.
 */

int 
XID_LoadCmap(XIDwin *win, char *filename, int overlay) 
  { 
  int i; FILE *fp;
  if( xid_verbose ) printf("Loading Cmap %s\n", filename);

  if((fp = fopen(filename,"r")) == NULL ) { 
    char buff[80]; char *basename; 
    basename = getenv("XID");
    if( basename == NULL ) { XIDerr("XID not defined"); return -1; }
    sprintf(buff, "%s/%s", basename,filename);
    if((fp = fopen(buff,"r")) == NULL ) 
       { XIDerr("Cannot open cmap file \n"); return -1; }
    }

  if( overlay )  {
    for(i=0; i<256; i++) 
      if( fscanf(fp,"%d%d%d", &(win->redOver[i]),&(win->grnOver[i]),&(win->bluOver[i])) != 3 ) 
        { XIDerr("Incomplete CMAP File"); return -1; }
    }
  else  {
    for(i=0; i<256; i++) 
      if( fscanf(fp,"%d%d%d", &(win->redCmap[i]),&(win->grnCmap[i]),&(win->bluCmap[i])) != 3 ) 
        { XIDerr("Incomplete CMAP File"); return -1; }
    }

  fclose(fp);
  return 0;
  }

#ifndef  DOXYGEN_SHOULD_SKIP_THIS

/*
 * Display Stats on window - pinched from xwininfo
 */


static binding _window_classes[] = {
	{ InputOutput, "InputOutput" },
	{ InputOnly, "InputOnly" },
        { 0, 0 } };

static binding _map_states[] = {
	{ IsUnmapped, "IsUnMapped" },
	{ IsUnviewable, "IsUnviewable" },
	{ IsViewable, "IsViewable" },
	{ 0, 0 } };

static binding _backing_store_states[] = {
	{ NotUseful, "NotUseful" },
	{ WhenMapped, "WhenMapped" },
	{ Always, "Always" },
	{ 0, 0 } };

static binding _visual_classes[] = {
	{ StaticGray, "StaticGray" },
	{ GrayScale, "GrayScale" },
	{ StaticColor, "StaticColor" },
	{ PseudoColor, "PseudoColor" },
	{ TrueColor, "TrueColor" },
	{ DirectColor, "DirectColor" },
	{ 0, 0 }};
#endif

/**
 * This is a diagnostic function that will display information about the 
 * current window 
 */

void XID_Info(XIDwin *win)
  {
  int junk;
  int rx, ry, xright, ybelow;
  XWindowAttributes wattr;
  XVisualInfo vistemplate, *vinfo;
  Window junkwin;

  if (!XGetWindowAttributes(display, win->id, &wattr))
    { printf("Can't get window attributes."); exit(-1); }

  vistemplate.visualid = XVisualIDFromVisual(wattr.visual);

  printf("  Width: %d\n",  wattr.width);
  printf("  Height: %d\n", wattr.height);
  printf("  Depth: %d\n",  wattr.depth);

  vinfo = XGetVisualInfo(display, VisualIDMask, &vistemplate, &junk);
  (void) XTranslateCoordinates (display, win->id, wattr.root, 
	-wattr.border_width, -wattr.border_width, 
	&rx, &ry, &junkwin);
				
  xright = (displayWidth - wattr.border_width * 2 - wattr.width);
  ybelow = (displayHeight - wattr.border_width * 2 - wattr.height);

  printf("\n");
  printf("  Border Width: %d\n", wattr.border_width);
  printf("  Visual Class: %s\n", Lookup(vinfo->class, _visual_classes));
  printf("  Class: %s\n", Lookup(wattr.class, _window_classes));
  printf("  Colormap: 0x%lx (%sinstalled)\n", wattr.colormap, wattr.map_installed ? "" : "not ");
  printf("  Backing Store State: %s\n", Lookup(wattr.backing_store, _backing_store_states));
  printf("  Save Under State: %s\n", wattr.save_under ? "yes" : "no");
  printf("  Map State: %s\n", Lookup(wattr.map_state, _map_states));
  printf("  Override Redirect State: %s\n", wattr.override_redirect ? "yes" : "no");
  printf("  Corners:  +%d+%d  -%d+%d  -%d-%d  +%d-%d\n", rx, ry, xright, ry, xright, ybelow, rx, ybelow);
  printf("\n");
  }

/**
 * Routine to let user select a window using the mouse
 * @Warning this is not working yet
 */

XIDwin * 
XID_WindowWithMouse()
  {
  int status;
  Cursor cursor;
  XEvent event;
  Window target_win = None;
  Window root = RootWindow(display,displayNumber);
  int buttons = 0;

  printf("Click mouse on window\n");
  cursor = XCreateFontCursor(display, XC_crosshair);
  status = XGrabPointer(display, root, False,
            ButtonPressMask|ButtonReleaseMask, GrabModeSync,
            GrabModeAsync, root, cursor, CurrentTime);

  if (status != GrabSuccess) { XIDerr("Can't grab the mouse"); return NULL; }

  while ((target_win == None) || (buttons != 0)) {

    if( xid_verbose ) printf("Mouse Event recieved\n");
    XAllowEvents(display, SyncPointer, CurrentTime); /* allow one more event */
    XWindowEvent(display, root, ButtonPressMask|ButtonReleaseMask, &event);

    switch (event.type) {

      case ButtonPress: 
	if (target_win == None) {
          target_win = event.xbutton.subwindow; /* window selected */
          if (target_win == None) target_win = root;
          }
        buttons++;
        break;

      case ButtonRelease: /* there may have been some down before we started */
        if (buttons > 0) buttons--;
        break;
      }
    }

  XUngrabPointer(display, CurrentTime);      /* Done with pointer */

  /* Framed 
   if (XGetGeometry (display, window, &root, &i, &i, &u, &u, &u, &u) 
	  && window != root) window = XmuClientWindow (display, window);
   if( xid_verbose ) printf("Client 0x%x\n", (unsigned int) window);
  */

  return  XID_WindowWithId(target_win, 1);
  }

/**
 * This function grabs the first window with the matching name
 * @return Pointer to window 
 * @param name Name of window
 * @warning This has not been tested yet.
 */

XIDwin *
XID_WindowWithName(char *name) {
  Window win, GrabWindowWithName(Window, char *);
  win = GrabWindowWithName(RootWindow(display,displayNumber), name);
  return XID_WindowWithId(win, 1);
  }

Window GrabWindowWithName(Window top, char *name)
  {
  int i; char *wname;
  unsigned int nc;
  Window *list, dummy, win = 0;
  if( XFetchName(display, top, &wname) && !strcmp(wname, name)) return(top);
  if( !XQueryTree(display, top, &dummy, &dummy, &list, &nc)) return(0);
  for(i=0; i<nc; i++) { win = GrabWindowWithName(list[i], name); if(win) break; }
  if( list ) XFree ((char *)list);
  return(win);
  }

/** 
 * This function grabs the first window with the matching size
 * @return Pointer to window
 * @param W desired width of window in pixels
 * @param H desired height of window in pixels
 * @warning This has not been tested yet.
 */

XIDwin *
XID_WindowWithSize(int H, int W) {
  Window win, GrabWindowWithSize(Window, int, int);
  win = GrabWindowWithSize(RootWindow(display,displayNumber), H, W);
  return XID_WindowWithId(win, 1);
  }

Window GrabWindowWithSize(Window top, int W, int H)
  {
  int i; unsigned int nc;
  Window *list, dummy, win = 0, root;
  int x, y;
  unsigned int w, h, b, d;

  if( !XGetGeometry(display, top, &root, &x, &y, &w, &h, &b, &d)) return 0;
  if( W == w && H == h ) return top;
  if( w  < W || h  < H ) return 0;
  if( xid_verbose ) printf("Window Size %d %d\n", w, h);

  if (!XQueryTree(display, top, &dummy, &dummy, &list, &nc)) return 0;
  for (i=0; i<nc; i++) { win = GrabWindowWithSize(list[i], W, H); if(win) break; }
  if (list) XFree ((char *)list);
  return win;
  }

/**
 * Internal Function to translate Keyboard Event to String. 
 * The XKeysymToString is needed for non ascii keys.
 * @param ev keyboard event 
 * @param s String to hold response
 * @param n length of string
 */

int
TranslateKeyCode(XEvent * ev, char *s, int n)
  {
  int count;
  char *tmp;
  KeySym ks;

  if( !ev ) return -1;
  count = XLookupString((XKeyEvent *) ev, s, n, &ks, NULL); s[count] = '\0';
  if( count == 0 ) { tmp = XKeysymToString(ks); if (tmp) strcpy(s, tmp); }
  return 0;
  }
