/**
 \example test-timer.c
 * 
 * CSIRO Autonomous Systems Laboratory
 * Queensland Centre for Advanced Technologies
 * PO Box 883, Kenmore, QLD 4069, Australia
 * http://www.ict.csiro.au/
 *  
 * Copyright (c) CSIRO 
 ***********************************************************************/

#include <stdlib.h>
#include <stdio.h>
#include <errno.h>

#include "../include/rtx/timer.h"
#include "../include/rtx/signal.h"
#include "../include/rtx/main.h"
#include "../include/rtx/error.h"
#include "../include/rtx/thread.h"
#include "../include/rtx/sem.h"
#include "../include/rtx/time.h"

int 		count = 0;
RtxSem		*sem;
int		testSigwait = 0;
RtxTimerClockId clkId = RTX_TIMER_CLOCK_REALTIME;
double		reqDelay = 0.01;
double		reqPeriod = 0.01;

void
timer_func (void * arg)
{
    fprintf (stderr, "timer_func: %d\n", count++);
    if (rtx_sem_post (sem) == -1) {
        perror ("timer_func");
    }
}

void *
test_thread (int *arg)
{
    RtxTimer	*tmr;
    RtxTime now;

    if (rtx_error_init ("test_thread", 0, NULL) == -1) {
        perror ("rtx_error_init");
	exit (1);
    }

    if (testSigwait) {
        if ((tmr = rtx_timer_create (reqDelay, reqPeriod, NULL, NULL, clkId)) == NULL) {
	    rtx_error_flush ("rtx_timer_create failed");
	    exit (1);
	}
    } else {
        if ((tmr = rtx_timer_create (reqDelay, reqPeriod, timer_func, NULL, clkId)) == NULL) {
	    rtx_error_flush ("rtx_timer_create failed");
	    exit (1);
	}
    }

    if (rtx_timer_start (tmr) == -1) {
	rtx_error_flush ("rtx_timer_start failed");
	exit (1);
    }
    fprintf (stderr, "thread: timer started\n");

    while (1) {
        if (testSigwait) {
	    if (rtx_timer_wait (tmr) == -1) {
	        rtx_error_flush ("rtx_timer_wait failed");
		exit (1);
	    }
	} else {
	    if (rtx_sem_wait_ignore_eintr (sem) == -1) {
	        rtx_error_flush ("rtx_sem_wait failed");
		exit (1);
	    }
	}
	if (rtx_time_get (&now) == -1) {
	    rtx_error_flush ("rtx_time_get failed");
	    exit (1);
	}
	fprintf (stderr, "test_thread %d: %ld.%09ld\n", *arg, now.seconds, now.nanoSeconds);
    }
}

void *
cancel_func(int *num)
{
	fprintf (stderr, "cancel num = %d\n", *num);

	return NULL;
}

int
main (int argc, char * argv[])
{
    RtxThread	*thread;

    if (argc > 1)
        testSigwait = atoi (argv[1]);

    if (argc > 2)
        clkId = atoi (argv[2]);

    if (clkId)
        clkId = RTX_TIMER_CLOCK_CUSTOM;
    else
        clkId = RTX_TIMER_CLOCK_REALTIME;

    fprintf (stderr, "main: using clock %d\n", clkId);

    if (rtx_error_init ("main", 0, NULL) == -1) {
        perror ("rtx_error_init");
	exit (1);
    }

    if ((sem = rtx_sem_init (NULL, 0, 0)) == NULL) {
        rtx_error_flush ("rtx_sem_init failed");
	exit (1);
    }

    if (rtx_signal_block_realtime () == -1) {
        rtx_error_flush ("rtx_signal_block_realtime failed");
	exit (1);
    }

    if ((thread = rtx_thread_create ("test_thread", 0, 0, 0, 0, RTX_THREAD_CANCEL_ASYNCHRONOUS, test_thread, &count, cancel_func, &count)) == NULL) {
        rtx_error_flush ("rtx_thread_create failed");
	exit (1);
    }

    if (rtx_main_wait_shutdown (0) == -1) {
        rtx_error_flush ("rtx_init_wait_shutdown failed");
	exit (1);
    }

    if (rtx_thread_destroy_sync(thread)) {
        rtx_error_flush ("rtx_thread_destroy_sync failed");
	exit (1);
    }

    fprintf (stderr, "main: exiting...\n");
    exit (0);
}

