/**
 \example test-threads.c
 * 
 * CSIRO Autonomous Systems Laboratory
 * Queensland Centre for Advanced Technologies
 * PO Box 883, Kenmore, QLD 4069, Australia
 * http://www.ict.csiro.au/
 *  
 * Copyright (c) CSIRO 
 ***********************************************************************/

#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <errno.h>

#include "../include/rtx/thread.h"
#include "../include/rtx/message.h"
#include "../include/rtx/error.h"
#include "../include/rtx/main.h"
#include "../include/rtx/time.h"
#include "../include/rtx/timer.h"
#include "../include/rtx/signal.h"
#include "../include/rtx/sem.h"

typedef struct _test_threads_arg {
  int thId;
  int useTimer;
  RtxTimer * tmr;
} TEST_THREADS_ARG;

static RtxSem * testSem;
RtxThread ** ths;
TEST_THREADS_ARG * thargs;

void test_thread_cancel (TEST_THREADS_ARG * arg)
{
  rtx_message ("test_thread_cancel(%d): ", arg->thId);
  if (arg->useTimer) {
    if (rtx_timer_stop (arg->tmr) == -1)
      rtx_error ("test_thread_cancel(%d): rtx_timer_stop() failed", arg->thId);
    if (rtx_timer_destroy (arg->tmr) == -1)
      rtx_error ("test_thread_cancel(%d): rtx_timer_destroy() failed", arg->thId);
  }
}

void * test_thread_func (TEST_THREADS_ARG * arg)
{
  int i = 0;

  rtx_message ("test_thread_func(%d): stack size = %d", arg->thId, 
	       rtx_thread_get_stack_size ());
  if (arg->useTimer) {
    if ((arg->tmr = rtx_timer_create (1.0, 1.0, NULL, NULL, 0)) == NULL) {
      rtx_error ("test_thread_func: rtx_timer_create() failed");
      arg->useTimer = 0;
    } else {
      if (rtx_timer_start (arg->tmr) == -1) {
	rtx_error ("test_thread_func: rtx_timer_start() failed");
	arg->useTimer = 0;
      }
    }
  }
  rtx_message ("test_thread_func(%d): starting", arg->thId);
  while (1) {
    rtx_thread_testcancel ();
    rtx_message ("test_thread_func(%d): loop %d", arg->thId, i++);
    if (arg->useTimer) {
      if (rtx_timer_wait (arg->tmr) == -1) {
	rtx_error_flush ("test_thread_func: rtx_timer_wait() failed");
	sleep (1);
      }
    } else {
      sleep (1);
    }
  }
  return (NULL);
}

int main (int argc, char * argv[])
{
  int i, numThreads = 10;
  char thname[128];
  int errs = 0;
  int useTimer = 0;

  if (argc > 1)
    useTimer = atoi (argv[1]);
  if (argc > 2)
    numThreads = atoi (argv[2]);

  rtx_message_init ("test_thread", RTX_MESSAGE_STDERR);

  rtx_message ("starting");

  if (rtx_error_init ("test_thread", RTX_ERROR_STDERR | RTX_ERROR_MESSAGE, NULL)) {
    perror ("main: rtx_error_init() failed: ");
    exit (1);
  }

  if ((testSem = rtx_sem_init (NULL, 0, 0)) == NULL) {
    rtx_error_flush ("main: rtx_sem_init() failed");
    exit (1);
  }

  if (rtx_signal_block_realtime () == -1) {
    rtx_error_flush ("main: rtx_signal_block_realtime() failed: ");
    exit (2);
  }

  rtx_message ("allocating memory");
  if ((ths = calloc (numThreads, sizeof (RtxThread *))) == NULL) {
    rtx_error_flush ("main: calloc() failed: ");
    exit (2);
  }
  if ((thargs = calloc (numThreads, sizeof (TEST_THREADS_ARG))) == NULL) {
    rtx_error_flush ("main: calloc() failed: ");
    exit (3);
  }
  
  rtx_message ("starting threads");
  for (i=0; i<numThreads; i++) {
    sprintf (thname, "test_thread_func(%d)", i);
    thargs[i].thId = i;
    thargs[i].useTimer = useTimer;
    if ((ths[i] = rtx_thread_create (thname,
				     1,
				     RTX_THREAD_SCHED_OTHER,
				     0,
				     0,
				     RTX_THREAD_CANCEL_ASYNCHRONOUS,
				     test_thread_func,
				     &(thargs[i]),
				     test_thread_cancel,
				     &(thargs[i]))) == NULL) {
    /*
    if (rtx_thread_create_legacy (&(ths[i].id), 0, 0, test_thread_func,
				  &(thargs[i])) == -1) {
				  }
    */
      rtx_error ("main: rtx_thread_create(%d) failed", i);
      errs++;
    }
    rtx_message ("thread %d started", i);
  }

  if (errs)
    rtx_error_traceback();
  rtx_message ("all threads up, waiting for shutdown");
  if (rtx_main_wait_shutdown (0) == -1)
    return (rtx_error ("main: rtx_main_init_shutdown() failed"));

  for (i=0; i<numThreads; i++) {
    if (rtx_thread_destroy_sync (ths[i]) == -1)
      rtx_error ("main: rtx_thread_destroy_sync() failed");
  }
  return 0;
}

