/**
 \example test-inet.c
 * 
 * CSIRO Autonomous Systems Laboratory
 * Queensland Centre for Advanced Technologies
 * PO Box 883, Kenmore, QLD 4069, Australia
 * http://www.ict.csiro.au/
 *  
 * Copyright (c) CSIRO 
 ***********************************************************************/

static char rcsid[] = "$Header$";

/**
 * \example test-inet.c
 * \author Pavan Sikka
 * A simple test/example program for the inet module.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <errno.h>

#include "../include/rtx/error.h"
#include "../include/rtx/main.h"
#include "../include/rtx/thread.h"
#include "../include/rtx/inet.h"
#include "../include/rtx/getopt.h"
#include "../include/rtx/signal.h"

char * mcAddr = "239.255.15.1";
int mcPort = 15000;
RtxInet * inetHandle = NULL;
int reader = 0;
int writer = 0;
char readBuf[8192];
char writeBuf[8192];

RtxGetopt myOpts[] = {
	{"-reader", "launch reader thread",
		{
			{RTX_GETOPT_SET, &reader, ""},
			RTX_GETOPT_END_ARG
		}
	},
	{"-writer", "launch writer thread",
		{
			{RTX_GETOPT_SET, &writer, ""},
			RTX_GETOPT_END_ARG
		}
	},
	RTX_GETOPT_END
};

	void *
test_reader_thread (void * arg)
{
	RtxInetEndpoint srcAddr;
	int n = 0;

	fprintf (stderr, "reader_thread: running\n");
	for ( ; ; ) {
		readBuf[0] = '\0';
		if ((n = rtx_inet_read (inetHandle->sock, readBuf,
						8191, &srcAddr)) == -1) {
			rtx_error_flush ("reader_thread: rtx_inet_read");
		} else {
			readBuf[n] = '\0';
			fprintf (stderr, "rx: %s\n", readBuf);
		}
	}
}

	void *
test_writer_thread (void * arg)
{
	RtxInetEndpoint destAddr;

	if (rtx_inet_get_net_addr (mcAddr, mcPort, &destAddr) == -1) {
		rtx_error_flush ("writer_thread: rtx_inet_get_net_addr");
		return (NULL);
	}

	for ( ; ; ) {
		writeBuf[0] = '\0';
#if 0
		writeBuf[1] = 0;
		if ((writeBuf[0] = getchar()) != EOF){
			if (rtx_inet_write (inetHandle->sock, writeBuf,
						strlen (writeBuf), &destAddr) == -1)
				rtx_error_flush ("writer_thread: rtx_inet_write");
		} else {
			perror("writer: getchar");
			usleep(100000);
		}
#else
		fprintf (stderr, "input: ");
		if (fgets (writeBuf, 8192, stdin)) {
			if (rtx_inet_write (inetHandle->sock, writeBuf,
						strlen (writeBuf), &destAddr) == -1)
				rtx_error_flush ("writer_thread: rtx_inet_write");
		} else {
			perror("writer: fgets");
			usleep(100000);
		}
#endif
	}
}

	int
main (int argc, char * argv[])
{
	int ret = -1;
	RtxThread	*readerThread = NULL;
	RtxThread	*writerThread = NULL;

	ret = RTX_GETOPT_CMD (myOpts, argc, argv, rcsid, NULL);
	if (ret == -1) {
		fprintf (stderr, "ret = %d\n", ret);
		RTX_GETOPT_PRINT (myOpts, argv[0], rcsid, NULL);
	}

	if (rtx_error_init ("test-inet", 0, NULL) == -1) {
		perror ("rtx_error_init");
		exit (1);
	}

	if (rtx_signal_block_realtime () == -1) {
		rtx_error_flush ("rtx_signal_block_realtime failed");
		exit (1);
	}

	if ((inetHandle = rtx_inet_init (RTX_INET_UDP_MULTICAST,
					NULL, 0,
					mcAddr, mcPort,
					NULL, NULL, NULL)) == NULL) {
		rtx_error_flush ("rtx_inet_init failed");
		exit (1);
	}
	if (rtx_inet_set_multicast_ttl (inetHandle, 5) == -1) {
		rtx_error_flush ("rtx_inet_set_multicast_ttl() failed");
		exit (1);
	}
	if (reader)
		if ((readerThread = rtx_thread_create ("reader_thread", 0, 0, 0, 0,
						RTX_THREAD_CANCEL_ASYNCHRONOUS,
						test_reader_thread, NULL, 
						NULL, NULL)) == NULL) {
			rtx_error_flush ("rtx_thread_create failed");
			exit (1);
		}
	if (writer)
		if ((writerThread = rtx_thread_create ("writer_thread", 0, 0, 0, 0,
						RTX_THREAD_CANCEL_ASYNCHRONOUS,
						test_writer_thread, NULL, 
						NULL, NULL)) == NULL) {
			rtx_error_flush ("rtx_thread_create failed");
			exit (1);
		}

	if (rtx_main_wait_shutdown (0) == -1) {
		rtx_error_flush ("rtx_init_wait_shutdown failed");
		exit (1);
	}

	if (readerThread != NULL)
		if (rtx_thread_destroy_sync(readerThread)) {
			rtx_error_flush ("rtx_thread_destroy_sync failed");
			exit (1);
		}
	if (writerThread != NULL)
		if (rtx_thread_destroy_sync(writerThread)) {
			rtx_error_flush ("rtx_thread_destroy_sync failed");
			exit (1);
		}
	if (inetHandle != NULL)
		if (rtx_inet_done (inetHandle)) {
			rtx_error_flush ("rtx_inet_done failed");
			exit (1);
		}

	return (0);
}

