/**
 \example test-httpd.c
 * 
 * CSIRO Autonomous Systems Laboratory
 * Queensland Centre for Advanced Technologies
 * PO Box 883, Kenmore, QLD 4069, Australia
 * http://www.ict.csiro.au/
 *  
 * Copyright (c) CSIRO 
 ***********************************************************************/

#include <signal.h>
#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>

#include "../include/rtx/httpd.h"
#include "../include/rtx/error.h"
#include "../include/rtx/message.h"


#ifdef i486_lynxos310
extern char *optarg;
extern int opterr;
#endif

char * xmlresp = NULL;
static int xmlresp_len =0;

RtxHttpd server;

static void sigpipeh()
{

}

static void siginth(int sig)
{
    rtx_httpd_server_destroy(&server);
}

int hello_handler(RtxHttpdReq * r)
{
    char * name;
    char hello[100];
    if (!(r->method == RTX_HTTPD_M_GET && strcmp(r->url,"/hello")==0 )){
        return 0;
    }
    if ((name = rtx_hash_find(r->args,"name"))){
	snprintf(hello,sizeof(hello),"Hello %s!",name);
    }else{
	strcpy(hello,"Hello World");
    }
    if(rtx_httpd_resp_status(r,200,NULL)!=0){
        return rtx_error("Could not set HTTP response status");
    }
    if(rtx_httpd_resp_add_hdr(r,"Content-Type: text/plain")!=0){
        return rtx_error("Could not set Content Type");
    }
    if(rtx_httpd_send_resp(r,hello,strlen(hello))!=0){
        return rtx_error("Could not send response");
    }
    printf("Request %d for %s handled\n",r->method, r->url);
    return 0;
}

int static_xml_handler(RtxHttpdReq * r)
{
    char tmp[10000];
    if (r->method != RTX_HTTPD_M_POST && strcmp(r->url,"/RPC2")!=0){
        return 0;
    }
    if (rtx_httpd_read(r->conn,tmp,r->content_length) !=r->content_length){
        return rtx_error("didn't receive full request");
    }
    if(rtx_httpd_resp_status(r,200,NULL)!=0){
        return rtx_error("Could not set HTTP response status");
    }

    if(rtx_httpd_resp_add_hdr(r,"Content-Type: text/xml")!=0){
        return rtx_error("Could not set Content Type");
    }
    if(rtx_httpd_send_resp(r,xmlresp,xmlresp_len)!=0){
        return rtx_error("Could not send response");
    }
#ifdef DEBUG
    if (r->conn->server->debug > 1){
	rtx_message_routine("xml_handler: request %d for %s handled\n",r->method, r->url);
    }
#endif
    return 0;
}


int main(int argc, char **argv)
{
    int ret;
    RtxHttpdConfig conf;
    struct stat xstat;
    char * xmlrespfile;
    FILE * file;
    size_t l;
    char * configdir;
    char * rootdir;
    char c;
	
    xmlrespfile=NULL;
    configdir="/data/aa/etc/";
    rootdir=".";
    conf.port=8000;
    conf.multi_thread=1;
    conf.debug=5;

    rtx_error_init("main",0,NULL);
    opterr = 0;
    while ((c = getopt (argc, argv, "d:sp:x:c:r:")) != -1){
        switch (c){
	case 'd':
	    conf.debug=atoi(optarg);
	    break;
        case 's':
	    printf("Enabled single thread mode\n");
            conf.multi_thread=0;
            break;
        case 'p':
            conf.port=atoi(optarg);
            break;
	case 'x':
	    xmlrespfile=optarg;
	    break;
	case 'c':
	    configdir=optarg;
	    break;
	case 'r':
	    rootdir=optarg;
	    break;
        case '?':
	    fprintf (stderr, "Unknown option `-%c'.\n", c);
            fprintf(stderr,
		    "Usage: %s options\n"
		    "options are:\n"
		    "   -p port       listen on this port (default:8000)\n"
		    "   -x file       return this file to every xmlrpc request recived) (default: none)\n"
		    "   -c configdir  read config from configdir (default: /usr/aa/etc/)\n"
		    "   -r rootdir    root directory for serving files (default: current dir)\n"
		    "   -d debuglevel 0-3 (default 0)\n"
		    "   -s            single threaded, only one connection allowed\n"
		    ,argv[0]);
	
            return 1;
        default:
            abort ();
        }
    }


    conf.hostfile=malloc(strlen(configdir) + 13);
    if (!conf.hostfile){
	perror("malloc:");
	return 0;
    }
    strcpy(conf.hostfile,configdir);
    strcat(conf.hostfile,"xmlrpchosts");

    conf.passwdfile=malloc(strlen(configdir) + 14);
    if (!conf.passwdfile){
	perror("malloc:");
	return 0;
    }
    strcpy(conf.passwdfile,configdir);
    strcat(conf.passwdfile,"xmlrpcpasswd");

    conf.mimefile=malloc(strlen(configdir) + 14);
    if (!conf.mimefile){
	perror("malloc:");
	return 0;
    }
    strcpy(conf.mimefile,configdir);
    strcat(conf.mimefile,"mime.types");
    conf.rootdir=rootdir;

    if (xmlrespfile){
	if (stat(xmlrespfile,&xstat)){
	    printf("Could not stat xml file: %s: %s Exiting.\n",xmlrespfile,strerror(errno));
	    exit(0);
	}
	xmlresp_len=xstat.st_size;
	xmlresp = malloc(xmlresp_len+1);
	if (!xmlresp){
	    perror("malloc");
	    return 1;
	}
	file = fopen(xmlrespfile,"r");
	if (!file){
	    perror("fopen of xmlfile failed");
	    return 1;
	}
	l = fread(xmlresp,xmlresp_len,1,file);
	if (l != 1){
	    perror("fread on xmlfile failed");
	    return 1;
	}
	xmlresp[xmlresp_len]='\0';
    }
    printf("Initialising HTTPD server\n");
    ret=rtx_httpd_server_init(&server, &conf);
    if (ret!=0){
        rtx_error_flush("httpd_server_init returned %d\n",ret);
        exit(1);
    }
    signal(SIGPIPE,sigpipeh);
/*    signal(SIGINT,siginth); */

    rtx_httpd_server_add_handler(&server,rtx_httpd_filehandler);
    rtx_httpd_server_add_handler(&server,hello_handler);
    if (xmlrespfile){
	rtx_httpd_server_add_handler(&server,static_xml_handler);
    }
    printf("Starting HTTPD server\n");
    if (rtx_httpd_server_loop(&server)!=0)
	rtx_error_flush("error during server execution");
    printf("back in main.");
    return 0;
}
