/**
 \example test-hash.c
 * 
 * CSIRO Autonomous Systems Laboratory
 * Queensland Centre for Advanced Technologies
 * PO Box 883, Kenmore, QLD 4069, Australia
 * http://www.ict.csiro.au/
 *  
 * Copyright (c) CSIRO 
 ***********************************************************************/

#include <stdio.h>
#include <assert.h>

#include "rtx/hash.h"

#define FORALL(assertion) for (i=0; test[i].key; i++) { assertion ;}

#define HASH_SIZE 4

struct test {
  void *key;
  int data;
};

struct test str_tests[] = {
  {"", 0},
  {"a really long key that forces the hash value to wrap", 0},
  {"bar", 0},
  {"baz", 0},
  {"mary", 0},
  {"jane", 0},
  {"bob", 0},
  {NULL, 0}};

struct test int_tests[] = {
  {(void*)1, 0},
  {(void*)2, 0},
  {(void*)3, 0},
  {(void*)4, 0},
  {(void*)5, 0},
  {(void*)6, 0},
  {(void*)7, 0},
  {NULL, 0}};

void inc_data(void *key,
	      void *value,
	      int i, int j,
	      int *stop_iter,
	      void *context) {

  (*(int *)value)++;
  (*(int *)context)++;
}


void stop_halfway(void *key,
		  void *value,
		  int i, int j,
		  int *stop_iter,
		  void *context) {
  (*(int *)context)++;

  if (i > j / 2) {
    *stop_iter = 1;
  }

}

void test(rtx_hash_key_hooks_t key_hooks, struct test *test) {
  rtx_hash_t hash;
  void *value;
  int count;
  int i;

  /** Tests on an unused hash. **/
  assert(hash = rtx_hash_alloc(HASH_SIZE, key_hooks, rtx_hash_shallow_values, (rtx_error_t)NULL));
  assert(rtx_hash_used(hash) == 0);
  FORALL(assert(rtx_hash_find(hash, test[i].key, (rtx_error_t)NULL) == NULL));
  FORALL(assert(rtx_hash_delete(hash, test[i].key, (rtx_error_t)NULL) == -1));

  FORALL(test[i].data = 77);
  count = 0;
  FORALL(rtx_hash_iter(hash, inc_data, &count));
  FORALL(assert(test[i].data == 77));
  assert(count == 0);

  rtx_hash_free(hash);

  /** Tests on setting elements. **/
  assert(hash = rtx_hash_alloc(HASH_SIZE, key_hooks, rtx_hash_shallow_values, (rtx_error_t)NULL));
  FORALL(assert(rtx_hash_set(hash, test[i].key, &test[i].data, (rtx_error_t)NULL) == 0));
  count=0;
  FORALL(count++);
  assert(rtx_hash_used(hash) == count);

  FORALL(assert(value = rtx_hash_find(hash, test[i].key, (rtx_error_t)NULL));
	 assert(test[i].data == *(int*)value));

  /* Modify the associated value. */
  FORALL(assert(rtx_hash_set(hash, test[i].key, &test[i + 1].data, (rtx_error_t)NULL) == 0));
  FORALL(assert(value = rtx_hash_find(hash, test[i].key, (rtx_error_t)NULL));
	 assert(test[i + 1].data == *(int*)value));

  rtx_hash_free(hash);

  /** Tests on removing elements. **/
  assert(hash = rtx_hash_alloc(HASH_SIZE, key_hooks, rtx_hash_shallow_values, (rtx_error_t)NULL));

  FORALL(assert(rtx_hash_set(hash, test[i].key, &test[i].data, (rtx_error_t)NULL) == 0));
  FORALL(assert(rtx_hash_delete(hash, test[i].key, (rtx_error_t)NULL) == 0));
  FORALL(assert(rtx_hash_delete(hash, test[i].key, (rtx_error_t)NULL) == -1));

  assert(rtx_hash_used(hash) == 0);

  /* Add data again. */
  FORALL(assert(rtx_hash_set(hash, test[i].key, &test[i].data, (rtx_error_t)NULL) == 0));

  rtx_hash_free(hash);

  /** Iterator tests. **/

  /* Full iteration. */
  assert(hash = rtx_hash_alloc(HASH_SIZE, key_hooks, rtx_hash_shallow_values, (rtx_error_t)NULL));

  FORALL(test[i].data = 50);
  FORALL(assert(rtx_hash_set(hash, test[i].key, &test[i].data, (rtx_error_t)NULL) == 0));

  count=0;
  FORALL(count++);

  rtx_hash_iter(hash, inc_data, &i);
  FORALL(assert(test[i].data == 51));

  i = 0;
  rtx_hash_iter(hash, stop_halfway, &i);
  assert(i < count);

  rtx_hash_free(hash);
}

int main(int argc, char *argv[]) {
  test(rtx_hash_shallow_string_keys, str_tests);
  test(rtx_hash_deep_string_keys, str_tests);
  test(rtx_hash_int_keys, int_tests);
  
  return (0);
}
