/***********************************************************************
 * 
 * CSIRO Autonomous Systems Laboratory
 * Queensland Centre for Advanced Technologies
 * PO Box 883, Kenmore, QLD 4069, Australia
 * http://www.ict.csiro.au/
 *  
 * Copyright (c) CSIRO 
 ***********************************************************************/
#include	<stdio.h>
#include	<stdlib.h>
#include	<math.h>
#include	"rtx/error.h"
#include	"rtx/quintic.h"
#include	"rtx/traj.h"

static char rcsid[] RTX_UNUSED = "$Id: traj.c 2271 2007-12-23 04:41:55Z roy029 $";

static void commence_seg(RtxTraj *t);

/**
 * Initialize a trajectory generator.  Generates motion at constant speed
 * between waypoints, with quintic polynomial blends at the transitions
 * to maintain position and velocity continuity with zero acceleration.
 *
 * @param t pointer to generator object to initialize.  If NULL, allocate
 * one.
 * @param dim dimension of the state space.
 * @return pointer to initialized generator object, or NULL on error.
 */
RtxTraj *
rtx_traj_init(RtxTraj *t, int dim)
{

	// allocate space for the trajectory object
	if (t == NULL) {
		t = (RtxTraj *)calloc(1, sizeof(RtxTraj));
		if (t == NULL)
			return NULL;
	}

	t->dim = dim;
	t->nsegs = 0;
	t->curseg = 0;
	t->seglist = NULL;
	t->q = (RtxQuintic *)calloc(dim, sizeof(RtxQuintic));
	t->x = (double *)calloc(dim, sizeof(double));
	t->xd = (double *)calloc(dim, sizeof(double));
	t->x2 = (double *)calloc(dim, sizeof(double));
	t->xd2 = (double *)calloc(dim, sizeof(double));
	t->speedFactor = 1.0;

	return t;
}


void
rtx_traj_reset(RtxTraj *t, double *x, double *xd)
{
	int	i;

	t->curseg = -1;
	t->tpath = 0;
	t->tseg = 0;
	t->state = IDLE;

	for (i=0; i<t->dim; i++) {
		t->x[i] = x[i];
		t->xd[i] = (xd == NULL) ? 0.0 : xd[i];
	}
}

/**
 * Evaluate the next step on the path.  Performs constant velocity or
 * blend polynomial as required. Moves along segment list.
 *
 * @param t pointer to the generator object
 * @param dt time step along the path, can be variable.
 * @return 0 if OK, else -1 on error.
 */
int
rtx_traj_eval(RtxTraj *t, double dt)
{
	static char	*func = "rtx_traj_eval";
	RtxTrajSeg	*s = t->seglist[t->curseg];
	int	i;

	switch (t->state) {
	case NOT_INITD:
		return rtx_error("%s: trajectory not initialized", func);
	case IDLE:
		
		commence_seg(t);
		break;
	case ACC:
	case STOPPING:

		for (i=0; i<t->dim; i++)
			rtx_quintic_evaluate2(&t->x[i], &t->xd[i],
				&t->q[i], t->tseg);

		if (t->tseg >= (s->tacc-dt)) {
			if (t->state == STOPPING)
				t->state = STOPPED;
			else
				t->state = CONSTVEL;
		}
		break;

	case CONSTVEL:
		/* update position based on constant velocity model */
		for (i=0; i<t->dim; i++)
			t->x[i] += t->xd[i] * dt;

		/* test if time to transition to new segment */
		if ((t->tsegmax - t->tseg) <= (s->tacc/2.0)) 
			commence_seg(t);

		break;

	case STOPPED:
		for (i=0; i<t->dim; i++)
			t->xd[i] = 0;

		return 1;
		
	}

	t->tseg += dt;
	t->tpath += dt;

	return 0;
}

void
rtx_traj_abort(RtxTraj *t, int seg)
{
	/* decelerate a stop */
}

/**
 * Append a motion segment to the path.
 *
 * @param t A trajectory object to which this segment is appended.
 * @param x A vector specifying the end-point of this segment.
 * @param speed The speed along the path in m/s.
 * @param tacc The acceleration time in s.  This is the time used to accelerate
 *   into this segment at its beginning.  Can be zero.
 * @param maxAcc The maximum acceleration allowed during the transition.
 * @return A segment number, zero or greater, or -1 on error.
 *
 * \todo Add functions to insert after particular segment, or to delete
 * a particular segment.
 */
int
rtx_traj_append(RtxTraj *t, double *x, double speed, double tacc, double accMax)
{
	RtxTrajSeg	*s;
	int		i;

	/* create a new segment */
	if ((s = (RtxTrajSeg *)calloc(1, sizeof(RtxTrajSeg))) == NULL)
		return rtx_error("rtx_traj_add: no memory");

	/* create storage for the coordinate */
	if ((s->x = calloc(t->dim, sizeof(double))) == NULL)
		return rtx_error("rtx_traj_add: no memory");

	/* copy params into the segment */
	for (i=0; i<t->dim; i++)
		s->x[i] = x[i];

	s->tacc = tacc;
	s->accMax = accMax;
	s->speed = speed;

	/* extend the segment list */
	t->seglist = (RtxTrajSeg **)realloc(t->seglist, (t->nsegs+1)*sizeof(RtxTrajSeg));
	if (t->seglist == NULL)
		return rtx_error("rtx_traj_add: no memory");

	t->seglist[t->nsegs] = s;

	t->nsegs++;

	return 0;
}

/**
 * Print a list of the motion segments.
 *
 * @param fp stream to output list on.
 * @param t pointer to the generator object
 */
void
rtx_traj_print(FILE *fp, RtxTraj *t)
{
	int	i, j;

	if (fp == NULL)
		fp = stdout;

	for (i=0; i<t->nsegs; i++) {
		fprintf(fp, "%3d: ", i);
		for (j=0; j<t->dim; j++)
			fprintf(fp, "%8.3f ", t->seglist[i]->x[j]);
		fprintf(fp, " (%.2fs)\n", t->seglist[i]->tacc);
	}
}

#define	curseg(t)	((t)->seglist[(t)->curseg])
#define	nextseg(t)	((t)->seglist[(t)->curseg+1])
#define	lastseg(t)	((t)->curseg == ((t)->nsegs-1))
#define	firstseg(t)	((t)->curseg == -1)

/**
 * Commence a new segment.  Handles generation of the blend polynomial.
 */
static void
commence_seg(RtxTraj *t)
{
	int	i;
	double	norm = 0.0;

	if (lastseg(t)) {
		/* this is the last segment */

		fprintf(stderr, "last segment\n");
		/* initialize the blend polynomial */
		for (i=0; i<t->dim; i++) {
			rtx_quintic_init(&t->q[i], curseg(t)->tacc/2,
				t->x[i], curseg(t)->x[i],
				t->xd[i], 0.0);
			t->state = STOPPING;
			t->tseg = 0;
		}
		return;
	}

	/*
	 * figure entry velocity for the next segment, based on motion from
	 * current destination to next segment destination
	 */


	fprintf(stderr, "transition\n");
	/* create unit vector */
	if (firstseg(t)) {
		for (i=0; i<t->dim; i++) {
			t->xd2[i] = nextseg(t)->x[i] - t->x[i];
			norm += t->xd2[i] * t->xd2[i];
		}
	} else {
		for (i=0; i<t->dim; i++) {
			t->xd2[i] = nextseg(t)->x[i] - curseg(t)->x[i];
			norm += t->xd2[i] * t->xd2[i];
		}
	}
	norm = sqrt(norm);

	/* convert to desired velocity */
	for (i=0; i<t->dim; i++) {
		t->xd2[i] = t->xd2[i] / norm * nextseg(t)->speed * t->speedFactor;
		if (firstseg(t))
			t->x2[i] = t->x[i] + t->xd2[i] * nextseg(t)->tacc/2.0;
		else {
			t->x2[i] = curseg(t)->x[i] + t->xd2[i] * nextseg(t)->tacc/2.0;
			fprintf(stderr, "nextseg->x=%f, t->xd2=%f, tacc=%f\n",
				nextseg(t)->x[i] , t->xd2[i], nextseg(t)->tacc/2.0);
		}

	}

	/* compute segment time */
	t->tsegmax = norm / nextseg(t)->speed / t->speedFactor;
	if (t->tsegmax < nextseg(t)->tacc)
		t->tsegmax = nextseg(t)->tacc;

	fprintf(stderr, "segtime %f, norm %f\n", t->tsegmax, norm);
	fprintf(stderr, "join point %f, %f\n", t->x2[0], t->x2[1]);
	fprintf(stderr, "join vel %f, %f\n", t->xd2[0], t->xd2[1]);

	/* initialize the blend polynomial */
	for (i=0; i<t->dim; i++)
		rtx_quintic_init(&t->q[i], nextseg(t)->tacc,
			t->x[i], t->x2[i],
			t->xd[i], t->xd2[i]);

	t->curseg++;
	t->state = ACC;
	t->tseg = 0;
}

#ifdef	MAIN

int
main(int ac, char *av[])
{
	RtxTraj	T;
	double	p0[2] = {0, 0};
	double	p1[2] = {1, 3};
	double	p2[2] = {2, 7};
	double	p3[2] = {1, 4};

	printf("bad %f\n", sqrt(-1.0));
	printf("bad %f\n", pow(-1.0, 1.0/3.0));

	rtx_traj_init(&T, 2);

	rtx_traj_append(&T, p1, 1.0, 1.0, 0.0);
	rtx_traj_append(&T, p2, 1.0, 1.0, 0.0);
	rtx_traj_append(&T, p3, 1.0, 1.0, 0.0);

	//rtx_traj_print(NULL, &T);

	rtx_traj_reset(&T, p0, NULL);

	while ( rtx_traj_eval(&T, 0.1) == 0)
		printf("%8.2f %8.2f %8.2f   %8.2f %8.2f %d\n", 
			T.tpath,
			T.x[0], T.x[1],
			T.xd[0], T.xd[1],
			T.state);

	return 0;
}
#endif
