/***********************************************************************
 * 
 * CSIRO Autonomous Systems Laboratory
 * Queensland Centre for Advanced Technologies
 * PO Box 883, Kenmore, QLD 4069, Australia
 * http://www.ict.csiro.au/
 *  
 * Copyright (c) CSIRO 
 ***********************************************************************/

/**
 * \file timer.c
 * \brief A simple interface to POSIX.1b timers
 * \author Jonathan Roberts
 * \author Pavan Sikka
 */

#include <stdlib.h>
#include <time.h>
#include <errno.h>
#include <math.h>

#ifdef i486_darwin
#define DARWIN
#endif

#include "rtx/defines.h"
#include "rtx/timer.h"
#include "rtx/time.h"
#include "rtx/signal.h"
#include "rtx/error.h"
#include "rtx/message.h"

static char rcsid[] RTX_UNUSED = "$Id: timer.c 2274 2007-12-23 05:37:32Z roy029 $";

#ifndef	DARWIN

static void
rtx_timer_sighandler (
		      int sigNum,
		      siginfo_t * info,
		      void * ignored
		      )
{
    RtxTimer * tmr;

    tmr = (RtxTimer *) info->si_value.sival_ptr;
    if (tmr->func != NULL) {
        tmr->func (tmr->arg);
    }
}

/**
 * Sleep for the specified period.
 *
 * @return 0 if OK, -1 on error.
 *
 * This function is a wrapper around the POSIX.1b nanosleep() function.
 * The function ignores EINTR
 * errors and returns only after the sleep is completed or an error occurs.
 */
int
rtx_timer_sleep (
		 double dt                   /**< period to sleep for */
		 )
{
    RtxTime t, r;
    int res;

    rtx_time_from_double (&t, dt);
    r = t;

    while ((r.seconds != 0) || (r.nanoSeconds != 0)) {
        res = nanosleep ((struct timespec *) &t, (struct timespec *) &r);
	if (res == -1) {
	    if (errno != EINTR) {
	        return (rtx_error_errno ("rtx_timer_sleep failed"));
	    } else {
	        t = r;
	    }
	} else {
  	    return (0);
	}
    }
    return (0);
}

/**
 * Sleep for the specified period (or until interrupted by a signal).
 *
 * @return 0 if OK, -1 on error.
 *
 * This function is a wrapper around the POSIX.1b nanosleep() function.
 * The function does not ignore signals and can return early if it is
 * interrupted by a signal.
 */
int
rtx_timer_sleep_eintr (
		 double dt                   /**< period to sleep for */
		 )
{
    RtxTime t, r;
    int res;

    rtx_time_from_double (&t, dt);
    res = nanosleep ((struct timespec *) &t, (struct timespec *) &r);
    if (res == -1)
        return (rtx_error_errno ("rtx_timer_sleep failed"));
    return (0);
}

/**
 * Get clock resolution of the specified clock
 *
 * @return clcok resolution, < 0 on error
 *
 */
double
rtx_timer_get_clock_resolution (
				RtxTimerClockId clockId       /**< clock id */
				)
{
    struct timespec resolution;
    RtxTime resRtx;
    clockid_t clkId;

    /*
     * Get the clock resolution and inform user of rounding errors
     */
    switch (clockId) {
        case RTX_TIMER_CLOCK_CUSTOM:
	    clkId = 1;
	    break;
        case RTX_TIMER_CLOCK_REALTIME:
        default:
	    clkId = CLOCK_REALTIME;
	    break;
    }
    if (clock_getres (clkId, &resolution) == -1)
        return ((double) rtx_error_errno ("rtx_timer_clock_get_resolution:"
					  " clock_getres failed"));
    resRtx.seconds = resolution.tv_sec;
    resRtx.nanoSeconds = resolution.tv_nsec;
    return (rtx_time_to_double (&resRtx));
}

/**
 * Create a Posix.1b timer.
 *
 * @return Pointer to RtxTimer structure if OK, NULL on error.
 *
 * This function creates and initializes a Posix.1b timer. However, the timer
 * is not run by this function.
 * @see rtx_timer_start
 */
RtxTimer * 
rtx_timer_create (
		  double initDelay,             /**< initial timer delay */
		  double period,                /**< subsequent timer delays */
		  void (* func) (void *),       /**< timer handler function */
		  void * arg,                   /**< argument to the handler */
		  RtxTimerClockId clockId       /**< clock to use for timer */
		  )
{
    RtxTimer * tmr = NULL;
    struct sigevent timerEvent;
    struct sigaction sa;
    clockid_t clkId;

    /* get memory for the handle */
    if ((tmr = calloc (1, sizeof (RtxTimer))) == NULL)
        return (rtx_error_errno_null ("rtx_timer_create: calloc failed"));

    /* set the times */
    tmr->arg = arg;
    tmr->func = func;
    tmr->clockId = clockId;
    rtx_timer_set (tmr, initDelay, period);

    /* Allocate a realtime signal and setup the sigevent structure */
    if ((tmr->sigNum = rtx_signal_alloc_realtime ()) == -1)
        return (rtx_error_null ("rtx_timer_create: "
				"rtx_signal_alloc_realtime failed"));
    timerEvent.sigev_notify = SIGEV_SIGNAL;
    timerEvent.sigev_signo = tmr->sigNum;
    timerEvent.sigev_value.sival_ptr = (void *) tmr;
    if (func == NULL) {
        /* User will be using rtx_timer_wait() to get the timer expirations */
        if (sigemptyset (&(tmr->tmrSig)) == -1) {
	    rtx_signal_free_realtime (tmr->sigNum);
	    free ((char *) tmr);
	    return (rtx_error_errno_null ("rtx_timer_create: sigemptyset()"
					  " failed"));
	}
	if (sigaddset (&(tmr->tmrSig), tmr->sigNum) == -1) {
	    rtx_signal_free_realtime (tmr->sigNum);
	    free ((char *) tmr);
	    return (rtx_error_errno_null ("rtx_timer_create: sigaddset()"
					  " failed"));
	}
	/* Block the realtime signal */
	if (rtx_signal_block (tmr->sigNum) == -1) {
	    rtx_signal_free_realtime (tmr->sigNum);
	    free ((char *) tmr);
	    return (rtx_error_null ("rtx_timer_create: rtx_signal_block failed"));
	}
    } else {
        /* User wants the function to be called from the signal handler */
        if (sigemptyset (&(sa.sa_mask)) == -1) {
	    rtx_signal_free_realtime (tmr->sigNum);
	    free (tmr);
	    return (rtx_error_errno_null ("rtx_timer_create: "
					  "sigemptyset failed"));
	}
	sa.sa_flags = SA_SIGINFO; /* real-time signal */
	sa.sa_sigaction = rtx_timer_sighandler;
	/* Setup the signal handler */
	if (sigaction (tmr->sigNum, &sa, &(tmr->old)) == -1) {
	    rtx_signal_free_realtime (tmr->sigNum);
	    free ((char *) tmr);
	    return (rtx_error_errno_null ("rtx_timer_create: sigaction failed"));
	}
	/* Unblock the realtime signal */
	if (rtx_signal_unblock (tmr->sigNum) == -1) {
	    rtx_signal_free_realtime (tmr->sigNum);
	    free ((char *) tmr);
	    return (rtx_error_null ("rtx_timer_create: rtx_signal_unblock"
				    " failed"));
	}
    }
    switch (tmr->clockId) {
        case RTX_TIMER_CLOCK_CUSTOM:
	    clkId = 1;
	    break;
        case RTX_TIMER_CLOCK_REALTIME:
        default:
	    clkId = CLOCK_REALTIME;
	    break;
    }
    /* now create the timer */
    if (timer_create(clkId, &timerEvent, &(tmr->timer)) == -1)
        return (rtx_error_errno_null ("rtx_timer_create: "
				      "timer_create failed"));

    /* the timer has been created but is not yet running */

    return (tmr);
}

/**
 * Wait for the timer to fire.
 *
 * @return 0 on success, -1 on failure.
 *
 * This function uses the sigwait() system call to wait for the
 * timer to fire.
 */
int
rtx_timer_wait (
		RtxTimer * tmr /**< timer */
		)
{
    int retVal, sigNum = -1;

    if ((retVal = sigwait (&(tmr->tmrSig), &sigNum)) != 0)
        return (rtx_error_errno2 (retVal, 
				  "rtx_timer_wait: sigwait() failed"));
    if (sigNum != tmr->sigNum)
        return (rtx_error ("rtx_timer_wait: signal = %d (expected %d)",
			   sigNum, tmr->sigNum));
    return (0);
}

/**
 * Destroy a Posix.1b timer.
 *
 * @return 0 if OK, -1 on error.
 */
int 
rtx_timer_destroy (
		   RtxTimer * tmr       /**< timer */
		   )
{
    int errs = 0;

    if (timer_delete (tmr->timer) == -1) {
        rtx_error_errno ("rtx_timer_destroy: timer_delete failed");
	errs++;
    }
    if (rtx_signal_free_realtime (tmr->sigNum) == -1) {
        rtx_error ("rtx_timer_destroy: rtx_signal_free_realtime failed");
	errs++;
    }

    free ((char *) tmr);
    
    if (errs)
        return (-1);
    return 0;
}

/**
 * Set timer parameters.
 *
 * @return 0 if OK, -1 on error.
 *
 * This function allows the application to change the delays associated with
 * a timer. Note that this only changes values in a structure associated with
 * the timer. You need to use rtx_timer_start and rtx_timer_stop to actually
 * control the timer itself.
 */
int 
rtx_timer_set (
	       RtxTimer * tmr,        /**< timer */
	       double initDelay, 
	       double period
	       )
{
    struct timespec resolution;
    RtxTime reqDelay;
    RtxTime reqPeriod;
    RtxTime clockRes;
    clockid_t clkId;

    /*
     * Get the clock resolution and inform user of rounding errors
     */
    switch (tmr->clockId) {
        case RTX_TIMER_CLOCK_CUSTOM:
	    clkId = 1;
	    break;
        case RTX_TIMER_CLOCK_REALTIME:
        default:
	    clkId = CLOCK_REALTIME;
	    break;
    }
    if (clock_getres (clkId, &resolution) == -1)
        return (rtx_error_errno ("rtx_timer_create: clock_getres failed"));
    clockRes.seconds = resolution.tv_sec;
    clockRes.nanoSeconds = resolution.tv_nsec;

    /* Set the timer */
    rtx_time_from_double (&reqDelay, initDelay);
    rtx_time_from_double (&reqPeriod, period);
    tmr->itimer.it_value.tv_sec = reqDelay.seconds;
    tmr->itimer.it_value.tv_nsec = reqDelay.nanoSeconds;
    tmr->itimer.it_interval.tv_sec = reqPeriod.seconds;
    tmr->itimer.it_interval.tv_nsec = reqPeriod.nanoSeconds;

    /* Check desired times (inform only) */
    if (reqPeriod.seconds == 0) {
        if ((reqPeriod.nanoSeconds < clockRes.nanoSeconds) &&
	    (reqPeriod.nanoSeconds > 0))
	    rtx_message_warning ("rtx_timer_set: requested period (%d ns) "
				 " is less than clock res (%d ns)",
				 reqPeriod.nanoSeconds, clockRes.nanoSeconds);
    }
    /* Could also check that the requested value is an exact multiple. */
    /* Could also check the requested delay for the same conditions. */

    return (0);
}

/**
 * Start the timer.
 *
 * @return 0 if OK, -1 on error.
 */
int 
rtx_timer_start (
		 RtxTimer * tmr       /**< timer */
		 )
{
    if (timer_settime(tmr->timer, 0, &(tmr->itimer), NULL) == -1)
        return (rtx_error_errno ("rtx_timer_start: timer_settime failed"));

    return (0);
}

/**
 * Stop the timer.
 *
 * @return 0 if OK, -1 on error.
 */
int 
rtx_timer_stop (
		RtxTimer * tmr       /**< timer */
		)
{
    struct itimerspec itimer = {{0, 0}, {0, 0}};
    
    if (timer_settime(tmr->timer, 0, &itimer, NULL) == -1)
        return (rtx_error_errno ("rtx_timer_stop: timer_settime failed"));
    return (0);
}

static void
rtx_timer_terminate(void * arg)
{
	RtxTimerThread * tmrTh = (RtxTimerThread *) arg;
	if (rtx_timer_stop (tmrTh->tmr) == -1) {
		rtx_error ("rtx_timer_periodic_thread: rtx_timer_stop failed\n");
	}
	if (rtx_timer_destroy (tmrTh->tmr) == -1) {
		rtx_error ("rtx_timer_periodic_thread: rtx_timer_stop failed\n");
	}
}

/**
 * Thread used to create the periodic timer and then run the specified function.
 *
 * @return Pointer to RtxTimerThread structure if OK, NULL on error.
 *
 * This is the thread used by the rtx_timer_create_thread() function.
 */
static void *
rtx_timer_periodic_thread (
			   void * arg
			   )
{
	RtxTimerThread * tmrTh = (RtxTimerThread *) arg;

	rtx_signal_block_realtime ();
	rtx_thread_setcancel (RTX_THREAD_CANCEL_DEFERRED);


	if ((tmrTh->tmr = rtx_timer_create (tmrTh->initDelay, tmrTh->period, NULL,
					NULL, tmrTh->clockId)) == NULL)
		return (NULL);

	rtx_thread_cleanup_push(rtx_timer_terminate,tmrTh);
	if (rtx_timer_start (tmrTh->tmr) == -1)
		return (NULL);

	while (! tmrTh->done) {
		if (rtx_timer_wait (tmrTh->tmr) == -1) {
			rtx_timer_sleep (0.05);
			continue;
		}
		if (tmrTh->done)
			continue;
		
		if (tmrTh->func != NULL)
			tmrTh->func (tmrTh->arg);
	}

	rtx_thread_cleanup_pop(0);

	if (rtx_timer_stop (tmrTh->tmr) == -1) {
		rtx_error ("rtx_timer_periodic_thread: rtx_timer_stop failed\n");
	}
	if (rtx_timer_destroy (tmrTh->tmr) == -1) {
		rtx_error ("rtx_timer_periodic_thread: rtx_timer_stop failed\n");
	}
	return ((void *) 0);

}

/**
 * Create a thread/timer that runs the specified function at the specified rate.
 *
 * @return Pointer to RtxTimerThread structure if OK, NULL on error.
 *
 * This function creates a thread. The thread creates and initializes a Posix.1b timer
 * and then uses the sigwait function to periodiaclly run the specified function at
 * the specified rate.
 */
RtxTimerThread * 
rtx_timer_create_thread (
		  double initDelay,             /**< initial timer delay */
		  double period,                /**< subsequent timer delays */
		  int thPrio,                   /**< thread priority */
		  void (* func) (void *),       /**< timer handler function */
		  void * arg,                   /**< argument to the handler */
		  RtxTimerClockId clockId       /**< clock to use for timer */
		  )
{
	int policy = RTX_THREAD_SCHED_OTHER;
    RtxTimerThread * tmrTh;

    /* get memory for the handle */
    if ((tmrTh = calloc (1, sizeof (RtxTimerThread))) == NULL)
        return (rtx_error_errno_null ("rtx_timer_create: calloc failed"));

    tmrTh->arg = arg;
    tmrTh->func = func;
	if (initDelay <= 0) initDelay = period;
    tmrTh->initDelay = initDelay;
    tmrTh->period = period;
    tmrTh->done = 0;
    tmrTh->clockId = clockId;

	switch (thPrio) {
		case 0:
		case RTX_THREAD_PRIO_MIN:
			policy = RTX_THREAD_SCHED_OTHER;
			break;
		default:
			policy = RTX_THREAD_SCHED_FIFO;
			break;
	}

    if ((tmrTh->th = rtx_thread_create ("rtx_timer_periodic_thread",
					0,
					policy,
					thPrio,
#ifdef i486_lynxos
					/* TODO: find what OS requires a value here */
					16,
#else
					0,
#endif
					RTX_THREAD_CANCEL_DEFERRED,
					rtx_timer_periodic_thread,
					tmrTh,
					NULL, NULL)) == NULL) {
        rtx_error_null ("could not create periodic thread");
        return (NULL);
    }
    
    return (tmrTh);
}

/**
 * Destroy a thread/timer.
 *
 * @return 0 if OK, -1 on error.
 */
int 
rtx_timer_destroy_thread_sync (
			  RtxTimerThread * perTh       /**< periodic thread */
			  )
{
    int errs = 0, retval;

    perTh->done = 1;
    if ((retval = pthread_kill (perTh->th->id, perTh->tmr->sigNum)) != 0) {
        errs++;
	rtx_error_errno2 (retval, "rtx_timer_destroy_thread: pthread_kill()"
			  " failed");
    }
    if (rtx_thread_join (perTh->th)) {
        errs++;
	rtx_error_errno ("rtx_timer_destroy_thread: rtx_thread_join() failed");
    }

    free ((char *) perTh);
    
    if (errs)
        return (-1);
    return 0;
}

/**
 * Destroy a thread/timer.
 *
 * @return 0 if OK, -1 on error.
 */
int 
rtx_timer_destroy_thread (
			  RtxTimerThread * perTh       /**< periodic thread */
			  )
{
    int errs = 0, retval;


    perTh->done = 1;
	if ((retval = rtx_thread_destroy_sync (perTh->th)) != 0) {
		errs++;
		rtx_error_errno2 (retval, "rtx_timer_destroy_thread: rtx_thread_destroy_sync()"
				" failed");
	}
    free ((char *) perTh);
    
    if (errs)
        return (-1);
    return 0;
}

#else
/**** mac version ****/

/**
 * Sleep for the specified period.
 *
 * @return 0 if OK, -1 on error.
 *
 * This function is a wrapper around the POSIX.1b nanosleep() function.
 * The function ignores EINTR
 * errors and returns only after the sleep is completed or an error occurs.
 */
int
rtx_timer_sleep (
		 double dt                   /**< period to sleep for */
		 )
{
    RtxTime t, r;
    int res;

    rtx_time_from_double (&t, dt);
    r = t;

    while ((r.seconds != 0) || (r.nanoSeconds != 0)) {
        res = nanosleep ((struct timespec *) &t, (struct timespec *) &r);
	if (res == -1) {
	    if (errno != EINTR) {
	        return (rtx_error_errno ("rtx_timer_sleep failed"));
	    } else {
	        t = r;
	    }
	} else {
  	    return (0);
	}
    }
    return (0);
}

/**
 * Sleep for the specified period (or until interrupted by a signal).
 *
 * @return 0 if OK, -1 on error.
 *
 * This function is a wrapper around the POSIX.1b nanosleep() function.
 * The function does not ignore signals and can return early if it is
 * interrupted by a signal.
 */
int
rtx_timer_sleep_eintr (
		 double dt                   /**< period to sleep for */
		 )
{
    RtxTime t, r;
    int res;

    rtx_time_from_double (&t, dt);
    res = nanosleep ((struct timespec *) &t, (struct timespec *) &r);
    if (res == -1)
        return (rtx_error_errno ("rtx_timer_sleep failed"));
    return (0);
}

/**
 * Get clock resolution of the specified clock
 *
 * @return clcok resolution, < 0 on error
 *
 */
double
rtx_timer_get_clock_resolution (
				RtxTimerClockId clockId       /**< clock id */
				)
{
    return rtx_error("rtx_timer_get_clock_resolution: not implemented on this platform");
}

/**
 * Create a Posix.1b timer.
 *
 * @return Pointer to RtxTimer structure if OK, NULL on error.
 *
 * This function creates and initializes a Posix.1b timer. However, the timer
 * is not run by this function.
 * @see rtx_timer_start
 */
RtxTimer * 
rtx_timer_create (
		  double initDelay,             /**< initial timer delay */
		  double period,                /**< subsequent timer delays */
		  void (* func) (void *),       /**< timer handler function */
		  void * arg,                   /**< argument to the handler */
		  RtxTimerClockId clockId       /**< clock to use for timer */
		  )
{
    return rtx_error_null("rtx_timer_create: not implemented on this platform");
}

/**
 * Wait for the timer to fire.
 *
 * @return 0 on success, -1 on failure.
 *
 * This function uses the sigwait() system call to wait for the
 * timer to fire.
 */
int
rtx_timer_wait (
		RtxTimer * tmr /**< timer */
		)
{
    return rtx_error("rtx_timer_wait: not implemented on this platform");
}

/**
 * Destroy a Posix.1b timer.
 *
 * @return 0 if OK, -1 on error.
 */
int 
rtx_timer_destroy (
		   RtxTimer * tmr       /**< timer */
		   )
{
    return rtx_error("rtx_timer_destroy: not implemented on this platform");
}

/**
 * Set timer parameters.
 *
 * @return 0 if OK, -1 on error.
 *
 * This function allows the application to change the delays associated with
 * a timer. Note that this only changes values in a structure associated with
 * the timer. You need to use rtx_timer_start and rtx_timer_stop to actually
 * control the timer itself.
 */
int 
rtx_timer_set (
	       RtxTimer * tmr,        /**< timer */
	       double initDelay, 
	       double period
	       )
{
    return rtx_error("rtx_timer_set: not implemented on this platform");
}

/**
 * Start the timer.
 *
 * @return 0 if OK, -1 on error.
 */
int 
rtx_timer_start (
		 RtxTimer * tmr       /**< timer */
		 )
{

    return rtx_error("rtx_timer_start: not implemented on this platform");
}

/**
 * Stop the timer.
 *
 * @return 0 if OK, -1 on error.
 */
int 
rtx_timer_stop (
		RtxTimer * tmr       /**< timer */
		)
{
    return rtx_error("rtx_timer_stop: not implemented on this platform");
}

/**
 * Create a thread/timer that runs the specified function at the specified rate.
 *
 * @return Pointer to RtxTimerThread structure if OK, NULL on error.
 *
 * This function creates a thread. The thread creates and initializes a Posix.1b timer
 * and then uses the sigwait function to periodiaclly run the specified function at
 * the specified rate.
 */
RtxTimerThread * 
rtx_timer_create_thread (
		  double initDelay,             /**< initial timer delay */
		  double period,                /**< subsequent timer delays */
		  int thPrio,                   /**< thread priority */
		  void (* func) (void *),       /**< timer handler function */
		  void * arg,                   /**< argument to the handler */
		  RtxTimerClockId clockId       /**< clock to use for timer */
		  )
{
    return rtx_error_null("rtx_timer_create_thread: not implemented on this platform");
}

/**
 * Destroy a thread/timer.
 *
 * @return 0 if OK, -1 on error.
 */
int 
rtx_timer_destroy_thread (
			  RtxTimerThread * perTh       /**< periodic thread */
			  )
{
    return rtx_error("rtx_timer_destroy_thread: not implemented on this platform");
}

#endif
