/***********************************************************************
 * 
 * CSIRO Autonomous Systems Laboratory
 * Queensland Centre for Advanced Technologies
 * PO Box 883, Kenmore, QLD 4069, Australia
 * http://www.ict.csiro.au/
 *  
 * Copyright (c) CSIRO 
 ***********************************************************************/

/**
 * \file time.c
 * \brief Time manipulation functions
 * \author Jonathan Roberts and Pavan Sikka
 */

#ifdef i486_darwin
#define DARWIN
#endif

#include	<stdio.h>
#ifdef	DARWIN
#include <sys/time.h>
#else
#include <time.h>
#endif

#include "rtx/defines.h"
#include "rtx/error.h"
#include "rtx/time.h"

static char rcsid[] RTX_UNUSED = "$Id: time.c 2274 2007-12-23 05:37:32Z roy029 $";

/**
 * get system time in seconds and nano seconds
 *
 * @return 0 on success, -1 on failure
 */
int
rtx_time_get (
	      RtxTime * tp      /**< RTX time structure */
	      )
{
#ifdef DARWIN
	struct timeval	tv;

	if (gettimeofday (&tv, NULL) == -1)
		return (rtx_error_errno ("rtx_time_get () failed: "));
	tp->seconds = tv.tv_sec;
	tp->nanoSeconds = tv.tv_usec*1000;
	return 0;
#else
    struct timespec ts;
    
    if (clock_gettime (CLOCK_REALTIME, &ts) == -1)
        return (rtx_error_errno ("rtx_time_get () failed: "));
    tp->seconds = ts.tv_sec;
    tp->nanoSeconds = ts.tv_nsec;
    return (0);
#endif
}

/**
 * get the time elapsed since tp0
 *
 * @return time elapsed as a double, a negative number indicates an error
 */
double
rtx_time_get_delta (
		    RtxTime * tp0
		    )
{
    RtxTime	ts;

    if (rtx_time_get (&ts) == -1) {
        rtx_error_errno ("rtx_time_get_delta () failed: ");
	return (-1.0);
    }

    return (rtx_time_subtract_to_double(&ts, tp0));
}

/**
 * convert RtxTime to a double.
 *
 * @param t RtxTime type
 * @return double precision representation of time in seconds 
 */
double
rtx_time_to_double (RtxTime *t)
{
        return ((double)t->seconds+(double)(t->nanoSeconds*1e-9));
}

/**
 * convert double to a timespec.
 *
 * @return t RtxTime type
 * @param double precision number of seconds.
 */
void
rtx_time_from_double (RtxTime *tp, double t)
{
        tp->seconds = (time_t) t;
        tp->nanoSeconds = (long)(1.0E9*(t-(int)t));
}

/**
 * add RtxTime types.
 *
 * @param t1 addend RtxTime
 * @param t2 addend RtxTime
 * @param result RtxTime representation of \c t1 + \c t2
 */
void
rtx_time_add (RtxTime *result, RtxTime *t1, RtxTime *t2)
{
	result->seconds = t1->seconds + t2->seconds;
	result->nanoSeconds = t1->nanoSeconds + t2->nanoSeconds;
	if (result->nanoSeconds >= 1000000000) {
	        result->nanoSeconds -= 1000000000;
		result->seconds++;
	}
}

/**
 * subtract RtxTime types.
 *
 * @param t1 a RtxTime
 * @param t2 a RtxTime
 * @param result RtxTime representation of \c t1 - \c t2
 */
void
rtx_time_subtract (RtxTime *result, RtxTime *t1, RtxTime *t2)
{
        result->nanoSeconds = t1->nanoSeconds - t2->nanoSeconds;
        result->seconds = t1->seconds - t2->seconds;
	if (result->nanoSeconds < 0) {
	        result->nanoSeconds +=  1000000000;
		result->seconds--;
	}
}

/**
 * subtract RtxTime types.
 *
 * @param t1 a RtxTime
 * @param t2 a RtxTime
 * @return \c t1 - \c t2 as a double
 */
double
rtx_time_subtract_to_double (RtxTime * t1, RtxTime * t2)
{
    RtxTime res;

    rtx_time_subtract (&res, t1, t2);
    return (rtx_time_to_double (&res));
}

/**
 * compare RtxTime.
 *
 * @param t1 a RtxTime.
 * @param t2 a RtxTime.
 * @return 0 if times are the same, 1 if t1 > t2, -1 otherwise.
 */
int
rtx_time_cmp(RtxTime *t1, RtxTime *t2)
{
	if (t1->seconds < t2->seconds)
	        return (-1);
	else if (t1->seconds > t2->seconds)
		return (1);
	else if (t1->nanoSeconds < t2->nanoSeconds)
		return (-1);
	else if (t1->nanoSeconds > t2->nanoSeconds)
		return (1);
        else
	        return (0);
}

/**
 * convert time to a string in date-time format
 *
 * @return 0 if OK, -1 on error
 */
int
rtx_time_conv_date_time (
			 char * buf,
			 int len,
			 char * prefix,
			 char * sep,
			 char * suffix,
			 RtxTime * ts
			 )
{
    char dateStr[128];
    char timeStr[128];

#ifdef DARWIN
    struct tm	lt;
    localtime_r(ts->seconds, &lt);
    strftime(timeStr, 128, "%H%M%S", &lt);
    strftime(dateStr, 128, "%Y%m%d", &lt);
#else
	time_t tsval = ts->seconds;
    strftime(timeStr, 128, "%H%M%S", localtime(&tsval));
    strftime(dateStr, 128, "%Y%m%d", localtime(&tsval));
#endif
    snprintf (buf, len, "%s%s%s%s%s", prefix, dateStr, sep, timeStr, suffix);
    return (0);
}

/**
 * convert time to a string in seconds-nanosenconds format
 *
 * @return 0 if OK, -1 on error
 */
int
rtx_time_conv_sec_nsec (
			char * buf,
			int len,
			char * prefix,
			char * sep,
			char * suffix,
			RtxTime * ts
			)
{
    snprintf (buf, len, "%s%ld%s%09ld%s", prefix, ts->seconds, sep, ts->nanoSeconds,
	      suffix);
    return (0);
}

