/***********************************************************************
 * 
 * CSIRO Autonomous Systems Laboratory
 * Queensland Centre for Advanced Technologies
 * PO Box 883, Kenmore, QLD 4069, Australia
 * http://www.ict.csiro.au/
 *  
 * Copyright (c) CSIRO 
 ***********************************************************************/

/**
 * \file thread.c
 * \brief Simpler interface to POSIX threads
 * \author Jonathan Roberts
 *
 * This module has been written to make using threads easy. Threads are
 * created using the rtx_thread_create() function. The user specifies a
 * function that will effectively 'be the thread'. The thread (and hence
 * the function) can be run with a specific scheduling policy and priority.
 * It is important that both the policy and priority are set appropriately.
 * Appropriate values are of course application specific. The user can
 * also specify a function to be called at thread cancelation time. This
 * may contain some code to cleanup some resources used by the main thread
 * function or may be used to signal another thread that this thread has
 * been cancelled.
 */

#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <pthread.h>
#include <errno.h>
#include <string.h>
#include <signal.h>

#include "rtx/defines.h"
#include "rtx/thread.h"
#include "rtx/sem.h"
#include "rtx/error.h"
#include "rtx/message.h"
#include "rtx/signal.h"

static char rcsid[] RTX_UNUSED = "$Id: thread.c 3086 2008-05-15 12:47:55Z roy029 $";

static void * rtx_thread_func(RtxThread *thread);
static void rtx_thread_cancel_func(RtxThread *thread);

static const int RTX_THREAD_STATUS_OK = 0;
static const int RTX_THREAD_STATUS_ERROR = 1;

static unsigned int rtx_thread_key_created = 0;
static pthread_key_t	rtx_thread_key;
static pthread_once_t   rtx_thread_once_init = {PTHREAD_ONCE_INIT};

/**
 * internal thread-specific data destructor
 */
static void
rtx_thread_key_destructor(void *p)
{
        if (p == NULL) {
  	        fprintf (stderr, "rtx_thread_key_destructor: NULL pointer\n");
		return;
	}
	free(p);
}

/**
 * create the key for holding thread-specific data
 */
static void
rtx_thread_init_once (void)
{
    int ret = -1;

    if ((ret = pthread_key_create((unsigned int *)&rtx_thread_key, 
				  rtx_thread_key_destructor)) != 0)
        fprintf (stderr, "rtx_thread_init_once: pthread_key_create() "
		 "failed: %s", strerror (errno));
    else
        rtx_thread_key_created = 1;
}


/** 
 * This function creates a thread with given attributes. The function blocks
 * until the thread is actually up-and-running. This is achieved using a
 * semaphore.
 *
 * The first argument specifies the scheduling policy of the thread. Possible
 * values are:
 * - RTX_THREAD_SCHED_OTHER
 * - RTX_THREAD_SCHED_FIFO
 * - RTX_THREAD_SCHED_RR
 * - RTX_THREAD_SCHED_INHERIT
 *
 * If set to RTX_THREAD_SCHED_INHERIT, then both the scheduling policy AND the
 * priority are inherited from the calling thread. Note that the default is
 * RTX_THREAD_SCHED_OTHER.
 *
 * If stackSize is set to 0, the function uses the system default stack value.
 *
 * If prio is set to zero, the function uses the calling threads priority value.
 *
 * If policy, threadPrio and stackSize are set to zero then the system defaults
 * are used for everything.
 *
 * Possible values of cancelType are:
 * - RTX_THREAD_CANCEL_DISABLE - cannot be cancelled
 * - RTX_THREAD_CANCEL_ASYNCHRONOUS - cancelled at any point
 * - RTX_THREAD_CANCEL_DEFERRED - cancelled when the thread issues 
 * a rtx_thread_test_cancel()
 *
 * Note that real-time signals are blocked by default when creating a thread.
 * Please use the rtx_signal_unblock_realtime() function to unblock them if
 * you need to.
 *
 * @return Pointer to an RtxThread structure on success, NULL on error
 *
 * @note To cast the function pointer to suit use \c (void * (*)(void*)) myfunc
*/
RtxThread *
rtx_thread_create(
	const char	*name,		/**< Name of the thread */
	int		debug,		/**< Debugging flag */
	RtxThreadSched	policy,		/**< Scheduling policy */
	RtxThreadPrio 	prio,		/**< The priority of the thread */
	int 		stackSize,	/**< Stack size in KB */
	RtxThreadCancel cancelType,	/**< Cancel type */
	void 	*(*func) (void *), 	/**< Pointer to thread function */
	void		*arg,		/**< Argument to thread function */
	void 	(*cancel_func) (void *),/**< Pointer to cancel function */
	void		*cancelArg	/**< Argument to cancel function */
	)
{
	RtxThread	*thread;
	pthread_attr_t  thread_attr, *tap;
	int             retval;
	int		schedPolicy = 0;
	int		threadPrio;

	pthread_once (&rtx_thread_once_init, rtx_thread_init_once);

	/*
	 * Allocate memory for thread structure
	 */
	if ((thread = (RtxThread *)calloc(1, sizeof(RtxThread))) == NULL)
		return (rtx_error_null ("rtx_thread_create: no memory"));

	/*
	 * Load thread attributes
	 */
	if ((thread->name = strdup(name)) == NULL)
		return (rtx_error_null ("rtx_thread_create: no memory"));
	thread->debug = debug;
	thread->initErrors = 0;
	thread->arg = arg;
	thread->func = func;
	thread->cancelArg = cancelArg;
	thread->cancel_func = cancel_func;
	thread->cancelType = cancelType;
	thread->cancelled = 0;
	thread->detached  = 0;
	thread->errorFlags = rtx_error_flags_get ();

	if (thread->debug>1) {
		rtx_message("Creating tread '%s'",name);
	}

	/*
	 * Work out scheduling policy
	 */
	switch (policy) {
	        case RTX_THREAD_SCHED_OTHER :
	                schedPolicy = SCHED_OTHER;
			break;
   	        case RTX_THREAD_SCHED_FIFO :
		        schedPolicy = SCHED_FIFO;
			break;
  	        case RTX_THREAD_SCHED_RR :
		        schedPolicy = SCHED_RR;
			break;
	        case RTX_THREAD_SCHED_INHERIT :
		        schedPolicy = SCHED_OTHER;
			break;
                default :
		        rtx_error ("rtx_thread_create: invalid scheduling policy %d", policy);
			free (thread);
			return NULL;
	}

	switch (prio) {
		case RTX_THREAD_PRIO_MIN:
			threadPrio = sched_get_priority_min(schedPolicy);
			break;
		case RTX_THREAD_PRIO_MAX:
			threadPrio = sched_get_priority_max(schedPolicy);
			break;
		default:
			threadPrio = rtx_thread_prio_to_system(prio);
			break;
	}


	if ((prio == 0) && (stackSize == 0) && (schedPolicy == 0)) {
		tap = NULL;
	} else {
		if ((retval = pthread_attr_init(&thread_attr))) {
			rtx_error_errno2 (retval, ("rtx_thread_create: pthread_attr_init: "));
			free (thread);
			return NULL;
		}
		tap = &thread_attr;
		if (thread->debug>1) {
			rtx_message("Attr initialised for thread '%s'",name);
		}

		if (stackSize > 0)
			pthread_attr_setstacksize(tap, stackSize * 1024);


		if (policy == RTX_THREAD_SCHED_INHERIT) {
			if ((retval = pthread_attr_setinheritsched(tap, PTHREAD_INHERIT_SCHED))) {
				rtx_error_errno2 (retval, "rtx_thread_create: pthread_attr_setinheritsched: ");
				free (thread);
				return NULL;
			}
		} else {
	       		struct sched_param	sp;
			int			max, min;
		
			/*
			 * Get calling threads priority 
			 */
			if (prio == 0)
				threadPrio = rtx_thread_prio_to_system(rtx_thread_prio_get());

			max = sched_get_priority_max(schedPolicy);
			min = sched_get_priority_min(schedPolicy);
			if ((threadPrio > max) || (threadPrio < min)) {
				free (thread);
				return (rtx_error_null ("rtx_thread_create: illegal priority %d (%d to %d)", threadPrio, min, max));
			}

			if ((retval = pthread_attr_setinheritsched(tap, PTHREAD_EXPLICIT_SCHED))) {
				rtx_error_errno2 (retval, "rtx_thread_create: pthread_attr_setinheritsched: ");
				free (thread);
				return NULL;
			}

			if ((retval = pthread_attr_setschedpolicy(tap, schedPolicy))) {
				rtx_error_errno2 (retval, "rtx_thread_create: pthread_attr_setschedpolicy: ");
				free (thread);
				return NULL;
			}
			sp.sched_priority = threadPrio;

			if ((retval = pthread_attr_setschedparam(tap, &sp))) {
				rtx_error_errno2 (retval, "rtx_thread_create: pthread_attr_setschedparam: ");
				free (thread);
				return NULL;
			}
		}
	}
	if (thread->debug>1) {
		rtx_message("Attribute setup for thread '%s'",name);
	}
	
	/*
	 * Create thread start and stop semaphore
	 */
	if (rtx_sem_init(&(thread->sem), 0, 0) == NULL) {
	        rtx_error ("rtx_thread_create: rtx_sem_init() failed");
		free (thread);
		return NULL;
	}
	if (thread->debug>1) {
		rtx_message("Ready to launch thread '%s'",name);
	}
	

	/*
	 * Launch the thread
	 */
	retval = pthread_create(&thread->id, tap, (void *(*)(void *)) rtx_thread_func, thread);
	//retval = pthread_create(&thread->id, NULL, (void *(*)(void *)) rtx_thread_func, thread);
	if (retval) {
		rtx_error_errno2(retval, "rtx_thread_create: pthread_create: ");
		free (thread);
		return NULL;
	}

	/*
	 * Wait to be signalled (via the semaphore) by the thread that it is
	 * alive and well
	 */
	if (rtx_sem_wait_ignore_eintr (&(thread->sem))) {
		rtx_error("rtx_thread_create: rtx_sem_wait_ignore_eintr() failed");
		free (thread);
		return NULL;
	}

	/*
	 * Destroy the semaphore
	 */
	if (rtx_sem_destroy (&(thread->sem))) {
        	rtx_error ("rtx_thread_create: rtx_sem_destroy() failed");
		free (thread);
		return NULL;
	}

	/*
	 * Check for any thread initialisation errors
	 */
	if (thread->initErrors) {
 	        rtx_error("rtx_thread_create: errors in thread initialisation "
			  "(0x%08X)", (unsigned) thread->initErrors);
		free (thread);
		return NULL;
	}

	return thread;
}

/**
 * This function detach a thread (a join will not be needed to free the
 * resources). It also set the detached flag of the structure
 * */
int 
rtx_thread_detach(RtxThread * thread)
{
	int retval;
	if (thread->detached) return 0;

	retval = pthread_detach(thread->id);
	switch (retval) {
		case 0:
		case EINVAL:  /* already detached */
		case ESRCH:   /* thread does not exist (probably terminated) */
			break;
		default :
		  return rtx_error_errno2(retval, "rtx_thread_detach failed");
	}
	thread->detached = 1;
	if (thread->debug)
		rtx_message_routine("%s: detached", thread->name);
	return 0;
}

/**
 * This function destroy a thread. rtx_thread_join can NOT be used
 * on it afterward. If you want to wait for the thread to terminate,
 * use \sa rtx_thread_destroy_sync
 * 
 * @return 0 on success, -1 on error
 */
int
rtx_thread_destroy(
	RtxThread	*thread   /**< Pointer to the thread structure */
	)
{
	int	retval;

	if (thread == NULL)
		return rtx_error("rtx_thread_destroy: NULL thread pointer");

	if (thread->debug)
		rtx_message_routine("%s: rtx_thread_destroy", thread->name);

	/*
	 * Check that thread can be cancelled
	 */
	if (thread->cancelType == RTX_THREAD_CANCEL_DISABLE)
		return rtx_error("rtx_thread_destroy: cannot destroy thread because its cancel state is set to RTX_THREAD_CANCEL_DISABLE");

	/*
	 * First detach the thread since we will not care about its termination
	 * otherwise we would have used rtx_thread_destroy_sync 
	 * */
	if (rtx_thread_detach(thread)) {
		rtx_error_flush("rtx_thread_destroy: rtx_thread_detach failed");
	}

	if (thread->cancelled) {
		/* The thread has already terminated, but not freed it's data
		 * otherwise thread would not be accessible */
		if (thread->debug)
			rtx_message_routine("%s: rtx_thread_destroy: freeing detached thread data", thread->name);
		free(thread->name);
		free(thread);
	} else {
		/*
		 * Kill the thread 
		 */
		retval = pthread_cancel(thread->id);
		switch (retval) {
			case ESRCH:
				if (thread->debug)
					rtx_message_routine("%s: rtx_thread_destroy: freeing detached thread data", thread->name);
				free(thread->name);
				free(thread);
				/* fall through */
			case 0:
				/* in this case, we don't need to free, since this will be done
				 * in the thread cleanup function */
				break;
			default:
				free(thread->name);
				free(thread);
				return rtx_error_errno2 (retval, "rtx_thread_destroy: pthread_cancel: ");
		}
	}

	return 0;
}

/**
 * This function destory a thread.  The function blocks until the thread 
 * has actually been cancelled. This is achieved using pthread_join.
 * 
 * @return 0 on success, -'ve on error
 */
int
rtx_thread_destroy_sync(
	RtxThread	*thread   /**< Pointer to the thread structure */
	)
{
	int	retval;
	void	*status;

	if (thread == NULL)
		return rtx_error("rtx_thread_destroy_sync: NULL thread pointer");

	if (thread->debug)
		rtx_message ("rtx_thread_destroy_sync[%s]: starting", thread->name);
	/*
	 * Check that thread can be cancelled
	 */
	if (thread->cancelType == RTX_THREAD_CANCEL_DISABLE)
		return rtx_error("rtx_thread_destroy_sync: cannot destroy thread because its cancel state is set to RTX_THREAD_CANCEL_DISABLE");

	/*
	 * Kill the thread 
	 */
	if (thread->debug)
		rtx_message ("rtx_thread_destroy_sync[%s]: destroying %d!", thread->name, thread->id);
	if ((retval = pthread_cancel(thread->id))) {
		if (thread->debug)
			rtx_message ("rtx_thread_destroy_sync: pthread_cancel failed with code %d ",retval);
		switch (retval) {
			case ESRCH:
				if (thread->debug)
					rtx_message ("rtx_thread_destroy_sync: pthread_cancel failed with ESRCH that's ok ");
				/** This thread was already dead, just use join to release the
				 * associated resources, we don't care about the result value
				 * since at this point nothing can be done about it.  **/
				pthread_join(thread->id, &status);
				break;
			default:
				free(thread->name);
				free(thread);
				return rtx_error_errno2 (retval, "rtx_thread_destroy_sync: pthread_cancel: ");
		}
	} else {
		/*
		 * Wait for thread to signal that it has been cancelled
		 */
		if (thread->debug)
			rtx_message ("rtx_thread_destroy_sync[%s]: waiting%d", thread->name, thread->id);
		if ((retval = pthread_join(thread->id, &status))) {
			free(thread->name);
			free(thread);
			return rtx_error_errno2(retval, "rtx_thread_destroy_sync: pthread_join: ");
		}
		if (thread->debug)
			rtx_message ("rtx_thread_destroy_sync[%s]: done%d", thread->name, thread->id);
	}

	free(thread->name);
	free(thread);

	return 0;
}

/**
 * This function waits for a thread to exit and then cleans up. The function blocks until
 * the thread 
 * has actually finished. This is achieved using pthread_join.
 * 
 * @return 0 on success, -'ve on error
 */
int
rtx_thread_join(
	RtxThread	*thread   /**< Pointer to the thread structure */
	)
{
	int	retval;
	void	*status;

	if (thread == NULL)
		return rtx_error("rtx_thread_join: NULL thread pointer");

	if (thread->debug)
	        rtx_message ("rtx_thread_join[%s]: starting", thread->name);
	/*
	 * Wait for thread to signal that it has been cancelled
	 */
	if (thread->debug)
	        rtx_message ("rtx_thread_join[%s]: waiting%d", thread->name, thread->id);
	if ((retval = pthread_join(thread->id, &status))) {
	        return rtx_error_errno2(retval, "rtx_thread_join: pthread_join: ");
	}
	if (thread->debug)
	        rtx_message ("rtx_thread_join[%s]: done%d", thread->name, thread->id);

	/* done in the cancellation wrapper */
	/*
	free(thread->name);
	free(thread);
	*/

	return 0;
}

/**
 * This function is a wrapper around the pthread_setcancel() functions
 * and can be used to specify the cancellation propertries of the thread.
 *
 * \return 0 on success, -1 on error
 */
int
rtx_thread_setcancel(RtxThreadCancel cancelType)
{
        int newType = 0, newState = 0, oldType = 0, oldState = 0;
	int retval, errs = 0;

	/*
	 * Work out cancel state and type
	 */
	switch (cancelType) {
	case RTX_THREAD_CANCEL_DISABLE:
		newType = PTHREAD_CANCEL_DEFERRED;
		newState = PTHREAD_CANCEL_DISABLE;
		break;
	case RTX_THREAD_CANCEL_DEFERRED:
		newType = PTHREAD_CANCEL_DEFERRED;
		newState = PTHREAD_CANCEL_ENABLE;
		break;
	case RTX_THREAD_CANCEL_ASYNCHRONOUS:
		newType = PTHREAD_CANCEL_ASYNCHRONOUS;
		newState = PTHREAD_CANCEL_ENABLE;
		break;
	}

	/*
	 * Set the cancel type of the thread
	 */
	retval = pthread_setcanceltype(newType, &oldType);
	if (retval) {
        	rtx_error_errno2 (retval, "rtx_thread_setcancel: pthread_setcanceltype: ");
		errs++;
	}

	/*
	 * Set the cancel state of the thread
	 */
	retval = pthread_setcancelstate(newState, &oldState);
	if (retval) {
        	rtx_error_errno2 (retval, "rtx_thread_setcancel: pthread_setcancelstate: ");
		errs++;
	}
	if (errs)
	        return (-1);
	if (oldState == PTHREAD_CANCEL_DISABLE)
	    return (RTX_THREAD_CANCEL_DISABLE);
	else if (oldType == PTHREAD_CANCEL_DEFERRED)
	    return (RTX_THREAD_CANCEL_DEFERRED);
	else 
	    return (RTX_THREAD_CANCEL_ASYNCHRONOUS);
	rtx_error ("rtx_therad_setcancel: funny prev state [%d, %d]",
		   oldState, oldType);
	return (-1);
}

/**
 * This function is a simple wrapper around the pthread_testcancel() function
 * and can be used to specify a cancelation point in a thread that has been
 * given an RTX_THREAD_CANCEL_DEFERRED cancelation type.
 *
 * \return Nothing
 */
void
rtx_thread_testcancel(void)
{
	pthread_testcancel();
}

/**
 * This function looks to see if a thread has been cancelled
 *
 * \return 0 if not cancelled, 1 if it has been cancelled
 */
int
rtx_thread_cancelled(
	RtxThread	*thread	/**< Pointer to the thread structure */
	)
{
	return (thread->cancelled);
}

/**
 * This function sets the user variable in the thread structure
 *
 * \return Nothing
 */
void
rtx_thread_set_var(
	RtxThread	*thread,	/**< Pointer to the thread structure */
	void		*var		/**< Pointer to a variable */
	)
{
	thread->var = var;
}

/**
 * This function gets the user variable in the thread structure
 *
 * \return Nothing
 */
void
rtx_thread_get_var(
	RtxThread	*thread,	/**< Pointer to the thread structure */
	void		*var		/**< Pointer to a variable */
	)
{
	var = thread->var;
}

/**
 * get current thread stack usage
 *
 * \return number of bytes
 */
unsigned int
rtx_thread_get_stack_size(void)
{
    unsigned int * bt;
    unsigned int tp;

    if (rtx_thread_key_created) {
        if ((bt = pthread_getspecific (rtx_thread_key)) != NULL) {
	    tp = (unsigned int) &tp;
	    if (tp > (* bt))
	        return ((unsigned int) (tp - (* bt)));
	    else
	        return ((unsigned int) ((* bt) - tp));
	}
    }
    return (0);
}

/**
 * This function converts an RtxThreadPrio value to a system prio value
 *
 * \return System priority
 */
int 
rtx_thread_prio_to_system(
	RtxThreadPrio	prio	/*!< Priority to convert */
	)
{
#ifdef i486_linux
	return prio;
#elif sparc_solaris
	return 0;
#else
	return prio;
#endif
}

/**
 * This function converts an RtxThreadPrio value to a system prio value
 *
 * \return System priority
 */
RtxThreadPrio 
rtx_thread_system_to_prio(
	int	prio	/*!< System priority to convert */
	)
{
	return prio;
}

/**
 * This function gets the priority of the calling thread
 *
 * \return Zero or positive priority on success, -1 on error
 */
RtxThreadPrio
rtx_thread_prio_get(
	)
{
        int                     retval;
	int			policy;
	struct sched_param	sp;

	if ((retval = pthread_getschedparam(pthread_self(), &policy, &sp)))
	  return rtx_error_errno2(retval, "rtx_thread_prio_get: pthread_getschedparam: ");

	return (rtx_thread_system_to_prio(sp.sched_priority));
}

/**
 * This function sets the priority of the calling thread
 *
 * \return Zero or positive priority on success, -1 on error
 */
int
rtx_thread_prio_set(
	RtxThreadPrio	prio	/*!< Priority */
	)
{
        int                     retval;
	int			policy;
	struct sched_param	sp;

	if ((retval = pthread_getschedparam(pthread_self(), &policy, &sp)))
	  return rtx_error_errno2(retval, "rtx_thread_prio_set: pthread_getschedparam: ");

	switch (prio) {
		case RTX_THREAD_PRIO_MIN:
			sp.sched_priority = sched_get_priority_min(policy);
			break;
		case RTX_THREAD_PRIO_MAX:
			sp.sched_priority = sched_get_priority_max(policy);
			break;
		default:
			sp.sched_priority = rtx_thread_prio_to_system(prio);
			break;
	}


	if ((retval = pthread_setschedparam(pthread_self(), policy, &sp)))
	  return rtx_error_errno2(retval, "rtx_thread_prio_set: pthread_setschedparam: ");

	return 0;
}

/**
 * This is the thread wrapper function. It initialises the thread before the
 * user supplied function is called
 *
 * \return Nothing
 */
static void *
rtx_thread_func(
	RtxThread	*thread   /**< Pointer to the thread structure */
	)
{
        int     z;   /* please declare all local variables
			  * after this
			  */
	int	newType = PTHREAD_CANCEL_ENABLE;
	int	newState = PTHREAD_CANCEL_DEFERRED;
	int	oldType, oldState, retval;
	unsigned int * p;

	if (thread->debug>1)
		rtx_message("rtx_thread_func: starting");


	thread->stackBottom = (void *) &z;
	if (rtx_thread_key_created) {
	    if ((p = calloc (1, sizeof (unsigned int))) != NULL) {
	        * p = (unsigned int) &z;
		pthread_setspecific (rtx_thread_key, p);
	    }
	}

	if (thread->debug)
		rtx_message_routine("%s: rtx_thread_func: thread started", thread->name);

	/*
	 * Initialise error handler
	 */
	if (rtx_error_init(thread->name, thread->errorFlags, NULL)) {
		rtx_error("%s: rtx_thread_func: rtx_error_init: failed", thread->name);
		thread->initErrors |= (1 << 0);
	}
	/*
	 * Block kill signals
	 */
	if (rtx_signal_block (SIGINT)) {
		rtx_error("rtx_thread_func: rtx_signal_block: failed");
		thread->initErrors |= (1 << 1);
	}
	if (rtx_signal_block (SIGQUIT)) {
		rtx_error("rtx_thread_func: rtx_signal_block: failed");
		thread->initErrors |= (1 << 2);
	}
	if (rtx_signal_block (SIGTSTP)) {
		rtx_error("rtx_thread_func: rtx_signal_block: failed");
		thread->initErrors |= (1 << 3);
	}
	if (rtx_signal_block (SIGTERM)) {
		rtx_error("rtx_thread_func: rtx_signal_block: failed");
		thread->initErrors |= (1 << 4);
	}
	if (rtx_signal_block (SIGUSR1)) {
		rtx_error("rtx_thread_func: rtx_signal_block: failed");
		thread->initErrors |= (1 << 5);
	}
	if (rtx_signal_block (SIGUSR2)) {
		rtx_error("rtx_thread_func: rtx_signal_block: failed");
		thread->initErrors |= (1 << 6);
	}
#ifdef __Lynx__
	if (rtx_signal_block (SIGBRK)) {
		rtx_error("rtx_thread_func: rtx_signal_block: failed");
		thread->initErrors |= (1 << 7);
	}
#endif

	/*
	 * Block real-time signals
	 */
#ifndef	i486_darwin
	if (rtx_signal_block_realtime()) {
		rtx_error("rtx_thread_func: rtx_signal_block: failed");
		thread->initErrors |= (1 << 8);
	}
#endif

	/*
	 * Work out cancel state and type
	 */
	switch (thread->cancelType) {
	case RTX_THREAD_CANCEL_DISABLE:
		newType = PTHREAD_CANCEL_DEFERRED;
		newState = PTHREAD_CANCEL_DISABLE;
		break;
	case RTX_THREAD_CANCEL_DEFERRED:
		newType = PTHREAD_CANCEL_DEFERRED;
		newState = PTHREAD_CANCEL_ENABLE;
		break;
	case RTX_THREAD_CANCEL_ASYNCHRONOUS:
		newType = PTHREAD_CANCEL_ASYNCHRONOUS;
		newState = PTHREAD_CANCEL_ENABLE;
		break;
	}

	/*
	 * Set the cancel type of the thread
	 */
	if (thread->debug)
		rtx_message_routine("%s: rtx_thread_func: setting cancel type", thread->name);
	retval = pthread_setcanceltype(newType, &oldType);
	if (retval) {
        	rtx_error_errno2 (retval, "rtx_thread_func: pthread_setcanceltype: ");
		thread->initErrors |= (1 << 9);
	}

	/*
	 * Set the cancel state of the thread
	 */
	if (thread->debug)
		rtx_message_routine("%s: rtx_thread_func: setting cancel state", thread->name);
	retval = pthread_setcancelstate(newState, &oldState);
	if (retval) {
        	rtx_error_errno2 (retval, "rtx_thread_func: pthread_setcancelstate: ");
		thread->initErrors |= (1 << 10);
	}

	/*
	 * Register the cancel function
	 */
	if (thread->debug)
		rtx_message_routine("%s: rtx_thread_func: registering cancel function", thread->name);
	pthread_cleanup_push((void (*)(void *))rtx_thread_cancel_func, thread);

	/*
	 * Signal the create function (via the semaphore) that this thread
	 * is running
	 */
	if (thread->debug)
		rtx_message_routine("%s: rtx_thread_func: posting semaphore", thread->name);
	if (rtx_sem_post (&(thread->sem)))
	    rtx_error("rtx_thread_func: rtx_sem_post() failed");

	/*
	 * Call the user thread function
	 */
	if (thread->debug)
		rtx_message_routine("%s: rtx_thread_func: calling user function", thread->name);
	if (thread->func != NULL)
		(*thread->func)(thread->arg);
	if (thread->debug)
		rtx_message_routine("%s: rtx_thread_func: user function terminated", thread->name);

	/* Pop the cleanup function as per the Pthreads standard */
	pthread_cleanup_pop (1);
	pthread_exit ((void *) &RTX_THREAD_STATUS_OK);
	return ((void *) &RTX_THREAD_STATUS_OK);
}

/**
 * This is the thread cancel wrapper function. It runs the user supplied 
 * cancel function and then posts the semaphore to signal that the thread
 * has been cancelled
 *
 * \return Nothing
 */
static void
rtx_thread_cancel_func(
	RtxThread	*thread   /**< Pointer to the thread structure */
	)
{

	/*
	* The verbose output is commented out as it causes a segfault
	* under Linux. Need to maintain a watching brief as we move
	* up the NPTL versions.
	*/
	if (thread->cancelled) {
		rtx_message_routine("%s: rtx_thread_cancel_func: ignoring second call", thread->name);
		return;
	}

	if (thread->debug)
		rtx_message_routine("%s: rtx_thread_cancel_func: canceling...", thread->name);

	/*
	 * Call the user cancel function
	 */
	if (thread->cancel_func != NULL) {
		if (thread->debug)
			rtx_message_routine("%s: rtx_thread_cancel_func: calling user thread cancel function", thread->name);
		/* WARNING: we can have a race condition here if the client triggers
		 * a rtx_thread_destroy */
		(*thread->cancel_func)(thread->cancelArg);
	}
	if (thread->debug)
		rtx_message_routine("%s: rtx_thread_cancel_func: cancelled", thread->name);

	thread->cancelled = 1;
	if (thread->detached) {
		if (thread->debug)
			rtx_message_routine("%s: rtx_thread_cancel_func: freeing detached thread data", thread->name);
		free(thread->name);
		free(thread);
	}
}


/** 
 * Create a thread with given attributes.
 *
 * If threadPrio and stackSize are both specified to be 0, the function
 * uses the system default stack values for both priority and stack size.
 *
 * @return 0 on success, -1 on error
 *
 * @note To cast the function pointer to suit use \c (void * (*)(void*)) myfunc
*/
int
rtx_thread_create_legacy(pthread_t * threadId,	/**< a pointer to the thread ID */

		  int threadPrio,	/**< the priority of the thread */

		  int stackSize,	/**< stack size in KB */

		  void *          (*func) (void *),
					    /**< the function name */

		  void *arg		/**< an argument to the function */
	)
{
	pthread_attr_t  thread_attr, *tap;
	int             retval;

	if ((threadPrio > 0) || (stackSize > 0)) {

  	    if ((retval = pthread_attr_init(&thread_attr)))
	        return ( rtx_error_errno2 (retval, 
			     ("rtx_thread_create_legacy: pthread_attr_init: ")));
	    tap = &thread_attr;

	    if (stackSize > 0)
	        pthread_attr_setstacksize(tap, stackSize * 1024);

	    if (threadPrio > 0) {
	        struct sched_param sp;
		
		if ((threadPrio > sched_get_priority_max(SCHED_FIFO))
		    || (threadPrio <
			sched_get_priority_min(SCHED_FIFO)))
		    return (rtx_error
			    ("rtx_thread_create_legacy: illegal priority %d (%d to %d)",
			     threadPrio,
			     sched_get_priority_min(SCHED_FIFO),
			     sched_get_priority_max(SCHED_FIFO)));
	        if ((retval = pthread_attr_setinheritsched(tap,
				PTHREAD_EXPLICIT_SCHED)))
		    return (rtx_error_errno2 (retval, 
				    "rtx_thread_create_legacy: "
			            "pthread_attr_setinheritsched() "));
		if ((retval = pthread_attr_setschedpolicy(tap, 
							  SCHED_FIFO)))
  		    return (rtx_error_errno2 (retval,
		     		  "rtx_thread_create_legacy: "
		       		  "pthread_attr_setschedpolicy() "));
		sp.sched_priority = threadPrio;
#ifdef sparc_solaris
		sp.sched_nice = threadPrio;
		sp.sched_nicelim = threadPrio;
#endif
		if ((retval = pthread_attr_setschedparam(tap, &sp)))
		    return (rtx_error_errno2 (retval, "rtx_thread_"
			     "create: pthread_attr_setschedparam()"));
	    } else {
		if ((retval = pthread_attr_setinheritsched(tap,
				             PTHREAD_INHERIT_SCHED)))
		    return (rtx_error_errno2 (retval,
			      	      "rtx_thread_create_legacy: "
			       	      "pthread_attr_setinheritsched() "));
	    }
	} else {
	    tap = NULL;
	}
	
	retval = pthread_create(threadId, tap, (void *(*)(void *)) func, arg);
	if (retval)
	    return (rtx_error_errno2(retval, "rtx_thread_create_legacy: "
				     "pthread_create: "));
	return 0;
}

/**
 * Destory a thread.
 * 
 * @return 0 on success, -1 on error
 */
int

rtx_thread_destroy_legacy(pthread_t * threadId	   /**< a pointer to the thread ID */
	)
{
    int retval;

    if ((retval = pthread_cancel(*threadId)))
        return (rtx_error_errno2 (retval,
				  "rtx_thread_destroy_legacy: pthread_cancel: "));
    return 0;
}
