/***********************************************************************
 * 
 * CSIRO Autonomous Systems Laboratory
 * Queensland Centre for Advanced Technologies
 * PO Box 883, Kenmore, QLD 4069, Australia
 * http://www.ict.csiro.au/
 *  
 * Copyright (c) CSIRO 
 ***********************************************************************/

/**
 * \file sync.c
 * \brief A simple interface to synchronization using condition variables
 * \author P Sikka
 */

#include <stdlib.h>
#include <unistd.h>
#include <pthread.h>
#include <string.h>
#include <errno.h>

#include "rtx/defines.h"
#include "rtx/time.h"
#include "rtx/timer.h"
#include "rtx/error.h"
#include "rtx/mutex.h"
#include "rtx/sync.h"

static char rcsid[] RTX_UNUSED = "$Id: sync.c 2274 2007-12-23 05:37:32Z roy029 $";

/**
 * Initialize a RtxSync structure.
 *
 * @return Pointer to a RtxSync struct on success, NULL on error
 *
 * It initializes a RtxSync structure which consists of
 * a mutex, a condition variable and a count variable that is
 * used as the condition associated with the condition variable.
 */
RtxSync *
rtx_sync_init (
        RtxSync * cp,      /**< pointer to sync struct */
	int mutexAttrs,    /**< Attributes for the mutex */
	int prio,          /**< Priority ceiling (if used) */
	int condAttrs      /**< Attributes for the cond var */
	)
{
	RtxSync			*p = cp;
	int 			retval;
	pthread_condattr_t	attr, * attrP = NULL;

	if (p == NULL) {
  	    if ((p = (RtxSync *)calloc(1, sizeof(RtxSync))) == NULL)
		return rtx_error_null("rtx_sync_init: calloc: no memory");
	    p->memAlloc = 1;
	} else
	    p->memAlloc = 0;

	if ((rtx_mutex_init (&p->mutex, mutexAttrs, prio)) == NULL)
		return (rtx_error_null ("rtx_sync_init: rtx_mutex_init() failed"));

	if (condAttrs != RTX_MUTEX_DEFAULT) {
	    attrP = &attr;
	    if ((retval = pthread_condattr_init (attrP))) {
	        rtx_error_errno2 (retval, "rtx_cond_init: pthread_condattr_init: ");
		return NULL;
	    }
	    if (condAttrs & RTX_MUTEX_PROCESS_SHARED) {
#ifdef _POSIX_THREAD_PROCESS_SHARED
	        if ((retval = pthread_condattr_setpshared (attrP, 
					      PTHREAD_PROCESS_SHARED))) {
		    rtx_error_errno2 (retval, "rtx_cond_init: pthread_condattr_setpshared: ");
		    return NULL;
		}
#else
		return (rtx_error_null ("rtx_cond_init: RTX_MUTEX_PROCESS_SHARED "
					"not supported for condition variables"));
#endif
	    }
	}
	if ((retval = pthread_cond_init (&(p->cond), attrP))) {
	    rtx_error_errno2 (retval, "rtx_cond_init: pthread_cond_init: ");
	    return NULL;
	}
    
	p->count = 0;
    
	return p;
}

/**
 * Destroy a RtxSync structure.
 *
 * @return 0 if OK, -1 on error
 *
 * This function provides a simple wrapper around PTHREADS condition 
 * variables. It destroys the mutex and the condition variable associated
 * with the RtxSync structure.
 */
int 
rtx_sync_destroy (
	RtxSync	*p    /**< Pointer to RtxSync struct to be destroyed */
	)
{
    int	retval = 0, errs = 0, destroyMutex = 1;  

    if ((retval = pthread_mutex_trylock (&(p->mutex.mutex))) == 0) {
        if ((retval = pthread_mutex_unlock (&(p->mutex.mutex))) != 0) {
	    destroyMutex = 0;
	    rtx_error ("rtx_sync_destroy: failed to unlock mutex");
	    errs++;
	}
    } else {
        if (retval == EBUSY) {
	    rtx_error ("rtx_sync_destroy: mutex locked");
	    errs++;
	    destroyMutex = 0;
	} else {
	    rtx_error_errno2 (retval, "rtx_sync_destroy: "
			      "pthread_mutex_trylock() failed");
	    errs++;
	    destroyMutex = 0;
	}
    }	  

    if (destroyMutex)
        if (rtx_mutex_destroy (&p->mutex) == -1) {
	    rtx_error_errno2 (retval, "rtx_sync_destroy: "
			      "rtx_mutex_destroy() failed");
	    errs++;
	}
    if ((retval = pthread_cond_destroy (&(p->cond)))) {
        rtx_error_errno2 (retval, "rtx_sync_destroy: "
				  "pthread_cond_destroy() failed");
	errs++;
    }
    if (p->memAlloc)
        free (p);
    if (errs)
        return (-1);
    return (0);
}

/**
 * Unlock the mutex associated with the sync struct.
 *
 * @return 0 if OK, -1 on error.
 *
 * This function is meant to be used in thread cleanup functions
 * to unlock a mutex that may have been re-acquired due to the
 * waiting thread being cancelled while in pthread_cond_wait().
 */
int 
rtx_sync_unlock (
		 RtxSync * p
		 )
{
  int	retval = 0, retval1 = 0;

    if ((retval = pthread_mutex_trylock (&(p->mutex.mutex))) == 0) {
        if ((retval = pthread_mutex_unlock (&(p->mutex.mutex))) != 0) {
	    return (rtx_error_errno2 (retval, "rtx_sync_unlock: "
				      "failed to unlock mutex"));
	} else {
	    return (0);
	}
    } else {
        if (retval == EBUSY) {
	    if ((retval1 = pthread_mutex_unlock (&(p->mutex.mutex)))
		    != 0) {
	        return (rtx_error_errno2 (retval1, "rtx_sync_unlock: "
					  "failed to unlock mutex"));
	    } else {
	        return (0);
	    }
	} else {
	    return (rtx_error_errno2 (retval1, "rtx_sync_unlock: "
				      "pthread_mutex_trylock() failed"));
	}
    }	 
    return (0);
}

/**
 * Wait for a condition variable to be signalled.
 *
 * @return 0 if OK, -1 on error.
 *
 * This function waits on the condition variable until the associated
 * condition is satisfied (happens when the count value is incremented.
 */
int
rtx_sync_wait (
	RtxSync	*p    /**< Pointer to RtxSync struct */
	)
{
        int retval, errs = 0, count;

	if ((retval = pthread_mutex_lock (&(p->mutex.mutex))))
        	return (rtx_error_errno2 (retval, "rtx_sync_wait: "
				  "pthread_mutex_lock failed"));
	count = p->count;
	while (count == p->count) {
	        if ((retval = pthread_cond_wait (&(p->cond), &(p->mutex.mutex)))) {
				rtx_error_errno2 (retval, "rtx_sync_wait: pthread_cond_wait: ");
		    		rtx_timer_sleep (0.01);
		    		errs++;
		}
	}

	if ((retval = pthread_mutex_unlock (&(p->mutex.mutex))))
		return (rtx_error_errno2 (retval, "rtx_sync_wait: pthread_mutex_unlock: "));
	if (errs)
        	return (-1);
	
	return (0);
}

/**
 * Wait (with timeout) for a condition variable to be signalled.
 *
 * @return 0 if OK, 1 if timed out, and -1 on error.
 *
 * This function waits on the condition variable until the associated
 * condition is satisfied (which happens when the associated counter
 * is incremented. It is different from the normal
 * wait in that a timeout for the wait is specified.
 */
int
rtx_sync_timedwait (
	RtxSync 	*p,	/**< Pointer to RtxSync struct */
	double		timeout /**< Timeout */
    )
{
	int		retval = 0, timedout = 0, errs = 0, count;
	RtxTime		tsEntry, tsNow, rtxtsExit;
	struct timespec	tsExit;

	rtx_time_get (&tsEntry);
	rtx_time_from_double (&rtxtsExit, timeout);
	rtx_time_add (&rtxtsExit, &tsEntry, &rtxtsExit);

	tsExit.tv_sec = rtxtsExit.seconds;
	tsExit.tv_nsec = rtxtsExit.nanoSeconds;
	

	if ((retval = pthread_mutex_lock (&(p->mutex.mutex))))
		return (rtx_error_errno2 (retval, "rtx_sync_timed_wait: "
				  "pthread_mutex_lock failed"));

	count = p->count;
	while (count == p->count) {
	
	        if ((retval = pthread_cond_timedwait (&(p->cond), &(p->mutex.mutex), &tsExit)) != 0) {
		        if (retval == ETIMEDOUT) {
		        	timedout = 1;
		        } else {
			        rtx_error_errno2 (retval, "rtx_sync_timed_wait: pthread_cond_wait: ");
				errs++;
			}
			break;
		} else {
		        rtx_time_get (&tsNow);
			if (rtx_time_cmp (&tsNow, &rtxtsExit) >= 0) {
				if (count != p->count)   {
					timedout = 1;
				}
				break;
			}
		}
	}

	if ((retval = pthread_mutex_unlock (&(p->mutex.mutex))))
		return (rtx_error_errno2 (retval, "rtx_sync_timed_wait: pthread_mutex_unlock: "));

	if (timedout)
		return (1);

	if (errs)
		return (-1);
		
	return (0);
}

/**
 * Signal a condition variable.
 *
 * @return 0 if OK, -1 on error
 *
 * This function provides a wrapper around the PTHREADS pthread_cond_signal()
 * function which returns an error number if an error occurs. This function
 * returns -1 on error and pushes the error number on the error stack using
 * rtx_error_errno2().
 */
int 
rtx_sync_signal (
	RtxSync	*p    /**< Pointer to RtxSync struct */
	)
{
    int retval;
    
    if ((retval = pthread_mutex_lock (&(p->mutex.mutex))))
        return (rtx_error_errno2 (retval, "rtx_sync_signal: "
				  "pthread_mutex_lock failed"));
    p->count++;
    if ((retval = pthread_cond_signal (&(p->cond))))
        return (rtx_error_errno2 (retval, "rtx_sync_signal: "
				  "pthread_cond_signal failed"));
    
    if ((retval = pthread_mutex_unlock (&(p->mutex.mutex))))
        return (rtx_error_errno2 (retval, "rtx_sync_signal: pthread_mutex_unlock: "));
    return (0);
}

/**
 * Broadcast a condition variable.
 *
 * @return 0 if OK, -1 on error
 *
 * This function provides a wrapper around the PTHREADS pthread_cond_broadcast()
 * function which returns an error number if an error occurs. This function
 * returns -1 on error and pushes the error number on the error stack using
 * rtx_error_errno2().
 */
int 
rtx_sync_broadcast (
	RtxSync	*p    /**< Pointer to RtxSync struct */
	)
{
    int retval;
    
    if ((retval = pthread_mutex_lock (&(p->mutex.mutex))))
        return (rtx_error_errno2 (retval, "rtx_sync_broadcast: "
				  "pthread_mutex_lock failed"));
    p->count++;
    if ((retval = pthread_cond_broadcast (&(p->cond))))
        return (rtx_error_errno2 (retval, "rtx_sync_broadcast: "
				  "pthread_cond_broadcast: "));
    if ((retval = pthread_mutex_unlock (&(p->mutex.mutex))))
        return (rtx_error_errno2 (retval, "rtx_sync_broadcast: pthread_mutex_unlock: "));
    
    return (0);
}


