/***********************************************************************
 * 
 * CSIRO Autonomous Systems Laboratory
 * Queensland Centre for Advanced Technologies
 * PO Box 883, Kenmore, QLD 4069, Australia
 * http://www.ict.csiro.au/
 *  
 * Copyright (c) CSIRO 
 ***********************************************************************/

/**
 * \file stats.c
 * \brief Simple statistical functions
 * \author Peter Corke and Jonathan Roberts
 */

#include <stdio.h>
#include <stdlib.h>
#include <float.h>
#include <math.h>

#include "rtx/defines.h"
#include "rtx/error.h"
#include "rtx/message.h"
#include "rtx/stats.h"

static char rcsid[] RTX_UNUSED = "$Id: stats.c 2274 2007-12-23 05:37:32Z roy029 $";

/**
 * Initialize a statistic data structure.
 * Clears all sums.
 *
 * @param s RtxStats type
 */
void
rtx_stats_init (RtxStats *s)
{
	s->n = s->sumx = s->sumx2 = 0;
	s->min = DBL_MAX;
	s->max = -DBL_MAX;
}


/**
 * Initialize a statistic data structure with histogramming.
 * Clears all sums.
 *
 * @param s RtxStats type
 * @param hmin Minimum value of data to be histogrammed.
 * @param hwidth Width of histogram bins.
 * @param nbins Number of histogram bins.
 */
int
rtx_stats_init_histogram (RtxStats *s, double hmin, double hwidth, int nbins)
{
	s->n = s->sumx = s->sumx2 = 0;
	s->min = DBL_MAX;
	s->max = -DBL_MAX;

	s->hmin = hmin;
	s->hwidth = hwidth;
	s->nbins = nbins;

	if ( (s->histbuf = (unsigned int *)calloc(nbins, sizeof(unsigned int))) == NULL )
		return rtx_error("rtx_stats_init: no memory");
	
	return 0;
}


/**
 * add a new sample to the statistic data structure.
 *
 * @param s RtxStats type
 * @param v value to be added to the statistics
 */
void
rtx_stats_update (RtxStats *s, double v)
{
	if (isnan(v)) {
		s->nnans++;
		return;
	}

	s->n++;
	s->sumx += v;
	s->sumx2 += v*v;

	if (v > s->max)
		s->max = v;
	else if (v < s->min)
		s->min = v;
	
	if (s->histbuf) {
		int	i;

		/* figure which bin */
		i = (v - s->min) / s->hwidth;

		/* clip the value */
		if (i >= s->nbins)
			i = s->nbins;
		else if (i < 0)
			i = 0;
		
		/* increment the bin */
		s->histbuf[i]++;
	}
}

/**
 * compute the mean.
 *
 * @param s RtxStats type.
 * @param mean returned mean value.
 * @return 0 if mean computed, -1 if zero samples in set.
 */
int
rtx_stats_mean (RtxStats *s, double *mean)
{
	if (s->n > 0) {
		*mean = (double)s->sumx / (double)s->n;
		return 0;
	} else
		return -1;
}

/**
 * compute the median.
 *
 * @param s RtxStats type.
 * @param median returned median value.
 * @return 0 if mean computed, -1 if zero samples in set.
 */
int
rtx_stats_median (RtxStats *s, double *median)
{
	unsigned int	total = 0, total2 = 0;
	int		i;

	if ((s->n > 0) && s->histbuf) {
		for (i=0; i<s->nbins; i++)
			total += s->histbuf[i];

		total /= 2;	/* halfway point */
		for (i=0; i<s->nbins; i++)
			if ((total2 += s->histbuf[i]) >= total)
				break;
		*median = s->hwidth * i + s->hmin;
		return 0;
	} else
		return -1;
}

/**
 * compute the variance.
 *
 * @param s RtxStats type.
 * @param var returned mean value.
 * @return 0 if mean computed, -1 if zero samples in set.
 */
int
rtx_stats_variance (RtxStats *s, double *var)
{
	if (s->n > 0) {
		*var = ((double)s->sumx2 - (double)(s->sumx)*s->sumx / (double)s->n) / (double)s->n;
		return 0;
	} else
		return -1;
}

void
rtx_stats_result (RtxStats *s)
{
	if (s->n > 0) {
		rtx_stats_mean(s, &s->mean);
		rtx_stats_variance(s, &s->var);
		s->std = sqrt(s->var);
		rtx_stats_median(s, &s->med);
	}
}

/**
 * print statistics to stdout.
 *
 * @param s RtxStats type.
 */
void
rtx_stats_print (RtxStats *s)
{
	return rtx_stats_fprint(stdout, s);
}

void
rtx_stats_fprint (FILE *fp, RtxStats *s)
{
	if (s->n == 0) {
		fprintf(fp, "<no samples>\n");
	} else {
		double	mean, var;

		rtx_stats_mean(s, &mean);
		rtx_stats_variance(s, &var);

		fprintf(fp, "mean %f, var %f, range %f to %f\n",
			mean, var, s->min, s->max);
	}
}

/**
 * print statistics to routine error channel.
 *
 * @param s RtxStats type.
 * @see routine
 */
void
rtx_stats_message(RtxStats *s) 
{
	if (s->n == 0) {
		rtx_message_routine ("stats: <no samples>\n");
	} else {
		double	mean, var;

		rtx_stats_mean(s, &mean);
		rtx_stats_variance(s, &var);

		rtx_message_routine ("stats: mean %f, var %f, "
				     "range %f to %f\n",
				     mean, var, s->min, s->max);
	}

}

int
rtx_stats_min_double(double *data, int nPoints, double *min)
{
	int	i;

	if (nPoints < 1) {
		return (rtx_error ("rtx_math_min_double: arg 2 must be "
				   "greater than zero\n"));
	}

	*min = data[0];

	for (i=1; i<nPoints; i++) {
		if (data[i] < *min)
			*min = data[i];
	}

	return 0;
}

int
rtx_stats_max_double(double *data, int nPoints, double *max)
{
	int	i;

	if (nPoints < 1) {
		return (rtx_error ("rtx_math_max_double: arg 2 must be "
				   "greater than zero\n"));
	}

	*max = data[0];

	for (i=1; i<nPoints; i++) {
		if (data[i] > *max)
			*max = data[i];
	}

	return 0;
}

