/***********************************************************************
 * 
 * CSIRO Autonomous Systems Laboratory
 * Queensland Centre for Advanced Technologies
 * PO Box 883, Kenmore, QLD 4069, Australia
 * http://www.ict.csiro.au/
 *  
 * Copyright (c) CSIRO 
 ***********************************************************************/

/**
 * \file srpc.c
 * \brief a simple interface for Tcl/Tk GUIs
 * \author Pavan Sikka
 *
 * RTC replacement
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "rtx/defines.h"
#include "rtx/hash.h"
#include "rtx/error.h"
#include "rtx/inet.h"
#include "rtx/srpc.h"

static char rcsid[] RTX_UNUSED = "$Id: srpc.c 3527 2008-06-17 06:02:42Z roy029 $";

/* forward declarations */
void * rtx_srpc_server_func (void *, void *);

/**
 * create a SRPC handle
 *
 * @return handle on success, NULL on failure
 */
RtxSrpc * 
rtx_srpc_init (
	      int port
	      )
{
    RtxSrpc * p = NULL;

    if ((p = (RtxSrpc *) calloc (1, sizeof (RtxSrpc))) == NULL)
        return (rtx_error_null ("rtx_srpc_init: no memory"));
    if ((p->ht = rtx_hash_alloc(32, rtx_hash_deep_string_keys, rtx_hash_shallow_values, (rtx_error_t)NULL)) == NULL)
        return (rtx_error_null ("rtx_srpc_init: rtx_hash_create"));
    if ((p->server = rtx_inet_init (RTX_INET_TCP_SERVER,
				    NULL, port + RTX_SRPC_PORT_OFFSET,
				    NULL, 0,
				    rtx_srpc_server_func, NULL,
				    p)) == NULL)
        return (rtx_error_null ("rtx_srpc_init: rtx_inet_init"));
    return (p);
}

/**
 * register a handler for a client function
 *
 * @return 0 on success, -1 on failure
 */
int
rtx_srpc_register (
		  RtxSrpc * p,
		  char * name,
		  RtxSrpcHandler func
		  )
{
  if (rtx_hash_set(p->ht, name, func, (rtx_error_t)NULL)) {
    return (rtx_error ("rtx_srpc_register: rtx_hash_set"));
  }

  return (0);
}

/**
 * cleanup the SRPC server
 *
 * @return 0 on success, -1 on failure
 */
int 
rtx_srpc_done (
	      RtxSrpc * p
	      )
{
    rtx_inet_done (p->server);
    free (p);
    return (0);
}

void * 
rtx_srpc_server_func (
		     void * p,
		     void * arg
		     )
{
    RtxInetTcpClient * clnt = (RtxInetTcpClient *) p;
    RtxSrpc * handle = (RtxSrpc *) arg;
    char * bufin = NULL, * bufout = NULL;
    int n = 0, i = 0, resp = -1;
    char * argv[256], * tok, * lasts;
    char * errorMsg = "RTX_SRPC_ERROR ";
    RtxSrpcHandler func;

    if ((bufin = calloc (1, 8192)) == NULL) {
        rtx_error_flush ("unable to allocate buffer");
	return (NULL);
    }
    if ((bufout = calloc (1, 8192)) == NULL) {
        rtx_error_flush ("unable to allocate buffer");
	return (NULL);
    }
    while (1) {
        if ((n = rtx_inet_readline (clnt->sock, bufin, 1024, NULL))
                == -1) {
	    rtx_error_flush ("rtx_inet_readline");
	    continue;  
        }
        if (n == 0) {
            /* end-of-file */
            rtx_error_traceback_depth (0);
	    return (NULL);
        }
        /* Have a complete message for processing */
	i = 0;
	tok = strtok_r (bufin, " \t\n\r", &lasts);
	while ((tok != NULL) && (i < 256)) {
	    argv[i++] = tok;
	    tok = strtok_r (NULL, " \t\n\r", &lasts);
	}
	if (i) {
	    if ((func = (RtxSrpcHandler) rtx_hash_find 
		 (handle->ht, argv[0], (rtx_error_t)NULL))	== NULL) {
	        rtx_error_flush ("%s: handler not found", argv[0]);
	    } else {
	        resp = func (bufout, 8192, i, argv);
		if (resp == 0) {
		    strcat (bufout, "\n");
		    if (rtx_inet_write (clnt->sock, bufout, strlen (bufout), 
					NULL) == -1)
		        rtx_error_flush ("rtx_inet_write");
		} else {
		    strcat (bufout, "\n");
		    if (rtx_inet_write (clnt->sock, errorMsg, strlen (errorMsg), 
					NULL) == -1)
		        rtx_error_flush ("rtx_inet_write");
		    if (rtx_inet_write (clnt->sock, bufout, strlen (bufout), 
					NULL) == -1)
		        rtx_error_flush ("rtx_inet_write");
		}
	    }
	} else {
	  /*
	    rtx_error_flush ("empty input[n = %d][%s]", n, bufin);
	  */
	}
    }
    return (NULL);
}

