/***********************************************************************
 * 
 * CSIRO Autonomous Systems Laboratory
 * Queensland Centre for Advanced Technologies
 * PO Box 883, Kenmore, QLD 4069, Australia
 * http://www.ict.csiro.au/
 *  
 * Copyright (c) CSIRO 
 ***********************************************************************/

/** 
 * \file
 * \brief Generic UDP socket 
 * \author Pavan Sikka
 * \author Elliot Duff
 *
 * This module has been written to make using UDP sockets easy. Sockets are
 * opened using the rtx_socket_open() function, which returns a pointer
 * a RtxSocket structure. Data can then be send with rtx_socket_send(), 
 * whilst data is recieved with rtx_socket_recv(). The socket can be 
 * closed with rtx_socket_close();
 *
 * \note In theory a single UDP socket can be used to send and recieve data,
 * but in this case it may be better to use TCP. In these functions, 
 * the sockets have either been set up to be a sender of reciever of data.
 *
 * \todo Add UDP broadcast and TCP - See Bottom of source code
 */

#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <pthread.h>
#include <string.h>
#include <errno.h>

#include "rtx/defines.h"
#include "rtx/socket.h"

static char rcsid[] RTX_UNUSED = "$Id: socket.c 2274 2007-12-23 05:37:32Z roy029 $";


char *rtx_socket_print_host_addr ( RtxSocket *com)
  { return (inet_ntoa (((struct sockaddr_in *) (com->addr))->sin_addr)); }

int   rtx_socket_print_host_port ( RtxSocket *com) {
    return (ntohs (((struct sockaddr_in *) (com->addr))->sin_port)); }
/**
 * Open a UDP socket 
 * \return If succesful RxtSocket, else NULL
 * \warning At the moment it is necessary to specify whether the socket is 
 * for sending or recieving. This is to overcome a problem on the same machine 
 * where it is not poosiblt to bind to the same socket. 
 * \todo  The restriction can be overcome by enabling REUSE of the socket.
 */

RtxSocket * 
rtx_socket_open (
	int portNum, 	/**< Ports 3000 to 10000 are available ? */
	char *hostname, /**< Name of the machine */
	int type)	/**< Whether there is a binding */ 
{
    RtxSocket *com;
    struct sockaddr_in addr;
    struct sockaddr_in s;
    struct ip_mreq mreq;
    struct hostent * h;

    if ((com = (RtxSocket *) calloc(1, sizeof(RtxSocket))) == NULL) return NULL;

    com->portNum = portNum;

    if( hostname ) strncpy(com->hostname, hostname, RTX_MAX_NAME_LEN); 
    else if(gethostname(com->hostname, RTX_MAX_NAME_LEN) < 0) return NULL; 

    /*
    ** Check if address is in the multicast domain
    */

    if( com->hostname[0] == '2' ) com->multicast = 1;

    /*
    ** Create the Socket 
    */

    if( (com->sockfd = socket (AF_INET, SOCK_DGRAM, IPPROTO_UDP)) == -1) return NULL;

    if( type == RTX_UDP_RECV  ) {

      /* 
      ** Bind the socket - Only the receiver "Needs" to do this  
      */

      addr.sin_family = AF_INET;
      addr.sin_addr.s_addr = htonl (INADDR_ANY);
      addr.sin_port = htons (portNum);
      if (bind (com->sockfd, (struct sockaddr *) &addr, sizeof (addr)) == -1) return NULL;

      if( com->multicast ) {
        mreq.imr_multiaddr.s_addr = inet_addr(hostname);
        mreq.imr_interface.s_addr = htonl(INADDR_ANY);
        if (setsockopt(com->sockfd, IPPROTO_IP, IP_ADD_MEMBERSHIP, &mreq, sizeof(mreq)) < 0)
	  { fprintf (stderr, "setsockopt (IP_ADD_MEMBERSHIP) failed: %s\n", 
	    strerror (errno)); return NULL; }
        }
      }

    else { 

      /*
      ** Setup Host Address - This information is sent to receiver 
      */
     
      if ((h = gethostbyname (com->hostname)) == NULL) return NULL;

      memset (&s, 0, sizeof (s));
      s.sin_family = AF_INET;
      s.sin_addr = * (struct in_addr *) (h->h_addr);
      s.sin_port = htons (com->portNum);

      memcpy (com->addr, &s, sizeof (s));
      com->addrLen = sizeof (s);
      }

    return com;
}

/**
 * Close a UDP socket
 * \return -1 on error 
 */

int 
rtx_socket_close
	( RtxSocket *com)	/**< Pointer to socket */
{
    int error = 0;

    if( com->multicast ) {
      struct ip_mreq mreq;
      mreq.imr_multiaddr.s_addr = inet_addr(com->hostname);
      mreq.imr_interface.s_addr = htonl(INADDR_ANY);
      if (setsockopt(com->sockfd, IPPROTO_IP, IP_ADD_MEMBERSHIP, &mreq, sizeof(mreq)) < 0) 
        error--;
      }

    if (shutdown (com->sockfd, 2) == -1) error--;
    if (close (com->sockfd) == -1) error--;
    free(com);
    return error;
}

/**
 * Send Data to UDP socket
 * \return Number of bytes sent
 */

int 
rtx_socket_send
	( RtxSocket *com, 	/**< Pointer to socket */
	char * buf, 		/**< Data buffer */
	int len)		/**< Length of Data buffer */
{
    int n;
    n = sendto (com->sockfd, buf, len, 0, (struct sockaddr *) 
	com->addr, com->addrLen);
    return n;
}

/**
 * Recieve Data from UDP socket
 * \return Number of bytes recieved
 */

int 
rtx_socket_recv ( 
	RtxSocket *com, 	/**< Pointer to socket */
	char * buf, 		/**< Data buffer */
	int len)		/**< Length of Data buffer */
{
    int n;
    com->addrLen = RTX_MAX_ADDR_LEN;
    n = recvfrom (com->sockfd, buf, len, 0, 
			(struct sockaddr *) com->addr, (unsigned int*)&(com->addrLen));
    return n;
}


/**
 * Get a sockaddr_in structure for the given multicast

int
rtx_socket_get_multicast_addr (
        RtxHostAddress * com
    )
{
    struct sockaddr_in s;

    memset (&s, 0, sizeof (s));
    s.sin_family = AF_INET;
    s.sin_addr.s_addr = inet_addr (com->hostname);
    s.sin_port = htons (com->portNum);
    memcpy (com->addr, &s, sizeof (s));
    com->addrLen = sizeof (s);

    return (0);
}

*/

