/***********************************************************************
 * 
 * CSIRO Autonomous Systems Laboratory
 * Queensland Centre for Advanced Technologies
 * PO Box 883, Kenmore, QLD 4069, Australia
 * http://www.ict.csiro.au/
 *  
 * Copyright (c) CSIRO 
 ***********************************************************************/


/**
 * \file signal.c
 * \brief A simple interface to signals
 * \author Pavan Sikka
 */

#include <unistd.h>
#include <stdarg.h>
#include <signal.h>
#include <errno.h>
#include <string.h>
#include <stdlib.h>
#include <pthread.h>

#ifdef i486_darwin
#define DARWIN
#endif

#include "rtx/defines.h"
#include "rtx/mutex.h"
#include "rtx/signal.h"
#include "rtx/error.h"

static char rcsid[] RTX_UNUSED = "$Id: signal.c 2274 2007-12-23 05:37:32Z roy029 $";

static unsigned int * rtxSignalRtSigStack = NULL;
static int rtxSignalSP = 0;
static int rtxSignalRtSigNum = 0;
static RtxMutex rtxSignalMutex;
#ifdef sparc_solaris
static pthread_once_t rtxSignalOnceBlock = {PTHREAD_ONCE_INIT};
#else
static pthread_once_t rtxSignalOnceBlock = PTHREAD_ONCE_INIT;
#endif

/**
 * Ignore the specified signal.
 *
 * @return 0 if OK, -1 on error.
 */
int
rtx_signal_ignore (
		   int sigNum       /**< Signal number */
		  )
{
    struct sigaction act, oldAct;

    memset (&act, 0, sizeof (act));
    memset (&oldAct, 0, sizeof (oldAct));
    act.sa_handler = SIG_IGN;
    if (sigaction (sigNum, &act, &oldAct) == -1)
        return (rtx_error_errno ("rtx_signal_ignore: sigaction() failed"));
    return (0);
}

/**
 * Block the specified signal.
 *
 * @return 0 if OK, -1 on error.
 */
int
rtx_signal_block (
		  int sigNum       /**< Signal number */
		  )
{
    sigset_t sigSet;
    int retval;

    /*
     * Prepare the signal set
     */
    if(sigemptyset(&sigSet)) {
        return (rtx_error_errno ("rtx_signal_block: sigemptyset() failed"));
    }
    /*
     * Load the signal to be blocked
     */
    if( sigaddset(&sigSet, sigNum)) {
        return (rtx_error_errno ("rtx_signal_block: sigaddset(%d) failed",
				 sigNum));
    }
    /* block the signal */
    if ((retval = pthread_sigmask (SIG_BLOCK, &sigSet, NULL))) {
        return (rtx_error_errno2 (retval, "rtx_signal_block: pthread_sigmask() failed"));
    }
    return (0);
}

/**
 * Unblock the specified signal.
 *
 * @return 0 if OK, -1 on error.
 */
int
rtx_signal_unblock (
		    int sigNum       /**< Signal number */
		    )
{
    sigset_t sigSet;
    int retval;

    /*
     * Prepare the signal set
     */
    if(sigemptyset(&sigSet)) {
        return (rtx_error_errno ("rtx_signal_unblock: sigemptyset() failed"));
    }
    /*
     * Load the signal to be unblocked
     */
    if( sigaddset(&sigSet, sigNum)) {
        return (rtx_error_errno ("rtx_signal_unblock: sigaddset(%d) failed",
				 sigNum));
    }
    /* unblock the signal */
    if ((retval = pthread_sigmask (SIG_UNBLOCK, &sigSet, NULL))) {
        return (rtx_error_errno2 (retval, "rtx_signal_unblock: "
				  "pthread_sigmask() failed"));
    }
    return (0);
}

/**
 * Block all realtime signals.
 *
 * @return 0 if OK, -1 on error.
 */
int
rtx_signal_block_realtime (void)
{
#ifdef	DARWIN
	return -1;
#else
    int sigNum;
    sigset_t rtSigSet;
    int retval;

    /*
     * Prepare the signal set
     */
    if(sigemptyset(&rtSigSet)) {
        return (rtx_error_errno ("rtx_signal_block_realtime: "
				 "sigemptyset() failed"));
    }
    /*
     * Load the signals
     */
    /* THIS WILL NOT BE REQUIRED ONCE REDHAT FIX GLIBC AGAIN */
#ifdef i486_linux
    for(sigNum=SIGRTMIN; sigNum < SIGRTMAX; sigNum++)
#else
    for(sigNum=SIGRTMIN; sigNum <= SIGRTMAX; sigNum++)
#endif
        if( sigaddset(&rtSigSet, sigNum)) {
	    return (rtx_error_errno ("rtx_signal_block_realtime: "
				     "sigaddset(%d) failed", sigNum));
	}
    /* block the signals */
    if ((retval = pthread_sigmask (SIG_BLOCK, &rtSigSet, NULL))) {
        return (rtx_error_errno2 (retval, "rtx_signal_block_realtime: "
				 "pthread_sigmask() failed"));
    }
    return (0);
#endif
}

/**
 * Unblock all realtime signals.
 *
 * @return 0 if OK, -1 on error.
 */
int
rtx_signal_unblock_realtime (void)
{
#ifdef	DARWIN
	return -1;
#else
    int sigNum;
    sigset_t rtSigSet;
    int retval;

    /*
     * Prepare the signal set
     */
    if(sigemptyset(&rtSigSet)) {
        return (rtx_error_errno ("rtx_signal_block_realtime: "
				 "sigemptyset() failed"));
    }
    /*
     * Load the signals
     */
    /* THIS WILL NOT BE REQUIRED ONCE REDHAT FIX GLIBC AGAIN */
#ifdef i486_linux
    for(sigNum=SIGRTMIN; sigNum < SIGRTMAX; sigNum++)
#else
    for(sigNum=SIGRTMIN; sigNum <= SIGRTMAX; sigNum++)
#endif
        if( sigaddset(&rtSigSet, sigNum)) {
	    return (rtx_error_errno ("rtx_signal_block_realtime: "
				     "sigaddset(%d) failed", sigNum));
	}
    /* unblock the signals */
    if ((retval = pthread_sigmask (SIG_UNBLOCK, &rtSigSet, NULL))) {
        return (rtx_error_errno2 (retval, "rtx_signal_block_realtime: "
				 "pthread_sigmask() failed"));
    }
    return (0);
#endif
}

/**
 * initialize the list of signals in the system.
 *
 * This function uses the pthread_once mechanism to initialize
 * some state stored in static variables.
 *
 * Reimplement using a list...
 */
static void
rtx_signal_alloc_realtime_init (void)
{
#ifdef	DARWIN
	return -1;
#else
    pthread_mutexattr_t attr;
#if (defined _POSIX_THREAD_PRIO_INHERIT) && (defined PTHREAD_MUTEX_SETPROTOCOL)
    int mutexProtocol;
#endif
    int i;

    /* initialize mutex attrs to support PRIO_INHERIT - this should be done
     * using the RTX mutex module.
     */
    pthread_mutexattr_init (&attr);
#if (defined _POSIX_THREAD_PRIO_INHERIT) && (defined PTHREAD_MUTEX_SETPROTOCOL)
    pthread_mutexattr_getprotocol (&attr, &mutexProtocol);
    if (mutexProtocol != PTHREAD_PRIO_INHERIT)
        pthread_mutexattr_setprotocol (&attr, PTHREAD_PRIO_INHERIT);
#endif
    pthread_mutex_init (&rtxSignalMutex.mutex, &attr);
    /* THIS WILL NOT BE REQUIRED ONCE REDHAT FIX GLIBC AGAIN */
#ifdef i486_linux
    rtxSignalRtSigNum = SIGRTMAX - SIGRTMIN;
#else
    rtxSignalRtSigNum = SIGRTMAX - SIGRTMIN + 1;
#endif
    /* Put all the real-time signals in a stack */
    rtxSignalRtSigStack = calloc (rtxSignalRtSigNum, sizeof (int));
    if (rtxSignalRtSigStack != NULL)
        for (i=0; i<rtxSignalRtSigNum; i++)
	    rtxSignalRtSigStack[i] = SIGRTMIN + i;
#endif
}

/**
 * Allocate a realtime signal.
 * 
 * @return Allocated signal number if OK, -1 on error.
 *
 * This function returns the signal number of a free realtime signal.
 */
int
rtx_signal_alloc_realtime (void)
{
    int retSigNum = -1;

    pthread_once (&rtxSignalOnceBlock, rtx_signal_alloc_realtime_init);
    /* check that stack is valid */
    if (rtxSignalRtSigStack == NULL)
        return (rtx_error ("rtx_signal_alloc_realtime: sig stack not initialized"));
    /* get the lock */
    if (rtx_mutex_lock (&rtxSignalMutex) == -1)
        return (rtx_error ("rtx_signal_alloc_realtime: rtx_mutex_lock"
			   " failed"));
    /* check for non-empty stack */
    if (rtxSignalSP >= rtxSignalRtSigNum)
        rtx_error ("rtx_signal_alloc_realtime: real-time signals exhausted");
    else {
        retSigNum = rtxSignalRtSigStack[rtxSignalSP];
	rtxSignalSP++;
    }
    if (rtx_mutex_unlock (&rtxSignalMutex) == -1)
        return (rtx_error ("rtx_signal_alloc_realtime: rtx_mutex_unlock"
			   " failed"));
    return (retSigNum);
}

/**
 * Free the specified realtime signal.
 *
 * @return 0 if OK, -1 on error.
 */
int
rtx_signal_free_realtime (
			  int sigNum       /**< Realtime signal to be freed */
			  )
{
#ifdef	DARWIN
	return -1;
#else
    int errs = 0, i;

    pthread_once (&rtxSignalOnceBlock, rtx_signal_alloc_realtime_init);
    /* check that stack is valid */
    if (rtxSignalRtSigStack == NULL)
        return (rtx_error ("rtx_signal_free_realtime: sig stack not initialized"));
    /* check signal is valid */
    if ((sigNum < SIGRTMIN) || (sigNum > SIGRTMIN + rtxSignalRtSigNum))
        return (rtx_error ("rtx_signal_free_realtime: invalid signal "
			   "(%d)", sigNum));
    if (rtx_mutex_lock (&rtxSignalMutex) == -1)
        return (rtx_error ("rtx_signal_free_realtime: rtx_mutex_lock"
			   " failed"));
    /* make sure signal is allocated */
    if (rtxSignalSP <= 0) {
        rtx_error ("rtx_signal_free_realtime: signal free");
	errs++;
    }
    if (errs == 0)
        for (i=rtxSignalSP; i<rtxSignalRtSigNum; i++)
	    if (sigNum == rtxSignalRtSigStack[i]) {
	        rtx_error ("rtx_signal_free_realtime: signal free");
		errs++;
	    }
    /* signal can be freed */
    rtxSignalSP--;
    rtxSignalRtSigStack[rtxSignalSP] = sigNum;
    if (rtx_mutex_unlock (&rtxSignalMutex) == -1)
        return (rtx_error ("rtx_signal_free_realtime: rtx_mutex_unlock"
			   " failed"));
    if (errs)
        return (-1);
    return (0);
#endif
}
