/***********************************************************************
 * 
 * CSIRO Autonomous Systems Laboratory
 * Queensland Centre for Advanced Technologies
 * PO Box 883, Kenmore, QLD 4069, Australia
 * http://www.ict.csiro.au/
 *  
 * Copyright (c) CSIRO 
 ***********************************************************************/

/**
 * \file serial.c
 * \brief serial data handling functions
 * \author Peter Corke
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#ifdef i486_darwin
// some issues with USB serial and MacOS, check the wiki
#define DARWIN
#endif
#if  !defined(i486_qnxrtp) && !defined(DARWIN)
#include <termio.h>
#endif
#include <termios.h>
#include <time.h>
#include <sys/types.h>
#include <fcntl.h>
#include <unistd.h>
#include <errno.h>
#include <signal.h>
#include <sys/ioctl.h>
#include <pthread.h>
#include <linux/serial.h>

#include "rtx/defines.h"
#include "rtx/error.h"
#include "rtx/time.h"
#include "rtx/serial.h"

static char rcsid[] RTX_UNUSED = "$Id: serial.c 2274 2007-12-23 05:37:32Z roy029 $";

/**
 * Open a serial data connection.
 * <p>
 * from man 4 tty:<p>
 *   The tty driver supports System V non-canonical mode input
 *   processing. To enter this mode, the System V-compatible TCSETA ioctl
 *   call must be used with the ISIG and ICANON bits in the c_lflag field
 *   of the termio structure turned off. When the tty driver is placed in
 *   System V non-canonical mode, the fifth and sixth control characters
 *   (i.e. c_cc[4] and c_cc[5]) of the termio structure take on the
 *   meanings of VMIN and VTIME, respectively. The VMIN and VTIME values
 *   dictate how input characters are processed. In general, a read
 *   request in System V non-canonical mode will not return until VMIN
 *   characters have been received or VTIME tenths of a second have
 *   elapsed. The point at which the timer begins depends on whether or
 *   not VMIN is zero. There are four distinct behaviors depending on the
 *   values of VMIN and VTIME:
 *   <p>
 *   VMIN>0, VTIME>0
 *                       In this case, VTIME represents an inter- character
 *                       timer and is not activated until the first
 *                       character is received. A read request will not
 *                       return until VMIN characters have been received or
 *                       VTIME tenths of a second have elapsed between the
 *                       receipt of two successive characters, whichever
 *                       comes first. The timer is reset upon receipt of
 *                       each character. A read will block indefinitely
 *                       until the first character is received.
 *   <p>
 *   VMIN>0, VTIME=0
 *                       In this case, there is no timer. A read request
 *                       will not return until VMIN characters have been
 *                       received.
 *   <p>
 *   VMIN=0, VTIME>0
 *                       In this case, VTIME represents a request timer and
 *                       is activated as soon as the read request is made.
 *                       A read request will return as soon as a character
 *                       is received or VTIME tenths of a second have
 *                       elapsed since the read request was made,
 *                       whichever comes first. Since the timer is
 *                       activated on the read and not on each received
 *                       character, it is possible in this case for the
 *                       read to return zero characters.
 *   <p>
 *   VMIN=0, VTIME=0
 *                       In this case, a read request always returns
 *                       immediately. The number of characters returned
 *                       will be the minimum of the number requested and
 *                       the number currently available, which could be
 *                       zero.
 *   <p>
 *   It is important to realize that VMIN does not serve as a maximum. For
 *   example, if VMIN is 5 and there are 10 characters available, a read
 *   request for 7 characters will return 7 characters (not 5). 
 * \return +'ve file descriptor on success, -1 on error.
 * @see open, tcsetattr
 */
int 
rtx_serial_open (
		 char *dev,      /**< pathname of the serial device to open */
		 int baud,       /**< baud rate for reading and writing */
		 int databits,   /**< number of data bits */
		 int stopbits,   /**< number of stop bits */
		 RtxSerialParity parity,  /**< parity */
		 RtxSerialFlow flow,      /**< flow control */
		 RtxSerialModem modem,    /**< modem control lines */
		 int vmin,   /**< minimum number of characters to read
				before returning */
		 int vtime   /**< minimum time to wait before returning */
		 )
{
	struct termios	ttyCfg;
	int		fd;
	static char	*func = "rtx_serial_open";

	if ((fd = open(dev, O_RDWR)) == -1)
		return rtx_error_errno ("%s: open(%s) failed", func, dev);
	/* zero the config structure */
#ifdef	DARWIN
	tcgetattr(fd, &ttyCfg);
#else
	memset (&ttyCfg, 0, sizeof (ttyCfg));
#endif

	/* set parity */
	switch (parity) {
  	    case RTX_SERIAL_PARITY_NOP :
	        break;
  	    case RTX_SERIAL_PARITY_NONE :
	        break;
  	    case RTX_SERIAL_PARITY_ODD :
	        ttyCfg.c_cflag |= PARODD | PARENB;
	        break;
  	    case RTX_SERIAL_PARITY_EVEN :
	        ttyCfg.c_cflag |= PARENB;
	        break;
  	    default :
	        return (rtx_error ("rtx_serial_open: invalid parity %d", 
				   parity));
	        break;
	}
#ifndef	DARWIN
	/* set flow control */
	switch (flow) {
  	    case RTX_SERIAL_FLOW_NOP :
	        break;
  	    case RTX_SERIAL_FLOW_NONE :
	        break;
  	    case RTX_SERIAL_FLOW_XONXOFF :
	        ttyCfg.c_iflag |= IXON | IXOFF;
	        break;
  	    case RTX_SERIAL_FLOW_HW :
#if defined (sparc_solaris)
		ttyCfg.c_cflag |= CRTSCTS | CRTSXOFF;
#endif
#if defined (i486_lynxos310) || defined (i486_linux)
	        ttyCfg.c_cflag |= CRTSCTS;
#endif
#if defined (i486_qnxrtp)
	        ttyCfg.c_cflag |= IHFLOW | OHFLOW;
#endif
#if defined (i486_darwin)
	        ttyCfg.c_cflag |= CRTSCTS | CRTS_IFLOW;
#endif
	        break;
  	    default :
	        return (rtx_error ("rtx_serial_open: invalid flow control %d", 
				   flow));
	        break;
	}
	/* set modem control */
	switch (modem) {
  	    case RTX_SERIAL_MODEM_NOP :
	        break;
  	    case RTX_SERIAL_MODEM_ON :
	        break;
  	    case RTX_SERIAL_MODEM_OFF :
	        ttyCfg.c_cflag |= CLOCAL;
	        break;
  	    default :
	        return (rtx_error ("rtx_serial_open: invalid modem "
				   "flag %d", modem));
	        break;
	}
#endif
	/* set data bits */
	switch (databits) {
  	    case 5 :
	        ttyCfg.c_cflag |= CS5;
	        break;
  	    case 6 :
	        ttyCfg.c_cflag |= CS6;
	        break;
  	    case 7 :
	        ttyCfg.c_cflag |= CS7;
	        break;
  	    case 8 :
	        ttyCfg.c_cflag |= CS8;
	        break;
  	    default :
	        return (rtx_error ("rtx_serial_open: invalid number of "
				   "data bits %d", stopbits));
	        break;
	}
	/* set stop bits */
	switch (stopbits) {
  	    case 1 :
	        break;
  	    case 2 :
	        ttyCfg.c_cflag |= CSTOPB;
	        break;
  	    default :
	        return (rtx_error ("rtx_serial_open: invalid number of "
				   "stop bits %d", stopbits));
	        break;
	}
	/* enable receiver */
	ttyCfg.c_cflag |= CREAD;

	ttyCfg.c_cc[VMIN] = vmin;
	ttyCfg.c_cc[VTIME] = vtime;
	cfmakeraw(&ttyCfg);

	if (tcsetattr (fd, TCSANOW, &ttyCfg) == -1)
		return rtx_error_errno ("%s: tcsetattr failed", func);

	if (rtx_serial_set_baudrate (fd, baud, baud) == -1)
		return rtx_error_errno ("%s: rtx_serial_set_baudrate(%d) failed",
					func, baud);

	return fd;
}

/**
 * This function is the same as rtx_serial_open() except that the priority
 * of the calling thread is changed to that of prio for the duration of the
 * open. Hence, the serial device driver will be opened at this priority
 * and so will all subsequent reads and writes.
 *
 * \return +'ve file descriptor on success, -1 on error.
 */
int
rtx_serial_open_with_prio(
		 char *dev,      /**< pathname of the serial device to open */
		 int baud,       /**< baud rate for reading and writing */
		 int databits,   /**< number of data bits */
		 int stopbits,   /**< number of stop bits */
		 RtxSerialParity parity,  /**< parity */
		 RtxSerialFlow flow,      /**< flow control */
		 RtxSerialModem modem,    /**< modem control lines */
		 int vmin,   /**< minimum number of characters to read
				before returning */
		 int vtime,   /**< minimum time to wait before returning */
		 RtxThreadPrio prio	/**< priority of open */
	)
{
	int			fd, oldPrio;

	/*
	 * Get current prio
	 */
	oldPrio = rtx_thread_prio_get();

	/*
	 * Change prio
	 */
	if (rtx_thread_prio_set(prio))
		return rtx_error_errno("rtx_serial_open_with_prio: rtx_thread_prio_set: failed");

	/*
	 * Open serial device
	 */
	if ((fd = rtx_serial_open(dev, baud, databits, stopbits, parity, flow, modem, vmin, vtime)) < 0)
		return rtx_error("rtx_serial_open_with_prio: rtx_serial_open() failed");

	/*
	 * Change prio back
	 */
	if (rtx_thread_prio_set(oldPrio))
		return rtx_error_errno("rtx_serial_open_with_prio: rtx_thread_prio_set: failed");

	return fd;
}

/**
 * This function closes a serial stream
 *
 * \return Zero on success, -1 on error.
 */
int
rtx_serial_close(
	int	fd	/** File descriptor */
	)
{
	return 0;
}

/**
 * alternative \c read() function.
 * @param fd the file descriptor to read from.
 * @param data a pointer to start of data buffer.
 * @param pktlen the number of bytes to read.
 *
 * Guaranteed to read \p pktlen bytes from the stream. It will not timeout.
 *
 * @return the number of bytes actually read, or -1 on error.
 */
int
rtx_serial_read (int fd, char *data, int pktlen)
{
	int	len, nread, count = 0;

	len = pktlen;

	while (len > 0) {
		nread = read(fd, data, len);
		if ((nread < 0) && (errno != EINTR))
			// handle errors, except for EINTR which is caused by a signal
			// causing premature return from read()
			return rtx_error_errno ("rtx_serial_read: read() failed");
		else {
			data += nread;
			len -= nread;
			count += nread;
		}
	}

	return count;
}

/**
 * alternative \c read() function.
 * @param fd the file descriptor to read from.
 * @param data a pointer to start of data buffer.
 * @param pktlen the number of bytes to read.
 * @param timeout return after timeout has expired
 *
 * You must set VMIN and VTIME properly for this to work
 *
 * @return the number of bytes actually read, or -1 on error.
 */
int
rtx_serial_read_timeout (int fd, char *data, int pktlen, double timeout)
{
        int             len, nread, count = 0;
        RtxTime initTime;

        len = pktlen;

	rtx_time_get (&initTime);
        while (len > 0) {
                nread = read(fd, data, len);
                if ((nread < 0) && (errno != EINTR)) {
                        return (rtx_error_errno ("rtx_serial_read_timeout:"
						 " read() failed "));
                } else {
                        data += nread;
                        len -= nread;
                        count += nread;
                }
		if (rtx_time_get_delta (&initTime) > timeout)
                        break;
        }

        return count;
}

/**
 * alternative \c read() function. If the read time exceeds VTIME for any
 * read then this function returns with the number of bytes read up to that
 * point.
 * @param fd the file descriptor to read from.
 * @param data a pointer to start of data buffer.
 * @param pktlen the number of bytes to read.
 *
 * You must set VMIN and VTIME properly for this to work
 *
 * @return the number of bytes actually read, or -1 on error.
 */
int
rtx_serial_read_vtimeout (int fd, char *data, int pktlen)
{
        int             len, nread, count = 0;

        len = pktlen;

        while (len > 0) {
                nread = read(fd, data, len);
		if (nread == 0) {
			break;
		} else if ((nread < 0) && (errno != EINTR)) {
                        return (rtx_error_errno ("rtx_serial_read_vtimeout:"
						 " read() failed "));
                } else {
                        data += nread;
                        len -= nread;
                        count += nread;
                }
        }

        return count;
}

/**
 * Read a packet with specified header from serial data stream.  Attempts
 * to synchronize with the packet stream using a specified header and length.
 *
 * @param fd the file descriptor to read from.
 * @param data pointer to buffer where packet will be placed.
 * @param packet_len the length of the packet.
 * @param header_pat a string of bytes that mark the start of the packet.
 * @param header_len the length of the header
 * @param ts timestamp associated with first byte read, not returned if NULL.
 * @return 1 if packet is read, 0 if header check failed, -1 on error.
 *
 * Function should be called in a loop until it returns 1.  If the first bytes
 * of the packet don't match the required header then it reads an extra byte
 * so as to move toward synchronization.
 *
 */
int
rtx_serial_read_packet(int fd,
	char *data, int packet_len, 
	char * header_pat, int header_len,
	RtxTime *ts)
{
	int	nread;
	static char	*func = "rtx_serial_read_packet";

	/*
	 * read the first byte and timestamp it
	 */
	nread = rtx_serial_read(fd, data, 1);

	if (nread < 0)
		return rtx_error_errno("%s: rtx_read_serial() failed", func);
	if (ts != NULL)
		rtx_time_get(ts);
	
	/*
	 * read the rest of the packet
	 */
	nread = rtx_serial_read(fd, &data[1], packet_len-1);
	if (nread < 0)
		return rtx_error_errno("%s: rtx_read_serial() failed", func);

	/*
	 * check the header, if wrong, slip a byte and go around
	 */
	if (memcmp(data, header_pat, header_len) == 0)
		return 1;

	/*
	 * read an extra byte in order to move toward synchronization
	 */
	nread = rtx_serial_read(fd, data, 1);

	return 0;
}


/**
 * alternative \c read() function.
 * @param fd the file descriptor to read from.
 * @param data a pointer to start of data buffer.
 * @param maxlen the maximum number of bytes to read.
 * @param stop_pat a string of bytes that mark the end of reading
 * @param pat_len length of the stop byte pattern
 * @return the number of bytes actually read.
 *
 * Read characters from a serial channel until the specified character
 * or specified length, \p maxlen, is encountered
 *
 */
int
rtx_serial_read_until(int fd, char *data, int maxlen, char *stop_pat, int pat_len )
{
	int	nread, count = 0;
	int	k = 0;

	while (count < maxlen) {
		nread = read(fd, data, 1);
		if ((nread < 0) && (errno != EINTR))
			return rtx_error_errno ("rtx_serial_read_until: read() failed");
		else {
			count += nread;
			if (*data == stop_pat[k]) {
				k++;
				if (k >= pat_len)
					return count;
			} else {
				k = 0;	/* start at the beginning */
			}
			data += nread;
		}
	}

	return count;
}

/**
 * change the number of data bits for the specified data stream
 *
 * @return 0 on success, else -1.
 */
int
rtx_serial_set_databits (
			 int fd, /**< file descriptor for serial stream */
			 int databits   /**< number of data bits */
			 )
{
	struct termios	ttyCfg;
	static char	*func = "rtx_serial_set_databits";

	if (tcgetattr (fd, &ttyCfg) == -1)
		return rtx_error_errno ("%s: tcgetattr() failed", func);

	/* reset the data bits field */
	ttyCfg.c_cflag &= (~ CSIZE);
	switch (databits) {
  	    case 5 :
	        ttyCfg.c_cflag |= CS5;
	        break;
  	    case 6 :
	        ttyCfg.c_cflag |= CS6;
	        break;
  	    case 7 :
	        ttyCfg.c_cflag |= CS7;
	        break;
  	    case 8 :
	        ttyCfg.c_cflag |= CS8;
	        break;
  	    default :
	        return (rtx_error ("%s: invalid number of data bits %d",
				   func, databits));
	        break;
	}

	if (tcsetattr (fd, TCSANOW, &ttyCfg) == -1)
		return rtx_error_errno ("%s: tcsetattr() failed", func);

	return 0;
}

/**
 * change the number of stop bits for the specified data stream
 *
 * @return 0 on success, else -1.
 */
int
rtx_serial_set_stopbits (
			 int fd, /**< file descriptor for serial stream */
			 int stopbits   /**< number of data bits */
			 )
{
	struct termios	ttyCfg;
	static char	*func = "rtx_serial_set_stopbits";

	if (tcgetattr (fd, &ttyCfg) == -1)
		return rtx_error_errno ("%s: tcgetattr() failed", func);

	switch (stopbits) {
  	    case 1 :
	        ttyCfg.c_cflag &= (~ CSTOPB);
	        break;
  	    case 2 :
	        ttyCfg.c_cflag |= CSTOPB;
	        break;
  	    default :
	        return (rtx_error ("%s: invalid number of data bits %d",
				   func, stopbits));
	        break;
	}

	if (tcsetattr (fd, TCSANOW, &ttyCfg) == -1)
		return rtx_error_errno ("%s: tcsetattr() failed", func);

	return 0;
}

/**
 * change the parity for the specified data stream
 *
 * @return 0 on success, else -1.
 */
int
rtx_serial_set_parity (
			 int fd, /**< file descriptor for serial stream */
			 RtxSerialParity parity   /**< parity */
			 )
{
	struct termios	ttyCfg;
	static char	*func = "rtx_serial_set_parity";

	if (tcgetattr (fd, &ttyCfg) == -1)
		return rtx_error_errno ("%s: tcgetattr() failed", func);

	switch (parity) {
  	    case RTX_SERIAL_PARITY_NOP :
	        break;
  	    case RTX_SERIAL_PARITY_NONE :
	        ttyCfg.c_cflag &= (~ (PARODD | PARENB));
	        break;
  	    case RTX_SERIAL_PARITY_ODD :
	        ttyCfg.c_cflag |= PARODD | PARENB;
	        break;
  	    case RTX_SERIAL_PARITY_EVEN :
	        ttyCfg.c_cflag |= PARENB;
		ttyCfg.c_cflag &= (~ PARODD);
	        break;
  	    default :
	        return (rtx_error ("%s: invalid parity %d", func, parity));
	        break;
	}

	if (tcsetattr (fd, TCSANOW, &ttyCfg) == -1)
		return rtx_error_errno ("%s: tcsetattr() failed", func);

	return 0;
}

/**
 * change the flow control for the specified data stream
 *
 * @return 0 on success, else -1.
 */
int
rtx_serial_set_flowcontrol (
			 int fd, /**< file descriptor for serial stream */
			 RtxSerialFlow flow   /**< flow control */
			 )
{
	struct termios	ttyCfg;
	static char	*func = "rtx_serial_set_flowcontrol";

	if (tcgetattr (fd, &ttyCfg) == -1)
		return rtx_error_errno ("%s: tcgetattr() failed", func);

	switch (flow) {
  	    case RTX_SERIAL_FLOW_NOP :
	        break;
  	    case RTX_SERIAL_FLOW_NONE :
	        ttyCfg.c_iflag &= (~ (IXON | IXOFF));
#if defined (sparc_solaris)
		ttyCfg.c_cflag &= (~ (CRTSCTS | CRTSXOFF));
#endif
#if defined (i486_lynxos310) || defined (i486_linux)
	        ttyCfg.c_cflag &= (~ CRTSCTS);
#endif
#if defined (i486_qnxrtp)
	        ttyCfg.c_cflag &= (~ (IHFLOW | OHFLOW));
#endif
	        break;
  	    case RTX_SERIAL_FLOW_XONXOFF :
#if defined (sparc_solaris)
		ttyCfg.c_cflag &= (~ (CRTSCTS | CRTSXOFF));
#endif
#if defined (i486_lynxos310) || defined (i486_linux)
	        ttyCfg.c_cflag &= (~ CRTSCTS);
#endif
#if defined (i486_qnxrtp)
	        ttyCfg.c_cflag &= (~ (IHFLOW | OHFLOW));
#endif
	        ttyCfg.c_iflag |= IXON | IXOFF;
	        break;
  	    case RTX_SERIAL_FLOW_HW :
	        ttyCfg.c_iflag &= (~ (IXON | IXOFF));
#if defined (sparc_solaris)
		ttyCfg.c_cflag &= (~ (CRTSCTS | CRTSXOFF));
#endif
#if defined (i486_lynxos310) || defined (i486_linux)
	        ttyCfg.c_cflag &= (~ CRTSCTS);
#endif
#if defined (i486_qnxrtp)
	        ttyCfg.c_cflag &= (~ (IHFLOW | OHFLOW));
#endif
	        break;
  	    default :
	        return (rtx_error ("%s: invalid flow control %d",
				   func, flow));
	        break;
	}

	if (tcsetattr (fd, TCSANOW, &ttyCfg) == -1)
		return rtx_error_errno ("%s: tcsetattr() failed", func);

	return 0;
}

/**
 * change the modem control for the specified data stream
 *
 * @return 0 on success, else -1.
 */
int
rtx_serial_set_modemcontrol (
			 int fd, /**< file descriptor for serial stream */
			 RtxSerialModem modem   /**< modem control */
			 )
{
	struct termios	ttyCfg;
	static char	*func = "rtx_serial_set_modemcontrol";

	if (tcgetattr (fd, &ttyCfg) == -1)
		return rtx_error_errno ("%s: tcgetattr() failed", func);

	switch (modem) {
  	    case RTX_SERIAL_MODEM_NOP :
	        break;
  	    case RTX_SERIAL_MODEM_ON :
	        ttyCfg.c_cflag &= (~ CLOCAL);
	        break;
  	    case RTX_SERIAL_MODEM_OFF :
	        ttyCfg.c_cflag |= CLOCAL;
	        break;
  	    default :
	        return (rtx_error ("%s: invalid modem flag %d", func, modem));
	        break;
	}

	if (tcsetattr (fd, TCSANOW, &ttyCfg) == -1)
		return rtx_error_errno ("%s: tcsetattr() failed", func);

	return 0;
}

/**
 * enable the receiver on the input data stream
 *
 * @return 0 on success, -1 on error
 */
int 
rtx_serial_enable_receiver (
			    int fd   /**< data stream */
			    )
{
    struct termios  ttyCfg;

    if (tcgetattr (fd, &ttyCfg) == -1) {
        return (rtx_error_errno ("rtx_serial_enable_receiver: tcgetattr "
				 "failed: "));
    }
    ttyCfg.c_cflag |= CREAD;
    if (tcsetattr (fd, TCSANOW, &ttyCfg) == -1) {
        return (rtx_error_errno ("rtx_serial_enable_receiver: tcsetattr "
				 "failed: "));
    }
    return 0;
}

/**
 * disable the receiver on the input data stream
 *
 * @return 0 on success, -1 on error
 */
int 
rtx_serial_disable_receiver (
			    int fd   /**< data stream */
			    )
{
    struct termios  ttyCfg;

    if (tcgetattr (fd, &ttyCfg) == -1) {
        return (rtx_error_errno ("rtx_serial_disable_receiver: tcgetattr "
				 "failed: "));
    }
    ttyCfg.c_cflag &= (~ CREAD);
    if (tcsetattr (fd, TCSANOW, &ttyCfg) == -1) {
        return (rtx_error_errno ("rtx_serial_disable_receiver: tcsetattr "
				 "failed: "));
    }
    return 0;
}

/**
 * Flush the serial input data stream.
 * @param fd file descriptor for existing serial stream.
 * @return 0 on success, else -1.
 */
int
rtx_serial_flush_input (int fd)
{
	static char	*func = "rtx_serial_inflush";

	if (tcflush(fd, TCIFLUSH) < 0)
		return rtx_error_errno ("%s: tcflush", func);
	return (0);
}

/**
 * Flush the serial output data stream.
 * @param fd file descriptor for existing serial stream.
 * @return 0 on success, else -1.
 */
int
rtx_serial_flush_output (int fd)
{
	static char	*func = "rtx_serial_outflush";

	if (tcflush(fd, TCOFLUSH) < 0)
		return rtx_error_errno ("%s: tcflush", func);
	return (0);
}

/**
 * Calculate CRC16
 * @return CRC value
 */
int
rtx_serial_crc16 (
	unsigned char	*p,	/*!< Pointer to data */
	int 		n	/*!< Length of data */
	) 
{
	unsigned short crc16 = 0, i;

	for(i=0;i<n;i++){
		crc16 = (unsigned char)(crc16 >> 8) | (crc16 << 8);
		crc16 ^= p[i];
		crc16 ^= (unsigned char)(crc16 & 0xff) >> 4;
		crc16 ^= (crc16 << 8) << 4;
		crc16 ^= ((crc16 & 0xff) << 4) << 1;
	}
	
	return(crc16);
}

/**
 * @param baud rate as an integer
 * @return the equivalent baudrate flag to be passed to \c ioctl().
 */
static int
int2baudflag(int baudrate, int * baudflag)
{
        int flg = 0;
	int customrate = 0;

	switch (baudrate) {
	case 0:	        flg = B0; break;
	case 50:	flg = B50; break;
	case 75:	flg = B75; break;
	case 110:	flg = B110; break;
	case 134:	flg = B134; break;
	case 150:	flg = B150; break;
	case 200:	flg = B200; break;
	case 300:	flg = B300; break;
	case 600:	flg = B600; break;
	case 1200:	flg = B1200; break;
	case 1800:	flg = B1800; break;
	case 2400:	flg = B2400; break;
	case 4800:	flg = B4800; break;
	case 9600:	flg = B9600; break;
	case 19200:	flg = B19200; break;
	case 38400:	flg = B38400; break;
	case 57600:	flg = B57600; break;
#if defined (sparc_solaris)
	case 76800:	flg = B76800; break;
#endif
	case 115200:	flg = B115200; break;
#if defined (sparc_solaris)
	case 153600:	flg = B153600; break;
#endif
#if defined (sparc_solaris) || defined (i486_linux)
	case 230400:	flg = B230400; break;
#endif
#if defined (sparc_solaris)
	case 307200:	flg = B307200; break;
	case 460800:	flg = B460800; break;
#endif
	default:
    	        customrate = 1;
		flg = B38400;
		break;
	}
	* baudflag = flg;
	return (customrate);
}

/**
 * change the baud rate of a serial connection.
 * @param fd file descriptor for existing serial stream.
 * @param newbaud desired baud rate as an integer.
 * @return 0 on success, else -1.
 *
 * The integer baudrate \p newbaud is mapped to a baud rate flag.
 * @see int2baudflag
 */
int
rtx_serial_set_baudrate (int fd, int readbaud, int writebaud)
{
	struct termios	ttyCfg;
	int inbaudflag = 0, outbaudflag = 0, customrate = 0;
	static char	*func = "rtx_serial_set_baudrate";

	customrate = int2baudflag (readbaud, &inbaudflag);
	customrate += int2baudflag (writebaud, &outbaudflag);

	if (tcgetattr (fd, &ttyCfg) == -1)
		return rtx_error_errno ("%s: tcgetattr() failed", func);

        if (cfsetispeed (&ttyCfg, inbaudflag) == -1)
                return rtx_error_errno ("%s: cfsetispeed(%d) failed",
                        func, readbaud);

        if (cfsetospeed (&ttyCfg, outbaudflag) == -1)
                return rtx_error_errno ("%s: cfsetospeed(%d) failed",
                        func, writebaud);

	if (tcsetattr (fd, TCSANOW, &ttyCfg) == -1)
		return rtx_error_errno ("%s: tcsetattr() failed", func);
	
	struct serial_struct ss;

	if (customrate) {
	        if (ioctl (fd, TIOCGSERIAL, &ss) == -1)
		        return rtx_error_errno ("%s: ioctl(TIOCGSERIAL) failed", func);
		ss.flags |= ASYNC_SPD_CUST;
		ss.custom_divisor = 24000000/500000;
		if (ioctl (fd, TIOCSSERIAL, &ss) == -1)
		        return rtx_error_errno ("%s: ioctl(TIOCSSERIAL) failed", func);
	}

	return 0;
}

