/***********************************************************************
 * 
 * CSIRO Autonomous Systems Laboratory
 * Queensland Centre for Advanced Technologies
 * PO Box 883, Kenmore, QLD 4069, Australia
 * http://www.ict.csiro.au/
 *  
 * Copyright (c) CSIRO 
 ***********************************************************************/

#ifndef USE_SYSV_API

/**
 * \file sem.c
 * \brief A simple interface to semaphores
 * \author Pavan Sikka
 */

#include <stdlib.h>
#include <signal.h>
#include <errno.h>

#include "rtx/defines.h"
#include "rtx/error.h"
#include "rtx/sem.h"
#include "rtx/timer.h"
#include "rtx/thread.h"

static char rcsid[] RTX_UNUSED = "$Id: sem.c 2274 2007-12-23 05:37:32Z roy029 $";



#define SEMID(X) (&(sem->id))

/**
 * Initialize a unnamed semaphore.
 *
 * @return Pointer to a semaphore on success, NULL on error.
 *
 * This function is included for completeness and is just a wrapper around
 * the POSIX.1b sem_init() function.
 */
RtxSem *
rtx_sem_init (
		RtxSem *        semP,           /*!< use memory if provided */
		int 		pshared,        /*!< 1 => semaphore can be shared
									  by multiple processes */
		unsigned int	initialValue	/*!< Initial value of semaphore */
		)
{
	RtxSem	*sem = semP;

	if (sem == NULL) {
		if ((sem = (RtxSem *)calloc(1, sizeof(RtxSem))) == NULL)
			return (rtx_error_null ("rtx_sem_init: calloc: no memory")); 
		sem->memAlloc = 1;
	} else {
		sem->memAlloc = 0;
	}

	if (sem_init (SEMID(sem), pshared, initialValue) == -1) {
		if (sem->memAlloc)
			free (sem);
		return (rtx_error_errno_null ("rtx_sem_init: sem_init: "));
	}

	return sem;
}

/**
 * Destroy a semaphore.
 *
 * @return 0 if OK, -1 on error.
 *
 * This function is included for completeness and is just a wrapper around
 * the POSIX.1b sem_destroy() function.
 */
int
rtx_sem_destroy (
		RtxSem *sem       /**< Pointer to semaphore */
		)
{
	int errs = 0;

	if (sem_destroy (SEMID(sem)) == -1) {
		rtx_error_errno ("rtx_sem_destroy: sem_destroy: ");
		errs++;
	}
	if (sem->memAlloc)
		free (sem);

	if (errs)
		return (-1);
	return (0);
}

/**
 * Wait on a semaphore.
 *
 * @return 0 if OK, -1 on error.
 *
 * This function is included for completeness and is just a wrapper around
 * the POSIX.1b sem_wait() function.
 */
int
rtx_sem_wait (
		RtxSem *sem       /**< Pointer to semaphore */
		)
{
	if (sem_wait (SEMID(sem)) == -1)
#ifdef i486_lynxos
		return (rtx_error_errno ("rtx_sem_wait (%d): failed", SEMID(sem).sem_off));
#else
	return (rtx_error_errno ("rtx_sem_wait: failed"));
#endif

	return (0);
}

typedef struct {
	RtxSem *sem;
	double timeout;
	int triggered;
} TimedWaitStruct;

static void* waiter(void *arg)
{
	TimedWaitStruct *tws = (TimedWaitStruct*)arg;
	rtx_timer_sleep_eintr(tws->timeout);
	tws->triggered = 1;
	rtx_sem_post(tws->sem);
	return NULL;
}

int
rtx_sem_timedwait(RtxSem *sem, double timeout)
{
	int ret = 0;
	TimedWaitStruct tws = {sem,timeout,0};
	RtxThread tictac;
	rtx_thread_create("sem timer",0,RTX_THREAD_SCHED_OTHER,0,
			0, RTX_THREAD_CANCEL_DISABLE, waiter, &tws,
			NULL,NULL);
	ret = rtx_sem_wait(sem);
	rtx_thread_destroy_sync(&tictac);
	if (tws.triggered) 
		return 1;
	return ret;
}

/**
 * Wait on a semaphore, ignore EINTR errors.
 *
 * @return 0 if OK, -1 on error.
 */
int
rtx_sem_wait_ignore_eintr (
		RtxSem *sem      /**< Pointer to semaphore */
		)
{
	while (sem_wait (SEMID(sem)) != 0)
		if (errno != EINTR) {
#ifdef i486_lynxos
			return (rtx_error_errno ("rtx_sem_wait (%d): failed", 
						SEMID(sem).sem_off));
#else
			return (rtx_error_errno ("rtx_sem_wait: failed"));
#endif
		}
	return (0);
}

int
rtx_sem_timedwait_ignore_eintr(RtxSem *sem, double timeout)
{
	int ret = 0;
	TimedWaitStruct tws = {sem,timeout,0};
	RtxThread tictac;
	rtx_thread_create("sem timer",0,RTX_THREAD_SCHED_OTHER,0,
			0, RTX_THREAD_CANCEL_DISABLE, waiter, &tws,
			NULL,NULL);
	ret = rtx_sem_wait_ignore_eintr(sem);
	rtx_thread_destroy_sync(&tictac);
	if (tws.triggered) 
		return 1;
	return ret;
}

/**
 * Post a semaphore.
 *
 * @return 0 if OK, -1 on error.
 *
 * This function is a little redundant since sem_post() cannot return with the
 * EINTR error. However, it still provides a wrapping function using the error
 * facility.
 */
int
rtx_sem_post (
		RtxSem *sem       /**< Pointer to semaphore */
		)
{
	if (sem_post (SEMID(sem)) == -1)
#ifdef i486_lynxos
		return (rtx_error_errno ("rtx_sem_post (%d) failed", SEMID(sem).sem_off));
#else
	return (rtx_error_errno ("rtx_sem_post failed"));
#endif

	return (0);
}

/**
 * Post a binary semaphore.
 *
 * @return 0 if OK, -1 on error.
 *
 * This function implements a binary semaphore using the standard POSIX.1b
 * semaphores. This function uses the sem_getvalue() function. Since the entire
 * operation is not atomic, there is the possibility of a race condition in the
 * event that the semaphore is posted by more than one process/thread. Therefore,
 * this function should not be used in these circumstances. If you have a need to
 * use this function, you are probably better off using PTHREADS mutexes and
 * condition variables.
 */
int
rtx_sem_post_binary (
		RtxSem	*sem,       /**< Pointer to semaphore */
		int	count
		)
{
	int sval;

#ifdef sparc_solaris
	if (count < 0)
		return (rtx_error ("rtx_sem_post_binary: sem_getvalue() doesnt support"
					" negative values for semaphores"));
#endif

	if (sem_getvalue (SEMID(sem), &sval) == -1)
		return (rtx_error_errno ("rtx_sem_post_binary: sem_getvalue() failed"));

	if (sval <= count)
		if (sem_post (SEMID(sem)) == -1)
			return (rtx_error_errno ("rtx_sem_post_binary: sem_post() failed"));

	return (0);
}

#else // USE_SYSV_API



/**
 * \file sem.c
 * \brief A simple interface to semaphores
 * \author Pavan Sikka
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <signal.h>
#include <unistd.h>
#include <fcntl.h>
#include <errno.h>

#include "rtx/error.h"
#include "rtx/sem.h"

#include <sys/types.h>
#include <sys/ipc.h>
#include <sys/sem.h>


#if (defined(__GNU_LIBRARY__) && !defined(_SEM_SEMUN_UNDEFINED)) || defined DARWIN
/* l'union semun est dfinie par l'inclusion de <sys/sem.h> */
#else
/* d'aprs X/OPEN il faut la dfinir nous-mmes */
union semun {
	int val;                  /* valeur pour SETVAL */
	struct semid_ds *buf;     /* buffer pour IPC_STAT, IPC_SET */
	unsigned short *array;    /* table  pour GETALL, SETALL */
	/* Spcificit Linux : */
	struct seminfo *__buf;    /* buffer pour IPC_INFO */
};
#endif

#define SEMID(X) (X->sysvid)

/**
 * Initialize a unnamed semaphore.
 *
 * @return Pointer to a semaphore on success, NULL on error.
 *
 * This function is included for completeness and is just a wrapper around
 * the POSIX.1b sem_init() function.
 */
RtxSem *
rtx_sem_init (
		RtxSem *        semP,           /*!< use memory if provided */
		int 		pshared,        /*!< 1 => semaphore can be shared
									  by multiple processes */
		unsigned int	initialValue	/*!< Initial value of semaphore */
		)
{
	union semun su;
	unsigned short tmp;
	RtxSem	*sem = semP;

	if (sem == NULL) {
		if ((sem = (RtxSem *)calloc(1, sizeof(RtxSem))) == NULL)
			return (rtx_error_null ("rtx_sem_init: calloc: no memory")); 
		sem->memAlloc = 1;
	} else {
		sem->memAlloc = 0;
	}

	SEMID(sem) = semget(IPC_PRIVATE,1,IPC_CREAT | 0600);
	//printf("Created %d\n",SEMID(sem));
	if (SEMID(sem) == -1) {
		if (sem->memAlloc)
			free (sem);
		return (rtx_error_errno_null ("rtx_sem_init: sem_init: "));
	}
	tmp = initialValue;
	su.array = &tmp;
	if (semctl(SEMID(sem), 0, SETALL, su) == -1) {
		if (sem->memAlloc)
			free (sem);
		return (rtx_error_errno_null ("rtx_sem_init: sem_init: "));
	}

	return sem;
}

/**
 * Destroy a semaphore.
 *
 * @return 0 if OK, -1 on error.
 *
 * This function is included for completeness and is just a wrapper around
 * the POSIX.1b sem_destroy() function.
 */
int
rtx_sem_destroy (
		RtxSem *sem       /**< Pointer to semaphore */
		)
{
	union semun su;
	int errs = 0;

	int result = -1;

	if ((result = semctl (SEMID(sem), 0, GETNCNT, su)) == -1) {
		rtx_error_errno_null ("rtx_post_sem: semctl: ");
	} else {
		//printf("sem_destroy %d : value %d\n",SEMID(sem),result);
		if (result > 0) {
			// Unlocking all waiting process
			//printf("sem_destroy %d : unlocking %d\n",SEMID(sem),result);
			struct sembuf op[1];
			op[0].sem_num = 0;
			op[0].sem_op = result;
			op[0].sem_flg = 0;

			result = semop (SEMID(sem), op, 1);
		}
	}

	if (semctl(SEMID(sem),0,IPC_RMID,su) == -1) {
		rtx_error_errno ("rtx_sem_destroy: sem_destroy: ");
		errs++;
	}
	//printf("sem_destroy %d : destroyed\n",SEMID(sem));
	if (sem->memAlloc)
		free (sem);

	if (errs)
		return (-1);
	return (0);
}


/**
 * Wait on a semaphore.
 *
 * @return 0 if OK, -1 on error.
 *
 * This function is included for completeness and is just a wrapper around
 * the POSIX.1b sem_wait() function.
 */
int
rtx_sem_wait (
		RtxSem *sem       /**< Pointer to semaphore */
		)
{
	struct sembuf op[1];
	int result;

	op[0].sem_num = 0;
	op[0].sem_op = -1;
	op[0].sem_flg = 0;

	//printf("Waiting %d\n",SEMID(sem));
	result = semop (SEMID(sem), op, 1);
	if (result < 0) 
		rtx_error_errno ("rtx_sem_wait: semop: ");
	return (result);
}

/**
 * Wait on a semaphore, ignore EINTR errors.
 *
 * @return 0 if OK, -1 on error.
 */
int
rtx_sem_wait_ignore_eintr (
		RtxSem *sem      /**< Pointer to semaphore */
		)
{
	struct sembuf op[1];
	int result;

	op[0].sem_num = 0;
	op[0].sem_op = -1;
	op[0].sem_flg = 0;

	while ((result = semop (SEMID(sem), op, 1)) == -1) {
		if (errno == EINTR) {
			continue;
		} else {
			break;
		}
	}
	return (result);
}

/**
 * Post a semaphore.
 *
 * @return 0 if OK, -1 on error.
 *
 * This function is a little redundant since sem_post() cannot return with the
 * EINTR error. However, it still provides a wrapping function using the error
 * facility.
 */
int
rtx_sem_post (
		RtxSem *sem       /**< Pointer to semaphore */
		)
{
	struct sembuf op[1];
	int result = -1;

	op[0].sem_num = 0;
	op[0].sem_op = 1;
	op[0].sem_flg = 0;

	while ((result = semop (SEMID(sem), op, 1)) == -1) {
		if (result == -1) {
			if (errno == EINTR)
				continue;
			else
				break;
		}
	}
	return (result);
}

/**
 * Post a binary semaphore.
 *
 * @return 0 if OK, -1 on error.
 *
 * This function implements a binary semaphore using the standard POSIX.1b
 * semaphores. This function uses the sem_getvalue() function. Since the entire
 * operation is not atomic, there is the possibility of a race condition in the
 * event that the semaphore is posted by more than one process/thread. Therefore,
 * this function should not be used in these circumstances. If you have a need to
 * use this function, you are probably better off using PTHREADS mutexes and
 * condition variables.
 */
int
rtx_sem_post_binary (
		RtxSem	*sem,       /**< Pointer to semaphore */
		int	count
		)
{
	union semun su;
	struct sembuf op[1];
	int result = -1;

	if ((result = semctl (SEMID(sem), 0, GETVAL, su)) == -1) {
		return (int)(rtx_error_errno_null ("rtx_post_sem: semctl: "));
	} else {
		if (result > 0)
			return (0);
	}

	op[0].sem_num = 0;
	op[0].sem_op = 1;
	op[0].sem_flg = 0;

	while ((result = semop (SEMID(sem), op, 1)) == -1) {
		if (result == -1) {
			if (errno == EINTR)
				continue;
			else
				break;
		}
	}
	return (result);
}


#endif
