/***********************************************************************
 * 
 * CSIRO Autonomous Systems Laboratory
 * Queensland Centre for Advanced Technologies
 * PO Box 883, Kenmore, QLD 4069, Australia
 * http://www.ict.csiro.au/
 *  
 * Copyright (c) CSIRO 
 ***********************************************************************/

/**
 * \file   quintic.c
 * \author Peter Corke
 * \brief  Simple quintic polynomial package.  
 * 
 * These functions are useful for motion planning.
 */

#include <stdio.h>
#include <math.h>

#include "rtx/defines.h"
#include "rtx/quintic.h"

static char rcsid[] RTX_UNUSED = "$Id: quintic.c 2274 2007-12-23 05:37:32Z roy029 $";

/**
 * Initialize a polynomial object.
 *
 * @param q pointer to a quintic polynomial object.
 * @param tmax duration of the motion.
 * @param q0 initial value
 * @param q1 final value
 * @param qd0 initial velocity
 * @param qd1 final velocity
 *
 * @note Boundary accelerations are zero.
 */
void
rtx_quintic_init(RtxQuintic *q, double tmax, double q0, double q1,
	double qd0, double qd1)
{
	q->tmax = tmax;
	q->A = 6*(q1 - q0) - 3*(qd1+qd0)*tmax;
	q->B = -15*(q1 - q0) + (8*qd0 + 7*qd1)*tmax;
	q->C = 10*(q1 - q0) - (6*qd0 + 4*qd1)*tmax;
	q->D = 0;
	q->E = qd0*tmax; 	/* as the t vector has been normalized */
	q->F = q0;
}

/**
 *  Evaluate the polynomial.
 *
 * @param q pointer to a quintic polynomial object.
 * @param t time along the path, 0 <= t <= tmax
 * @param flag pointer to a flag which is set to zero at end of trajectory (can
 * be null).
 *
 * @return the position value
 *
 * The value of \p t is clipped to the range 0 to \p tmax.  When t > \p tmax
 * the flag pointed to by \p flag has zero written to it.
 */
double
rtx_quintic_evaluate(RtxQuintic *q, double t, int *flag)
{
	t = t / q->tmax;
	if (t < 0)
		t = 0.0;
	if (t> 1.0) {
		t = 1.0;
		if (flag)
			*flag = 0;
	}
	/* express it in Horner form for efficiency */
	return q->F+(q->D+(q->C+(q->B+q->A*t)*t)*t)*t*t+q->E*t;
}

/**
 *  Evaluate the polynomial, return position and velocity.
 *
 * @param p pointer in which position is stored
 * @param v pointer in which velocity is stored
 * @param q pointer to a quintic polynomial object.
 * @param t time along the path, 0 <= t <= tmax
 *
 * The value of \p t is clipped to the range 0 to \p tmax.
 */
void
rtx_quintic_evaluate2(double *p, double *v, RtxQuintic *q, double t)
{
	t = t / q->tmax;
	if (t < 0)
		t = 0.0;
	if (t> 1)
		t = 1.0;
	/* express it in Horner form for efficiency */
	*p = q->F+(q->D+(q->C+(q->B+q->A*t)*t)*t)*t*t+q->E*t;
	*v = ((((5*q->A*t + 4*q->B)*t + 3*q->C)*t + 2*q->D)*t + q->E) / q->tmax;
}

/**
 *  Evaluate the polynomial, return position, velocity and acceleration.
 *
 * @param p pointer in which position is stored
 * @param v pointer in which velocity is stored
 * @param a pointer in which acceleration is stored
 * @param q pointer to a quintic polynomial object.
 * @param t time along the path, 0 <= t <= tmax
 *
 * The value of \p t is clipped to the range 0 to \p tmax.
 */
void
rtx_quintic_evaluate3(double *p, double *v, double *a, RtxQuintic *q, double t)
{
	t = t / q->tmax;
	if (t < 0)
		t = 0.0;
	if (t> 1)
		t = 1.0;
	/* express it in Horner form for efficiency */
	*p = ((((((q->A*t + q->B)*t + q->C)*t + q->D)*t) + q->E)*t + q->F);
	*v = ((((5*q->A*t + 4*q->B)*t + 3*q->C)*t + 2*q->D)*t + q->E) / q->tmax;
	*a = (((20*q->A*t + 12*q->B)*t + 6*q->C)*t + 2*q->D) / (q->tmax*q->tmax);
}

/**
 * Determine the acceleration behaviour of the quintic.  Quintic polynomials
 * are generally used to create smooth motion paths that conform to boundary
 * conditions.  In some cases, large position change but small final velocity,
 * the generated acceleration profile changes sign during the motion.  This
 * function determines the location of the acceleration zero-crossing, if
 * it exists, and also the acceleration peak time and value.
 *
 * @param q pointer to a quintic polynomial object.
 * @param tmax pointer in which acceleration peak time is stored.
 * @param amax pointer in which acceleration peak value is stored.
 * @param tzero pointer in which acceleration zero time is stored.
 * 
 * \ref A new approach to solving the cubic: Cardan's solution revealed.
 *  R.W.D. Nickalls, The Mathematical Gazette, 1993, Vol 77, pp 354-358.
 *
 */
void
rtx_quintic_accel(RtxQuintic *q, double *tmax, double *amax, double *tzero)
{
	double	root[3];
	double	a, b, c, d;
	double	xn, yn, yn2, h, h2, delta;
	int	i, nroots = 0;

	/* acceleration peak */
	if (tmax)
		*tmax = -q->B/q->A/4.0;

	if (amax)
		*amax = -0.75/q->A*q->B*q->B + 2.0*q->C;

	if (tzero == NULL)
		return;

	/* acceleration zero crossings */

	if (q->A >= 0) {
		a = q->A;
		b = q->B;
		c = q->C;
		d = q->D;
	} else {
		a = -q->A;
		b = -q->B;
		c = -q->C;
		d = -q->D;
	}

	xn = -b/(3*a);
	yn = ((a*xn+b)*xn+c)*xn+d;	// Horner form
	yn2 = yn*yn;
	delta = sqrt( (b*b-3*a*c)/(9*a*a) );
	h = 2*a*delta*delta*delta;
	h2 = h*h;

	if (yn2 > h2) {
		// 1 real root
		nroots = 1;
		root[0] = xn + cbrt(1.0/(2.0*a)*(-yn+sqrt(yn2-h2)))
			+ cbrt(1.0/(2.0*a)*(-yn-sqrt(yn2-h2)));

	} else if (yn2 < h2) {
		// 3 real roots
		double	th;

		nroots = 3;
		th = acos(-yn/h)/3.0;
		root[0] = xn + 2.0*delta*cos(th);
		root[1] = xn + 2.0*delta*cos(th+2.0*M_PI/3.0);
		root[2] = xn + 2.0*delta*cos(th+4.0*M_PI/3.0);
	} else {
		// repeated real roots
		nroots = 2;
		root[0] = delta;		// repeated
		root[1] = -2.0*delta;
	}

	// check for roots within bound, return first root found that lies
	// within the range [0, tmax]
	for (i=0; i<nroots; i++)
		if ((root[i] > 0.0) && (root[i] < q->tmax)) {
			*tzero = root[i];
			return;
		}
}
