/***********************************************************************
 * 
 * CSIRO Autonomous Systems Laboratory
 * Queensland Centre for Advanced Technologies
 * PO Box 883, Kenmore, QLD 4069, Australia
 * http://www.ict.csiro.au/
 *  
 * Copyright (c) CSIRO 
 ***********************************************************************/

#include	<stdio.h>
#include	<stdlib.h>
#include	<math.h>
#include	"rtx/error.h"
#include	"rtx/time.h"
#include	"rtx/message.h"
#include	"rtx/param.h"
#include	"rtx/export.h"
#include	"rtx/pid.h"

static char rcsid[] RTX_UNUSED = "$Id: pid.c 3819 2008-08-15 04:30:09Z pra082 $";

static char * loop_param_string(RtxParamStream *pf, char *loopName, char *param, int required);
static int loop_param_double(RtxParamStream *pf, char *loopName, char *param, double *p, int required);

static int loop_export_double(char *name, double *p, RtxPid *P);
#if 0 /*not used for now */
static int loop_param_int(RtxParamStream *pf, char *loopName, char *param, int *p, int required);
static int loop_export_int(char *name, int *p, RtxPid *P);
#endif

/*
 * 1 debugs from initialization
 * 2 debugs from control loop
 */

/**
 * Initialize a proportional–integral–derivative (PID) control loop.
 *
 * @param paramFile pointer to parameter stream containing configuration
 * @param name name of the loop, used to find parameters
 * @param dt sample interval for this loop, if zero, use parameter dt.
 * @param verbosity level of verbosity, 0 is quiet, more is more
 *
 * A data structure is initialized and parameters are read from the stream.  For
 * example if name = "wrist" then parameters would be named wrist/P, wrist/D etc.
 *
 * @return 0 if OK, -1 on error
 */
RtxPid *
rtx_pid_init(RtxPid *P, RtxParamStream *pf, char *name, double dt, int verbosity)
{
	static char	*func = "rtx_pid_init";
	int	s = 0;

	if (P == NULL) {
		if ((P = (RtxPid *)calloc(1, sizeof(RtxPid))) == NULL)
			return rtx_error_null("%s: no memory", func);
	}

	P->verbose = verbosity;
	if (P->verbose)
		rtx_message("loading parameters for loop %s\n", name);
	P->name = name;

	/*
	 * read optional loop parameters, if not given they default to zero.
	 */
	P->units = loop_param_string(pf, P->name, "units", 0);
	loop_param_double(pf, P->name, "ratelimit", &P->ratelimit, 0);
	loop_param_double(pf, P->name, "deadband", &P->deadband, 0);
	loop_param_double(pf, P->name, "intband", &P->intband, 0);
	loop_param_double(pf, P->name, "offset", &P->offset, 0);
	loop_param_double(pf, P->name, "Kff", &P->Kff, 0);

	/* now read mandatory loop parameters */
	s += loop_param_double(pf, P->name, "P", &P->P, 1);
	s += loop_param_double(pf, P->name, "I", &P->I, 1);
	s += loop_param_double(pf, P->name, "D", &P->D, 1);
	s += loop_param_double(pf, P->name, "min", &P->min, 1);
	s += loop_param_double(pf, P->name, "max", &P->max, 1);
	s += loop_param_double(pf, P->name, "Imin", &P->Imin, 1);
	s += loop_param_double(pf, P->name, "Imax", &P->Imax, 1);
	if (dt == 0.0)
		s += loop_param_double(pf, P->name, "dt", &P->dt, 1);
	else
		P->dt = dt;
	/* Set previous demand to Inf (flag for first call to pid_eval) */
	/* Yes this is strange but NAN doesn't seem to be defined! */
	P->dmd_prev = sqrt(-1);
	rtx_pid_integral_reset(P);
	if (s != 0)
		return rtx_error_null("%s: missing parameters for loop %s",
			func, name);

	return P;
}

/**
 * Implement control law for one loop.
 *
 * @param P loop to receive the demand
 * @param y current plant output
 * @param ystar demand value
 * @param ystar_d demand value rate, used for feedforward only.
 * @param option options flag associated with the demand
 * @return the demand to the plant
 *
 */
double
rtx_pid_eval(RtxPid *P, double y, double ystar, double ystar_d)
{
	static char	*func = "loop_control";
	double	d, u, e, ed;

	P->feedback = y;	/* stash current plant output */
	
	/* 
	 * Set value of previous dmd to prevent weird things happening
	 * with the rate limit
	 */
	if(isnan(P->dmd_prev))
		P->dmd_prev = ystar;

	/*
	 * apply rate limit to demand
	 */
	if (P->ratelimit != 0.0) {
		d = (ystar - P->dmd_prev) / P->dt;
		if (d > P->ratelimit)
			d = P->ratelimit;
		else if (d < -P->ratelimit)
			d = -P->ratelimit;
		P->dmd = P->dmd_prev + d * P->dt;
		P->dmd_prev = P->dmd;

		if (ystar_d > P->ratelimit)
			ystar_d = P->ratelimit;
		else if (ystar_d < -P->ratelimit)
			ystar_d = -P->ratelimit;
	}else{
        P->dmd = ystar;
    }
        

	/*
	 * control
	 */
	P->error = e = P->dmd - P->feedback;


	/* apply deadband to error */
	if (P->deadband != 0.0) {
		if (fabs(e) < P->deadband)
			e = 0;
	}

	/* compute integral */
	if (P->Ienable == 0)
		P->integral = 0;
	else {
		P->integral += P->I * e;
		if (P->integral > P->Imax)
			P->integral = P->Imax;
		else if (P->integral < P->Imin)
			P->integral = P->Imin;
	}

	/* compute derivative */
	ed = P->e_prev - e;
	P->e_prev = e;

	/* control law */
	u = P->P * e + P->integral + P->D * ed + P->offset + P->Kff * ystar_d;

	//rtx_message("dmd=%f, fb=%f e=%f u=%f", P->dmd, *P->feedback, e, u);

	/* clip it */
	if (u > P->max)
		u = P->max;
	else if (u < P->min)
		u = P->min;

	if (P->verbose > 2)
		rtx_message("%s: loop %s, dmd=%f, fb=%f",
			func, P->name, P->dmd, y);

	return u;
}

/**
 * Reset integral action on this loop.
 */
void
rtx_pid_integral_reset(RtxPid *P)
{
	P->integral = 0.0;
}

/**
 * Enable integral action on this loop.
 */
void
rtx_pid_integral_enable(RtxPid *P)
{
	P->Ienable = 1;
}

/**
 * Disable integral action on this loop.
 */
void
rtx_pid_integral_disable(RtxPid *P)
{
	P->Ienable = 0;
}


/**
 * Export variables from this loop.
 */
int
rtx_pid_export(RtxPid *P)
{
	loop_export_double("dmd", &P->dmd, P);
	loop_export_double("err", &P->error, P);
	loop_export_double("actual", &P->feedback, P);

	return 0;
}


/**
 * Reset the controller (resets integral and rate limit stuff).
 */
void
rtx_pid_reset(RtxPid *P)
{
	rtx_pid_integral_reset(P);
	/* Set previous demand to Inf (flag for first call to pid_eval) */
	/* Yes this is strange but NAN doesn't seem to be defined! */
	P->dmd_prev = sqrt(-1);
}


/*************************************************************************
 *    s u p p o r t    f u n c t i o n s
 ************************************************************************/

static int
loop_export_double(char *name, double *p, RtxPid *P)
{
	char	b[BUFSIZ];

	sprintf(b, "%s_%s", P->name, name);
	return rtx_export_double(b, p, 1,
		RTX_EXPORT_POLL | RTX_EXPORT_READONLY, P->units);
}

static char *
loop_param_string(RtxParamStream *pf, char *loopName, char *param, int required)
{
	char	b[BUFSIZ];
	char	*p;

	sprintf(b, "%s/%s", loopName, param);
	if ((p = rtx_param_get_string(pf, b, NULL, 0)) == NULL) {
		if (required) {
			return rtx_error_null("loop parameter [%s] not found", b);
		}
	}
	return p;
}

static int
loop_param_double(RtxParamStream *pf, char *loopName, char *param, double *p, int required)
{
	char	b[BUFSIZ];

	sprintf(b, "%s/%s", loopName, param);
	if (rtx_param_get_double(pf, b, p, 1) < 0) {
		if (required) {
			return rtx_error("loop parameter [%s] not found", b);
		}
	}
	//rtx_export_double(b, p, 1, 0, "");
	
	return 0;
}

#if 0
/* not used for now */
static int
loop_export_int(char *name, int *p, RtxPid *P)
{
	char	b[BUFSIZ];

	sprintf(b, "%s_%s", P->name, name);
	return rtx_export_int(b, p, 1,
		RTX_EXPORT_POLL | RTX_EXPORT_READONLY, NULL);
}

static int
loop_param_int(RtxParamStream *pf, char *loopName, char *param, int *p, int required)
{
	char	b[BUFSIZ];

	sprintf(b, "%s/%s", loopName, param);
	if (rtx_param_get_int(pf, b, p, 1) < 0) {
		if (required) {
			return rtx_error("loop parameter [%s] not found", b);
		}
	}

	return 0;
}
#endif
