/***********************************************************************
 * 
 * CSIRO Autonomous Systems Laboratory
 * Queensland Centre for Advanced Technologies
 * PO Box 883, Kenmore, QLD 4069, Australia
 * http://www.ict.csiro.au/
 *  
 * Copyright (c) CSIRO 
 ***********************************************************************/

/**
 * \file parse.c
 * \brief Parser for simple 'C' data-structures
 * \author Pavan Sikka
 *
 * The parser module provides a complete set of functions to deal with C
 * data structures. Typically, the user should first use rtx_parse_init()
 * to initialize the symbol tables for the current architecture. Then, the
 * user can use rtx_parse() to get parse-trees for simple C declarations.
 *
 * The user can use rtx_parse_init_remote() to initialize the symbol tables
 * for a remote architecture. The two sets of symbol tables can then be used
 * separately to produce parse trees for the different architectures. These
 * parse trees, coupled with the symbol tables, can be used to translate data
 * between the two architectures.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <string.h>

#include "rtx/defines.h"
#include "rtx/message.h"
#include "rtx/error.h"
#include "rtx/parse.h"
#include "rtx/hash.h"

static char rcsid[] RTX_UNUSED = "$Id: parse.c 3087 2008-05-15 12:49:35Z roy029 $";

/* forward declarations */

static int rtx_parse_skip_white_space (char * s, int * p);
static int rtx_parse_get_name (char * s, int * p, RtxParseToken * t);
static int rtx_parse_get_number (char * s, int * p, RtxParseToken * t);
static int rtx_parse_get_next_token (char * s, int * p, RtxParseToken * t);

static int rtx_parse_var (char * s, int * p, RtxParseToken * t, 
			  RtxParseVar * v, RtxParseSymTabs * symTabs);
static int rtx_parse_type (char * s, int * p, RtxParseToken * t, 
			   RtxParseVar * v, RtxParseSymTabs * symTabs);
static int rtx_parse_decl_list (char * s, int * p, RtxParseToken * t,
				RtxParseVar * v, RtxParseSymTabs * symTabs);
static int rtx_parse_old_simple_decl (char * s, int * p, RtxParseToken * t,
				      RtxParseVar * v, RtxParseSymTabs * symTabs);
static int rtx_parse_old_struct_decl (char * s, int * p, RtxParseToken * t,
				      RtxParseVar * v, RtxParseSymTabs * symTabs);

/**
 * get the size/alignment of basic compiler data types
 *
 */
void
rtx_parse_get_compiler_parms (
			      int dataSize[],     /**< size of data-type */
			      int dataAlign[]     /**< alignemnt of data type */
			      )
{
  /* Normally, the sizeof() and __alignof__() operators should have been
   * used. However, the compiler under Linux reports an alignment of 8
   * for double but uses an alignment of 4. Hence, the following hack.
   */
    char charData;
    short shortData;
    int intData;
    long longData;
    float floatData;
    double doubleData;
    
    struct {
	char a;
	char b;
    } z1;

    struct {
	char a;
	short b;
    } z2;

    struct {
	char a;
	int b;
    } z3;

    struct {
	char a;
	long b;
    } z4;

    struct {
	char a;
	float b;
    } z5;

    struct {
	char a;
	double b;
    } z6;

    
    dataSize[rtx_char_t] = sizeof (charData);
    dataSize[rtx_short_t] = sizeof (shortData);
    dataSize[rtx_int_t] = sizeof (intData);
    dataSize[rtx_long_t] = sizeof (longData);
    dataSize[rtx_float_t] = sizeof (floatData);
    dataSize[rtx_double_t] = sizeof (doubleData);
    dataAlign[rtx_char_t] = (char *) &z1.b - (char *) &z1;
    dataAlign[rtx_short_t] = (char *) &z2.b - (char *) &z2;
    dataAlign[rtx_int_t] = (char *) &z3.b - (char *) &z3;
    dataAlign[rtx_long_t] = (char *) &z4.b - (char *) &z4;
    dataAlign[rtx_float_t] = (char *) &z5.b - (char *) &z5;
    dataAlign[rtx_double_t] = (char *) &z6.b - (char *) &z6;
}


/**
 * Release memory allocated by rtx_parse_init
 */
void rtx_parse_finish(RtxParseSymTabs * symtabs)
{
	rtx_hash_free(symtabs->constTable);
	rtx_hash_free(symtabs->typeTable);
	rtx_hash_free(symtabs->varTable);
	free(symtabs);
}

/**
 * initialize symbol tables with basic data-types
 *
 * @return pointer to sturucture containing symbol table, NULL on error
 */
RtxParseSymTabs *
rtx_parse_init (void)
{
    RtxParseVar * v;
    int dataSize[8], dataAlign[8];
    RtxParseSymTabs * symTabs;

    if ((symTabs = calloc (1, sizeof (RtxParseSymTabs))) == NULL) {
      return (rtx_error_null ("rtx_parse_init: calloc() failed"));
    }

    if ((symTabs->constTable = rtx_hash_alloc(MAX_SYM_TAB_SIZE, 
					      rtx_hash_shallow_string_keys, rtx_hash_shallow_values,
					      (rtx_error_t)NULL)) == NULL) {
      return (rtx_error_null ("rtx_parse_init: rtx_hash_alloc() failed"));
    }

    if ((symTabs->typeTable = rtx_hash_alloc(MAX_SYM_TAB_SIZE,
					     rtx_hash_shallow_string_keys, rtx_hash_shallow_values,
					     (rtx_error_t)NULL)) == NULL) {
        return (rtx_error_null("rtx_parse_init: rtx_hash_alloc() failed"));
    }

    if ((symTabs->varTable = rtx_hash_alloc(MAX_SYM_TAB_SIZE, rtx_hash_shallow_string_keys, rtx_hash_shallow_values,
					    (rtx_error_t)NULL)) == NULL) {
      return (rtx_error_null("rtx_parse_init: rtx_hash_alloc() failed"));
    }
    
    symTabs->typeNumber = rtx_struct_t;

    rtx_parse_get_compiler_parms (dataSize, dataAlign);
    /* add rtx_struct_t */
    if ((v = (RtxParseVar *) calloc (1, sizeof (RtxParseVar))) == NULL)
        return (rtx_error_errno_null ("rtx_parse_init: "
				 "calloc(rtx_struct_t) failed"));
    strcpy (v->name, "struct");
    v->type = rtx_struct_t;
    v->dim = 0;
    v->alignment = 0;
    v->offset = 0;
    v->size = 0;
    v->next = NULL;
    v->subVar = NULL;
    if (rtx_parse_add_type_symbol (v, 1, symTabs->typeTable, &(symTabs->typeNumber)))
        return (rtx_error_null("rtx_parse_init: "
			   "rtx_parse_add_type_symbol(rtx_struct_t) failed"));
    /* add rtx_char_t */
    if ((v = (RtxParseVar *) calloc (1, sizeof (RtxParseVar))) == NULL)
        return (rtx_error_errno_null ("rtx_parse_init: "
				 "calloc(rtx_char_t) failed"));
    strcpy (v->name, "char");
    v->type = rtx_char_t;
    v->dim = 0;
    v->alignment = dataAlign[rtx_char_t];
    v->offset = 0;
    v->size = dataSize[rtx_char_t];
    v->next = NULL;
    v->subVar = NULL;
    if (rtx_parse_add_type_symbol (v, 1, symTabs->typeTable, &(symTabs->typeNumber)))
        return (rtx_error_null("rtx_parse_init: "
			   "rtx_parse_add_type_symbol(rtx_char_t) failed"));
    /* add rtx_short_t */
    if ((v = (RtxParseVar *) calloc (1, sizeof (RtxParseVar))) == NULL)
        return (rtx_error_errno_null ("rtx_parse_init: "
				 "calloc(rtx_short_t) failed"));
    strcpy (v->name, "short");
    v->type = rtx_short_t;
    v->dim = 0;
    v->alignment = dataAlign[rtx_short_t];
    v->offset = 0;
    v->size = dataSize[rtx_short_t];
    v->next = NULL;
    v->subVar = NULL;
    if (rtx_parse_add_type_symbol (v, 1, symTabs->typeTable, &(symTabs->typeNumber)))
        return (rtx_error_null("rtx_parse_init: "
			   "rtx_parse_add_type_symbol(rtx_short_t) failed"));
    /* add rtx_int_t */
    if ((v = (RtxParseVar *) calloc (1, sizeof (RtxParseVar))) == NULL)
        return (rtx_error_errno_null ("rtx_parse_init: "
				 "calloc(rtx_int_t) failed"));
    strcpy (v->name, "int");
    v->type = rtx_int_t;
    v->dim = 0;
    v->alignment = dataAlign[rtx_int_t];
    v->offset = 0;
    v->size = dataSize[rtx_int_t];
    v->next = NULL;
    v->subVar = NULL;
    if (rtx_parse_add_type_symbol (v, 1, symTabs->typeTable, &(symTabs->typeNumber)))
        return (rtx_error_null("rtx_parse_init: "
			   "rtx_parse_add_type_symbol(rtx_int_t) failed"));
    /* add rtx_long_t */
    if ((v = (RtxParseVar *) calloc (1, sizeof (RtxParseVar))) == NULL)
        return (rtx_error_errno_null ("rtx_parse_init: "
				 "calloc(rtx_long_t) failed"));
    strcpy (v->name, "long");
    v->type = rtx_long_t;
    v->dim = 0;
    v->alignment = dataAlign[rtx_long_t];
    v->offset = 0;
    v->size = dataSize[rtx_long_t];
    v->next = NULL;
    v->subVar = NULL;
    if (rtx_parse_add_type_symbol (v, 1, symTabs->typeTable, &(symTabs->typeNumber)))
        return (rtx_error_null("rtx_parse_init: "
			   "rtx_parse_add_type_symbol(rtx_long_t) failed"));
    /* add rtx_float_t */
    if ((v = (RtxParseVar *) calloc (1, sizeof (RtxParseVar))) == NULL)
        return (rtx_error_errno_null ("rtx_parse_init: "
				 "calloc(rtx_float_t) failed"));
    strcpy (v->name, "float");
    v->type = rtx_float_t;
    v->dim = 0;
    v->alignment = dataAlign[rtx_float_t];
    v->offset = 0;
    v->size = dataSize[rtx_float_t];
    v->next = NULL;
    v->subVar = NULL;
    if (rtx_parse_add_type_symbol (v, 1, symTabs->typeTable, &(symTabs->typeNumber)))
        return (rtx_error_null("rtx_parse_init: "
			   "rtx_parse_add_type_symbol(rtx_float_t) failed"));
    /* add rtx_double_t */
    if ((v = (RtxParseVar *) calloc (1, sizeof (RtxParseVar))) == NULL)
        return (rtx_error_errno_null ("rtx_parse_init: "
				 "calloc(rtx_double_t) failed"));
    strcpy (v->name, "double");
    v->type = rtx_double_t;
    v->dim = 0;
    v->alignment = dataAlign[rtx_double_t];
    v->offset = 0;
    v->size = dataSize[rtx_double_t];
    v->next = NULL;
    v->subVar = NULL;
    if (rtx_parse_add_type_symbol (v, 1, symTabs->typeTable, &(symTabs->typeNumber)))
        return (rtx_error_null("rtx_parse_init: "
			   "rtx_parse_add_type_symbol(rtx_double_t) failed"));
    return (symTabs);
}

/**
 * initialize symbol tables with basic data-types
 *
 * @return pointer to sturucture containing symbol table, NULL on error
 */
RtxParseSymTabs *
rtx_parse_init_remote (
		       int typeAlign[],   /**< alignment of data-type */
		       int typeSize[]     /**< size of data-type */
		       )
{
    RtxParseVar * v;
    RtxParseSymTabs * symTabs;

    if ((symTabs = calloc (1, sizeof (RtxParseSymTabs))) == NULL) {
      return (rtx_error_null("rtx_parse_init: calloc() failed"));
    }

    if ((symTabs->constTable = rtx_hash_alloc(MAX_SYM_TAB_SIZE, 
					      rtx_hash_shallow_string_keys, rtx_hash_shallow_values,
					      (rtx_error_t)NULL)) == NULL) {
        return (rtx_error_null ("rtx_parse_init: rtx_hash_alloc() failed"));
    }

    if ((symTabs->typeTable = rtx_hash_alloc(MAX_SYM_TAB_SIZE,
					     rtx_hash_shallow_string_keys, rtx_hash_shallow_values,
					     (rtx_error_t)NULL)) == NULL) {
      return (rtx_error_null("rtx_parse_init: rtx_hash_alloc() failed"));
    }

    if ((symTabs->varTable = rtx_hash_alloc(MAX_SYM_TAB_SIZE, rtx_hash_shallow_string_keys, rtx_hash_shallow_values,
					    (rtx_error_t)NULL)) == NULL) {
            return (rtx_error_null("rtx_parse_init: rtx_hash_alloc() failed"));
    }

    symTabs->typeNumber = rtx_struct_t;

    /* add rtx_struct_t */
    if ((v = (RtxParseVar *) calloc (1, sizeof (RtxParseVar))) == NULL)
        return (rtx_error_errno_null ("init_type_table_remote: "
				 "calloc(rtx_struct_t) failed"));
    strcpy (v->name, "struct");
    v->type = rtx_struct_t;
    v->dim = 0;
    v->alignment = 0;
    v->offset = 0;
    v->size = 0;
    v->next = NULL;
    v->subVar = NULL;
    if (rtx_parse_add_type_symbol (v, 1, symTabs->typeTable, &(symTabs->typeNumber)))
        return (rtx_error_null("init_type_table_remote: "
			   "rtx_parse_add_type_symbol(rtx_struct_t) failed"));
    /* add rtx_char_t */
    if ((v = (RtxParseVar *) calloc (1, sizeof (RtxParseVar))) == NULL)
        return (rtx_error_errno_null ("init_type_table_remote: "
				 "calloc(char) failed"));
    strcpy (v->name, "char");
    v->type = rtx_char_t;
    v->dim = 0;
    v->alignment = typeAlign[rtx_char_t];
    v->offset = 0;
    v->size = typeSize[rtx_char_t];
    v->next = NULL;
    v->subVar = NULL;
    if (rtx_parse_add_type_symbol (v, 1, symTabs->typeTable, &(symTabs->typeNumber)))
        return (rtx_error_null("init_type_table_remote: "
			   "rtx_parse_add_type_symbol(rtx_char_t) failed"));
    /* add rtx_short_t */
    if ((v = (RtxParseVar *) calloc (1, sizeof (RtxParseVar))) == NULL)
        return (rtx_error_errno_null ("init_type_table_remote: "
				 "calloc(rtx_short_t) failed"));
    strcpy (v->name, "short");
    v->type = rtx_short_t;
    v->dim = 0;
    v->alignment = typeAlign[rtx_short_t];
    v->offset = 0;
    v->size = typeSize[rtx_short_t];
    v->next = NULL;
    v->subVar = NULL;
    if (rtx_parse_add_type_symbol (v, 1, symTabs->typeTable, &(symTabs->typeNumber)))
        return (rtx_error_null("init_type_table_remote: "
			   "rtx_parse_add_type_symbol(rtx_short_t) failed"));
    /* add rtx_int_t */
    if ((v = (RtxParseVar *) calloc (1, sizeof (RtxParseVar))) == NULL)
        return (rtx_error_errno_null ("init_type_table_remote: "
				 "calloc(rtx_int_t) failed"));
    strcpy (v->name, "int");
    v->type = rtx_int_t;
    v->dim = 0;
    v->alignment = typeAlign[rtx_int_t];
    v->offset = 0;
    v->size = typeSize[rtx_int_t];
    v->next = NULL;
    v->subVar = NULL;
    if (rtx_parse_add_type_symbol (v, 1, symTabs->typeTable, &(symTabs->typeNumber)))
        return (rtx_error_null("init_type_table_remote: "
			   "rtx_parse_add_type_symbol(rtx_int_t) failed"));
    /* add rtx_long_t */
    if ((v = (RtxParseVar *) calloc (1, sizeof (RtxParseVar))) == NULL)
        return (rtx_error_errno_null ("init_type_table_remote: "
				 "calloc(rtx_long_t) failed"));
    strcpy (v->name, "long");
    v->type = rtx_long_t;
    v->dim = 0;
    v->alignment = typeAlign[rtx_long_t];
    v->offset = 0;
    v->size = typeSize[rtx_long_t];
    v->next = NULL;
    v->subVar = NULL;
    if (rtx_parse_add_type_symbol (v, 1, symTabs->typeTable, &(symTabs->typeNumber)))
        return (rtx_error_null("init_type_table_remote: "
			   "rtx_parse_add_type_symbol(rtx_long_t) failed"));
    /* add rtx_float_t */
    if ((v = (RtxParseVar *) calloc (1, sizeof (RtxParseVar))) == NULL)
        return (rtx_error_errno_null ("init_type_table_remote: "
				 "calloc(rtx_float_t) failed"));
    strcpy (v->name, "float");
    v->type = rtx_float_t;
    v->dim = 0;
    v->alignment = typeAlign[rtx_float_t];
    v->offset = 0;
    v->size = typeSize[rtx_float_t];
    v->next = NULL;
    v->subVar = NULL;
    if (rtx_parse_add_type_symbol (v, 1, symTabs->typeTable, &(symTabs->typeNumber)))
        return (rtx_error_null("init_type_table_remote: "
			   "rtx_parse_add_type_symbol(rtx_float_t) failed"));
    /* add rtx_double_t */
    if ((v = (RtxParseVar *) calloc (1, sizeof (RtxParseVar))) == NULL)
        return (rtx_error_errno_null ("init_type_table_remote: "
				 "calloc(rtx_double_t) failed"));
    strcpy (v->name, "double");
    v->type = rtx_double_t;
    v->dim = 0;
    v->alignment = typeAlign[rtx_double_t];
    v->offset = 0;
    v->size = typeSize[rtx_double_t];
    v->next = NULL;
    v->subVar = NULL;
    if (rtx_parse_add_type_symbol (v, 1, symTabs->typeTable, &(symTabs->typeNumber)))
        return (rtx_error_null("init_type_table_remote: "
			   "rtx_parse_add_type_symbol(rtx_double_t) failed"));
    return (symTabs);
}

/**
 * compare two RtxParseVar structures
 *
 * @return 0 if same, -1 otherwise
 */
int
rtx_parse_cmp_types (
		     RtxParseVar * v1,    /**< first var */
		     RtxParseVar * v2     /**< second var */
		     )
{
    if (v1 == v2)
        return (0);
    if ((v1 == NULL) && (v2 != NULL))
        return (-1);
    if ((v1 != NULL) && (v2 == NULL))
        return (-1);

    if (v1->type != v2->type)
        return (-1);
    if (v1->dim != v2->dim)
        return (-1);
    if (v1->alignment != v2->alignment)
        return (-1);
    if (v1->offset != v2->offset)
        return (-1);
    if (v1->size != v2->size)
        return (-1);

    if (rtx_parse_cmp_types (v1->subVar, v2->subVar))
        return (-1);
    return (rtx_parse_cmp_types (v1->next, v2->next));
}

/**
 * free a var structure
 *
 */
void
rtx_parse_free_var (
        RtxParseVar * v
    )
{
    if (v == NULL)
        return;
    if (v->subVar != NULL)
        rtx_parse_free_var (v->subVar);
    if (v->next != NULL)
        rtx_parse_free_var (v->next);
    free (v);
}

/**
 * copy a var node
 *
 * @return pointer to node copy, NULL on error
 */
static RtxParseVar *
rtx_parse_copy_var_node (
			 RtxParseVar * v    /**< node to be copied */
			 )
{
    RtxParseVar * v1;

    if (v == NULL)
        return (NULL);

    if ((v1 = (RtxParseVar *) calloc (1, sizeof (RtxParseVar))) == NULL) {
        return (rtx_error_errno_null ("copy_var_node (%s): "
				      "calloc() failed", v->name));
    }

    memcpy (v1, v, sizeof (RtxParseVar));
    return (v1);
}

/**
 * copy a var structure
 *
 * @return pointer to structure copy, NULL on error
 */
RtxParseVar *
rtx_parse_copy_var (
		    RtxParseVar * v      /**< top-level var of structure to copy */
		    )
{
    RtxParseVar * v1;

    if ((v1 = rtx_parse_copy_var_node (v)) == NULL) {
        return (rtx_error_null ("rtx_parse_copy_var (%s): "
				"copy_var_node() failed", v->name));
    }

    if (v->next != NULL) {
        if ((v1->next = rtx_parse_copy_var (v->next)) == NULL) {
	    rtx_error_null ("rtx_parse_copy_var (%s): copy_var(%s) failed",
			    v->name, v->next->name);
  	    rtx_parse_free_var (v1);
	    return (NULL);
	}
    }
    if (v->subVar != NULL) {
        if ((v1->subVar = rtx_parse_copy_var (v->subVar)) == NULL) {
	    rtx_error_null ("rtx_parse_copy_var (%s): copy_var(%s) failed",
			    v->name, v->subVar->name);
  	    rtx_parse_free_var (v1);
	    return (NULL);
	}
    }
    return (v1);
}

/**
 * expand a var structure
 *
 * @return 0 if OK, -1 on error
 */
int
rtx_parse_expand_var_definition (
				 RtxParseVar * v,     /**< var to expand */
				 rtx_hash_t symTab     /**< type table */
				 )
{
	RtxParseTypeTableEntry * s;
	int i;
	RtxParseVar * v2;

	if (v == NULL)
		return (0);


	switch (v->type) {
		case rtx_char_t :
		case rtx_short_t :
		case rtx_int_t :
		case rtx_long_t :
		case rtx_float_t :
		case rtx_double_t :
			/* Expand the next node, if any */
			if (v->next != NULL) {
				if (rtx_parse_expand_var_definition (v->next, symTab) == -1)
					return (rtx_error ("rtx_parse_expand_var_definition(%s): failed",
								v->name));
				else
					return (0);
			}
			return (0);
			break;
		case rtx_struct_t :
			/* Expand the next node, if any */
			if (v->next != NULL) {
				if (rtx_parse_expand_var_definition (v->next, symTab) == -1)
					return (rtx_error ("rtx_parse_expand_var_definition(%s): failed",
								v->name));
				else
					return (0);
			}
			/* Expand sub-structure */
			if (rtx_parse_expand_var_definition (v->subVar, symTab) == -1)
				return (rtx_error ("rtx_parse_expand_var_definition(%s): failed",
							v->name));
			else
				return (0);
			break;
		default :
			if ((s = rtx_parse_lookup_type (v->type, symTab)) == NULL) {
				return (rtx_error ("rtx_parse_expand_var_definition(%s): "
							"not found type %d", v->name, v->type));
			}
			switch (s->t->type) {
				case rtx_char_t :
				case rtx_short_t :
				case rtx_int_t :
				case rtx_long_t :
				case rtx_float_t :
				case rtx_double_t :
					/* copy the defn to v, no further expansion */
					v->type = s->t->type;
					v->dim += s->t->dim;
					for (i=0; i<s->t->dim; i++)
						v->arrayDim[i+v->dim] = s->t->arrayDim[i];
					v->elemSize = s->t->elemSize;
					/* Expand the next node, if any */
					if (v->next != NULL) {
						if (rtx_parse_expand_var_definition (v->next, symTab) == -1)
							return (rtx_error ("rtx_parse_expand_var_definition(%s): failed",
										v->name));
						else
							return (0);
					}
					return (0);
					break;
				case rtx_struct_t :
					if ((v2 = rtx_parse_copy_var (s->t)) == NULL)
						return (-1);
					v->type = v2->type;
					v->dim += s->t->dim;
					for (i=0; i<s->t->dim; i++)
						v->arrayDim[i+v->dim] = s->t->arrayDim[i];
					v->elemSize = s->t->elemSize;
					v->subVar = v2->subVar;
					v2->subVar = NULL;
					v2->next = NULL;
					rtx_parse_free_var (v2);
					/* Expand the next node, if any */
					if (rtx_parse_expand_var_definition (v->next, symTab) == -1)
						return (rtx_error ("rtx_parse_expand_var_definition(%s): failed",
									v->name));
					/* Expand sub-structure */
					if (rtx_parse_expand_var_definition (v->subVar, symTab) == -1)
						return (rtx_error ("rtx_parse_expand_var_definition(%s): failed",
									v->name));
					return (0);
					break;
				default :
					/* copy the defn to v, then expand v */
					v->type = s->t->type;
					v->dim += s->t->dim;
					for (i=0; i<s->t->dim; i++)
						v->arrayDim[i+v->dim] = s->t->arrayDim[i];
					v->elemSize = s->t->elemSize;
					if (rtx_parse_expand_var_definition (v, symTab) == -1)
						return (rtx_error ("rtx_parse_expand_var_definition(%s): failed",
									v->name));
					else
						return (0);
					break;
			}
	}
	return (rtx_error ("rtx_parse_expand_var_definition(%s): serious problem - should "
				"never have reached this point", v->name));
}

/* This kludge is required because of the need to lookup from the
 * hash table based on an attribute other than name.
 */

struct _rtx_parse_hash_comp_type_t {
    RtxParseTypeTableEntry * tp;
    int typeNumber;
};

static void
rtx_parse_hash_comp_type_func (void *key,
			       void *data,
			       int i, int j,
			       int *stop_iter,
			       void *context)
{
  RtxParseTypeTableEntry * tp = (RtxParseTypeTableEntry *) data;
  struct _rtx_parse_hash_comp_type_t * xp;

    xp = (struct _rtx_parse_hash_comp_type_t *) context;
    if (xp->typeNumber == tp->typeNum)
        xp->tp = tp;
}

/**
 * lookup a type by number in the type table
 *
 * @return pointer to type table entry, NULL if not found
 */
RtxParseTypeTableEntry *
rtx_parse_lookup_type (
		       int typeNumber,
		       rtx_hash_t symTab
		       )
{
    struct _rtx_parse_hash_comp_type_t x;

    x.tp = NULL;
    x.typeNumber = typeNumber;
    rtx_hash_iter(symTab, rtx_parse_hash_comp_type_func, &x);

    return (x.tp);
}

/**
 * delete a symbol from the const table
 *
 * 0 if OK
 */
int
rtx_parse_del_const_symbol (
			    char * name,        /**< entry to delete */
			    rtx_hash_t symTab    /**< hash table to delete from */
			    )
{
    RtxParseConstTableEntry * tp;
    int errs = 0;

    if ((tp = rtx_hash_find (symTab, name, (rtx_error_t)NULL)) != NULL) {
      if (! rtx_hash_delete (symTab, name, (rtx_error_t)NULL)) {
  	    rtx_error ("rtx_parse_del_const_symbol: rtx_hash_delete() failed");
	    errs++;
	}
	free (tp);
    }
    return (errs);
}

/**
 * delete a symbol from the type table
 *
 * 0 if OK
 */
int
rtx_parse_del_type_symbol (
			   char * name,        /**< entry to delete */
			   rtx_hash_t symTab    /**< hash table to delete from */
			   )
{
    if (! rtx_hash_delete (symTab, name, (rtx_error_t)NULL)) {
      rtx_error ("rtx_parse_del_type_symbol: rtx_hash_delete() failed");
      return -1;
    }
    return 0;
}

/**
 * delete a symbol from the var table
 *
 * 0 if OK
 */
int
rtx_parse_del_var_symbol (
			  char * name,        /**< entry to delete */
			  rtx_hash_t symTab    /**< hash table to delete from */
			  )
{
    RtxParseVarTableEntry * tp;
    int errs = 0;

    if ((tp = rtx_hash_find (symTab, name, (rtx_error_t)NULL)) != NULL) {
      if (! rtx_hash_delete(symTab, name, (rtx_error_t)NULL)) {
  	    rtx_error ("rtx_parse_del_var_symbol: rtx_hash_delete() failed");
	    errs++;
	}
	rtx_parse_free_var (tp->v);
	free (tp);
    }
    return (errs);
}


/**
 * add an entry to the const table
 *
 * 0 if OK
 */
int
rtx_parse_add_const_symbol (
			    char * name,       /**< name of entry to add */
			    int val,           /**< value of entry to add */
			    rtx_hash_t symTab   /**< table to add to */
    )
{
    RtxParseConstTableEntry * tp;

    if ((tp = (RtxParseConstTableEntry *) calloc (1, sizeof (RtxParseConstTableEntry)))
	    == NULL) {
        return (rtx_error_errno ("rtx_parse_add_const_symbol (%s): calloc() failed",
				 name));
    }
    strcpy (tp->name, name);
    tp->value = val;
    if (rtx_hash_set(symTab, tp->name, tp, (rtx_error_t)NULL)) {
	  return (rtx_error ("rtx_parse_add_const_symbol (%s): rtx_hash_set() failed",
						 name));
	}

    return (0);
}

/**
 * add an entry to the type table
 *
 * 0 if OK
 */
int
rtx_parse_add_type_symbol (
			   RtxParseVar * v,        /**< var structure for the type */
			   int primitiveType,      /**< basic/user-defined type */
			   rtx_hash_t symTab,       /**< table to add to */
			   int * typeNumber        /**< type number */
    )
{
    RtxParseTypeTableEntry * tp;

    if ((tp = (RtxParseTypeTableEntry *) calloc (1, sizeof (RtxParseTypeTableEntry)))
	    == NULL) {
        return (rtx_error_errno ("rtx_parse_add_type_symbol (%s): calloc() failed",
				 v->name));
    }
    strcpy (tp->name, v->name);
    tp->typeNum = *typeNumber;
    (* typeNumber)++;
    tp->t = v;
    tp->primitiveType = primitiveType;
    if (rtx_hash_set(symTab, v->name, tp, (rtx_error_t)NULL)) {
        return (rtx_error ("rtx_parse_add_type_symbol (%s): rtx_hash_set() failed",
			   v->name));
	}
    return (0);
}

/**
 * add an entry to the var table
 *
 * 0 if OK
 */
int
rtx_parse_add_var_symbol (
			  RtxParseVar * v,   /**< var structure */
			  int varId,         /**< var id */
			  int offset,        /**< shared memory offset for data */
			  int headerOffset,  /**< shared memory offset for header */
			  char * varDecl,    /**< var declaration (text) */
			  rtx_hash_t symTab   /**< table to add to */
    )
{
    RtxParseVarTableEntry * tp;

    if ((tp = (RtxParseVarTableEntry *) calloc (1, sizeof (RtxParseVarTableEntry)))
	    == NULL) {
        return (rtx_error_errno ("rtx_parse_add_var_symbol (%s): calloc() failed",
				 v->name));
    }
    strcpy (tp->name, v->name);
    tp->v = v;
    tp->varId = varId;
    tp->offset = offset;
    tp->headerOffset = headerOffset;
    if ((tp->decl = strdup (varDecl)) == NULL) {
        free (tp);
        return (rtx_error_errno ("rtx_parse_add_var_symbol (%s): strdup() failed",
				 v->name));
    }
    if (rtx_hash_set(symTab, v->name, tp, (rtx_error_t)NULL)) {
	  return (rtx_error ("rtx_parse_add_var_symbol (%s): rtx_hash_set() failed",
						 v->name));
	}

    return (0);
}

/**
 * skip comments from the declaration
 *
 * @return 0 if input exhausted, 1 otherwise
 */
static int
rtx_parse_skip_comment (
			char * s,     /**< declaration being processed */
			int * p       /**< current offset into the declaration string */
			)
{
    int i = * p, n = MAX_STR_LEN;

    i = * p;
    while ((s[i] != '\n') && n) {
        i++;
	n--;
    }
    * p = i;
    if (s[i] == '\0')
        return (0);
    else
        return (1);
}

/**
 * skip all white space from the declaration string
 *
 * @return 0 if input exhausted, 1 otherwise
 */
static int
rtx_parse_skip_white_space (
			    char * s,     /**< declaration being processed */
			    int * p       /**< current offset into the decl string */
			    )
{
    int i = * p, n = MAX_STR_LEN;

    i = * p;
    while ((isspace (s[i]) || (s[i] == '%') || (s[i] == '#')) && n) {
        if ((s[i] == '%') || (s[i] == '#')) {
	    rtx_parse_skip_comment (s, &i);
	} else {
	    i++;
	    n--;
	}
    }
    * p = i;
    if (s[i] == '\0')
        return (0);
    else
        return (1);
}

/**
 * get name token from declaration string
 *
 * @return token id
 */
static int
rtx_parse_get_name (
		    char * s,     /**< declaration being processed */
		    int * p,       /**< current offset into the declaration string */
		    RtxParseToken * t    /**< token */
		    )
{
    int i = * p, n = MAX_TOK_LEN, j = 0;
    t->tokStr[j++] = s[i++];

    /* skip to end of name */
    while ((isalpha (s[i]) || isdigit (s[i]) || (s[i] == '_')) &&
	   (s[i] != '\0') && n) {
        t->tokStr[j++] = s[i++];
	n--;
    }
    t->tokStr[j] = '\0';
    * p = i;

    /* check if its a reserverd word */
    if (strcmp (t->tokStr, "struct") == 0) {
	t->tokId = TOK_STRUCT;
    } else if (strcmp (t->tokStr, "char") == 0) {
        t->tokId = TOK_CHAR;
    } else if (strcmp (t->tokStr, "short") == 0) {
        t->tokId = TOK_SHORT;
    } else if (strcmp (t->tokStr, "int") == 0) {
        t->tokId = TOK_INT;
    } else if (strcmp (t->tokStr, "long") == 0) {
        t->tokId = TOK_LONG;
    } else if (strcmp (t->tokStr, "float") == 0) {
        t->tokId = TOK_FLOAT;
    } else if (strcmp (t->tokStr, "double") == 0) {
        t->tokId = TOK_DOUBLE;
    } else if (strcmp (t->tokStr, "bool") == 0) {
        t->tokId = TOK_BOOL;
    } else if (strcmp (t->tokStr, "real") == 0) {
        t->tokId = TOK_REAL;
    } else if (strcmp (t->tokStr, "byte") == 0) {
        t->tokId = TOK_BYTE;
    } else if (strcmp (t->tokStr, "typedef") == 0) {
        t->tokId = TOK_TYPEDEF;
    } else if (strcmp (t->tokStr, "unsigned") == 0) {
        t->tokId = TOK_UNSIGNED;
    } else {
        t->tokId = TOK_NAME;
    }
    return (t->tokId);
}

/**
 * get number token from declaration string
 *
 * @return token id
 */
static int
rtx_parse_get_number (
		      char * s,     /**< declaration being processed */
		      int * p,       /**< current offset into the declaration string */
		      RtxParseToken * t    /**< token */
		      )
{
    int i = * p, n = MAX_TOK_LEN, j = 0;
    t->tokStr[j++] = s[i++];

    /* skip to end of initial number part */
    while (isdigit (s[i]) && n) {
        t->tokStr[j++] = s[i++];
	n--;
    }
    if ((s[i] != '.') && (s[i] != 'e') && (s[i] != 'E')) {
        /* we probably have an integer */
        if ((! isdigit (t->tokStr[0])) && (strlen (t->tokStr) == 1)) {
  	    /* we have an error */
	    return (rtx_error ("rtx_parse_get_number (%s): invalid token",
			       t->tokStr));
	}
        * p = i;
	t->tokStr[j] = '\0';
	t->tokVal.intVal = atoi (t->tokStr);
	t->tokId = TOK_INT_NUMBER;
	return (t->tokId);
    }

    /* get fraction part */
    if (s[i] == '.') {
        t->tokStr[j++] = s[i++];
	while (isdigit (s[i]) && n) {
	    t->tokStr[j++] = s[i++];
	    n--;
	}
    }

    if ((s[i] != 'e') && (s[i] != 'E')) {
        /* we dont have an exponent */
        * p = i;
	t->tokStr[j] = '\0';
	t->tokVal.realVal = atof (t->tokStr);
	t->tokId = TOK_REAL_NUMBER;
	return (t->tokId);
    }

    /* get exponent */
    t->tokStr[j++] = s[i++];
    if ((s[i] == '+') || (s[i] == '-'))
        t->tokStr[j++] = s[i++];
    if (! isdigit (s[i])) {
        /* we have an error */
        return (rtx_error ("rtx_parse_get_number (%s): invalid token [%s]",
			   t->tokStr, &(s[i])));
    }
    while (isdigit (s[i]) && n) {
        t->tokStr[j++] = s[i++];
	n--;
    }

    * p = i;
    t->tokStr[j] = '\0';
    t->tokVal.realVal = atof (t->tokStr);
    t->tokId = TOK_REAL_NUMBER;
    return (t->tokId);
}

/**
 * get next token from declaration string
 *
 * @return token id, 0 if input exhausted
 */
static int
rtx_parse_get_next_token (
			  char * s,     /**< declaration being processed */
			  int * p,       /**< current offset into the decl string */
			  RtxParseToken * t    /**< token */
			  )
{
    int i;
    char * argStr;

    argStr = &(s[*p]);

    /* get to the first significant character of the next token */
    if (rtx_parse_skip_white_space (s, p) == 0)
        return (0);
    
    i = * p;
    /* check if its a punctuation */
    t->tokStr[0] = s[i];
    switch (s[i]) {
        case '{' :
	    t->tokId = TOK_LEFT_BRACE;
	    t->tokStr[1] = '\0';
	    * p = i + 1;
	    return (t->tokId);
        case '}' :
  	    t->tokId = TOK_RIGHT_BRACE;
	    t->tokStr[1] = '\0';
	    * p = i + 1;
	    return (t->tokId);
        case '[' :
  	    t->tokId = TOK_LEFT_BRACKET;
	    t->tokStr[1] = '\0';
	    * p = i + 1;
	    return (t->tokId);
        case ']' :
  	    t->tokId = TOK_RIGHT_BRACKET;
	    t->tokStr[1] = '\0';
	    * p = i + 1;
	    return (t->tokId);
        case ';' :
  	    t->tokId = TOK_SEMICOLON;
	    t->tokStr[1] = '\0';
	    * p = i + 1;
	    return (t->tokId);
        case '.' :
  	    t->tokId = TOK_DOT;
	    t->tokStr[1] = '\0';
	    * p = i + 1;
	    return (t->tokId);
    }

    /* check if its a type or a name */
    if (isalpha (s[i]) || (s[i] == '_')) {
        return (rtx_parse_get_name (s, p, t));
    }

    /* check if its a number */
    if (isdigit (s[i]) || (s[i] == '+') || (s[i] == '-')) {
        return (rtx_parse_get_number (s, p, t));
    }

    /* input is invalid */
    return (rtx_error ("rtx_parse_get_next_token (%s): invlaid token", argStr));
}

/* Recursive descent parser for store data structures */

/**
 * parse a 'C' declaration
 *
 * @return parse-tree if OK, NULL on error
 */
RtxParseVar *
rtx_parse (
	   char * s,                   /**< declaration string */
	   RtxParseSymTabs * symTabs   /**< symbol tables */
	   )
{
    int p = 0;
    RtxParseToken t;
    RtxParseVar * v = NULL;

    memset (&t, 0, sizeof (t));
    if ((v = (RtxParseVar *) calloc (1, sizeof (RtxParseVar))) == NULL)
        return (rtx_error_null ("no memory"));
    if (rtx_parse_decl (s, &p, &t, v, symTabs) == -1)
        return (rtx_error_null ("rtx_parse_decl() failed"));
    return (v);
}

/**
 * recursively parse a 'C' declaration
 *
 * @return 0 if OK, -1 on error
 */
int
rtx_parse_decl (
		char * s,                   /**< declaration string */
		int * p,                    /**< current offset into decl string */
		RtxParseToken * t,          /**< current parse token */
		RtxParseVar * v,            /**< var structure being built */
		RtxParseSymTabs * symTabs   /**< symbol tables */
		)
{
	RtxParseToken t1;
	int i, n = 1;
	int ret;
	int rem = 0;
	char * argStr;

	if (symTabs == NULL)
		return (rtx_error ("rtx_parse_decl: cannot parse without tables"));

	argStr = &(s[*p]);

	if ((ret = rtx_parse_type (s, p, &t1, v, symTabs)) == 0) {
		return (0);
	}
	if (ret == -1) {
		return (rtx_error ("rtx_parse_decl: rtx_parse_type failed [%s]", argStr));
	}
	if (rtx_parse_var (s, p, &t1, v, symTabs) == -1) {
		return (rtx_error ("rtx_parse_decl: rtx_parse_var failed [%s]", argStr));
	}
	if ((rem = v->elemSize % v->alignment)) 
		v->elemSize = v->elemSize + (v->alignment - rem);
	if (v->dim) {
		for (i=0; i<v->dim; i++) {
			n *= v->arrayDim[i];
		}
		if (n <= 0) {
			return (rtx_error ("rtx_parse_decl (%s): array size %d (< 0)",
						v->name));
		}
		v->size = n * v->elemSize;
	} else {
		v->size = v->elemSize;
	}
	if (rtx_parse_get_next_token (s, p, &t1) == TOK_SEMICOLON) {
		return (1);
	}
	return (rtx_error ("rtx_parse_decl (%s): missing semicolon", v->name));
}

/**
 * parse a 'C' declaration list (as in a structure)
 *
 * @return 0 if OK, -1 on error
 */
static int
rtx_parse_decl_list (
		     char * s,                   /**< declaration string */
		     int * p,                    /**< current offset into decl string */
		     RtxParseToken * t,          /**< current parse token */
		     RtxParseVar * v,            /**< var structure being built */
		     RtxParseSymTabs * symTabs   /**< symbol tables */
		     )
{
    RtxParseToken t1;
    int p1;
    int done = 0, res;
    RtxParseVar * v1, * v2;
    int ret;
    char * argStr;

    argStr = &(s[*p]);

    if ((v1 = (RtxParseVar *) calloc (1, sizeof (RtxParseVar))) == NULL)
        return (rtx_error_errno ("rtx_parse_decl_list: no memory [%s]",
				 argStr));

    if ((ret = rtx_parse_decl (s, p, &t1, v1, symTabs)) == -1) {
        rtx_error ("rtx_parse_decl_list: rtx_parse_decl [%s]", argStr);
	rtx_parse_free_var (v1);
        return (-1);
    }
    if (ret == 0) {
        rtx_error ("rtx_parse_decl_list: unexpected end of input [%s]",
		   argStr);
	rtx_parse_free_var (v1);
        return (-1);
    }
    v->subVar = v1;
    v->next = NULL;
    v->elemSize = v1->size;
    v1->offset = v->offset;
    if (v1->alignment > v->alignment)
        v->alignment = v1->alignment;

    p1 = * p;
    while (! done) {
        if ((res = rtx_parse_get_next_token (s, &p1, &t1)) < 0) {
	    rtx_error ("rtx_parse_decl_list: unexpected end of input [%s]",
		       argStr);
	    return (-1);
	}
	if (res == 0) {
	    return (0);
	}
	switch (t1.tokId) {
            case TOK_CHAR :
            case TOK_SHORT :
            case TOK_INT :
            case TOK_LONG :
            case TOK_FLOAT :
            case TOK_DOUBLE :
	    case TOK_STRUCT :
	    case TOK_UNSIGNED :
   	    case TOK_NAME :
	        if ((v2 = (RtxParseVar *) calloc (1, sizeof (RtxParseVar))) == NULL) {
		    rtx_error_errno ("rtx_parse_decl_list: calloc() failed [%s]",
				     argStr);
		    return (-1);
		}
	        if ((ret = rtx_parse_decl (s, p, &t1, v2, symTabs)) == -1) {
		    rtx_error_errno ("rtx_parse_decl_list: rtx_parse_decl() failed [%s]",
				     argStr);
		    rtx_parse_free_var (v2);
		    return (-1);
		}
	        if (ret == 0) {
		    rtx_error ("rtx_parse_decl_list: unexpected end of input [%s]",
			       argStr);
		    rtx_parse_free_var (v2);
		    return (-1);
		}
		v2->offset = v1->offset + v1->size;
		if (v2->offset % v2->alignment)
		    v2->offset = (v2->offset / v2->alignment + 1) *
		        v2->alignment;
		if (v2->alignment > v->alignment)
		    v->alignment = v2->alignment;
		v->elemSize += v2->size;
		v1->next = v2;
		v1 = v2;
		p1 = * p;
		break;
	    default :
	        v->elemSize = v1->size + v1->offset;
		return (0);
	}
    }
    return (0);
}

/**
 * parse the type-part of a 'C' declaration
 *
 * @return 0 if OK, -1 on error
 */
static int
rtx_parse_type (
		char * s,                   /**< declaration string */
		int * p,                    /**< current offset into decl string */
		RtxParseToken * t,          /**< current parse token */
		RtxParseVar * v,            /**< var structure being built */
		RtxParseSymTabs * symTabs   /**< symbol tables */
		)
{
	RtxParseToken t1;
	RtxParseTypeTableEntry * sym = NULL;
	int ret;
	char * argStr;

	argStr = &(s[*p]);

	if ((ret = rtx_parse_get_next_token (s, p, &t1)) == 0) {
		return (0);
	}
	if (ret == -1) {
		return (rtx_error ("rtx_parse_type: invalid token [%s]", argStr));
	}
	if (t1.tokId == TOK_TYPEDEF) {
		if (rtx_parse_get_next_token (s, p, &t1) <= 0) {
			return (rtx_error ("rtx_parse_type: invalid/no token [%s]", argStr));
		}
	}
	if (t1.tokId == TOK_UNSIGNED) {
		if (rtx_parse_get_next_token (s, p, &t1) <= 0) {
			return (rtx_error ("rtx_parse_type: invalid/no token [%s]", argStr));
		}
	}
	switch (t1.tokId) {
		case TOK_CHAR :
		case TOK_SHORT :
		case TOK_INT :
		case TOK_LONG :
		case TOK_FLOAT :
		case TOK_DOUBLE :
		case TOK_STRUCT :
		case TOK_NAME :
			if ((sym = (RtxParseTypeTableEntry *) rtx_hash_find 
			     (symTabs->typeTable, t1.tokStr, (rtx_error_t)NULL)) == NULL) {
				return (rtx_error ("rtx_parse_type: undefined type = %s [%s]",
							t1.tokStr, argStr));
			}
			break;
		default :
			return (rtx_error ("rtx_parse_type: undefined type = %s [%s]",
						t1.tokStr, argStr));
	}
	if (t1.tokId != TOK_STRUCT) {
		v->elemSize = sym->t->size;
		v->alignment = sym->t->alignment;
		v->type = sym->typeNum;
		v->subVar = NULL;
		return (1);
	}
	v->size = 0;
	v->elemSize = 0;
	v->type = sym->typeNum;
	if (rtx_parse_get_next_token (s, p, &t1) == TOK_NAME) {
		strcpy (v->tag, t1.tokStr);
		if (rtx_parse_get_next_token (s, p, &t1) != TOK_LEFT_BRACE) {
			return (rtx_error ("rtx_parse_type: expecting { [%s]", argStr));
		}
	} else if (t1.tokId != TOK_LEFT_BRACE) {
		return (rtx_error ("rtx_parse_type: expecting { [%s]", argStr));
	}
	if (rtx_parse_decl_list (s, p, &t1, v, symTabs) == -1) {
		return (rtx_error ("rtx_parse_type: rtx_parse_decl() failed [%s]", argStr));

	}
	if (rtx_parse_get_next_token (s, p, &t1) != TOK_RIGHT_BRACE) {
		return (rtx_error ("rtx_parse_type: expecting } [%s]", argStr));
	}
	return (1);
}

/**
 * parse the var-part of a 'C' declaration
 *
 * @return 0 if OK, -1 on error
 */
static int
rtx_parse_var (
	       char * s,                   /**< declaration string */
	       int * p,                    /**< current offset into decl string */
	       RtxParseToken * t,          /**< current parse token */
	       RtxParseVar * v,            /**< var structure being built */
	       RtxParseSymTabs * symTabs   /**< symbol tables */
	       )
{
    RtxParseToken t1;
    RtxParseConstTableEntry * sym = NULL;
    int symval;
    int p1, done = 0;
    char * argStr;

    argStr = &(s[*p]);

    if (rtx_parse_get_next_token (s, p, &t1) != TOK_NAME) {
        return (rtx_error ("rtx_parse_var: invalid name = %s [%s]",
			   t1.tokStr, argStr));
    }
    strcpy (v->name, t1.tokStr);
    v->dim = 0;
    while (! done) {
	p1 = * p;
	if (rtx_parse_get_next_token (s, &p1, &t1) != TOK_LEFT_BRACKET) {
	    return (0);
	}
	if (v->dim >= MAX_NUM_ARRAY_DIMS) {
	    return (rtx_error ("rtx_parse_var: dims (%d) > max (%d) [%s]",
			       v->dim, MAX_NUM_ARRAY_DIMS, argStr));
	}
	* p = p1;
	if (rtx_parse_get_next_token (s, p, &t1) == -1) {
	    return (rtx_error ("rtx_parse_var: unexpected end of "
			       "input [%s]", argStr));
	}
	if (t1.tokId == TOK_INT_NUMBER) {
	    if (t1.tokVal.intVal < 0) {
	        return (rtx_error ("rtx_parse_var: array size (%d) "
				   "< 0 [%s]", t1.tokVal.intVal,
				   argStr));
	    }
	    v->arrayDim[v->dim] = t1.tokVal.intVal;
	} else if (t1.tokId == TOK_NAME) {
	    sym = (RtxParseConstTableEntry *) rtx_hash_find
	      (symTabs->constTable, t1.tokStr, (rtx_error_t)NULL);
	    if (sym == NULL) {
	        return (rtx_error ("rtx_parse_var: undefined CONST = %s [%s]",
				   t1.tokStr, argStr));
	    }
	    symval = sym->value;
	    if (symval < 0) {
	        return (rtx_error ("rtx_parse_var: array size (%d) "
				   "< 0 [%s]", symval, argStr));
	    }
	    v->arrayDim[v->dim] = symval;
	}
	v->dim++;
	if (rtx_parse_get_next_token (s, p, &t1) != TOK_RIGHT_BRACKET) {
	    return (rtx_error ("rtx_parse_var: expecting ] [%s]", argStr));
	}
    }
    return (0);
}

/* mangle names */

static void
convert_name (
        char * name
    )
{
    char * p;

    if (strlen (name) > MAX_STR_LEN-1)
        name[MAX_STR_LEN-1] = '\0';
    for (p = name; p != NULL; ) {
        if ((p = strchr (p, '.')) != NULL) {
  	    * p = '~';
	    p++;
	}
    }
    for (p = name; p != NULL; ) {
        if ((p = strchr (p, '[')) != NULL) {
  	    * p = '$';
	    p++;
	}
    }
    for (p = name; p != NULL; ) {
        if ((p = strchr (p, ']')) != NULL) {
  	    * p = '^';
	    p++;
	}
    }
}

/* tag var structures */

static void
tag_var (
        RtxParseVar * v
    )
{
    RtxParseVar * v1;

    v->attribs |= VAR_ATTR_TAG;
    if (v->dim)
        v->attribs |= VAR_ATTR_ENTIRE_ARRAY;
    if (v->subVar != NULL)
        for (v1 = v->subVar; v1 != NULL; v1 = v1->next)
	    tag_var (v1);
}

/**
 * parse a 'C' variable name (including structure elements and arrays)
 *
 * @return a var structure, NULL on error
 */
RtxParseVar *
rtx_parse_name_list (
		     char * s,                   /**< declaration string */
		     int * p,                    /**< current offset into decl string */
		     RtxParseToken * t,          /**< current parse token */
		     RtxParseVar * v,            /**< var structure being built */
		     RtxParseSymTabs * symTabs   /**< symbol tables */
		     )
{
    RtxParseToken t1;
    RtxParseVarTableEntry * stEntry = NULL;
    int done = 0, found;
    int i, res;
    RtxParseVar * vEntry, * vin, * v1;
    char * argStr;

    argStr = &(s[*p]);
    
    if ((vin = (RtxParseVar *) calloc (1, sizeof (RtxParseVar))) == NULL) {
        return (rtx_error_errno_null ("rtx_parse_name_list: no memory [%s]",
				      argStr));
    }
    if (rtx_parse_var (s, p, t, vin, symTabs) == -1) {
        rtx_error ("rtx_parse_name_list: rtx_parse_var() failed [%s]", argStr);
	rtx_parse_free_var (vin);
	return (NULL);
    }
    strcpy (v->name, vin->name);
    if ((stEntry = (RtxParseVarTableEntry *) rtx_hash_find 
	 (symTabs->varTable, vin->name, (rtx_error_t)NULL)) == NULL) {
        rtx_error ("rtx_parse_name_list: top-level var (%s) not found [%s]",
		   v->name, argStr);
	rtx_parse_free_var (vin);
	return (NULL);
    }
    vEntry = stEntry->v;
    vEntry->attribs |= VAR_ATTR_TAG;
    if (vin->dim) {
        for (i=0; i<vin->dim; i++) {
	    if (vin->arrayDim[i] >= vEntry->arrayDim[i]) {
	        rtx_error ("rtx_parse_name_list: array index %d out of "
			   "bounds (%d) [%s]", vin->arrayDim[i],
			   vEntry->arrayDim[i], argStr);
		rtx_parse_free_var (vin);
		return (NULL);
	    }
	    vEntry->arrayIndex[i] = vin->arrayDim[i];
	    v->offset += vEntry->elemSize * vin->arrayDim[i];
	}
	v->size = vEntry->elemSize;
    } else {
        vEntry->attribs |= VAR_ATTR_ENTIRE_ARRAY;
        v->offset = vEntry->offset;
	v->size = vEntry->size;
    }
    while (! done) {
	res = rtx_parse_get_next_token (s, p, &t1);
        if (res == -1) {
	    rtx_error ("rtx_parse_name_list: invalid token [%s]", argStr);
	    rtx_parse_free_var (vin);
	    return (NULL);
	}
        if (res == 0) {
  	    done = 1;
	    continue;
	}
        if (res != TOK_DOT) {
	    rtx_error ("rtx_parse_name_list: missing . [%s]", argStr);
	    rtx_parse_free_var (vin);
	    return (NULL);
	}
        if (rtx_parse_var (s, p, t, vin, symTabs) == -1) {
	    rtx_error ("rtx_parse_name_list: rtx_parse_var() failed [%s]", argStr);
	    rtx_parse_free_var (vin);
	    return (NULL);
	}
	found = 0;
	for (v1 = vEntry->subVar; v1 != NULL; v1 = v1->next) {
	    if (strcmp (v1->name, vin->name) == 0) {
	        found = 1;
		break;
	    }
	}
	if (found == 0) {
	    rtx_error ("rtx_parse_name_list: %s not an element [%s]", 
		       vin->name, argStr);
	    rtx_parse_free_var (vin);
	    return (NULL);
	}
        vEntry = v1;
	vEntry->attribs |= VAR_ATTR_TAG;
	if (vin->dim) {
	    for (i=0; i<vin->dim; i++) {
	        if (vin->arrayDim[i] >= vEntry->arrayDim[i]) {
		    rtx_error ("rtx_parse_name_list: array index %d out of "
			       "bounds (%d) [%s]", vin->arrayDim[i],
			       vEntry->arrayDim[i], argStr);
		    rtx_parse_free_var (vin);
		    return (NULL);
		}
		vEntry->arrayIndex[i] = vin->arrayDim[i];
		v->offset += vEntry->elemSize * vin->arrayDim[i];
	    }
	    v->size = vEntry->elemSize;
	} else {
	    vEntry->attribs |= VAR_ATTR_ENTIRE_ARRAY;
	    v->offset = vEntry->offset;
	    v->size = vEntry->size;
	}
    }
    if (vEntry->subVar != NULL)
        for (v1 = vEntry->subVar; v1 != NULL; v1 = v1->next)
  	    tag_var (v1);
    v1 = rtx_parse_copy_var (vEntry);
    strcpy (v1->name, s);
    convert_name (v1->name);
    rtx_parse_free_var (vin);
    return (v1);
}

/* Recursive descent parser for old logger data structures */

/**
 * parse a variable declaration for the old logger
 *
 * @return 0 if OK, -1 on error
 */
int
rtx_parse_old_decl (
		    char * s,                   /**< declaration string */
		    int * p,                    /**< current offset into decl string */
		    RtxParseToken * t,          /**< current parse token */
		    RtxParseVar * v,            /**< var structure being built */
		    RtxParseSymTabs * symTabs   /**< symbol tables */
		    )
{
    RtxParseToken t1;
    int ret;
    char * argStr;

    if (symTabs == NULL)
        return (rtx_error ("rtx_parse_old_decl: cannot parse without tables"));

    argStr = &(s[*p]);

    if ((ret = rtx_parse_get_next_token (s, p, &t1)) == -1) {
        return (rtx_error ("rtx_parse_old_decl: invalid token [%s]", argStr));
    }
    if (ret == 0)
        return (0);
    switch (t1.tokId) {
        case TOK_BYTE :
        case TOK_SHORT :
        case TOK_BOOL :
        case TOK_INT :
        case TOK_REAL :
  	    return (rtx_parse_old_simple_decl (s, p, &t1, v, symTabs));
        case TOK_NAME :
	    strcpy (v->name, t1.tokStr);
  	    return (rtx_parse_old_struct_decl (s, p, &t1, v, symTabs));
        default :
	    return (rtx_error ("rtx_parse_old_decl: invalid type %s [%s]", 
			       t1.tokStr, argStr));
    }
}

/**
 * parse a simple variable declaration for the old logger
 *
 * @return 0 if OK, -1 on error
 */
static int
rtx_parse_old_simple_decl (
			   char * s,                   /**< declaration string */
			   int * p,                    /**< current offset into decl string */
			   RtxParseToken * t,          /**< current parse token */
			   RtxParseVar * v,            /**< var structure being built */
			   RtxParseSymTabs * symTabs   /**< symbol tables */
			   )
{
    RtxParseToken t1;
    int i, n = 1;
    char * argStr;

    argStr = &(s[*p]);

    v->subVar = NULL;
    switch (t->tokId) {
        case TOK_BYTE :
  	    v->elemSize = sizeof (char);
	    v->type = rtx_char_t;
	    break;
        case TOK_SHORT :
  	    v->elemSize = sizeof (short);
	    v->type = rtx_short_t;
	    break;
        case TOK_BOOL :
  	    v->elemSize = sizeof (int);
	    v->type = rtx_int_t;
	    break;
        case TOK_INT :
  	    v->elemSize = sizeof (int);
	    v->type = rtx_int_t;
	    break;
        case TOK_REAL :
  	    v->elemSize = sizeof (float);
	    v->type = rtx_float_t;
	    break;
    }
    if (rtx_parse_var (s, p, &t1, v, symTabs) == -1) {
        return (rtx_error ("rtx_parse_old_simple_decl: rtx_parse_var() failed [%s]",
			   argStr));
    }
    if (v->dim) {
        for (i=0; i<v->dim; i++) {
	    n *= v->arrayDim[i];
	}
	if (n <= 0) {
	    return (rtx_error ("rtx_parse_old_simple_decl: array size (%d) "
			       "< 0 [%s]", n, argStr));
	}
	v->size = n * v->elemSize;
    } else {
	v->size = v->elemSize;
    }
    v->offset = 0;
    if (rtx_parse_get_next_token (s, p, &t1) == TOK_SEMICOLON) {
	return (0);
    }
    return (rtx_error ("rtx_parse_old_simple_decl: missing ; [%s]", argStr));
}

/**
 * parse a structure declaration for the old logger
 *
 * @return 0 if OK, -1 on error
 */
static int
rtx_parse_old_struct_decl (
			   char * s,                   /**< declaration string */
			   int * p,                    /**< current offset into decl string */
			   RtxParseToken * t,          /**< current parse token */
			   RtxParseVar * v,            /**< var structure being built */
			   RtxParseSymTabs * symTabs   /**< symbol tables */
			   )
{
    RtxParseToken t1;
    int res;
    RtxParseVar * v1, * v2;
    char * argStr;

    argStr = &(s[*p]);
    
    if ((v1 = (RtxParseVar *) calloc (1, sizeof (RtxParseVar))) == NULL) {
        return (rtx_error_errno ("rtx_parse_old_struct_decl: no memory [%s]",
				 argStr));
    }
    if (rtx_parse_get_next_token (s, p, &t1) != TOK_LEFT_BRACE) {
        return (rtx_error ("rtx_parse_old_struct_decl: expecting { [%s]",
			   argStr));
    }
    v->type = rtx_struct_t;
    v->size = 0;
    v->elemSize = 0;
    v->subVar = v1;
    v->next = NULL;
    v1->offset = 0;

    if (rtx_parse_get_next_token (s, p, &t1) <= 0) {
        return (rtx_error ("rtx_parse_old_struct_decl: unexpected end of "
			   "input or invalid token [%s]", argStr));
    }
    while (t1.tokId != TOK_RIGHT_BRACE) {
        if (rtx_parse_old_simple_decl (s, p, &t1, v1, symTabs) == -1) {
	    return (rtx_error ("rtx_parse_old_struct_decl: rtx_parse_old_simple"
			       "_decl() failed [%s]", argStr));
	}
	v1->offset = v->size;
	v->size += v1->size;
        if ((res = rtx_parse_get_next_token (s, p, &t1)) <= 0) {
	    return (rtx_error ("rtx_parse_old_struct_decl: unexpected end of "
			       "input or invalid token [%s]", argStr));
	}
	if (t1.tokId == TOK_RIGHT_BRACE) {
     	    v1->next = NULL;
  	    continue;
	}
	if ((v2 = (RtxParseVar *) calloc (1, sizeof (RtxParseVar))) == NULL) {
	    return (rtx_error_errno ("rtx_parse_old_struct_decl: no memory [%s]",
				     argStr));
	}
	v1->next = v2;
	v1 = v2;
    }
    if (rtx_parse_get_next_token (s, p, &t1) != TOK_SEMICOLON) {
        return (rtx_error ("rtx_parse_old_struct_decl: missing ; [%s]",
			   argStr));
    }
    return (0);
}

/**
 * print the value of a var structure
 *
 * @return 0 if OK, -1 on error
 */
int
rtx_parse_print_var (
		     RtxParseVar * v,     /**< var structure to print */
		     void * din,          /**< data for the var structure */
		     char * s             /**< string to print to */
		     )
{
    int i, n;
    char * charp;
    short * shortp;
    int * intp;
    long * longp;
    float * floatp;
    double * doublep;
    char * d = (char *) din;
    RtxParseVar * v1;

    if (v == NULL)
        return (0);
    n = 1;
    for (i=0; i<v->dim; i++) {
        n *= v->arrayDim[i];
    }
    switch (v->type) {
        case rtx_char_t :
	    charp = (char *) &(d[v->offset]);
	    for (i=0; i<n; i++)
	        sprintf (s, "%s%d ", s, charp[i]);
	    break;
	case rtx_short_t :
	    shortp = (short *) &(d[v->offset]);
	    for (i=0; i<n; i++)
		sprintf (s, "%s%d ", s, shortp[i]);
	    break;
	case rtx_int_t :
	    intp = (int *) &(d[v->offset]);
	    for (i=0; i<n; i++)
		sprintf (s, "%s%d ", s, intp[i]);
	    break;
	case rtx_long_t :
	    longp = (long *) &(d[v->offset]);
	    for (i=0; i<n; i++)
		sprintf (s, "%s%ld ", s, longp[i]);
	    break;
	case rtx_float_t :
	    floatp = (float *) &(d[v->offset]);
	    for (i=0; i<n; i++)
		sprintf (s, "%s%f ", s, floatp[i]);
	    break;
	case rtx_double_t :
	    doublep = (double *) &(d[v->offset]);
	    for (i=0; i<n; i++)
		sprintf (s, "%s%f ", s, doublep[i]);
	    break;
	case rtx_struct_t :
	    for (i=0; i<n; i++)
	        for (v1 = v->subVar; v1 != NULL; v1 = v1->next)
		    rtx_parse_print_var (v1, &(d[v->elemSize*i+v->offset]), s);
	    break;
        default :
	    return (rtx_error ("rtx_parse_print_var (%s): invalid type (%d)",
			       v->name, v->type));
    }
    return (0);
}

/* static global variables used to convert a list of values into a var
 * structure
 */
static char * tok;                  /* check the need for this ! */
static char * sep = " \t\r\n";      /* read-only string */

/**
 * convert a text string (list of numbers) into the value of a var structure
 *
 * @return 0 if OK, -1 on error
 */
int
rtx_parse_scan_var (
		    RtxParseVar * v,      /**< var structure to be used */
		    void * din,           /**< data for the var structure */
		    char * s,             /**< text string (value-list) */
		    char ** lasts,        /**< used by strtok_r */
		    int initCall          /**< flag to indicate first call */
		    )
{
    int i, n;
    char * charp;
    short * shortp;
    int * intp;
    long * longp;
    float * floatp;
    double * doublep;
    char * d = (char *) din;
    RtxParseVar * v1;

    if (v == NULL)
        return (0);
    n = 1;
    for (i=0; i<v->dim; i++) {
        n *= v->arrayDim[i];
    }
    if (initCall) {
        tok = strtok_r (s, sep, lasts);
    }
    switch (v->type) {
        case rtx_char_t :
	    charp = (char *) &(d[v->offset]);
	    for (i=0; i<n; i++) {
	        sscanf (tok, "%c", charp);
		charp++;
		tok = strtok_r (NULL, sep, lasts);
	    }
	    break;
	case rtx_short_t :
	    shortp = (short *) &(d[v->offset]);
	    for (i=0; i<n; i++) {
	        sscanf (tok, "%hd", shortp);
		shortp++;
		tok = strtok_r (NULL, sep, lasts);
	    }
	    break;
	case rtx_int_t :
	    intp = (int *) &(d[v->offset]);
	    for (i=0; i<n; i++) {
	        sscanf (tok, "%d", intp);
		intp++;
		tok = strtok_r (NULL, sep, lasts);
	    }
	    break;
	case rtx_long_t :
	    longp = (long *) &(d[v->offset]);
	    for (i=0; i<n; i++) {
	        sscanf (tok, "%ld", longp);
		longp++;
		tok = strtok_r (NULL, sep, lasts);
	    }
	    break;
	case rtx_float_t :
	    floatp = (float *) &(d[v->offset]);
	    for (i=0; i<n; i++) {
	        sscanf (tok, "%f", floatp);
		floatp++;
		tok = strtok_r (NULL, sep, lasts);
	    }
	    break;
	case rtx_double_t :
	    doublep = (double *) &(d[v->offset]);
	    for (i=0; i<n; i++) {
	        sscanf (tok, "%lf", doublep);
		doublep++;
		tok = strtok_r (NULL, sep, lasts);
	    }
	    break;
	case rtx_struct_t :
	    for (i=0; i<n; i++)
	        for (v1 = v->subVar; v1 != NULL; v1 = v1->next)
		    rtx_parse_scan_var (v1, &(d[v->elemSize*i+v->offset]), s, lasts, 0);
	    break;
        default :
	    return (rtx_error ("rtx_parse_scan_var (%s): invalid type %d", 
			       v->name, v->type));
    }
    return (0);
}
/**
 * generate a 'C' declaration from a var structure
 *
 * @return 0 if OK, -1 on error
 */
int
rtx_parse_generate_var_decl (
			     RtxParseVar * v,      /**< var structure */
			     char * buf            /**< buffer to write the declaration */
			     )
{
	int i, n;
	char locString[32];
	RtxParseVar * v1;

	/* printf("IN  rtx_parse_generate_var_decl: '%s'\n",buf); */
	/* print type */
	switch (v->type) {
		case rtx_char_t :
			strcat (buf, " char ");
			break;
		case rtx_short_t :
			strcat (buf, " short ");
			break;
		case rtx_int_t :
			strcat (buf, " int ");
			break;
		case rtx_long_t :
			strcat (buf, " long ");
			break;
		case rtx_float_t :
			strcat (buf, " float ");
			break;
		case rtx_double_t :
			strcat (buf, " double ");
			break;
		case rtx_struct_t :
			sprintf (locString, " struct %s { ", v->tag);
			strcat (buf, locString);
			n = strlen (buf);
			for (v1 = v->subVar; v1 != NULL; v1 = v1->next) {
				rtx_parse_generate_var_decl (v1, buf);
			}
			strcat (buf, " } ");
			break;
		default :
			sprintf(locString, " blabla(%d) ",v->type);
			strcat (buf, locString);
			break;
	}
	sprintf (locString, " %s ", v->name);
	strcat (buf, locString);
	for (i=0; i<v->dim; i++) {
		sprintf (locString, "[ %d ]", v->arrayDim[i]);
		strcat (buf, locString);
	}
	strcat (buf, " ; ");
	/* printf("OUT rtx_parse_generate_var_decl: '%s'\n",buf); */
	return (0);
}

/*
 *
 */

static void
rtx_parse_swap_data (
	void * d,
	int len
    )
{
    unsigned char * p, * p1, t0;
    p = (unsigned char *) d;
    p1 = &(((unsigned char *) d) [len-1]);

    for ( ; p1 > p; p++, p1--) {
            t0 = * p;
	    * p = * p1;
	    * p1 = t0;
    }
}

/**
 * convert a binary value of a var structure to a different architecture
 *
 * Written with the assumption that primitive data sizes are the same.
 * This is true of the four platforms we use:
 *    - sparc-solaris
 *    - i486-linux
 *    - i486-lynxos
 *    - i486-qnx
 * 
 * Should be generalized to take sizes into account. That will be an
 * issue once we start using JAVA to access the store ...

 * @return 0 if OK, -1 on error
 */
int
rtx_parse_translate_data (
			  void * d,               /**< target data */
			  RtxParseVar * v,        /**< target var structure */
			  void * dRemote,         /**< source data */
			  RtxParseVar * vRemote,  /**< source var structure */
			  int swapType            /**< endian conversion required */
			  )
{
    int i, n;
    int srcOffset, destOffset;
    /*
    char charData;
    short shortData;
    int intData;
    long longData;
    float floatData;
    double doubleData;
    */
    char * din = (char *) dRemote;
    char * dout = (char *) d;
    RtxParseVar * v1, * v2;

    if (v == NULL) {
        return (0);
    }
    n = 1;
    if (v->dim) {
        for (i=0; i<v->dim; i++) {
  	    n *= v->arrayDim[i];
	}
    } else {
        n = 1;
    }

    switch (v->type) {
        case rtx_char_t :
	case rtx_short_t :
	case rtx_int_t :
	case rtx_long_t :
	case rtx_float_t :
	case rtx_double_t :
	    for (i=0; i<n; i++) {
	        memcpy (dout, din, v->elemSize);
		if (swapType)
		        rtx_parse_swap_data (dout, v->elemSize);
	        din += vRemote->elemSize;
	        dout += v->elemSize;
	    }
	    break;
#if 0
	case rtx_short_t :
	    for (i=0; i<n; i++) {
	        memcpy (&shortData, (din + i*v->elemSize), v->size);
		rtx_parse_swap_data (&shortData, v->elemSize, swapword);
		((short *) dout) [i] = shortData;
		/*
	        memcpy ((dout + i*v->size), &shortData, v->size);
		*/
	    }
	    break;
	case rtx_int_t :
	    for (i=0; i<n; i++) {
	        memcpy (&intData, (din + i*v->size), v->size);
		rtx_parse_swap_data (&intData, v->elemSize, swapword);
		((int *) dout) [i] = intData;
		/*
	        memcpy ((dout + i*v->size), &intData, v->size);
		*/
	    }
	    break;
	case rtx_long_t :
	    for (i=0; i<n; i++) {
	        memcpy (&longData, (din + i*v->size), v->size);
		rtx_parse_swap_data (&longData, v->elemSize, swapword);
		((long *) dout) [i] = longData;
		/*
	        memcpy ((dout + i*v->size), &longData, v->size);
		*/
	    }
	    break;
	case rtx_float_t :
	    for (i=0; i<n; i++) {
	        memcpy (&floatData, (din + i*v->size), v->size);
		rtx_parse_swap_data (&floatData, v->elemSize, swapword);
		((float *) dout) [i] = floatData;
		/*
	        memcpy ((dout + i*v->size), &floatData, v->size);
		*/
	    }
	    break;
	case rtx_double_t :
	    for (i=0; i<n; i++) {
	        memcpy (&doubleData, (din + i*v->size), v->size);
		rtx_parse_swap_data (&doubleData, v->elemSize, swapword);
		((double *) dout) [i] = doubleData;
		/*
	        memcpy ((dout + i*v->size), &doubleData, v->size);
		*/
	    }
	    break;
#endif
	case rtx_struct_t :
	    for (i=0; i<n; i++) {
	        srcOffset = vRemote->elemSize * i;
	        destOffset = v->elemSize * i;
	        for (v1 = v->subVar, v2 = vRemote->subVar; v1 != NULL;
		        v1 = v1->next, v2 = v2->next)
		    rtx_parse_translate_data (&(dout[destOffset + v1->offset]), v1,
				    &(din[srcOffset + v2->offset]), v2,
				    swapType);
	    }
	    break;
        default :
  	    return (rtx_error ("rtx_parse_translate_data (%s): invalid type %d",
			       v->name, v->type));
    }
    return (0);
}

/**
 * print a var structure
 *
 */
void
rtx_parse_print_decl (
		      RtxParseVar * v,      /**< var structure to print */
		      int indent            /**< white space to print */
		      )
{
    if (v == NULL)
        return;
    if (v->type == rtx_struct_t) {
        rtx_message ("Begin Structure: %s size %d offset %d align %d\n",
		v->name, v->size, v->offset, v->alignment);
	rtx_parse_print_decl (v->subVar, indent + 2);
	rtx_message ("End Structure: %s\n", v->name);
    } else {
        rtx_message ("Variable %s: type %d dim %d size %d offset %d "
			     "align %d\n", v->name, v->type, v->dim,
			     v->size, v->offset, v->alignment);
    }
    rtx_parse_print_decl (v->next, indent);
}

static void
print_const_table_entry(void *key,
			void *value,
			int i, int j,
			int *stop_iter,
			void *context)
{
    int * entryNum = (int *) context;
    RtxParseConstTableEntry * tp = (RtxParseConstTableEntry *) value;

    rtx_message ("Entry %d - %s [%d]", * entryNum, key, tp->value);
    (* entryNum)++;
}

static void
print_type_table_entry (void *key,
			void *value,
			int i, int j,
			int *stop_iter,
			void *context)
{
    int * entryNum = (int *) context;
    RtxParseTypeTableEntry * tp = (RtxParseTypeTableEntry *) value;

    rtx_message ("Entry %d - %s", * entryNum, key);
    rtx_parse_print_decl (tp->t, 2);
    (* entryNum)++;
}


static void
print_var_table_entry(void *key,
		      void *value,
		      int i, int j,
		      int *stop_iter,
		      void *context)
{
    int * entryNum = (int *) context;
    RtxParseVarTableEntry * tp = (RtxParseVarTableEntry *)value;

    rtx_message ("Entry %d - %s", * entryNum, key);
    rtx_parse_print_decl (tp->v, 2);
    (* entryNum)++;
}

/**
 * print the const table
 *
 */
void
rtx_parse_print_const_table (
			     rtx_hash_t constTable
			     )
{
    int i = 0;

    rtx_message("CONST TABLE");
    rtx_hash_iter(constTable, print_const_table_entry, &i);
}

/**
 * print the type table
 *
 */
void
rtx_parse_print_type_table (
			     rtx_hash_t typeTable
			     )
{
    int i = 0;

    rtx_message("TYPE TABLE");
    rtx_hash_iter(typeTable, print_type_table_entry, &i);
}

/**
 * print the var table
 *
 */
void
rtx_parse_print_var_table (
			     rtx_hash_t varTable
			     )
{
    int i = 0;

    rtx_message("VAR TABLE");
    rtx_hash_iter(varTable, print_var_table_entry, &i);
}
