/***********************************************************************
 * 
 * CSIRO Autonomous Systems Laboratory
 * Queensland Centre for Advanced Technologies
 * PO Box 883, Kenmore, QLD 4069, Australia
 * http://www.ict.csiro.au/
 *  
 * Copyright (c) CSIRO 
 ***********************************************************************/

/**
 * \file paramfile.c
 * \brief parameter access module stream functions
 * \author Peter Corke
 *
 * Bottom half of the general purpose parameter access library.
 * Reads parameters come from a simple text format file.
 *
 * @see param.c
 *
 */

#include	<stdio.h>
#include	<stdlib.h>
#include	<string.h>

#include "rtx/defines.h"
#include "rtx/error.h"
#include "rtx/list.h"
#include "rtx/param.h"

static char rcsid[] RTX_UNUSED = "$Id: paramfile.c 2274 2007-12-23 05:37:32Z roy029 $";

static char *rtx_param_get_line (RtxParamStream *pf, char *b, const char *param, int n);
static char *rtx_param_get_line_preprocessed (RtxParamStream *pf, char *b, const char *param, int n);

/**
 * open a parameter stream.
 *
 * @param	fname	name of parameter file.
 * @param	verb	verbosity, if > 0 print messages for every parameter read.
 * @param	errfunc	handler to be called when a read error happens, 
 *		could be because  parameter not found, bad syntax, 
 *		insufficient number values in file for an array type.
 * @return a parameter file type.
 */
RtxParamStream *
rtx_param_open(const char *fname, int verb, void (* errfunc)(const char *name, const char *err) )
{
	RtxParamStream	*pf;
	FILE		*fp;

	if ((pf = (RtxParamStream *)calloc(1, sizeof(RtxParamStream))) == NULL)
		return NULL;

	fp = fopen(fname, "r");

	if (fp == NULL)
		return NULL;
	pf->verbose = verb;
	pf->fp = fp;
	pf->errfunc = errfunc;
	pf->param_get_line = rtx_param_get_line;

	return pf;
}

/**
 * open a parameter stream filtered through the C pre-processor
 *
 * @param	fname	name of parameter file.
 * @param	verb	verbosity, if > 0 print messages for every parameter read.
 * @param	errfunc	handler to be called when a read error happens, 
 *		could be because  parameter not found, bad syntax, 
 *		insufficient number values in file for an array type.
 * @param       argc    number of definitions to be passed on to the pre-processor
 * @param       argv    list of definitions to be passed on to the pre-processor
 * @return a parameter file type.
 */
RtxParamStream *
rtx_param_open_preprocessed(const char *fname, int verb, void (* errfunc)(const char *name, const char *err), int argc, char * argv[] )
{
	RtxParamStream	*pf;
	FILE		*fp;
	char            b[BUFSIZ];
	char            b2[BUFSIZ];
	int             i;
	char	        *t, *l;
	char            *lasts=NULL;
	int	        len = 0;

	if ((pf = (RtxParamStream *)calloc(1, sizeof(RtxParamStream))) == NULL)
		return NULL;
	if ((pf->lines = rtx_list_init ()) == NULL) {
	        free (pf);
		return (NULL);
	}

#ifdef sparc_solaris
	sprintf (b, "/usr/local/bin/cpp -P ");
#else
	sprintf (b, "/usr/bin/cpp -P ");
#endif
	for (i=0; i<argc; i++) {
	    strcat (b, "-D");
	    strcat (b, argv[i]);
	    strcat (b, " ");
	}
	strcat (b, fname);

	if (verb)
	        fprintf (stderr, "rtx_param_open_preprocessed: %s\n", b);
	fp = popen(b, "r");
	pf->fp = fp;

	if (fp == NULL)
		return NULL;
	
	while (fgets(b, BUFSIZ, pf->fp) != NULL) {
		/* skip comment lines */
		if (strchr("/#%\n", b[0]) != NULL)
			continue;

		len = strlen (b);
		t = strtok_r(b, " \t", &lasts);
		/*
		 * handle line continuation, lines ending with backslash
		 */
		while (b[len-1] == '\\') {
			if (fgets(b2, BUFSIZ, pf->fp) == NULL)
				return NULL;	/* EOF in continuation */
			
			/* will it fit ? */
			if ((strlen(b2) + len) >= BUFSIZ)
				return NULL;
			strcat(b, b2);
			len += strlen(b2);
		}
		if ((l = strdup (lasts)) == NULL)
		        return (NULL);
		if (rtx_list_add (pf->lines, t, l) == -1)
		        return (NULL);
	}
	pclose (fp);
	pf->usePreprocessor = 1;
	pf->verbose = verb;
	pf->fp = NULL;
	pf->errfunc = errfunc;
	pf->param_get_line = rtx_param_get_line_preprocessed;

	return pf;
}

/**
 * close a parameter file
 * 
 * @param pf parameter file type.
 */
void
rtx_param_close(RtxParamStream *pf)
{
        if (pf->usePreprocessor)
	        rtx_list_destroy (pf->lines, 1);
	else
	        fclose(pf->fp);
	free(pf);
}


/**
 * return a line from the stream that starts with the specified parameter
 * name.  This is the main primitive used by the top-level functions 
 * \c paramGetInt(), \c paramGetReal() and \c paramGetString().
 *
 * @param	fp	a file pointer.
 * @param	b	where to return the line from the file.
 * @param	param	name of parameter to search for.
 * @param	n	maximum length of string to return
 *
 * the stream is rewound and examined line by line, the lines are assumed
 * be of the form:<pre>
 *	parametername val1 val2 val3 ...
 * </pre>
 * Comment lines are ignored (start with # or %).  Trailing comments
 * will be returned with the line.
 *
 * The standard line continuation syntax (line ends with backslash) is
 * obeyed.
 *
 * @return a pointer to the returned string or NULL on any of the errors:
 *	- the specified paramater is not found.
 *	- the line in the file would exceed the return buffer length \p n
 *	- an end-of-file is encountered when a continuation line was expected
 */
char *
rtx_param_get_line(RtxParamStream *pf, char *b, const char *param, int n)
{
	char	*t;
	char    *lasts=NULL;
	int	len = 0;

        rewind(pf->fp);
        for (;;) {
                if (fgets(b, n, pf->fp) == NULL)
                        break;

		/* skip comment lines */
		if (strchr("/#%\n", b[0]) != NULL)
			continue;

		t = strtok_r(b, " \t", &lasts);
		if (strcmp(t, param) != 0)
			continue;

		len = strlen(b);

		/*
		 * handle line continuation, lines ending with backslash
		 */
		while (b[len-1] == '\\') {
			char	b2[BUFSIZ];

			if (fgets(b2, BUFSIZ, pf->fp) == NULL)
				return NULL;	/* EOF in continuation */
			
			/* will it fit ? */
			if ((strlen(b2) + len) >= n)
				return NULL;
			strcat(b, b2);
			len += strlen(b2);
		}

		return t+strlen(t)+1;
        }

	return NULL;
}

char *
rtx_param_get_line_preprocessed(RtxParamStream *pf, char *b, const char *param, int n)
{
        return (rtx_list_lookup (pf->lines, param));
}

