/***********************************************************************
 * 
 * CSIRO Autonomous Systems Laboratory
 * Queensland Centre for Advanced Technologies
 * PO Box 883, Kenmore, QLD 4069, Australia
 * http://www.ict.csiro.au/
 *  
 * Copyright (c) CSIRO 
 ***********************************************************************/

/**
 * \file param.c
 * \brief Parameter access module
 * \author Peter Corke
 *
 * General purpose parameter access library. API allows access to
 * int, double and string valued parameters. Parameters come from a
 * simple text format file, eg.
 * \code
 *	bufSize 1024
 *	propGain 0.78
 * \endcode 
 * Parameters can also be arrays:
 * \code
 *	gains  0.1 0.2 0.3
 * 
 * \endcode
 * Long lines can be terminated with a backslash.
 * <p>
 * The module is in two parts: param.c and paramfile.c.  The former is the
 * interface to user code, the latter is the interface to the file and could
 * be replaced if parameters are to come from a different source or 
 * a different format file.
 *
 * @see paramfile.c
 */

#include	<stdio.h>
#include	<stdlib.h>
#include	<stdarg.h>
#include	<string.h>
#include	<strings.h>
#include	<ctype.h>

#include "rtx/defines.h"
#include "rtx/param.h"
#include "rtx/error.h"

static char rcsid[] RTX_UNUSED = "$Id: param.c 2274 2007-12-23 05:37:32Z roy029 $";

#define RTX_PARAM_ERR_NOT_FOUND       "PARAMETER NOT FOUND"
#define RTX_PARAM_ERR_TOO_FEW_VALS    "TOO FEW INITIALIZERS IN FILE"
#define RTX_PARAM_ERR_CONVERSION      "CONVERSION ERROR"
#define RTX_PARAM_ERR_MALFORMED_STR   "MALFORMED STRING PARAMETER"

#ifndef DOXYGEN_SHOULD_SKIP_THIS
#define	PARAM_ERROR(err)	{ \
		if (pf->verbose) \
			fprintf(stderr, "%s\n", err); \
		if (pf->errfunc) \
			(*pf->errfunc)(param, err); \
		return -1; \
		}

#define	PARAM_ERROR_NULL(err)	{ \
		if (pf->verbose) \
			fprintf(stderr, "%s\n", err); \
		if (pf->errfunc) \
			(*pf->errfunc)(param, err); \
		return NULL; \
		}
#define	param_get_line	(pf->param_get_line)
#endif

/**
 * return value of named short parameter.
 *
 * @param	pf	handle for parameter file
 * @param	param	name of parameter to search for
 * @param	p	pointer to storage for short
 * @param	n	number of elements to read
 * @return 0 on success, else -1
 * @see rtx_param_open()
 */
int
rtx_param_get_short(RtxParamStream *pf, const char *param, short *p, int n)
{
	char	*bp, b[BUFSIZ], *t;
	char    *lasts=NULL;


	if (pf->verbose)
		fprintf(stderr, "rtx_param_get_short: %s: ", param);

	if ((bp = param_get_line(pf, b, param, BUFSIZ)) == NULL) {
		PARAM_ERROR(RTX_PARAM_ERR_NOT_FOUND);
	}

	while ((t = strtok_r(bp, " \t", &lasts)) && (n-- > 0)) {
		int	v_i;
		short	v;

		if (sscanf(t, "%d", &v_i) != 1) {
			PARAM_ERROR(RTX_PARAM_ERR_CONVERSION);
		}
		v = (short)v_i;
		if (pf->verbose)
			fprintf(stderr, "%d ", v);
		*p++ = v;
		bp = NULL;
	}

	if (n > 0) {
		PARAM_ERROR(RTX_PARAM_ERR_TOO_FEW_VALS);
	}

	if (pf->verbose)
		fputc('\n', stderr);

	return 0;
}

/**
 * return value of named integer parameter.
 *
 * @param	pf	handle for parameter file
 * @param	param	name of parameter to search for
 * @param	p	pointer to storage for int
 * @param	n	number of elements to read
 * @return 0 on success, else -1
 * @see rtx_param_open()
 */
int
rtx_param_get_int(RtxParamStream *pf, const char *param, int *p, int n)
{
	char	*bp, b[BUFSIZ], *t;
	char    *lasts=NULL;


	if (pf->verbose)
		fprintf(stderr, "rtx_param_get_int: %s: ", param);

	if ((bp = param_get_line(pf, b, param, BUFSIZ)) == NULL) {
		PARAM_ERROR(RTX_PARAM_ERR_NOT_FOUND);
	}

	while ((t = strtok_r(bp, " \t", &lasts)) && (n-- > 0)) {
		int	v;

		if (sscanf(t, "%i", &v) != 1) {
			PARAM_ERROR(RTX_PARAM_ERR_CONVERSION);
		}
		if (pf->verbose)
			fprintf(stderr, "%d ", v);
		*p++ = v;
		bp = NULL;
	}

	if (n > 0) {
		PARAM_ERROR(RTX_PARAM_ERR_TOO_FEW_VALS);
	}

	if (pf->verbose)
		fputc('\n', stderr);

	return 0;
}

/**
 * return value of named real/float parameter.
 *
 * @param	pf	handle for parameter file
 * @param	param	name of parameter to search for
 * @param	p	pointer to storage for float
 * @param	n	number of elements to read
 * @return 0 on success, else -1
 * @see paramOpen()
 */
int
rtx_param_get_float(RtxParamStream *pf, const char *param, float *p, int n)
{
	char	*bp, b[BUFSIZ], *t;
	char    *lasts=NULL;


	if (pf->verbose)
		fprintf(stderr, "rtx_param_get_float: %s: ", param);
	if ((bp = param_get_line(pf, b, param, BUFSIZ)) == NULL) {
		PARAM_ERROR(RTX_PARAM_ERR_NOT_FOUND);
	}

	while ((t = strtok_r(bp, " \t", &lasts)) && (n-- > 0)) {

		float	v;

		/*	
		printf("t=\"%s\"\n", t);
		*/
		if (sscanf(t, "%f", &v) != 1) {
			PARAM_ERROR(RTX_PARAM_ERR_CONVERSION);
		}
		if (pf->verbose)
			fprintf(stderr, "%f ", v);
		*p++ = v;
		bp = NULL;
	}

	if (n > 0) {
		PARAM_ERROR(RTX_PARAM_ERR_TOO_FEW_VALS);
	}
	if (pf->verbose)
		fputc('\n', stderr);

	return 0;
}

/**
 * return value of named real/double parameter.
 *
 * @param	pf	handle for parameter file
 * @param	param	name of parameter to search for
 * @param	p	pointer to storage for double
 * @param	n	number of elements to read
 * @return 0 on success, else -1
 * @see paramOpen()
 */
int
rtx_param_get_double(RtxParamStream *pf, const char *param, double *p, int n)
{
	char	*bp, b[BUFSIZ], *t;
	char    *lasts=NULL;


	if (pf->verbose)
		fprintf(stderr, "rtx_param_get_double: %s: ", param);
	if ((bp = param_get_line(pf, b, param, BUFSIZ)) == NULL) {
		PARAM_ERROR(RTX_PARAM_ERR_NOT_FOUND);
	}

	while ((t = strtok_r(bp, " \t", &lasts)) && (n-- > 0)) {

		double	v;

		/*	
		printf("t=\"%s\"\n", t);
		*/
		if (sscanf(t, "%lf", &v) != 1) {
			PARAM_ERROR(RTX_PARAM_ERR_CONVERSION);
			return -1;	/* conversion error */
		}
		if (pf->verbose)
			fprintf(stderr, "%f ", v);
		*p++ = v;
		bp = NULL;
	}

	if (n > 0) {
		PARAM_ERROR(RTX_PARAM_ERR_TOO_FEW_VALS);
	}
	if (pf->verbose)
		fputc('\n', stderr);

	return 0;
}

/**
 * return value of named string parameter 
 *
 * @param	pf	handle for parameter file
 * @param	param	name of parameter to search for
 * @param	p	pointer to storage for string
 * @param	n	maximum length of string
 * @return 0 on success, else -1
 * @see paramOpen()
 */
char *
rtx_param_get_string(RtxParamStream *pf, const char *param, char *p, int n)
{
	char	*bp, b[BUFSIZ], *s;
	char	*t;


	if (pf->verbose)
		fprintf(stderr, "rtx_param_get_string: param %s\n", param);
	if ((bp = param_get_line(pf, b, param, BUFSIZ)) == NULL) {
		PARAM_ERROR_NULL(RTX_PARAM_ERR_NOT_FOUND);
	}
	if (pf->verbose > 1)
		fprintf(stderr, "rtx_param_get_string: got <%s>\n", bp);

	/* eliminate white space at front of string */
	while (isspace((int) (*bp)))
		bp++;
	if (pf->verbose > 1)
		fprintf(stderr, "rtx_param_get_string: removed leading space: <%s>\n", bp);

	if (*bp == '"') {
		/* string is quote delimited */
		char	*q, *q2;

		q2 = q = ++bp;	/* skip initial quote */

		while (1) {
			switch (*q) {
			case '\0':
			case '\n':
				/* no terminating quote found */
				PARAM_ERROR_NULL(RTX_PARAM_ERR_MALFORMED_STR);
			case '"':
				*q2 = '\0';
				goto done;
			case '\\':
				if (q[1] == '"')
					q++;
				/* fall through */
			default:
				*q2++ = *q++;	/* keep searching */
				break;
			}
		}
	} else {

		/* eliminate newline at end of string */
		t = strrchr(bp, '\n');
		if (t)
			*t = '\0';
	}

done:
	/* if no buffer provided, dynamically allocate storage */
	if (p == NULL) {
		if (pf->verbose > 1)
		        fprintf(stderr, "rtx_param_get_string: allocating space\n");
		s = strdup(bp);
	} else {
		if (pf->verbose > 1)
		        fprintf(stderr, "rtx_param_get_string: copying to buffer [%d]\n", n);
		s = strncpy(p, bp, n);
	}
	
	if (pf->verbose)
	        fprintf(stderr, "rtx_param_get_string: returning: <%s>\n", s);
	return s;
}

#ifdef	notdef
/**
 * scanf style parsing of parameter line
 *
 * @param	pf	handle for parameter file
 * @param	param	name of parameter to search for
 * @param	fmt	pointer to storage for string
 * @return number of arguments parsed, else -1
 * @see paramOpen()
 */
char *
rtx_param_scanf(RtxParamStream *pf, const char *param, char *fmt, ...)
{
	char	*bp, b[BUFSIZ], *s;
	char	*t;
	va_list		ap;

	va_start(ap, fmt);

	if (pf->verbose)
		fprintf(stderr, "paramGetString: %s: ", param);
	if ((bp = param_get_line(pf, b, param, BUFSIZ)) == NULL) {
		PARAM_ERROR_NULL(RTX_PARAM_ERR_NOT_FOUND);
	}

	/* eliminate white space at front of string */
	while (isspace((int) (*bp)))
		bp++;

	return vsscanf(bp, fmt, ap);
}
#endif
