/***********************************************************************
 * 
 * CSIRO Autonomous Systems Laboratory
 * Queensland Centre for Advanced Technologies
 * PO Box 883, Kenmore, QLD 4069, Australia
 * http://www.ict.csiro.au/
 *  
 * Copyright (c) CSIRO 
 ***********************************************************************/
/**
 ********************************************************************
 *
 * \file nmea.c
 * \brief Handler to process NMEA-0183 messages
 * \author Craig Worthington
 *
 * This module will process and decode NMEA-0183 based messages.
 *
 ********************************************************************
 */

#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <ctype.h>

#include "../include/rtx/nmea.h"

static char rcsid[] RTX_UNUSED = "$Id: nmea.c 2271 2007-12-23 04:41:55Z roy029 $";


/*
 * states for processing NMEA messages
 *  STATE_LOOK_FOR_START - search for the '$' starting NMEA messages
 *  STATE_LOOK_FOR_END - search for the '*' indicating the end of NMEA mesgs
 *  STATE_LOOK_FOR_CHECKSUM - get 2 byte NMEA checksum
 */
enum states { STATE_LOOK_FOR_START, STATE_LOOK_FOR_END, STATE_LOOK_FOR_CHECKSUM };


/* Assumption: this is a correct NMEA msg */
static int rtx_nmea_parse(RtxNMEA *msg) 
{
	char * p = msg->nmeaMesg;
	unsigned int i;
	msg->numFields = 0;
	for (i=0;i<82;i++)
		msg->nmeaField[i] = NULL;
	
	msg->nmeaField[msg->numFields] = p;
	msg->numFields += 1;
	while (*p && !isspace(*p)) {
		if ((p - msg->nmeaMesg) > 82) return -1;
		if ((*p == ',')||(*p == '*')) {
			msg->nmeaField[msg->numFields] = p+1;
			msg->numFields += 1;
			*p = 0;
		}
		p++;
	}
	msg->numFields += 1;
	*p = 0;
	return 0;
}
	

/**
 * Wait for and read NMEA-0183 packets on the configured port.
 * If no packet recieved for 2 seconds an error will
 * be returned.
 *
 * @param buffer Pointer to character buffer
 * @param len  length of the buffer
 * @param mesg Pointer to RtxNMEA structure
 *
 * @return 0  if read NMEA packet
 *         -1 if read error occured
 *         -2 if checksum failed
 *
 **/
int rtx_nmea_read_buffer( char *buffer, unsigned int len, RtxNMEA *mesg )
{
  int mesgLoop = 1, i;
  int checksum;
  int currLength = 0, checksumCalc = 0;
  char charBuff, currChecksum[2];
  enum states currState;

  currState = STATE_LOOK_FOR_START;     /* define starting state */
  
  /* read full NMEA message from port */
  for (i=0;i<82;i++) {
	  charBuff = buffer[i];
	  if (charBuff==0) break;

	  switch( currState )
	  {
		  case STATE_LOOK_FOR_START:        /** look for start of NMEA message **/
			  if( charBuff == '$' )           /* check character for start '$' */
			  {
				  currState = STATE_LOOK_FOR_END;
				  currLength = 0;
				  checksumCalc = 0;
			  }
			  break;
		  case STATE_LOOK_FOR_END:          /** while reading in message for for end **/
			  if( charBuff == '*' )       /* check for start of checksum character */
				  currState = STATE_LOOK_FOR_CHECKSUM;
			  else
			  {
				  mesg->nmeaMesg[currLength] = charBuff; /* add character to buffer */
				  checksumCalc ^= mesg->nmeaMesg[currLength]; /* calc checksum */
				  currLength++;
			  }
			  break;
		  case STATE_LOOK_FOR_CHECKSUM:

			  currChecksum[0] = charBuff;
			  currChecksum[1] = buffer[i+1];
			  currChecksum[2] = '\0';
			  mesgLoop = 0;    /* exit messaging loop */
			  break;
	  }
	  if (!mesgLoop) break;
  }

  /* check message checksum */
  mesg->nmeaMesg[currLength] = '\0';
  sscanf (currChecksum, "%x", &checksum); /* convert message checksum to int */

  /** compare calc checksum with mesg checksum **/
  if( checksum != checksumCalc ) {
	  printf("Expected checksum: %02X received %02X (%s)\n",checksumCalc,checksum,currChecksum);
    return( -2 );
  }
  return rtx_nmea_parse(mesg);
}

/**
 * Wait for and read NMEA-0183 packets on the configured port.
 * If no packet recieved for 2 seconds an error will
 * be returned.
 *
 * @param port Pointer to configuration comms port
 * @param mesg Pointer to RtxNMEA structure
 *
 * @return 0  if read NMEA packet
 *         -1 if read error occured
 *         -2 if checksum failed
 *
 **/
int rtx_nmea_read( int port, RtxNMEA *mesg )
{
  int mesgLoop = 1;
  int checksum;
  int currLength = 0, checksumCalc = 0;
  char charBuff, currChecksum[2];
  enum states currState;

  currState = STATE_LOOK_FOR_START;     /* define starting state */
  
  /* read full NMEA message from port */
  while( mesgLoop == 1 )
    {
      if( rtx_serial_read_timeout(port,&charBuff,1,2.0) < 0 )
	return( - 1 );

      switch( currState )
	{
	case STATE_LOOK_FOR_START:        /** look for start of NMEA message **/
	  if( charBuff == '$' )           /* check character for start '$' */
	    {
	      currState = STATE_LOOK_FOR_END;
	      currLength = 0;
	      checksumCalc = 0;
	    }
	  break;
	case STATE_LOOK_FOR_END:          /** while reading in message for for end **/
	  if( charBuff == '*' )       /* check for start of checksum character */
	    currState = STATE_LOOK_FOR_CHECKSUM;
	  else
	    {
	      mesg->nmeaMesg[currLength] = charBuff; /* add character to buffer */
	      checksumCalc ^= mesg->nmeaMesg[currLength]; /* calc checksum */
	      currLength++;
	    }
	  break;
	case STATE_LOOK_FOR_CHECKSUM:

	  currChecksum[0] = charBuff;

	  if( rtx_serial_read_timeout(port,&charBuff,1,2.0) < 0 )
	    return( - 1 );

	  currChecksum[1] = charBuff;

	  currChecksum[2] = '\0';
	  mesgLoop = 0;    /* exit messaging loop */
	  break;
	}
    }

  /* check message checksum */
  mesg->nmeaMesg[currLength] = '\0';
  sscanf (currChecksum, "%x", &checksum); /* convert message checksum to int */

  /** compare calc checksum with mesg checksum **/
  if( checksum != checksumCalc )
    return( -2 );
  return rtx_nmea_parse(mesg);

}



/**
 * Write a NMEA-0183 formatted packet in a buffer
 *
 * @param buffer Mem buffer to write to
 * @param message - the NMEA message to write in a list of character 
 *                         strings terminated by a NULL
 *                         eg. "PLEIR","BAT","0","200",NULL
 *
 * @return -1 on error
 *
 **/
int rtx_nmea_write_buffer( char buffer[82], ... )
{
	char *tmpBuff ;
	int argcount = 82;
	u_char checkSum = 0;
	int i,l,fl;
	va_list args;

	va_start( args, buffer );

	buffer[0] = '$';
	buffer[1] = 0;
	fl=2;

	for( i = 0; i < argcount; i++ )
	{
		tmpBuff = va_arg(args,char*);
		if( tmpBuff == NULL )
			break;
		l = strlen(tmpBuff);
		if (fl + l + 1 > 82) {
			return -1;
		}

		if( i > 0 ) strcat (buffer, ",");
		strcat( buffer, tmpBuff );
		fl += l + 1;
	}

	if (fl + 5 > 82) {
		return -2;
	}
	for (i=1; i<strlen(buffer); i++)
		checkSum ^= buffer[i];
	sprintf (buffer,"%s*%2X%c%c", buffer, checkSum, '\xd', '\xa');

	va_end( args );  
	return( 0 );
}

/**
 * Write a NMEA-0183 formatted packet on the comms port
 *
 * @param port Comms port to write to
 * @param message - the NMEA message to write in a list of character 
 *                         strings terminated by a NULL
 *                         eg. "PLEIR","BAT","0","200",NULL
 *
 * @return -1 on error
 *
 **/
int rtx_nmea_write( int port, ... )
{
	char *tmpBuff , buffer[82];
	int argcount = 82;
	u_char checkSum = 0;
	int i,l,fl;
	va_list args;

	va_start( args, port );

	buffer[0] = '$';
	buffer[1] = 0;
	fl=2;

	for( i = 0; i < argcount; i++ )
	{
		tmpBuff = va_arg(args,char*);
		if( tmpBuff == NULL )
			break;
		l = strlen(tmpBuff);
		if (fl + l + 1 > 82) {
			return -1;
		}

		if( i > 0 ) strcat (buffer, ",");
		strcat( buffer, tmpBuff );
		fl += l + 1;
	}

	if (fl + 5 > 82) {
		return -2;
	}
	for (i=1; i<strlen(buffer); i++)
		checkSum ^= buffer[i];
	sprintf (buffer,"%s*%2X%c%c", buffer, checkSum, '\xd', '\xa');


	if (write (port, buffer, strlen (buffer)) == -1) 
		return (-1);

	va_end( args );  
	return( -1 );
}



/**
 * extract part of a NMEA-0183 message
 *
 * @param mesg - NMEA message read
 * @param fieldNo - no of field in message to read
 * @param value - characters of field extracted
 *
 * @return -1 on error 
 *
 **/
int rtx_nmea_extract( RtxNMEA *mesg, int fieldNo, char *value )
{
#if 0
  int cnt = 0;
  char *p = mesg->nmeaMesg;
  char *q = value;

  while (fieldNo-- > 0)
    while (*p++ != ',');

  /* strcpy(value, p); */
  /* p = value; */

  while (*p && *p != ',' && *p != '*' && *p != '\r')
    {
      *(q) = *(p);
      q++;
      p++;
      cnt++;
    }
  
  *q = '\0';
  
  return 0; 
#else
  if (fieldNo >= mesg->numFields) return -1;
  if (mesg->nmeaField[fieldNo] == NULL) return -2;
  strcpy(value, mesg->nmeaField[fieldNo]);
  return 0;
#endif
}

/**
 * extract part of a NMEA-0183 message and return as short
 *
 * @param mesg - NMEA message read
 * @param fieldNo - no of field in message to read
 * @param value - field extracted as a short
 *
 * @return -1 on error 
 *
 **/
int rtx_nmea_extract_short( RtxNMEA *mesg, int fieldNo, short *value )
{
  char charBuff[82];

  rtx_nmea_extract( mesg, fieldNo, charBuff );

  if( sscanf(charBuff,"%hd",value) != 1 )
    return( -1 );
  
  return( 0 );
}



/**
 * extract part of a NMEA-0183 message and return as int
 *
 * @param mesg - NMEA message read
 * @param fieldNo - no of field in message to read
 * @param value - field extracted as a int
 *
 * @return -1 on error 
 *
 **/
int rtx_nmea_extract_int( RtxNMEA *mesg, int fieldNo, int *value )
{
  char charBuff[82];

  rtx_nmea_extract( mesg, fieldNo, charBuff );

  if( sscanf(charBuff,"%d",value) != 1 )
    return( -1 );
  
  return( 0 );
}


/**
 * extract part of a NMEA-0183 message and return as float
 *
 * @param mesg - NMEA message read
 * @param fieldNo - no of field in message to read
 * @param value - field extracted as a float
 *
 * @return -1 on error 
 *
 **/
int rtx_nmea_extract_float( RtxNMEA *mesg, int fieldNo, float *value )
{
  char charBuff[82];

  rtx_nmea_extract( mesg, fieldNo, charBuff );

  if( sscanf(charBuff,"%f",value) != 1 )
    return( -1 );
  
  return( 0 );
}



/**
 * extract part of a NMEA-0183 message and return as double
 *
 * @param mesg - NMEA message read
 * @param fieldNo - no of field in message to read
 * @param value - field extracted as a double
 *
 * @return -1 on error 
 *
 **/
int rtx_nmea_extract_double( RtxNMEA *mesg, int fieldNo, double *value )
{
  char charBuff[82];

  rtx_nmea_extract( mesg, fieldNo, charBuff );

  if( sscanf(charBuff,"%lf",value) != 1 )
    return( -1 );
  
  return( 0 );
}
