/***********************************************************************
 * 
 * CSIRO Autonomous Systems Laboratory
 * Queensland Centre for Advanced Technologies
 * PO Box 883, Kenmore, QLD 4069, Australia
 * http://www.ict.csiro.au/
 *  
 * Copyright (c) CSIRO 
 ***********************************************************************/

/**
 * \file mutex.c
 * \brief A simple interface to mutexes
 * \author J Roberts
 *
 * A mutex is a synchronization primitive that is typically used to protect a
 * shared data structure.  The mutex associated with the data is "locked" 
 * while the data is being modified, and then "unlocked".   If the mutex
 * is locked by one thread then another thread cannot obtain the lock until
 * the first thread has finished and unlocked it.
 *
 * Importantly a mutex can only be locked and unlocked by the same thread.
 *
 * \see cond, sem
 */

#include <stdio.h>
#include <unistd.h>
#include <pthread.h>
#include <string.h>
#include <stdlib.h>
#include <errno.h>

#include "rtx/defines.h"
#include "rtx/error.h"
#include "rtx/mutex.h"
#include "rtx/message.h"

static char rcsid[] RTX_UNUSED = "$Id: mutex.c 2274 2007-12-23 05:37:32Z roy029 $";

#define	rtx_mutex_debug_hook	if (rtx_mutex_debug_proc) (*rtx_mutex_debug_proc)

/* pointer to debug handler */
static void (*rtx_mutex_debug_proc)(RtxMutex *m, enum _rtx_mutex_debug_op op) = NULL;

static void rtx_mutex_debug_message(RtxMutex *m, enum _rtx_mutex_debug_op op);


/**
 * Debug mutexes used by this process.  Install a global mutex debug callback
 * function which is called for all mutex operations.   The function is 
 * invoked as:
 *
 *    func(RtxMutex *m, enum _rtx_mutex_debug_op)
 *
 * @param enable boolean to control whether mutex messages are logged
 * \see message
 */
void
rtx_mutex_debug_set_callback(
	void (*func)(RtxMutex *m, enum _rtx_mutex_debug_op op))
{
	rtx_mutex_debug_proc = func;
}

/**
 * Debug mutexes used by this process.  A global mutex debug handler is 
 * installed which logs all mutex operations via the message module.
 *
 * @param enable boolean to control whether mutex messages are logged
 * \see message
 */
void
rtx_mutex_debug(int enable)
{
	if (enable)
		rtx_mutex_debug_proc = rtx_mutex_debug_message;
	else
		rtx_mutex_debug_proc = NULL;
}

static void
rtx_mutex_debug_message(RtxMutex *m, enum _rtx_mutex_debug_op op)
{
	switch (op) {
	case RTX_MUTEX_DEBUG_INIT:
		rtx_message("mutex init id=%s", m->name);
		break;
	case RTX_MUTEX_DEBUG_DESTROY:
		rtx_message("mutex destroy id=%s", m->name);
		break;
	case RTX_MUTEX_DEBUG_UNLOCK:
		rtx_message("mutex unlock id=%s", m->name);
		break;
	case RTX_MUTEX_DEBUG_LOCK_REQUEST:
		rtx_message("mutex lockrequest id=%s", m->name);
		break;
	case RTX_MUTEX_DEBUG_LOCK_GRANTED:
		rtx_message("mutex lockgrant id=%s", m->name);
		break;
	}
}
/**
 * Initialize a mutex.  The mutex is initially unlocked.
 *
 * @return Pointer to a mutex if OK, NULL on error
 *
 * If \p ptr is NULL then a a Mutex is dynamically allocated.
 *
 * Permissible values for \p mutexAttrs are:
 *	- RTX_MUTEX_DEFAULT default settings
 *
 * else a logical or of the following:
 *	- RTX_MUTEX_PRIO_CEILING defines the  priority  ceiling  of
 *      initialized  mutexes, which is the minimum priority level at
 *      which the critical section guarded by the mutex is executed.
 *      In order to avoid priority inversion, the priority ceiling
 *      of the mutex, \p prio, must be set to a priority higher than or equal
 *	to  the  highest  priority  of all the threads that may lock that mutex.
 *	See pthread_mutexattr_setprioceiling().
 *	- RTX_MUTEX_PRIO_INHERITANCE the  calling  thread is blocked
 *         because the mutex is owned by  another  thread,  that  owner
 *         thread will inherit the priority level of the calling thread
 *         as long as it continues to own the mutex. 
 *	   See pthread_mutexattr_setprotocol().
 *	- RTX_MUTEX_PROCESS_SHARED is set to permit a mutex to be 
 *	  operated upon by any thread that has access to the memory
 *	  where the mutex is allocated, even if the mutex is allocated
 *	  in memory that is shared  by  multiple  processes. 
 *	  See pthread_mutexattr_setpshared().
 *
 * Note that RTX_MUTEX_PRIO_CEILING and RTX_MUTEX_PRIO_INHERITANCE cannot
 * be used together.

 * This function provides a wrapper around the PTHREADS pthread_mutex_init()
 * function which returns an error number if an error occurs.
 */
RtxMutex *
rtx_mutex_init2 (
		char	*name,			/**< name of the mutex for debugging */
		RtxMutex * mutexP,               /**< ptr to mutex */
		int mutexAttrs,                  /**< mutex attributes */
		int prio                         /**< prio ceiling, only used if \p  RTX_MUTEX_PRIO_CEILING is set */
		)
{
    RtxMutex		*mutex = mutexP;
    pthread_mutexattr_t attr, * attrP = NULL;
    int 		retval;
    char		defname[20];

    if (mutexP == NULL) {
        if ((mutex = (RtxMutex *)calloc(1, sizeof(RtxMutex))) == NULL)
	    return (rtx_error_null ("rtx_mutex_init: calloc: no memory"));
	mutex->memAlloc = 1;
    } else
        mutex->memAlloc = 0;

    if (name == NULL) {
	    sprintf(defname, "0x%08x", (unsigned int)mutex);
	    mutex->name = strdup(defname);
    } else {
	    mutex->name = strdup(name);
    }
    if (mutexAttrs != RTX_MUTEX_DEFAULT) {
        attrP = &attr;
        if ((retval = pthread_mutexattr_init (attrP))) {
  	    if (mutex->memAlloc)
	        free (mutex);
	    rtx_error_errno2 (retval, "rtx_mutex_init: "
				      "pthread_mutexattr_init() failed");
	    return NULL;
	}
	if (mutexAttrs & RTX_MUTEX_PROCESS_SHARED) {
#ifdef _POSIX_THREAD_PROCESS_SHARED
  	    if ((retval = pthread_mutexattr_setpshared (attrP,
					       PTHREAD_PROCESS_SHARED))) {
  	        if (mutex->memAlloc)
		    free (mutex);
	        rtx_error_errno2 (retval, "rtx_mutex_init: "
					  "pthread_mutexattr_setpshared()"
					  "failed");
		return NULL;
	    }
#else
  	    if (mutex->memAlloc)
	        free (mutex);
	    return (rtx_error_null ("rtx_mutex_init: RTX_MUTEX_PROCESS_SHARED "
			       "not supported"));
#endif
	}
	if ((mutexAttrs & RTX_MUTEX_PRIO_INHERITANCE) &&
 	        (mutexAttrs & RTX_MUTEX_PRIO_CEILING)) {
  	    if (mutex->memAlloc)
	        free (mutex);
	    return (rtx_error_null ("rtx_mutex_init: RTX_MUTEX_PRIO_CEILING "
			       "and RTX_MUTEX_PRIO_INHERITANCE cannot be "
			       "specified together"));
	}
	if (mutexAttrs & RTX_MUTEX_PRIO_CEILING) {
#if (defined _POSIX_THREAD_PRIO_PROTECT) && (defined PTHREAD_MUTEX_SETPROTOCOL)
  	    if ((retval = pthread_mutexattr_setprotocol (attrP, 
				 PTHREAD_PRIO_PROTECT))) {
	        if (mutex->memAlloc)
	            free (mutex);
	        rtx_error_errno2 (retval, "rtx_mutex_init: "
					  "pthread_mutexattr_setprotocol()"
					  "failed");
		return NULL;
	    }
  	    if ((retval = pthread_mutexattr_setprioceiling (attrP, prio))) {
	        if (mutex->memAlloc)
	            free (mutex);
	        rtx_error_errno2 (retval, "rtx_mutex_init: "
					  "pthread_mutexattr_setprioceiling()"
					  "failed");
		return NULL;
	    }
#else
  	    if (mutex->memAlloc)
	        free (mutex);
	    return (rtx_error_null ("rtx_mutex_init: RTX_MUTEX_PRIO_CEILING "
			       "not supported"));
#endif
	}
	if (mutexAttrs & RTX_MUTEX_PRIO_CEILING) {
#if (defined _POSIX_THREAD_PRIO_INHERIT) && (defined PTHREAD_MUTEX_SETPROTOCOL)
  	    if ((retval = pthread_mutexattr_setprotocol (attrP, 
				 PTHREAD_PRIO_INHERIT))) {
  	        if (mutex->memAlloc)
		    free (mutex);
	        rtx_error_errno2 (retval, "rtx_mutex_init: "
					  "pthread_mutexattr_setprotocol()"
					  "failed");
		return NULL;
	    }
#else
  	    if (mutex->memAlloc)
	        free (mutex);
	    return (rtx_error_null ("rtx_mutex_init: RTX_MUTEX_PRIO_INHERIT "
			       "not supported"));
#endif
	}
    }
    if ((retval = pthread_mutex_init (&mutex->mutex, attrP))) {
        rtx_error_errno2 (retval, "rtx_mutex_init: "
				  "pthread_mutex_init() failed");
	if (mutex->memAlloc)
	    free (mutex);
	return NULL;
    }

    rtx_mutex_debug_hook(mutex, RTX_MUTEX_DEBUG_INIT);

    return mutex;
}

/**
 * Initialize a mutex.  The mutex is initially unlocked.
 *
 * @return Pointer to a mutex if OK, NULL on error
 *
 * If \p ptr is NULL then a a Mutex is dynamically allocated.
 *
 * Permissible values for \p mutexAttrs are:
 *	- RTX_MUTEX_DEFAULT default settings
 *
 * else a logical or of the following:
 *	- RTX_MUTEX_PRIO_CEILING defines the  priority  ceiling  of
 *      initialized  mutexes, which is the minimum priority level at
 *      which the critical section guarded by the mutex is executed.
 *      In order to avoid priority inversion, the priority ceiling
 *      of the mutex, \p prio, must be set to a priority higher than or equal
 *	to  the  highest  priority  of all the threads that may lock that mutex.
 *	See pthread_mutexattr_setprioceiling().
 *	- RTX_MUTEX_PRIO_INHERITANCE the  calling  thread is blocked
 *         because the mutex is owned by  another  thread,  that  owner
 *         thread will inherit the priority level of the calling thread
 *         as long as it continues to own the mutex. 
 *	   See pthread_mutexattr_setprotocol().
 *	- RTX_MUTEX_PROCESS_SHARED is set to permit a mutex to be 
 *	  operated upon by any thread that has access to the memory
 *	  where the mutex is allocated, even if the mutex is allocated
 *	  in memory that is shared  by  multiple  processes. 
 *	  See pthread_mutexattr_setpshared().
 *
 * Note that RTX_MUTEX_PRIO_CEILING and RTX_MUTEX_PRIO_INHERITANCE cannot
 * be used together.

 * This function provides a wrapper around the PTHREADS pthread_mutex_init()
 * function which returns an error number if an error occurs.
 */
RtxMutex *
rtx_mutex_init (
		RtxMutex * mutexP,               /**< ptr to mutex */
		int mutexAttrs,                  /**< mutex attributes */
		int prio                         /**< prio ceiling, only used if \p  RTX_MUTEX_PRIO_CEILING is set */
		)
{
    return rtx_mutex_init2(NULL, mutexP, mutexAttrs, prio);
}

/**
 * Destroy mutex.
 *
 * @return 0 if OK, -1 on error
 *
 * This function provides a wrapper around the PTHREADS pthread_mutex_destroy()
 * function which returns an error number if an error occurs.  If the mutex
 * was dynamically allocated the memory will be freed.
 */
int 
rtx_mutex_destroy (
		   RtxMutex * mutex     /**< Mutex to be destroyed */
		   )
{
    int retval;  

    rtx_mutex_debug_hook(mutex, RTX_MUTEX_DEBUG_DESTROY);
    if ((retval = pthread_mutex_destroy (&mutex->mutex))) {
        return (rtx_error_errno2 (retval, "rtx_mutex_destroy: "
				  "pthread_mutex_destroy(%s) failed",mutex->name));
	}
    /* free the name field */
    if (mutex->name)
	    free(mutex->name);

    /* free the RTX structure */
    if (mutex->memAlloc)
        free (mutex);
    return (0);
}

/**
 * Lock mutex.  Attempt to obtain an exclusive lock, will block if somebody
 * else already has it.
 *
 * @return 0 if OK, -1 on error
 *
 * This function provides a wrapper around the PTHREADS pthread_mutex_lock()
 * function which returns an error number if an error occurs.
 */
int
rtx_mutex_lock (
		RtxMutex * mutex	/**< Pointer to mutex */
		)
{
    int retval;  

    rtx_mutex_debug_hook(mutex, RTX_MUTEX_DEBUG_LOCK_REQUEST);
#ifdef i486_lynxos
    if ((retval = pthread_mutex_lock (&mutex->mutex)))
        return (rtx_error_errno2 (retval, "rtx_mutex_lock: %d mutex"
				 "lock error", mutex->mutex->id));
#else
    if ((retval = pthread_mutex_lock (&mutex->mutex)))
        return (rtx_error_errno2 (retval, "rtx_mutex_lock: mutex"
				 "lock error"));
#endif

    rtx_mutex_debug_hook(mutex, RTX_MUTEX_DEBUG_LOCK_GRANTED);
    return (0);
}

/**
 * Unlock mutex.  Release the exclusive lock.
 *
 * @return 0 if OK, -1 on error
 *
 * This function provides a wrapper around the PTHREADS pthread_mutex_unlock()
 * function which returns an error number if an error occurs.
 */
int
rtx_mutex_unlock (
		  RtxMutex * mutex	/**< Pointer to mutex */
		  )
{
    int retval;

#ifdef i486_lynxos
    if ((retval = pthread_mutex_unlock (&mutex->mutex)))
        return (rtx_error_errno2 (retval, "rtx_mutex_unlock: %d mutex"
				  " unlock error", mutex->mutex->id));
#else
    if ((retval = pthread_mutex_unlock (&mutex->mutex)))
        return (rtx_error_errno ("rtx_mutex_unlock: mutex"
				 " unlock error"));
#endif


    rtx_mutex_debug_hook(mutex, RTX_MUTEX_DEBUG_UNLOCK);
    return (0);
}
