/***********************************************************************
 * 
 * CSIRO Autonomous Systems Laboratory
 * Queensland Centre for Advanced Technologies
 * PO Box 883, Kenmore, QLD 4069, Australia
 * http://www.ict.csiro.au/
 *  
 * Copyright (c) CSIRO 
 ***********************************************************************/

/**
 * \file message.c
 * \brief Interface to real-time message logging facility
 * \author Jonathan Roberts
 *
 * This module provides several functions that allow applications to
 * write messages to the message daemon. The message daemon is a 
 * daemon that reads messages from a global system-wide queue and then
 * appends them to a message file.
 *
 * This is now in need for revamping in light of the new error
 * handling facility.
 */

#include        <unistd.h>
#include        <stdarg.h>      /* must be before stdio */
#include        <string.h>
#include        <strings.h>
#include        <stdlib.h>
#include        <stdio.h>
#include        <errno.h>
#include	<syslog.h>
#include        <sys/types.h>
#include        <sys/stat.h>
#include        <fcntl.h>
#include 	<pthread.h>
#include        <errno.h>

#ifdef HAVE_MESSAGE_QUEUE
#include	<mqueue.h>
#endif

#include "rtx/defines.h"
#include "rtx/thread.h"
#include "rtx/error.h"
#include "rtx/time.h"
#include "rtx/message.h"

static char rcsid[] RTX_UNUSED = "$Id: message.c 3066 2008-05-15 02:08:15Z roy029 $";

#define QUEUE_JAM_WAIT (1.0) /* wait 1 second when the queue is jammed */

static pthread_mutex_t errorMutex;
static pthread_once_t 	inited_once = { PTHREAD_ONCE_INIT };

/* 
 * If the following flag is set, the messages are sent to the standard error
 * file instead of the messages queue. By default, it's reset.
 */

static char	        message_pname[BUFSIZ];
static int		message_destination = RTX_MESSAGE_STDERR;
static RtxMessageCustomFunc message_custom_destination = NULL;
#ifdef HAVE_MESSAGE_QUEUE
static mqd_t     mqIn;
#else
static int messagedFifoFd;
#endif

/*
 * Forward defines
 */
#ifdef HAVE_MESSAGE_QUEUE
static int rtx_message_to_mq (RtxMessageErrorType severity, RtxTime * tp,
			      const char *s);
#else
static int rtx_message_to_fifo(RtxMessageErrorType severity, RtxTime * tp,
			       const char *s);
#endif
static int rtx_message_common(RtxMessageErrorType severity, const char *fmt, va_list ap);
static void message_to_func( void (*func)(), RtxMessageErrorType severity,
			     RtxTime * tp, int	arg1, char	*s);

/**
 * Initialize the message system
 *
 * @param pname	Name of process/thread to prepend to messages.
 * @param destination bit mask indicating where messages go.  Can be one or
 *	more of  RTX_MESSAGE_STDERR, RTX_MESSAGE_STDOUT, RTX_MESSAGE_MESSAGE, RTX_MESSAGE_SYSLOG
 *
 * Default destination is stderr.
 */
void
rtx_message_init(const char *pname, int destination)
{
	strncpy(message_pname, pname,BUFSIZ);
	message_destination = destination;
}

/**
 * Define the custom function, used when the destination is RTX_MESSAGE_CUSTOM
 * */
void rtx_message_set_custom_destination(RtxMessageCustomFunc f)
{
	message_custom_destination = f;
}

/**
 * Send a routine message to the message stream.
 * 
 * @param fmt A printf-like format string.
 */
void 
rtx_message(const char *fmt, ...)
{
	va_list	ap;

	va_start(ap, fmt);
	rtx_message_common(RTX_MESSAGE_ERR, fmt, ap);
}

/**
 * Send a routine message to the message daemon.
 * 
 * @param fmt A printf-like format string.
 */
void 
rtx_message_routine(const char *fmt, ...)
{
	va_list	ap;

	va_start(ap, fmt);
	rtx_message_common(RTX_MESSAGE_ERR, fmt, ap);
}

/**
 * Send a warning message to the message daemon.
 * 
 * @param fmt A printf-like format string.
 */
void 
rtx_message_warning(const char *fmt, ...)
{
	va_list	ap;

	va_start(ap, fmt);
	rtx_message_common(RTX_MESSAGE_WARNING, fmt, ap);
}

/**
 * Send an error message to the message daemon.
 * 
 * @param fmt A printf-like format string.
 */
void 
rtx_message_error(const char *fmt, ...)
{
	va_list	ap;

	va_start(ap, fmt);
	rtx_message_common(RTX_MESSAGE_WARNING, fmt, ap);
}

/**
 * Send a shutdown message to the message daemon and exits.
 * 
 * @param fmt A printf-like format string.
 */
void 
rtx_message_shutdown(const char *fmt, ...)
{
	va_list	ap;

	va_start(ap, fmt);
	rtx_message_common(RTX_MESSAGE_SHUTDOWN, fmt, ap);
	exit(1);
}

/**
 * Log a process startup message that includes all the command line
 * options.
 * 
 * @param ac the number of command line arguments
 * @param av the command line strings
 */
void
rtx_message_argv(int ac, char **av)
{
        char    b[BUFSIZ];

        b[0] = '\0';
        while (ac-- > 0) {
                if ((strlen(b)+strlen(*av)+1) >= BUFSIZ)
                        break;
                strcat(b, *av++);
                strcat(b, " ");
        }
        rtx_message("startup: %s", b);
}

/**
 * Sets the message output to local.
 * 
 * This function causes the message functions to print messages on the
 * console.
 *
 * @note Superseded by rtx_message_init()
 */
void
rtx_message_local()
{
	message_destination = RTX_MESSAGE_STDERR;
}

/**
 * Sets the message output to the message queue.
 * 
 * This function causes the message functions to send messages to the message
 * queue.
 *
 * @note Superseded by rtx_message_init()
 */
void
rtx_message_global()
{
	message_destination = RTX_MESSAGE_MESSAGE;
}

/**
 * Sets the global application name that is prepended to all messages.
 *
 * @note Superseded by rtx_message_init()
 */
void
rtx_message_set_pname(const char *name)
{
	strcpy(message_pname, name);
}

/**
 * Render an error string to a function.
 *
 * @param func	function to invoke with string, called as func(arg1, buf)
 * @param severity error severity code
 * @param time time at which error occurred
 * @param arg1 first argument passed to function
 * @param s error string
 */
static void
message_to_func(
	void (*func)(),
	RtxMessageErrorType severity,
	RtxTime * tp,
	int	arg1,
	char	*s)
{
	char 	stringTime[128], head = 'R';
	char	buf[RTX_MESSAGE_SIZE];
	struct tm locTimeStruct;

	/*
	fprintf (stderr, "stack size = %d\n", rtx_thread_get_stack_size());
	*/

	switch (severity) {
	case RTX_MESSAGE_ERR:
		head = 'R';
		break;
	case RTX_MESSAGE_WARNING:
		head = 'W';
		break;
	case RTX_MESSAGE_SHUTDOWN:
		head = 'P';
		break;
	}

	stringTime[127] = '\0';
	if (tp != NULL) {
	    memset (&locTimeStruct, 0, sizeof (locTimeStruct));
	    localtime_r (&(tp->seconds), &locTimeStruct);
	    strftime(stringTime, 127, "%b %d %H:%M:%S", &locTimeStruct);
#ifdef i486_lynxos310
	    sprintf(buf, "%c: %s.%06d: %s\n", head, stringTime, 
		    (int)tp->nanoSeconds/1000, s);
#else
	    snprintf(buf, RTX_MESSAGE_SIZE, "%c: %s.%06d: %s\n", head, 
		     stringTime, (int)tp->nanoSeconds/1000, s);
	    buf[RTX_MESSAGE_SIZE-2] = '\n';
	    buf[RTX_MESSAGE_SIZE-1] = '\0';
#endif
	    /*
	    sprintf(buf, "%c: %d.%06d: %s\n", head, tp->seconds,
		    (int)tp->nanoSeconds/1000, s);
	    */
	} else {
#ifdef i486_lynxos310
	    sprintf(buf, "%c: %s\n", head, s);
#else
	    snprintf(buf, RTX_MESSAGE_SIZE, "%c: %s\n", head, s);
	    buf[RTX_MESSAGE_SIZE-2] = '\n';
	    buf[RTX_MESSAGE_SIZE-1] = '\0';
#endif
	}
	    
	(func)(arg1, buf);
}

void message_to_custom(int d, const char * buf)
{
	if (message_custom_destination) {
		message_custom_destination(buf);
	}
}

/**
 * Message handler for the message daemon
 */
static int
rtx_message_common(RtxMessageErrorType severity, const char *fmt, va_list ap)
{
	char		b[RTX_MESSAGE_SIZE];
	RtxTime	localTime;

	/* get the time */
	if (rtx_time_get (&localTime))
                return rtx_error("rtx_message_common: rtx_time_get() failed");

	/*
	 * start message with process/thread name if given and convert
	 * format string + args to a single string.
	 */
#ifdef i486_lynxos310
	if (strcmp(message_pname, "")) {
		sprintf(b, "%s: ", message_pname);
		vsprintf(&b[strlen(message_pname)+2], fmt, ap);
	}
	else
		vsprintf(b, fmt, ap);
#else
	if (strcmp(message_pname, "")) {
		snprintf(b, RTX_MESSAGE_SIZE, "%s: ", message_pname);
		vsnprintf(&b[strlen(message_pname)+2], RTX_MESSAGE_SIZE, 
			 fmt, ap);
		b[RTX_MESSAGE_SIZE-2] = '\n';
		b[RTX_MESSAGE_SIZE-1] = '\0';
	}
	else {
		vsnprintf(b, RTX_MESSAGE_SIZE, fmt, ap);
	        b[RTX_MESSAGE_SIZE-2] = '\n';
		b[RTX_MESSAGE_SIZE-1] = '\0';
	}
#endif

	/*
	 * now send the message to various destinations
	 */
#ifdef HAVE_MESSAGE_QUEUE
	if (message_destination & RTX_MESSAGE_MESSAGE)
		rtx_message_to_mq(severity, &localTime, b);
#else
	if (message_destination & RTX_MESSAGE_MESSAGE)
		rtx_message_to_fifo(severity, &localTime, b);
#endif
	if (message_destination & RTX_MESSAGE_SYSLOG) {
		message_to_func((void (*)())syslog, severity, &localTime, LOG_WARNING, b);
	}
	if (message_destination & RTX_MESSAGE_STDERR) {
		message_to_func((void (*)())fprintf, severity, &localTime, (int)stderr, b);
	}
	if (message_destination & RTX_MESSAGE_STDOUT) {
		message_to_func((void (*)())fprintf, severity, &localTime, (int)stdout, b);
	}
	if (message_destination & RTX_MESSAGE_CUSTOM) {
		message_to_func((void (*)())message_to_custom, severity, &localTime, 0, b);
	}


	return 0;
}

/**
 * Message handler for the message daemon
 */
void
rtx_message_rtx_error(RtxTime * tp, const char *msg)
{
#ifdef HAVE_MESSAGE_QUEUE
        rtx_message_to_mq(RTX_MESSAGE_WARNING, tp, msg);
#else
        rtx_message_to_fifo(RTX_MESSAGE_WARNING, tp, msg);
#endif
}

#ifdef HAVE_MESSAGE_QUEUE
/*
 * Connect to an existing message queue, return the queue id or error.
 */
static mqd_t
mq_connect(char *name)
{
	char	b[BUFSIZ];
	mqd_t	mq;
	int	oldmask;

	if (strlen(name) > (BUFSIZ-5)) {
		fprintf(stderr, "mq_connect : message queue name too long\n");
		return (mqd_t)-1;
	}
	sprintf(b, "/mq-%s", name);

	/* 
	 * Open the message queue 
	 */
	oldmask = umask(0);
	if ( (mq = mq_open(b, O_RDWR | O_NONBLOCK))  == (mqd_t)-1 ) {
		fprintf(stderr, "mq_connect : mq_open failed (%s) on mq %s\n", strerror(errno), b);
		umask(oldmask);
		return (mqd_t)-1;
	}
	umask(oldmask);

	return mq;
}

static double lastQueueJamTime = -1;
static int queueJam = 0;

static void
error_mq_init()
{
	if (pthread_mutex_init(&errorMutex, NULL))
		fprintf(stderr, "error_mutex_init: pthread_mutex_init() failed: %s\n", strerror(errno));

	/*
	 * Connect to the message queue if this is the first call
	 */
	if ((mqIn = mq_connect (RTX_MESSAGE_MQ_NAME)) == (mqd_t)-1) {
		fprintf(stderr, "error: mq_connect () failed\n");
		message_destination &= ~RTX_MESSAGE_MESSAGE;
		message_destination |= RTX_MESSAGE_STDERR;
		fprintf(stderr, "using local message\n");
	}

	lastQueueJamTime = -1;
	queueJam = 0;
}


/**
 * Render an error string to the message daemon.
 *
 * @param severity error severity code
 * @param time time at which error occurred
 * @param s error string
 */
int
rtx_message_to_mq(
		  RtxMessageErrorType severity, 
		  RtxTime * tp,
		  char *s)
{
	int ret;
	RtxTime tsnow;
	double tnow;
	RtxMessage  msg;

	pthread_once(&inited_once, error_mq_init);

	/* If there is no message Q, dont attempt to write to it !
	 */
	if (mqIn == (mqd_t)-1)
	        return (0);

	if (queueJam) {
		rtx_time_get(&tsnow);
		tnow = rtx_time_to_double(&tsnow);
		if ((tnow - lastQueueJamTime) > QUEUE_JAM_WAIT) {
			queueJam = 0;
			rtx_message("message queue reopened after jamming\n");
		} else {
			return (+1);
		}
	}

	pthread_mutex_lock(&errorMutex);

	/*
	 * Construct message
	 */
	msg.type = severity;
	if (tp == NULL)
	        rtx_time_get (&msg.timestamp);
	else
	        msg.timestamp = *tp;
	msg.string[RTX_MESSAGE_SIZE-1] = '\0';
	strncpy(msg.string, s, RTX_MESSAGE_SIZE-1);


	/*
	 * Send the message to messaged
	 */
	ret = mq_send(mqIn, (char *)&msg, sizeof(msg), 0);
	switch (errno) {
		case 0:
			/* no error */
			ret = 0;
			break;
		case EAGAIN:
		case ETIMEDOUT:
		case EINTR:
			fprintf(stderr, "warning: queue jamed: %d:%s\n", errno,strerror(errno));
			/* the queue is jammed */
			queueJam = 1;
			rtx_time_get(&tsnow);
			lastQueueJamTime = rtx_time_to_double(&tsnow);
			ret = 1;
			break;
		default:
			fprintf(stderr, "error: mq_send() failed: %d:%s\n", errno,strerror(errno));
			/* dont send to message queue anymore */
			message_destination &= ~RTX_MESSAGE_MESSAGE;
			message_destination |= RTX_MESSAGE_STDERR;
			ret = -1;
			break;
	}
	pthread_mutex_unlock(&errorMutex);

	return ret;
}

#else

/* use fifo */

static void
fifo_init (void)
{
    if (pthread_mutex_init(&errorMutex, NULL)) {
        /*
        fprintf(stderr, "error_mutex_init: pthread_mutex_init() failed: %s\n", strerror(errno));
	*/
	/* dont send to message queue anymore */
	message_destination &= ~RTX_MESSAGE_MESSAGE;
	message_destination |= RTX_MESSAGE_STDERR;
	return;
    }

    if ((messagedFifoFd = open (RTX_MESSAGE_FIFO_PATH, O_NONBLOCK | O_WRONLY)) == -1) {
        /*
        fprintf(stderr, "error: open() failed: %s\n", strerror(errno));
	*/
	/* dont send to message queue anymore */
	message_destination &= ~RTX_MESSAGE_MESSAGE;
	message_destination |= RTX_MESSAGE_STDERR;
	return;
    }
}

static int 
rtx_message_to_fifo(
		    RtxMessageErrorType severity, 
		    RtxTime * tp,
		    const char *s
		    )
{
	RtxMessage  msg;
	void * p1, * p2;
	int n, msgSize;

	pthread_once(&inited_once, fifo_init);
	pthread_mutex_lock(&errorMutex);

	/*
	 * Construct message
	 */
	msg.type = severity;
	if (tp == NULL)
	        rtx_time_get (&msg.timestamp);
	else
	        msg.timestamp = *tp;
	msg.string[RTX_MESSAGE_SIZE-1] = '\0';
	strncpy(msg.string, s, RTX_MESSAGE_SIZE-1);

	p1 = &msg;
	p2 = &(msg.string[strlen (msg.string)]);
	msgSize = p2 - p1 + 1;
	if (msgSize > sizeof (msg))
  	    msgSize = sizeof (msg);
	/*
	 * Send the message to messaged
	 */
	if ((n = write (messagedFifoFd, &msg, sizeof (msg))) == -1) {
	        /*
		fprintf(stderr, "error: write(%d) failed: %s\n", 
			messagedFifoFd, strerror(errno));
		*/
		/* dont send to message queue anymore */
		message_destination &= ~RTX_MESSAGE_MESSAGE;
		message_destination |= RTX_MESSAGE_STDERR;
	}
	pthread_mutex_unlock(&errorMutex);

	return 0;
}

#endif
