/***********************************************************************
 * 
 * CSIRO Autonomous Systems Laboratory
 * Queensland Centre for Advanced Technologies
 * PO Box 883, Kenmore, QLD 4069, Australia
 * http://www.ict.csiro.au/
 *  
 * Copyright (c) CSIRO 
 ***********************************************************************/

/**
 * \file matrix.c
 * \brief Matrix manipulation functions
 * \author Peter Corke
 */

#include	<stdio.h>
#include	<stdlib.h>
#include	<math.h>
#include	<string.h>

#include "rtx/defines.h"
#include "rtx/error.h"
#include "rtx/matrix.h"

static char rcsid[] RTX_UNUSED = "$Id: matrix.c 2274 2007-12-23 05:37:32Z roy029 $";

static double ** read_matlab_file(FILE *fp, int rows, int cols, int order);
static void mat_free(double **x, int n);
static void double_swap(double *xd);
static void long_swap(unsigned long *xi);

static int	swapFlag;

/**
 * Load a matrix from a MATLAB v4 binary format file.
 * @param file name of Matlab MAT format file.
 * @param matname name of the particular matrix to read from the file.
 * @return a Matrix handle or NULL if the file cannot be opened or the
 * specified matrix cannot be found.
 * Handles big/little endian conversion if required.
 */
RtxMatrix *
rtx_matrix_load_matlab_file (char *file, char *matname)
{
	FILE	*fp;
	int	mn, order;
	char	nambuf[20];
	/* Fmatrix x; */
	RtxMatrix	*m;
	struct {
	     long type;   /**< type. */
	     long mrows;  /**< row dimension. */
	     long ncols;  /**< column dimension. */
	     long imagf;  /**< flag indicating imaginary part. */
	     long namlen; /**< name length (including NULL). */
	} x;

	if ((fp = fopen(file, "r")) == NULL) {
		return (rtx_error_null ("rtx_matrix_load_matlab_file: "
					"cant open file %s\n", file));
	}
	
	/*
	 * Get FRtxMatrix structure from file
	 */
	/*
	while (fread((char *)&x, sizeof(FRtxMatrix), 1, fp) == 1) {
	*/
	while (fread((char *)&x, sizeof(x), 1, fp) == 1) {

		if ((x.type != 1000) & (x.type != 1001))
			swapFlag = 1;

		if (swapFlag) {
			long_swap((unsigned long*)&x.type);
			long_swap((unsigned long*)&x.mrows);
			long_swap((unsigned long*)&x.ncols);
			long_swap((unsigned long*)&x.imagf);
			long_swap((unsigned long*)&x.namlen);
		}
			
		/*
		 * Get RtxMatrix name from file
		 */
		if (fread(nambuf, sizeof(char), x.namlen, fp) != x.namlen) {
			fclose(fp);
			return NULL;
		}

		mn = x.mrows * x.ncols;

		if (strncmp(nambuf, matname, x.namlen) != 0) {
			/*
			 * dont want this RtxMatrix
			 */
			if (x.imagf)
				mn *= 2;
			fseek(fp, mn*sizeof(double), 1);
			continue;
		}

		m = (RtxMatrix *)malloc(sizeof(RtxMatrix));
		order = (x.type/100) % 10;	/* RtxMatrix data order */

		m->r = read_matlab_file(fp, x.mrows, x.ncols, order);
		/*
		if (x.imagf)
			m->i = read_matlab_file(fp, x.mrows, x.ncols, order);
		*/
		m->rows = x.mrows;
		m->cols = x.ncols;
		fclose(fp);
		return m;
	}
	fclose(fp);
	return NULL;
}

static double **
read_matlab_file(FILE *fp, int rows, int cols, int order)
{
	double	**x;
	int	i, j;

	x = (double **) malloc(rows*sizeof(double *));
	for (i=0; i<rows; i++) {
		x[i] = (double *)malloc(cols*sizeof(double));
		if (order == 1)
			fread(x[i], sizeof(double), cols, fp);
	}

	if (order == 0) {
		for (j=0; j<cols; j++)
			for (i=0; i<rows; i++) {
				fread(&x[i][j], sizeof(double), 1, fp);
				if (swapFlag)
					double_swap(&x[i][j]);
			}
	}
	return x;
}

/**
 * Allocate storage for RtxMatrix data.
 * @param rows number of rows for this RtxMatrix.
 * @param cols number of columns for this RtxMatrix.
 */
static double ** 
mat_alloc(int rows, int cols)
{
	double	**x;
	int	i;

	x = (double **) malloc(rows*sizeof(double *));
	for (i=0; i<rows; i++) {
		x[i] = (double *)malloc(cols*sizeof(double));
	}
	return x;
}

/*
 * create a new RtxMatrix structure and allocate storage for the RtxMatrix data.
 * @param rows number of rows for this RtxMatrix.
 * @param cols number of columns for this RtxMatrix.
 */
RtxMatrix *
rtx_matrix_new (int rows, int cols)
{
	RtxMatrix	*m;

	m = (RtxMatrix *)malloc(sizeof(RtxMatrix));
	if (m == NULL)
		return m;
	m->r = mat_alloc(rows, cols);
	m->rows = rows;
	m->cols = cols;
	return m;
}

/**
 * Free a RtxMatrix data structure.
 * @param m the RtxMatrix to be freed.
 */
void
rtx_matrix_free (RtxMatrix *m)
{
	mat_free(m->r, m->rows);
	free(m);
}

static void 
mat_free(double **x, int n)
{
	int	i;

	for (i=0; i<n; i++)
		free(x[i]);
	free(x);
}

/**
 * Compute product of two RtxMatrix types.
 * @param p the product: \p p = \p a * \p b.
 * @param a multiplicand.
 * @param b multiplicand.
 * Storage for the product is allocated if none presently exists.
 * @todo allocate or reallocate storage in \p p as required.
 * @todo check for RtxMatrix dimension conformance.
 */
void
rtx_matrix_mult (RtxMatrix *p, RtxMatrix *a, RtxMatrix *b)
{
	int	i, j, k;
	double	t;

	if (p->r == NULL)
		p->r = mat_alloc(a->rows, b->cols);
	p->rows = a->rows;
	p->cols = b->cols;
	for (i=0; i<a->rows; i++)
		for (j=0; j<b->cols; j++) {
			t = 0.0;
			for (k=0; k<a->cols; k++)
				t += a->r[i][k] * b->r[k][j];
			p->r[i][j] = t;
		}
}

/**
 * Compute sum of two RtxMatrix types.
 * @param p the sum: \p p = \p a + \p b.
 * @param a addend.
 * @param b addend.
 * Storage for the sum is allocated if none presently exists.
 * @todo allocate or reallocate storage in \p p as required.
 * @todo check for RtxMatrix dimension conformance.
 */
void
rtx_matrix_add (RtxMatrix *s, RtxMatrix *a, RtxMatrix *b)
{
	int	i, j;

	if (s->r == NULL)
		s->r = mat_alloc(a->rows, a->cols);
	s->rows = a->rows;
	s->cols = a->cols;
	for (i=0; i<s->rows; i++)
		for (j=0; j<s->cols; j++)
			s->r[i][j] = a->r[i][j] + b->r[i][j];
}

/**
 * Print a RtxMatrix to file.
 * @param fp the file pointer on which to print.
 * @param s the RtxMatrix to print.
 * Print the RtxMatrix in simple grid layout.
 * @todo allow variable format for display.
 * @todo perform line wrapping.
 */
void
rtx_matrix_fprint (FILE *fp, RtxMatrix *s)
{
	int	i, j;

	for (i=0; i<s->rows; i++) {
		for (j=0; j<s->cols; j++)
			fprintf(fp, "%8.2f ", s->r[i][j]);
		fputc('\n', fp);
	}
}

/**
 * Print a RtxMatrix to stdout.
 * @param s the RtxMatrix to print.
 * Print the RtxMatrix in simple grid layout.
 * @todo allow variable format for display.
 * @todo perform line wrapping.
 */
void
rtx_matrix_print (RtxMatrix *s)
{
	rtx_matrix_fprint (stdout, s);
}

#ifdef	TEST
main()
{
	RtxMatrix	*w1;

	w1 = rtx_matrix_load_matlab_file ("weights.mat", "W1");
	
	rtx_matrix_print (w1);
	putchar('\n');
}
#endif

static void
double_swap(double *xd)
{
	unsigned long	*x = (unsigned long *)xd, t;

	long_swap(&x[0]);
	long_swap(&x[1]);
	t = x[0];
	x[0] = x[1];
	x[1] = t;
}

static void
long_swap(unsigned long *xi)
{
	unsigned char	*x = (unsigned char *)xi, t;

	t = x[0];
	x[0] = x[3];
	x[3] = t;

	t = x[1];
	x[1] = x[2];
	x[2] = t;
}

