/***********************************************************************
 * 
 * CSIRO Autonomous Systems Laboratory
 * Queensland Centre for Advanced Technologies
 * PO Box 883, Kenmore, QLD 4069, Australia
 * http://www.ict.csiro.au/
 *  
 * Copyright (c) CSIRO 
 ***********************************************************************/

/**
 * \file main.c
 * \brief Startup/shutdown helper functions
 * \author Jonathan Roberts and Pavan Sikka
 *
 * This module provides helper functions that do common startup and
 * shutdown tasks in applications developed within the Automation
 * group.
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <signal.h>
#include <sched.h>
#include <string.h>
#include <errno.h>

#include "rtx/defines.h"
#include "rtx/message.h"
#include "rtx/error.h"
#include "rtx/signal.h"
#include "rtx/thread.h"
#include "rtx/timer.h"

static char rcsid[] RTX_UNUSED = "$Id: main.c 2274 2007-12-23 05:37:32Z roy029 $";

#ifndef DARWIN

/**
 * initialize the main process
 *
 * @return 0 on success, -1 on error
 */
int
rtx_main_init (
		const char * appName,
		int msgDest
		)
{
	int initErrors = 0;

	if (msgDest == 0) msgDest = RTX_ERROR_STDERR;

	/*
	 * Initialise message handler
	 */
	rtx_message_init(appName, msgDest);
	/*
	 * Initialise error handler
	 */
	if (rtx_error_init(appName, msgDest, NULL)) {
		rtx_error("rtx_main_init: rtx_error_init() failed");
		initErrors |= (1 << 0);
	}
	/*
	 * Block kill signals
	 */
	if (rtx_signal_block (SIGINT)) {
		rtx_error("rtx_main_init: rtx_signal_block: failed");
		initErrors |= (1 << 1);
	}
	if (rtx_signal_block (SIGQUIT)) {
		rtx_error("rtx_main_init: rtx_signal_block: failed");
		initErrors |= (1 << 2);
	}
	if (rtx_signal_block (SIGTERM)) {
		rtx_error("rtx_main_init: rtx_signal_block: failed");
		initErrors |= (1 << 4);
	}
	if (rtx_signal_block (SIGUSR1)) {
		rtx_error("rtx_main_init: rtx_signal_block: failed");
		initErrors |= (1 << 5);
	}
	if (rtx_signal_block (SIGUSR2)) {
		rtx_error("rtx_main_init: rtx_signal_block: failed");
		initErrors |= (1 << 6);
	}

	/*
	 * Block real-time signals
	 */
	if (rtx_signal_block_realtime()) {
		rtx_error("rtx_main_init: rtx_signal_block_realtime: failed");
		initErrors |= (1 << 8);
	}

	if (initErrors)
		return (-1);
	return (0);
}

static int signalReceived = 0;
static void failsafeHdl(int n) {
	signalReceived += 1;
	if (signalReceived > 2) {
		fprintf(stderr,"Received repeated stop request. Sending SIGKILL\n");
#ifndef i486_linux
		kill(getpid(),SIGKILL);
#else
		kill(0,SIGKILL);
#endif
	}
}

/**
 * wait for the shutdown signal
 *
 * @return 0 on success, -1 on error
 */
int
rtx_main_wait_shutdown (
		RtxThreadPrio prio     /**< priority of the waiting thread */
		)
{

	int     signals[] = {SIGINT, SIGQUIT, SIGTERM, 0};
	sigset_t signalSet;
	int	signalIndex;
#ifndef i486_linux
	struct sched_param	schedulingParameters;
	int	prioSystem;
#endif
	int     retVal, signo = -1;
	pid_t   myPid, myParentPid;
	char * xp = NULL;
	union sigval val;

	/* Block all realtime signals in the main thread */
	rtx_signal_block_realtime ();

	/* Ignore the signals that might be generated by network
	 * sockets */
	rtx_signal_ignore (SIGIO);
	rtx_signal_ignore (SIGPIPE);
	rtx_signal_ignore (SIGURG);

	myPid = getpid ();
	myParentPid = getppid ();
	xp = getenv ("LAUNCHER");

	/*
	 * Prepare the signal set
	 */
	if(sigemptyset(&signalSet)) {
		return (rtx_error_errno ("rtx_main_wait_shutdown (): "
					"sigemptyset () failed: "));
	}

	/*
	 * Load the killing signals
	 */
	for(signalIndex=0; signals[signalIndex] != 0; signalIndex++)
		if( sigaddset(&signalSet, signals[signalIndex])) {
			return (rtx_error_errno ("rtx_main_wait_shutdown (): "
						"sigaddset () failed: "));

		}

/*
#ifndef i486_linux
why not for linux ?? pic */

#if !defined(i486_linux) && !defined(DARWIN)

	/*
	 * Set the priority of the thread to prio
	 */

	/* we shouldnt really change the prio or the scheduler for the main thread. */
	prioSystem = rtx_thread_prio_to_system(prio);
	if (prioSystem > 0) {
		schedulingParameters.sched_priority = prioSystem;
		if (sched_setscheduler(getpid(), SCHED_FIFO, &schedulingParameters)<0) {
			return (rtx_error_errno ("rtx_main_wait_shutdown (): "
						"sched_setscheduler () failed: "));
		}
	}
#endif

	val.sival_int = (int) myPid;
	if ((xp != NULL) && (strcmp (xp, "TRUE") == 0)) {
		if (sigqueue (myParentPid, SIGRTMIN, val) == -1)
			rtx_error_errno_flush ("rtx_main_wait_shutdown: "
					"sigqueue");
	}

	/*
	 * Wait to receive a kill signal and then shut everything down
	 */
	do {
		retVal = sigwait( &signalSet, &signo);
		signalReceived = 0;
		for(signalIndex=0; signals[signalIndex] != 0; signalIndex++) {
			struct sigaction s;
			int r;
			rtx_signal_unblock(signals[signalIndex]);
			s.sa_handler = failsafeHdl;
			sigemptyset(&s.sa_mask);
			s.sa_flags = 0;
			r = sigaction(signals[signalIndex],&s,NULL);
		}

		if (retVal == -1) {
			return (rtx_error_errno ("rtx_main_wait_shutdown (): "
						"sigwait () failed"));
		}
		if (sigismember (&signalSet, signo) == 1) {
			return (0);
		}
		rtx_timer_sleep (0.1);
	} while (1);

	return 0;
}

/**
 * send the shutdown signal to the application
 *
 */
	void
rtx_main_signal_shutdown(void)
{
#ifndef i486_linux
	kill(getpid(),SIGQUIT);
#else
	kill(0,SIGQUIT);
#endif
}

/**
 * RTC server function used by RTC clients to send the shutdown signal
 *
 * @return string containing return status ("ERROR" or "OK")
 */
char *
rtx_main_signal_rtc_shutdown (
		int argc,        /**< number of arguments */
		char * argv[]    /**< arguments */
		)
{
	static char res[10];
	int pid;

	if (argc != 2) {
		strcpy (res, "ERROR");
		return res;
	}

	pid = getpid ();
	fprintf (stderr, "rtx_main_signal_rtc_shutdown (): process %d "
			"received signal %s", pid, argv[1]);

	/* find the signal specified in the command */
	if (strcmp (argv[1], "SIGINT") == 0) {
		if (kill (pid, SIGINT) == 0) {
			strcpy (res, "OK");
			return res;
		} else {
			strcpy (res, "ERROR");
			return res;
		}
	} else if (strcmp (argv[1], "SIGQUIT") == 0) {
		if (kill (pid, SIGQUIT) == 0) {
			strcpy (res, "OK");
			return res;
		} else {
			strcpy (res, "ERROR");
			return res;
		}
	}
	else if (strcmp (argv[1], "SIGTERM") == 0) {
		if (kill (pid, SIGTERM) == 0) {
			strcpy (res, "OK");
			return res;
		} else {
			strcpy (res, "ERROR");
			return res;
		}
	}

	strcpy (res, "ERROR");
	return res;
}

/**
 * SRPC server function used by SRPC clients to send the shutdown signal
 *
 * @return string containing return status ("ERROR" or "OK")
 */
int
rtx_main_signal_srpc_shutdown (
		char * buf,      /**< buffer to use for response */
		int len,         /**< size of response buffer */
		int argc,        /**< number of arguments */
		char * argv[]    /**< arguments */
		)
{
	int pid;

	if (argc != 2) {
		strcpy (buf, "ERROR");
		return (0);
	}

	pid = getpid ();
	fprintf (stderr, "rtx_main_signal_rtc_shutdown (): process %d "
			"received signal %s", pid, argv[1]);

	/* find the signal specified in the command */
	if (strcmp (argv[1], "SIGINT") == 0) {
		if (kill (pid, SIGINT) == 0) {
			strcpy (buf, "OK");
			return (0);
		} else {
			strcpy (buf, "ERROR");
			return (0);
		}
	} else if (strcmp (argv[1], "SIGQUIT") == 0) {
		if (kill (pid, SIGQUIT) == 0) {
			strcpy (buf, "OK");
			return (0);
		} else {
			strcpy (buf, "ERROR");
			return (0);
		}
	}
	else if (strcmp (argv[1], "SIGTERM") == 0) {
		if (kill (pid, SIGTERM) == 0) {
			strcpy (buf, "OK");
			return (0);
		} else {
			strcpy (buf, "ERROR");
			return (0);
		}
	}

	strcpy (buf, "ERROR");
	return (0);
}

/**
 * signal the process launcher that the application is running
 *
 * Note: this function is now subsumed by rtx_main_wait_shutdown()
 * and therefore should be used only if rtx_main_wait_shutdown()
 * is not being used.
 *
 * @return 0 on success, -1 on error
 */
	int
rtx_main_signal_launcher (void)
{
	pid_t   myPid, myParentPid;
	char * xp = NULL;
	union sigval val;

	myPid = getpid ();
	myParentPid = getppid ();
	xp = getenv ("LAUNCHER");
#if 1
	val.sival_int = (int) myPid;
	if ((xp != NULL) && (strcmp (xp, "TRUE") == 0)) {
		if (sigqueue (myParentPid, SIGRTMIN, val) == -1)
			rtx_error_errno_flush ("rtx_main_wait_shutdown: "
					"sigqueue");
	}
#else
	// Old code, apparently not working anymore -- cp
	if ((xp != NULL) && (strcmp (xp, "TRUE") == 0))
		if (kill (myParentPid, SIGUSR1) == -1) 
			return (rtx_error_errno ("rtx_main_signal_launcher: kill() failed"));
#endif
	return (0);
}

#else

#warning "Building Mac version"

/**
 * initialize the main process
 *
 * @return 0 on success, -1 on error
 */
int
rtx_main_init (
	       char * appName,
	       int msgDest
	       )
{
        int initErrors = 0;

	/*
	 * Initialise error handler
	 */
	if (rtx_error_init(appName, msgDest, NULL)) {
  	        rtx_error("rtx_main_init: rtx_error_init() failed");
		initErrors |= (1 << 0);
	}
	/*
	 * Block kill signals
	 */
	if (rtx_signal_block (SIGINT)) {
		rtx_error("rtx_main_init: rtx_signal_block: failed");
		initErrors |= (1 << 1);
	}
	if (rtx_signal_block (SIGQUIT)) {
		rtx_error("rtx_main_init: rtx_signal_block: failed");
		initErrors |= (1 << 2);
	}
	if (rtx_signal_block (SIGTERM)) {
		rtx_error("rtx_main_init: rtx_signal_block: failed");
		initErrors |= (1 << 4);
	}
	if (rtx_signal_block (SIGUSR1)) {
		rtx_error("rtx_main_init: rtx_signal_block: failed");
		initErrors |= (1 << 5);
	}
	if (rtx_signal_block (SIGUSR2)) {
		rtx_error("rtx_main_init: rtx_signal_block: failed");
		initErrors |= (1 << 6);
	}

	if (initErrors)
	        return (-1);
	return (0);
}

/**
 * wait for the shutdown signal
 *
 * @return 0 on success, -1 on error
 */
int
rtx_main_wait_shutdown (
			RtxThreadPrio prio     /**< priority of the waiting thread */
			)
{

	int     signals[] = {SIGINT, SIGQUIT, SIGTERM, 0};
	sigset_t signalSet;
	int	signalIndex;
#ifndef i486_linux
	struct sched_param	schedulingParameters;
	int	prioSystem;
#endif
	int     retVal, signo = -1;
	pid_t   myPid, myParentPid;
	char * xp = NULL;
	union sigval val;

	/* Ignore the signals that might be generated by network
	 * sockets */
	rtx_signal_ignore (SIGIO);
	rtx_signal_ignore (SIGPIPE);
	rtx_signal_ignore (SIGURG);

	myPid = getpid ();
	myParentPid = getppid ();
	xp = getenv ("LAUNCHER");

	/*
	 * Prepare the signal set
	 */
	if(sigemptyset(&signalSet)) {
		return (rtx_error_errno ("rtx_main_wait_shutdown (): "
					"sigemptyset () failed: "));
	}

	/*
	 * Load the killing signals
	 */
	for(signalIndex=0; signals[signalIndex] != 0; signalIndex++)
		if( sigaddset(&signalSet, signals[signalIndex])) {
			return (rtx_error_errno ("rtx_main_wait_shutdown (): "
						"sigaddset () failed: "));

		}

	/*
#ifndef i486_linux
?? pic
*/
#if !defined(i486_linux) && !defined(DARWIN)
	/*
	 * Set the priority of the thread to prio
	 */

	/* we shouldnt really change the prio or the scheduler for the main thread. */
	prioSystem = rtx_thread_prio_to_system(prio);
	if (prioSystem > 0) {
		schedulingParameters.sched_priority = prioSystem;
		if (sched_setscheduler(getpid(), SCHED_FIFO, &schedulingParameters)<0) {
			return (rtx_error_errno ("rtx_main_wait_shutdown (): "
						"sched_setscheduler () failed: "));
		}
	}
#endif

	/*
	 * Wait to receive a kill signal and then shut everything down
	 */
	do {
		retVal = sigwait( &signalSet, &signo);
		if (retVal == -1) {
			return (rtx_error_errno ("rtx_main_wait_shutdown (): "
						"sigwait () failed"));
		}
		if (sigismember (&signalSet, signo) == 1)
			return (0);
		rtx_timer_sleep (0.1);
	} while (1);

	return 0;
}

/**
 * send the shutdown signal to the application
 *
 */
void
rtx_main_signal_shutdown(void)
{
	kill(0,SIGQUIT);
}

/**
 * RTC server function used by RTC clients to send the shutdown signal
 *
 * @return string containing return status ("ERROR" or "OK")
 */
char *
rtx_main_signal_rtc_shutdown (
			      int argc,        /**< number of arguments */
			      char * argv[]    /**< arguments */
    )
{
    static char res[10];
    int pid;

    if (argc != 2) {
        strcpy (res, "ERROR");
        return res;
    }

    pid = getpid ();
    fprintf (stderr, "rtx_main_signal_rtc_shutdown (): process %d "
	     "received signal %s", pid, argv[1]);

    /* find the signal specified in the command */
    if (strcmp (argv[1], "SIGINT") == 0) {
        if (kill (pid, SIGINT) == 0) {
	    strcpy (res, "OK");
	    return res;
	} else {
	    strcpy (res, "ERROR");
	    return res;
	}
    } else if (strcmp (argv[1], "SIGQUIT") == 0) {
        if (kill (pid, SIGQUIT) == 0) {
	    strcpy (res, "OK");
	    return res;
	} else {
	    strcpy (res, "ERROR");
	    return res;
	}
    }
    else if (strcmp (argv[1], "SIGTERM") == 0) {
        if (kill (pid, SIGTERM) == 0) {
	    strcpy (res, "OK");
	    return res;
	} else {
	    strcpy (res, "ERROR");
	    return res;
	}
    }

    strcpy (res, "ERROR");
    return res;
}

/**
 * SRPC server function used by SRPC clients to send the shutdown signal
 *
 * @return string containing return status ("ERROR" or "OK")
 */
int
rtx_main_signal_srpc_shutdown (
			       char * buf,      /**< buffer to use for response */
			       int len,         /**< size of response buffer */
			       int argc,        /**< number of arguments */
			       char * argv[]    /**< arguments */
			       )
{
    int pid;

    if (argc != 2) {
        strcpy (buf, "ERROR");
        return (0);
    }

    pid = getpid ();
    fprintf (stderr, "rtx_main_signal_rtc_shutdown (): process %d "
	     "received signal %s", pid, argv[1]);

    /* find the signal specified in the command */
    if (strcmp (argv[1], "SIGINT") == 0) {
        if (kill (pid, SIGINT) == 0) {
	    strcpy (buf, "OK");
	    return (0);
	} else {
	    strcpy (buf, "ERROR");
	    return (0);
	}
    } else if (strcmp (argv[1], "SIGQUIT") == 0) {
        if (kill (pid, SIGQUIT) == 0) {
	    strcpy (buf, "OK");
	    return (0);
	} else {
	    strcpy (buf, "ERROR");
	    return (0);
	}
    }
    else if (strcmp (argv[1], "SIGTERM") == 0) {
        if (kill (pid, SIGTERM) == 0) {
	    strcpy (buf, "OK");
	    return (0);
	} else {
	    strcpy (buf, "ERROR");
	    return (0);
	}
    }

    strcpy (buf, "ERROR");
    return (0);
}

/**
 * signal the process launcher that the application is running
 *
 * Note: this function is now subsumed by rtx_main_wait_shutdown()
 * and therefore should be used only if rtx_main_wait_shutdown()
 * is not being used.
 *
 * @return 0 on success, -1 on error
 */
int
rtx_main_signal_launcher (void)
{
    pid_t   myPid, myParentPid;
    char * xp = NULL;

    myPid = getpid ();
    myParentPid = getppid ();
    xp = getenv ("LAUNCHER");
    if ((xp != NULL) && (strcmp (xp, "TRUE") == 0))
        if (kill (myParentPid, SIGUSR1) == -1) 
	    return (rtx_error_errno ("rtx_main_signal_launcher: kill() failed"));
    return (0);
}
#endif
