/***********************************************************************
 * 
 * CSIRO Autonomous Systems Laboratory
 * Queensland Centre for Advanced Technologies
 * PO Box 883, Kenmore, QLD 4069, Australia
 * http://www.ict.csiro.au/
 *  
 * Copyright (c) CSIRO 
 ***********************************************************************/

/**
 ********************************************************************
 *
 * \file logread.c
 * \brief reader module for log files created by the data-logging module
 * \author Pavan Sikka
 * \author Felix Ruess 25/01/2008
 *
 *
 * This module provides a simple interface to read logfiles created by
 * the RTX log module. Typically, a user needs to use the rtx_logread_init()
 * function to open the logfile (which could be a gzipped file). The user
 * can then use the rtx_logread_tag_var() to tag the variables of interest.
 * The rtx_logread_next() returns a pointer to the RtxLogreadVar structure
 * holding the next record read from the logfile. The user can use the
 * attribs field of the RtxParseVar field (v) to test for the VAR_ATTR_TAG
 * flag. The user can use the rtx_logread_close() function when the
 * logfile is no longer needed. The rtx_logread_rewind() can be used to
 * rewind to the beginning of the first record in the logfile.
 *
 * Can deal with streams (e.g. sockets) instead of only files now too.
 *
 ********************************************************************
 */

#include <unistd.h>
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <signal.h>
#include <stddef.h>
#include <dirent.h>
#include <time.h>
#include <sys/file.h>
#include <sys/types.h>
#include <fcntl.h>
#include <string.h>
#include <errno.h>
#include <pthread.h>
#include <semaphore.h>
#include <zlib.h>

#include "rtx/defines.h"
#include "rtx/message.h"
#include "rtx/thread.h"
#include "rtx/mutex.h"
#include "rtx/sem.h"
#include "rtx/error.h"
#include "rtx/time.h"
#include "rtx/timer.h"
#include "rtx/log.h"
#include "rtx/parse.h"
#include "rtx/logread.h"

#define SLEEPTIME 0.01

static char rcsid[] RTX_UNUSED = "$Id: logread.c 3085 2008-05-15 12:45:51Z roy029 $";

/* forward defines */

static int      rtx_logread_init_vars(RtxLogread * l);
static int      rtx_logread_read_header(RtxLogread * l);
static int 		rtx_logread_read(void *ptr, int n, RtxLogread *l);
static char*	rtx_logread_readline(char *s, int size, RtxLogread *l);
static void     swap_data(void *d, int length, int swapword);

/**
 * rtx_logread_init - setup the specified log file for reading.
 *
 * This function opens the specified log file. It then reads the header
 * and sets up all the data structures required to read data from the
 * log file. The rtx_logread_next() function can be used to read the next
 * from the log file.
 *
 * \param filename name of the log file
 * \param verbose an integer specifying the verbose level
 */
RtxLogread *
rtx_logread_init(char *filename, int verbose)
{
	RtxLogreadVar        *rv;
	RtxLogread * l;

	if ((l = (RtxLogread *) calloc (1, sizeof (RtxLogread))) == NULL) {
	        fprintf(stderr, "rtx_logread_init: no memory\n");
		return (NULL);
	}
	l->varc.vh.id = -1;
	l->varc.v = &(l->commentVar);;
	strcpy(l->varc.v->name, "comment");
	l->varc.d = l->comment;
	l->verbose = verbose;

	if (verbose)
	        fprintf (stderr, "rtx_logread_init: opening %s\n", filename);
	if (strstr(filename, ".gz") != NULL) {
	        l->gzippedLogFile = 1;
	        l->fileDescriptor = 0;
		if (verbose)
		        fprintf (stderr, "rtx_logread_init: opening compressed file\n");
	}
	
	if (filename == NULL) {
		l->logFileP = stdin;
		if ((l->logFile = strdup ("stdin")) == NULL) {
	        	fprintf(stderr, "rtx_logread_init: no memory\n");
			return (NULL);
		}
	} else {
	        if (l->gzippedLogFile) {
		        if ((l->gzLogFile = gzopen(filename, "r")) == NULL) {
			        fprintf(stderr, "rtx_logread_init: gzopen %s failed: %s\n",
					filename, strerror(errno));
				return (NULL);
			}
		} else {
		        if ((l->logFileP = fopen(filename, "r")) == NULL) {
			        fprintf(stderr, "rtx_logread_init: fopen %s failed: %s\n",
					filename, strerror(errno));
				return (NULL);
			}
		}
		if ((l->logFile = strdup (filename)) == NULL) {
		        fprintf(stderr, "rtx_logread_init: no memory\n");
			return (NULL);
		}
	}
	if (verbose)
  	        fprintf (stderr, "rtx_logread_init: processing header\n");
	if (rtx_logread_init_vars(l) == -1) {
		fprintf(stderr, "rtx_logread_init: rtx_logread_init_vars failed\n");
		return (NULL);
	}
	if (verbose) {
		fprintf(stderr, "remoteData = %d\n", l->remoteData);
		fprintf(stderr, "swapType   = %d\n", l->swapType);
		fprintf(stderr, "oldLogFile = %d\n", l->oldLogFile);
	}

	/*
	 * Create Map of Variables 
	 */

	if (verbose)
		fprintf(stderr, "%d top level variables in file\n", l->numVars);
	if ((l->varmap = (RtxLogreadVar **) calloc(l->numVars, sizeof(RtxLogreadVar *)))
	                == NULL) {
	        fprintf(stderr, "rtx_logread_init: no memory\n");
		return (NULL);
	}

	for (rv=l->vars; rv!=NULL; rv=rv->next)
		l->varmap[rv->vh.id] = rv;

	if (verbose) {
	        for (rv=l->vars; rv!=NULL; rv=rv->next)
		        rtx_parse_print_decl(rv->v, 0);
		if (l->remoteData)
		        for (rv=l->vars; rv!=NULL; rv=rv->next)
			        rtx_parse_print_decl(rv->vRemote, 0);
	}

	l->max_id = l->numVars;
	if (verbose)
		fprintf(stderr, "Max Id %d\n", l->max_id);
	return (l);
}

/**
 * rtx_logread_init_fd - setup the specified file descriptor for reading.
 *
 * This function reads the header and sets up all the data structures 
 * required to read data from the file descriptor (e.b. socket).
 * The rtx_logread_next() function can be used to read the next
 * from the fd. (blocking read)
 *
 * \param fd socket file descriptor
 * \param verbose an integer specifying the verbose level
 */
RtxLogread *
rtx_logread_init_fd(int fd, int verbose)
{
	RtxLogread * l;
	RtxLogreadVar        *rv;

	if ((l = (RtxLogread *) calloc (1, sizeof (RtxLogread))) == NULL) {
	        fprintf(stderr, "rtx_logread_init: no memory\n");
		return (NULL);
	}
	l->varc.vh.id = -1;
	l->varc.v = &(l->commentVar);;
	strcpy(l->varc.v->name, "comment");
	l->varc.d = l->comment;
	l->verbose = verbose;

	if (verbose)
	        fprintf (stderr, "rtx_logread_init: opening socket\n");
	l->gzippedLogFile = 0;
	l->fileDescriptor = 1;
	l->fd = fd;
	l->logFileP = NULL;
	l->logFile = NULL;

	if (verbose)
  	        fprintf (stderr, "rtx_logread_init: processing header\n");
	if (rtx_logread_init_vars(l) == -1) {
		fprintf(stderr, "rtx_logread_init: rtx_logread_init_vars failed\n");
		return (NULL);
	}
	if (verbose) {
		fprintf(stderr, "remoteData = %d\n", l->remoteData);
		fprintf(stderr, "swapType   = %d\n", l->swapType);
		fprintf(stderr, "oldLogFile = %d\n", l->oldLogFile);
	}

	/*
	 * Create Map of Variables 
	 */

	if (verbose)
		fprintf(stderr, "%d top level variables in file\n", l->numVars);
	if ((l->varmap = (RtxLogreadVar **) calloc(l->numVars, sizeof(RtxLogreadVar *)))
	                == NULL) {
	        fprintf(stderr, "rtx_logread_init: no memory\n");
		return (NULL);
	}

	for (rv=l->vars; rv!=NULL; rv=rv->next)
		l->varmap[rv->vh.id] = rv;

	if (verbose) {
	        for (rv=l->vars; rv!=NULL; rv=rv->next)
		        rtx_parse_print_decl(rv->v, 0);
		if (l->remoteData)
		        for (rv=l->vars; rv!=NULL; rv=rv->next)
			        rtx_parse_print_decl(rv->vRemote, 0);
	}

	l->max_id = l->numVars;
	if (verbose)
		fprintf(stderr, "Max Id %d\n", l->max_id);
	return (l);
}

/**********************************************************************/

/**
 * rtx_logread_rewind - rewind the log file to the first data record.
 *
 * This function rewinds the log file to the beginning of the first
 * data record in the file.
 * Returns an error if called for a socket.
 *
 * \param l handle for the log file being read
 *
 */
int     
rtx_logread_rewind(RtxLogread * l)
{
    if (l->gzippedLogFile)
        return (gzseek (l->gzLogFile, l->initOffset, SEEK_SET));
    else if (l->fileDescriptor) {
    	fprintf(stderr, "rtx_logread_rewind: Can not rewind on socket!");
    	return (-1);
    }
    else
        return (fseek (l->logFileP, l->initOffset, SEEK_SET));
}

/**********************************************************************/

/**
 * rtx_logread_close - close the log file.
 *
 * This function closes the log file being read.
 *
 * \param l handle for the log file being read
 *
 */
void
rtx_logread_close(RtxLogread * l)
{
    if (l->gzippedLogFile)
	gzclose(l->gzLogFile);
	else if (l->fileDescriptor)
		close(l->fd);
    else
	fclose(l->logFileP);
}

/**********************************************************************/

/**
 * rtx_logread_tag - tag the variable for printing.
 *
 * This function tags the specified logger variable for printing.
 *
 * \param l handle for the log file being read
 * \param name name of the variable to be tagged
 *
 */
RtxLogreadVar        *
rtx_logread_tag_var(RtxLogread * l, char *name)
{
	RtxParseVar            *v = NULL, *v1;
	int             parseIndex = 0;
	RtxParseToken           t;
	RtxParseVarTableEntry *stEntry;

	if (!strcmp(name, "comment")) {
		l->varc.v->attribs |= VAR_ATTR_TAG;
		return &(l->varc);
	}

	memset(&t, 0, sizeof(t));
	if ((v = calloc(1, sizeof(RtxParseVar))) == NULL) {
		fprintf(stderr, "mem alloc failed\n");
		return (NULL);
	}
	/*
	 * This call will tag the nodes for printing 
	 */
	if (l->verbose)
		fprintf(stderr, "rtx_logread_tag: tagging %s\n", name);
	if ((v1 = rtx_parse_name_list(name, &parseIndex, &t, v, l->symTabs)) == NULL) {
		if (l->verbose)
			fprintf(stderr, "name (%s) not found\n", name);
		return (NULL);
	}
	rtx_parse_free_var(v1);
	if ((stEntry = (RtxParseVarTableEntry *) rtx_hash_find 
	     (l->symTabs->varTable, v->name, (rtx_error_t)NULL)) == NULL) {
		if (l->verbose)
			fprintf(stderr, "name (%s) not found\n", v->name);
		return (NULL);
	}
	if (l->verbose)
		fprintf(stderr, "rtx_logread_tag: done\n");
	return (l->varmap[stEntry->varId]);
}

/**********************************************************************/

/**
 * rtx_logread_next - read the next record from the log file.
 *
 * This function reads the next data record from the log file.
 *
 * \param l handle for the log file being read
 *
 */
RtxLogreadVar        *
rtx_logread_next(RtxLogread * l)
{
	RtxLogItemHeader vh;
	RtxLogreadVar        *rv;

	/*
	 * Read the Header 
	 */

	if (l->verbose)
		fprintf(stderr, "rtx_logread_next: reading header = %d bytes\n",
			sizeof(RtxLogItemHeader));
	if (rtx_logread_read((unsigned char *) &vh, sizeof(RtxLogItemHeader), l) <= 0)
		        return NULL;
	swap_data(&vh, sizeof(RtxLogItemHeader), l->swapword);

	/*
	 * Get the Starting Time Stamp 
	 */

	if (l->start.seconds == 0) {
		l->start.seconds = vh.ts.seconds;
		l->start.nanoSeconds = vh.ts.nanoSeconds;
	}

	/*
	 * Check the Header ID 
	 */

	if (l->verbose)
		fprintf(stderr, "Next ID %d\n", vh.id);
	if (vh.id >= l->max_id || vh.id < -1) {
		fprintf(stderr, "Unknown ID %d\n", vh.id);
		return NULL;
	}

	/*
	 * Read a Comment 
	 */
	/*
	 * Needs to read in comment, rather than data.s because it is
	 * character 
	 */

	if (vh.id == -1) {
		l->varc.vh.ts = vh.ts;
		if (vh.spare >= 512) {
			fprintf(stderr, "Comment too long\n");
			return (NULL);
		}
		if (rtx_logread_read(l->comment, vh.spare, l) <= 0)
			        return NULL;
		l->comment[vh.spare] = 0;
		return &(l->varc);
	}

	/*
	 * Read Some Data 
	 */

	rv = l->varmap[vh.id];
	rv->vh.ts = vh.ts;
	if (l->remoteData) {
		if (l->verbose)
			fprintf(stderr,
				"rtx_logread_next: reading %s data = %d bytes\n",
				rv->vRemote->name, rv->vRemote->size);
		if (rtx_logread_read(rv->dRemote, rv->vRemote->size, l) <= 0)
			        return NULL;
		if (l->verbose)
		        fprintf (stderr, "rtx_logread_next: translating data\n");
		rtx_parse_translate_data(rv->d, rv->v, rv->dRemote, rv->vRemote,
					 l->swapType);
	}
	else {
		if (l->verbose)
			fprintf(stderr,
				"rtx_logread_next: reading %s data = %d bytes\n",
				rv->v->name, rv->v->size);
		if (rtx_logread_read(rv->d, rv->v->size, l) <= 0)
				return NULL;
	}
	return rv;
}

/**
 * read the next tagged record from the log file.
 *
 * \param l handle for the log file being read
 *
 */
RtxLogreadVar        *
rtx_logread_next_tagged(RtxLogread * l)
{
	RtxLogreadVar        *rv;

	while ((rv = rtx_logread_next (l)) != NULL)
	    if (rv->v->attribs & VAR_ATTR_TAG)
	        return (rv);
	return rv;
}

/************************ support functions ******************************/

static int
rtx_logread_init_vars(RtxLogread * l)
{
	RtxParseVar            *v = NULL;
	RtxParseVar            *v1 = NULL;
	RtxLogreadVar        *rv;
	int             parseIndex;
	int             parseIndexLocal;
	RtxParseToken           t;
	int             done = 0;
	int             numVars = 0;
	int             ret;

	l->header[0] = '\0';

	if (rtx_logread_read_header(l) == -1) {
		fprintf(stderr, "invalid header\n");
		return (-1);
	}
	if (l->verbose)
	        fprintf (stderr, "rtx_logread_init_vars: header read\n");
	if ((l->symTabs = rtx_parse_init ()) == NULL) {
		fprintf(stderr, "failed to initialize type table\n");
		return (-1);
	}
	if (l->remoteData) {
	        if ((l->remSymTabs = rtx_parse_init_remote 
		     (l->remoteAlign, l->remoteSize)) == NULL) {
			fprintf(stderr, "failed to initialize type table\n");
			return (-1);
		}
	}
	parseIndex = 0;
	do {
		memset(&t, 0, sizeof(t));
		parseIndexLocal = parseIndex;
		if ((v = calloc(1, sizeof(RtxParseVar))) == NULL) {
			fprintf(stderr, "mem alloc failed\n");
			return (-1);
		}
		if (l->oldLogFile) {
		        if (rtx_parse_old_decl(l->header, &parseIndex, &t, v, 
					       l->symTabs) == -1) {
				done = 1;
				continue;
			}
		}
		else {
			if ((ret = rtx_parse_decl(l->header, &parseIndex, &t, v, 
					      l->symTabs)) == 0) {
				done = 1;
				continue;
			}
			if (ret == -1) {
			        return (-1);
			}
		}
		if (rtx_parse_add_var_symbol(v, 0, 0, numVars, "", 
					     l->symTabs->varTable) == -1) {
			fprintf(stderr, "add_var_symbol failed\n");
			return (-1);
		}
		if ((rv = calloc(1, sizeof(RtxLogreadVar))) == NULL) {
			fprintf(stderr, "mem alloc failed\n");
			return (-1);
		}
		rv->vh.id = numVars;
		rv->next = l->vars;
		l->vars = rv;
		rv->v = v;
		numVars++;
		if ((rv->d = (void *) calloc(1, v->size)) == NULL) {
			fprintf(stderr, "mem alloc failed\n");
			return (-1);
		}
		if (l->remoteData) {
			memset(&t, 0, sizeof(t));
			if ((v1 = calloc(1, sizeof(RtxParseVar))) == NULL) {
				fprintf(stderr, "mem alloc failed\n");
				return (-1);
			}
			if (l->oldLogFile) {
				if (rtx_parse_old_decl (l->header,
						    &parseIndexLocal, &t, v1, 
						    l->remSymTabs) == -1) {
					fprintf(stderr,
						"couldnt parse structure already parsed !\n");
					return (-1);
				}
			}
			else {
				if (rtx_parse_decl (l->header,
						&parseIndexLocal, &t, v1,
						l->remSymTabs) == -1) {
					fprintf(stderr,
						"couldnt parse structure already parsed !\n");
					return (-1);
				}
			}
			if ((rv->dRemote = (void *) calloc(1, v1->size)) ==
			    NULL) {
				fprintf(stderr, "mem alloc failed\n");
				return (-1);
			}
			rv->vRemote = v1;
		}
	} while (!done);

	l->numVars = numVars;
	return (numVars);
}

static char*
rtx_logread_readline(char *s, int size, RtxLogread *l)
{
	if (l->gzippedLogFile) {
		if (l->verbose)
			fprintf (stderr, "rtx_logread_readline: reading gzipped file\n");
		return gzgets(l->gzLogFile, s, size);
	} else if (l->fileDescriptor) {
		int i = 0, n = 0;
		char ch;
		
		while (i < size - 1) {
			n = read (l->fd, &ch, 1);
			if(n == -1) { /* read error */
				if ((errno == EAGAIN) || (errno == EINTR)) {/* no data available (non-blocking I/O) or read was interrupted by a signal*/
					fprintf (stderr, "rtx_logread_readline: read() interrupted or no data available");
					rtx_timer_sleep(SLEEPTIME);
					continue;
				}
				else {
					fprintf(stderr, "rtx_logread_readline: read() failed\n");
					return NULL;
				}
			} else if (n == 0) {   /* socket closed */
				break;
			}
			s[i++] = ch;
			if (ch == '\n')
				break;
		}
		s[i] = '\0';
		return s;
	} else {
		return fgets(s, size, l->logFileP);
	}
	return NULL;
}

static int
rtx_logread_read(void *ptr, int n, RtxLogread *l)
{
	if (l->gzippedLogFile) {
		return gzread(l->gzLogFile, ptr, n);
	} else if (l->fileDescriptor) {
		int i = 0, n_read = 0;
		char ch;
		char *p = ptr;
		
		while (i < n) {
			n_read = read (l->fd, &ch, 1);
			if(n_read == -1) { /* read error */
				if ((errno == EAGAIN) || (errno == EINTR)) {/* no data available (non-blocking I/O) or read was interrupted by a signal*/
					fprintf (stderr, "rtx_logread_read: read() interrupted or no data available");
					rtx_timer_sleep(SLEEPTIME);
					continue;
				}
				else {
					fprintf (stderr, "rtx_logread_read: read() failed");
					return (-1);
				}
			} else if (n_read == 0) {   /* socket closed */
				break;
			}
			p[i++] = ch;
		}
		return i;
	} else {
		return fread(ptr, 1, n, l->logFileP);
	}
	return -1;
}

static int
rtx_logread_read_header(RtxLogread * l)
{
	int             headerLen = 0, haveValidHeader = 0;
	static char     headerLine[BUFSIZ];
	int             i, locSwap;

	while (headerLen < RTX_LOG_MAX_HEADER_SIZE) {
		if (rtx_logread_readline(headerLine, BUFSIZ, l) == NULL)
			        break;
		if (l->verbose)
		        fprintf (stderr, "rtx_logread_read_header: %s\n",
				 headerLine);
		if (strcmp(headerLine, RTX_LOG_HEADER_EOH) == 0) {
			haveValidHeader = 1;
			break;
		}
		if (headerLine[0] == '%') {
			if (strstr(headerLine, "%%primitiveDataSize:") != NULL) {
				if (sscanf(headerLine, "%*s %d %d %d %d %d %d",
					   &(l->remoteSize[rtx_char_t]), &(l->remoteSize[rtx_short_t]),
					   &(l->remoteSize[rtx_int_t]), &(l->remoteSize[rtx_long_t]),
					   &(l->remoteSize[rtx_float_t]),
					   &(l->remoteSize[rtx_double_t])) == 6) {
					l->remoteData++;
				}
			} else if (strstr(headerLine, "%%primitiveDataAlignment:") != NULL) {
				if (sscanf(headerLine, "%*s %d %d %d %d %d %d",
				     &(l->remoteAlign[rtx_char_t]), &(l->remoteAlign[rtx_short_t]),
				     &(l->remoteAlign[rtx_int_t]), &(l->remoteAlign[rtx_long_t]),
				     &(l->remoteAlign[rtx_float_t]),
				     &(l->remoteAlign[rtx_double_t])) == 6) {
					l->remoteData++;
				}
			} else if (strstr(headerLine, "%%CatalogName:") != NULL) {
				sscanf(headerLine, "%*s %s", (l->catalogName));
			}
		}
		strcat(l->header, headerLine);
		headerLen += strlen(headerLine);
	}
	if (l->verbose)
	    fprintf (stderr, "HEADER:\n%s\n", l->header);
	if (haveValidHeader) {
  	        rtx_parse_get_compiler_parms (l->localSize, l->localAlign);
		switch (l->remoteData) {
		case 0:
		        for (i=0; i<8; i++) {
			        l->remoteSize[i] = l->localSize[i];
			        l->remoteAlign[i] = l->localAlign[i];
			}
			break;
		case 2:
			break;
		default:
			printf("Invalid size/alignment info in header\n");
			return (-1);
		}
		l->remoteData = 0;
		for (i=rtx_char_t; i<=rtx_double_t; i++) {
		        if (l->localSize[i] != l->remoteSize[i]) {
			        l->remoteData = 1;
				break;
			}
		        if (l->localAlign[i] != l->remoteAlign[i]) {
			        l->remoteData = 1;
				break;
			}
		}
		rtx_logread_read(&(l->swapword), sizeof(int), l);
		if (l->verbose)
			fprintf(stderr, "swap word is 0x%08x\n", l->swapword);
		l->oldLogFile = 0;
		switch (l->swapword) {
		case 0x01020304:
			l->oldLogFile = 1;
			locSwap = 0x01020304;
			break;
		case 0x04030201:
			l->oldLogFile = 1;
			locSwap = 0x04030201;
			break;
		case 0x02030405:
			l->oldLogFile = 0;
			locSwap = 0x01020304;
			break;
		case 0x05040302:
			l->oldLogFile = 0;
			locSwap = 0x04030201;
			break;
		default:
			printf("Unknown swap order 0x%08x\n", l->swapword);
			return (-1);
		}
		if (locSwap == 0x01020304)
			l->swapType = 0;
		else
			l->swapType = 1;
		if (l->swapType)
			l->remoteData = 1;
		l->headerLen = headerLen;
		if (l->gzippedLogFile) {
			l->initOffset = gztell (l->gzLogFile);
		} else if (l->fileDescriptor) {
			l->initOffset = 0;
		} else {
			l->initOffset = ftell (l->logFileP);
		}
		return (headerLen);
	}
	return (-1);
}

void
rtx_logread_print_var(
			 FILE * fp, 
			 RtxParseVar * v, 
			 char *d, 
			 int offset
			 )
{
	int             i, m, n, k;
	char           *charp;
	short          *shortp;
	int            *intp;
	long           *longp;
	float          *floatp;
	double         *doublep;
	RtxParseVar            *v1;

	if (v == NULL)
		return;
	if (!(v->attribs & VAR_ATTR_TAG))
		return;
	m = 0;
	n = 1;
	if (v->dim) {
		if (v->attribs & VAR_ATTR_ENTIRE_ARRAY) {
			for (i = 0; i < v->dim; i++) {
				n *= v->arrayDim[i];
			}
		}
		else {
			m = 0;
			k = 1;
			for (i = v->dim - 1; i >= 0; i--) {
				m += (k * v->arrayIndex[i]);
				k *= v->arrayDim[i];
			}
			n = m + 1;
		}
	}
	switch (v->type) {
	case rtx_char_t:
		charp = (char *) &(d[v->offset + offset]);
		for (i = m; i < n; i++)
			fprintf(fp, "%d ", charp[i]);
		break;
	case rtx_short_t:
		shortp = (short *) &(d[v->offset + offset]);
		for (i = m; i < n; i++)
			fprintf(fp, "%d ", shortp[i]);
		break;
	case rtx_int_t:
		intp = (int *) &(d[v->offset + offset]);
		for (i = m; i < n; i++)
			fprintf(fp, "%d ", intp[i]);
		break;
	case rtx_long_t:
		longp = (long *) &(d[v->offset + offset]);
		for (i = m; i < n; i++)
			fprintf(fp, "%ld ", longp[i]);
		break;
	case rtx_float_t:
		floatp = (float *) &(d[v->offset + offset]);
		for (i = m; i < n; i++)
			fprintf(fp, "%f ", floatp[i]);
		break;
	case rtx_double_t:
		doublep = (double *) &(d[v->offset + offset]);
		for (i = m; i < n; i++)
			fprintf(fp, "%f ", doublep[i]);
		break;
	case rtx_struct_t:
		for (i = m; i < n; i++) {
			for (v1 = v->subVar; v1 != NULL; v1 = v1->next) {
				rtx_logread_print_var(fp, v1, d,
						  offset + v->offset + i * v->elemSize);
			}
		}
		break;
	default:
		fprintf(stderr, "rtx_logread_print_var: invalid type\n");
		return;
	}
}

void
rtx_logread_print(FILE * fp, RtxLogreadVar * rv)
{
	if (!(rv->v->attribs & VAR_ATTR_TAG))
		return;

	if (rv->vh.id == -1) {
		fprintf(fp, " %s\n", (char *) rv->d);
		return;
	}

	rtx_logread_print_var(fp, rv->v, rv->d, 0);
}

/* 
** Return Time 
*/

double
rtx_logread_time(RtxLogreadVar * rv)
{
	return ((double) rv->vh.ts.seconds +
		(double) ((rv->vh.ts.nanoSeconds) * 1e-9));
}

/* 
** Return Time as a string "Month day hours:minutes:seconds.nanoseconds"
*/

char           *
rtx_logread_stime(RtxLogreadVar * rv)
{
	char     timeString[64];
	char     retString[64];
	struct tm locTm;

        memset (&locTm, 0, sizeof (locTm));
	localtime_r (&(rv->vh.ts.seconds), &locTm);
	strftime(timeString, 64, "%b %d %H:%M:%S", &locTm);
	sprintf(retString, "%s.%09ld", timeString, rv->vh.ts.nanoSeconds);
	return (strdup (retString));
}

/* 
** Return Relative Time 
*/

double
rtx_logread_rtime(RtxLogreadVar * rv, RtxTime start)
{
	time_t          sec;
	long            nsec;
	sec = rv->vh.ts.seconds - start.seconds;
	nsec = rv->vh.ts.nanoSeconds - start.nanoSeconds;
	if (nsec < 0) {
		nsec += 1000000000;
		sec--;
	}
	return ((double) sec + (double) (nsec * 1e-9));
}

/*
 * find all logfiles in the specified directory and sort them into descending
 * order, ie. latest through to oldest.
 */

static int
sortfunc(void **ap, void **bp)
{
	char           *a = (char *) *ap;
	char           *b = (char *) *bp;
	return strcmp(b, a);
}

char          **
rtx_logread_get_files(char *dir, int *nf)
{
	DIR            *d;
	struct dirent  *de;
	char          **names = NULL;
	int             nfiles = 0;
	char            exten[BUFSIZ];
	char            exten1[BUFSIZ];
	char            logno[BUFSIZ];

	if ((d = opendir(dir)) == NULL)
		return (NULL);

	while ((de = readdir(d)) != NULL) {
	        exten[0] = '\0';
	        exten1[0] = '\0';
		if (sscanf(de->d_name, "%[0-9].%s", logno, exten) != 2) {
		        if (sscanf(de->d_name, "%[0-9].%s.%s", logno, exten, exten1) != 3) {
			        continue;
			} else {
			        if ((strncmp (exten, "log", 3) != 0) ||
				    (strncmp (exten1, "gz", 2) != 0))
				        continue;
			}
		} else {
		        if (strncmp(exten, "log", 3) != 0)
			        continue;
		}
		if (names == NULL)
			names = malloc(16 * sizeof(char *));
		else
			names = realloc(names, (nfiles + 1) * sizeof(char *));
		if (names == NULL)
			return (NULL);
		//names[nfiles] = strdup(de->d_name);
		names[nfiles] = (char*)malloc((strlen(dir)+strlen(de->d_name)+2)*sizeof(char));
		strcpy(names[nfiles],dir);
		strcat(names[nfiles],"/");
		strcat(names[nfiles],de->d_name);
		nfiles++;
	}

	closedir(d);
	qsort((void *) names, nfiles, sizeof(char *),
	      (int (*)(const void *, const void *)) sortfunc);

	*nf = nfiles;
	return names;
}

/* 
** To maintain Device independance - Swap Bytes
*/

static void
swap_data(void *d, int length, int swapword)
{
	unsigned char  *p, t0, t1;
	p = (unsigned char *) d;

	switch (swapword) {

	case 0x01020304:
	case 0x02030405:
		return;
	case 0x02010403:	/*
				 * byte swapped 
				 */
	case 0x03020504:
		for (; length > 0; length -= 2, p += 2) {
			t0 = p[0];
			p[0] = p[1];
			p[1] = t0;
		}
		return;
	case 0x04030201:	/*
				 * byte+word swapped 
				 */
	case 0x05040302:
		for (; length > 0; length -= 4, p += 4) {
			t0 = p[0];
			p[0] = p[3];
			p[3] = t0;
			t1 = p[1];
			p[1] = p[2];
			p[2] = t1;
		}
		return;
	default:
		fprintf(stderr, "Unknown swap order 0x%x\n", swapword);
		return;
	}
}

