/***********************************************************************
 * 
 * CSIRO Autonomous Systems Laboratory
 * Queensland Centre for Advanced Technologies
 * PO Box 883, Kenmore, QLD 4069, Australia
 * http://www.ict.csiro.au/
 *  
 * Copyright (c) CSIRO 
 ***********************************************************************/

/**
 ********************************************************************
 *
 * \file log.c
 * \brief Data-logging module
 * \author Pavan Sikka
 * \anchor log
 *
 * This module provides applications with a buffered data-logging 
 * facility. The rtx_log_open() function creates buffers in memory
 * and starts up a file-writer thread. Applications then use the
 * rtx_log_write set of functions which write data to the internal
 * memory buffers. The file-writer thread empties the internal
 * memory buffers to the log file. The module rolls over log files
 * based on age and size.
 *
 ********************************************************************
 */

#include <unistd.h>
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <signal.h>
#include <stddef.h>
#include <dirent.h>
#include <time.h>
#include <sys/file.h>
#include <sys/types.h>
#include <fcntl.h>
#include <string.h>
#include <errno.h>
#include <pthread.h>
#include <semaphore.h>

#include "rtx/defines.h"
#include "rtx/message.h"
#include "rtx/thread.h"
#include "rtx/mutex.h"
#include "rtx/sem.h"
#include "rtx/error.h"
#include "rtx/time.h"
#include "rtx/timer.h"
#include "rtx/log.h"
#include "rtx/parse.h"

static char rcsid[] RTX_UNUSED = "$Id: log.c 2912 2008-04-09 06:29:51Z roy029 $";

/* Forward declarations of functions defined and used within this module */

static int      rtx_log_write_logfile_thread (RtxLog * logger);
static int      rtx_log_write_logbuf_to_file (RtxLog * logger);
static int      rtx_log_init_logger (RtxLog * logger);
static int      rtx_log_create_logfile (RtxLog * logger);
static int      rtx_log_alloc_logbufs (RtxLog * logger);
static void     rtx_log_free_logbufs (RtxLog * logger);
static void     rtx_log_write_logfile_thread_cleanup (RtxLog * logger);

/**
 * initialize the data-logger (create the buffers etc, start the writer thread)
 *
 * The logFile argument allows the application to specify the format of the
 * log file name. In general, it should be a string of the type 
 * "<prefix>:<suffix>". The file-writing module replaces the ":" with
 * a timestamp. If the logFile argument contains no ":", the argument
 * is used as is. By default, the string ":.log" is used.
 *
 * The appendCreateComment argument causes the logger to add a file-creation
 * comment to the beginning of every log file. This comment is of the form:
 * "% <appName>: created <time> by <user>@<host>".
 *
 * @return a pointer to the RtxLog structure on success, NULL on failure
 */
RtxLog *
rtx_log_open (
		char * logDir,            /**< directory where log files are created */
		char * logFile,           /**< log file name */
		int timeLimit,            /**< maximum age of file, in minutes */
		int sizeLimit,            /**< maximum size of file, in Kbytes */
		int numBufs,              /**< number of internal memory buffers */
		int bufSize,              /**< internal memory buffer size, in Kbytes */
		int writerThreadPrio,     /**< writer thread priority */
		int gzip,                 /**< gzip compression level */
		int verbose,              /**< debug level */
		unsigned char * header,   /**< log file header */
		int headerSize,           /**< log file header length */
		char * appName,           /**< appliaction name, used in the header */
		int appendCreateComment,  /**< add file-creation data to the header */
		char * indexFile          /**< update an index file with created files */
		)
{
	RtxLog * l;
	char * p;
	int len1, len2;

	if (verbose)
		rtx_message ("rtx_log_open: starting");

	if ((l = calloc (1, sizeof (RtxLog))) == NULL) {
		return (rtx_error_null ("rtx_log_open (%s:%s): calloc() failed",
					logDir, logFile));
	}
	l->done = 0;
	strcpy (l->parms.myHostName, "unknown");
	strcpy (l->parms.myUserName, "unknown");
	gethostname (l->parms.myHostName, BUFSIZ);
	p = getlogin();
	strncpy(l->parms.myUserName, (p ? p : "unknown") ,BUFSIZ);

	if (indexFile) {
		strncpy(l->parms.indexFileName,indexFile,MAX_FILENAME_LEN-1);
		l->parms.indexFileName[MAX_FILENAME_LEN-1] = 0;
	} else {
		l->parms.indexFileName[0] = 0;
	}
#ifdef USE_GZIP_COMPRESSION
	l->parms.gzip = gzip;
#endif
	l->parms.noTimeLimit = 0;
	l->parms.timeLimit = timeLimit * 60;
	if (timeLimit == 0)
		l->parms.noTimeLimit = 1;
	if (timeLimit < 0) {
		free (l);
		return (rtx_error_null ("rtx_log_open (%s:%s): time limit specified (%d)"
					" is < 0", logDir, logFile, timeLimit));
	}

	l->parms.noSizeLimit = 0;
	l->parms.sizeLimit = sizeLimit * 1024;
	if (sizeLimit == 0)
		l->parms.noSizeLimit = 1;
	if (sizeLimit < 0) {
		free (l);
		return (rtx_error_null ("rtx_log_open (%s:%s): size limit specified (%d)"
					" is < 0", logDir, logFile, sizeLimit));
	}

	if (numBufs <= 1) {
		free (l);
		return (rtx_error_null ("rtx_log_open (%s:%s): number of buffers "
					"specified (%d) is < 1", logDir, logFile, numBufs));
	}
	l->parms.numBufs = numBufs;

	l->parms.verbose = verbose;
	if (verbose < 0) {
		rtx_message_warning ("rtx_log_open (%s:%s): verbose (%d) < 0, setting to 0",
				logDir, logFile, verbose);
		l->parms.verbose = 0;
	}

	l->parms.writerThreadPrio = writerThreadPrio;
#if 0 
	if (writerThreadPrio <= 0) {
		free (l);
		return (rtx_error_null ("rtx_log_open (%s:%s): writer thread prio "
					"specified (%d) is < 1", logDir, logFile, 
					writerThreadPrio));
	}
#endif

	if (verbose)
		rtx_message ("rtx_log_open: header len = %d", headerSize);
	l->parms.headerLen = headerSize;
	if (headerSize < 0) {
		free (l);
		return (rtx_error_null ("rtx_log_open (%s:%s): header size (%d) < 0", 
					logDir, logFile, headerSize));
	}

	if (bufSize < 0) {
		free (l);
		return (rtx_error_null ("rtx_log_open (%s:%s): buf size (%d) < 0", 
					logDir, logFile, bufSize));
	}
	l->parms.bufSize = bufSize * 1024;
	if (l->parms.bufSize <= l->parms.headerLen) {
		free (l);
		return (rtx_error_null ("rtx_log_open (%s:%s): buf size must be "
					"greater than header size", logDir, logFile));
	}
	if (headerSize > 0) {
		if ((l->parms.header = calloc (1, headerSize)) == NULL) {
			free (l);
			return (rtx_error_errno_null ("rtx_log_open (%s:%s): calloc() failed",
						logDir, logFile));
		}
		memcpy (l->parms.header, header, headerSize);
		l->parms.headerLen = headerSize;
	} else {
		l->parms.header = NULL;
		l->parms.headerLen = 0;
	}

	l->parms.appendCreateComment = 0;
	if (appendCreateComment)
		l->parms.appendCreateComment = 1;
	if (verbose)
		rtx_message ("rtx_log_open: app name = %s", appName);

	if (appName != NULL) {
		if ((l->parms.appName = strdup (appName)) == NULL) {
			free (l);
			return (rtx_error_errno_null ("rtx_log_open (%s:%s): strdup() failed",
						logDir, logFile));
		}
	} else
		l->parms.appName = "";
	if (strlen (logDir) > MAX_PATHNAME_LEN) {
		free (l);
		return (rtx_error_null ("rtx_log_open (%s:%s): pathname too long "
					"(%d chars)", logDir, logFile, 
					strlen (logDir)));
	}
	if (verbose)
		rtx_message ("rtx_log_open: log dir = %s", logDir);
	if (logDir == NULL)
		strcpy (l->parms.loggerDir, "./");
	else {
		strcpy (l->parms.loggerDir, logDir);
		if (l->parms.loggerDir[strlen (logDir) - 1] != '/')
			strcat (l->parms.loggerDir, "/");
	}

	if (verbose)
		rtx_message ("rtx_log_open: log file = %s", logFile);
	/* Parse the log file name */
	if (strlen (logFile) > MAX_FILENAME_LEN) {
		free (l);
		return (rtx_error_null ("rtx_log_open (%s:%s): filename too long "
					"(%d chars)", logDir, logFile, strlen (logFile)));
	}
	if (logFile == NULL) {
		l->parms.simpleLogFileName = 0;
		strcpy (l->parms.loggerFileNamePrefix, "");
		strcpy (l->parms.loggerFileNameSuffix, ".log");
	} else if ((p = strstr (logFile, ":")) == NULL) {
		l->parms.simpleLogFileName = 1;
		strcpy (l->parms.loggerFileName, logFile);
	} else {
		if (verbose)
			rtx_message ("rtx_log_open: parsing log file name");
		l->parms.simpleLogFileName = 0;
		len1 = strlen (logFile);
		len2 = strlen (p);
		memcpy (l->parms.loggerFileNamePrefix, logFile, len1 - len2);
		l->parms.loggerFileNamePrefix[len1 - len2] = '\0';
		strcpy (l->parms.loggerFileNameSuffix, p+1);
	}

	if (verbose)
		rtx_message ("rtx_log_open: initializing logger");
	if (rtx_log_init_logger (l) == -1) {
		return (rtx_error_null ("rtx_log_open (%s:%s): rtx_log_init_logger () "
					"failed", logDir, logFile));
	}
	if (verbose)
		rtx_message ("rtx_log_open: done");

	return (l);
}

/**************************************************************************
 * rtx_log_init_logger - prepare the logger for data logging
 *
 */
int
rtx_log_init_logger (
		RtxLog * logger
		)
{
	time_t timeNow;
	char timestamp[32];
	int i, j, n = 0;
	RtxLogBuf * curLogBuf;
	char * timeStr;
	char headerComment[256];
	struct tm locTm;
	char ctimeBuf[32];

	if (rtx_log_alloc_logbufs (logger) == -1) {
		return (rtx_error ("rtx_log_init_logger: rtx_log_alloc_logbufs() "
					"failed"));;
	}
	if (logger->parms.verbose)
		rtx_message ("rtx_log_init_logger: initializing mutexes");
	if ((logger->lock = rtx_mutex_init (NULL, RTX_MUTEX_DEFAULT, 0)) == NULL) {
		return (rtx_error ("rtx_log_init_logger: rtx_mutex_init (logger) failed"));;
	}
	for (i = 0; i < logger->parms.numBufs; i++) {
		if ((logger->bufs[i].lock = rtx_mutex_init (NULL, RTX_MUTEX_DEFAULT, 0)) == NULL) {
			for (j = 0; j < i; j++)
				rtx_mutex_destroy (logger->bufs[j].lock);
			return (rtx_error ("rtx_log_init_logger: rtx_mutex_init (buf) failed"));;
		}
	}
	if (logger->parms.verbose)
		rtx_message ("rtx_log_init_logger: creating bufFull semaphore");
	if ((logger->bufFull = rtx_sem_init (NULL, 0, 0)) == NULL) {
		return (rtx_error ("rtx_log_init_logger: rtx_sem_init(bufFull) failed"));;
	}

	time (&timeNow);
	if (logger->parms.simpleLogFileName) {
		strcpy (logger->pathname, logger->parms.loggerDir);
		strcat (logger->pathname, logger->parms.loggerFileName);
	} else {
		memset (&locTm, 0, sizeof (locTm));
		localtime_r (&timeNow, &locTm);
		strftime (timestamp, 32, "%Y%m%d%H%M%S", &locTm);
		strcpy (logger->pathname, logger->parms.loggerDir);
		strcat (logger->pathname, logger->parms.loggerFileNamePrefix);
		strcat (logger->pathname, timestamp);
		strcat (logger->pathname, logger->parms.loggerFileNameSuffix);
	}

	if (rtx_log_create_logfile (logger) == -1) {
		return (rtx_error ("rtx_log_init_logger: rtx_log_create_file() failed"));;
	}
	logger->curBuf = 0;
	curLogBuf = &(logger->bufs[logger->curBuf]);
	if (logger->parms.appendCreateComment) {
		timeStr = ctime_r (&timeNow, ctimeBuf);
		timeStr[24] = '\0';
		sprintf (headerComment, "%% %s: created %s by %s@%s\n", 
				logger->parms.appName, timeStr, logger->parms.myUserName,
				logger->parms.myHostName);
		n = strlen (headerComment);
		memcpy (curLogBuf->buf, headerComment, n);
		curLogBuf->curPos = n;
	}
	memcpy (&(curLogBuf->buf[n]), logger->parms.header, 
			logger->parms.headerLen);
	curLogBuf->curPos += logger->parms.headerLen;
	logger->curFileSize = logger->parms.headerLen + n;
	logger->curFileCreated = timeNow;

	if (logger->parms.verbose)
		rtx_message ("rtx_log_init_logger: launching writer thread");
	if ((logger->writerThread = rtx_thread_create ("rtx_log_thread",
					logger->parms.verbose,
					RTX_THREAD_SCHED_OTHER,
					logger->parms.writerThreadPrio,
					0,
					RTX_THREAD_CANCEL_DEFERRED,
					(void *(*) (void *)) rtx_log_write_logfile_thread,
					logger,
					(void (*) (void *)) rtx_log_write_logfile_thread_cleanup,
					(void *) logger)) == NULL) {
		return (rtx_error ("rtx_log_init_logger: rtx_thread_create() failed"));;
	}
	return (0);
}

/**************************************************************************
 * rtx_log_close - logger is terminating, shutdown this module
 *
 * This function signals the file_writer thread to terminate and then waits
 * for it. After that, it frees all the memory, mutexes and semaphores.
 *
 * @return 0 if OK, -1 if errors
 */
int
rtx_log_close (
		RtxLog * logger,          /**< handler */
		RtxLogStats * stats       /**< storage for logger stats */
		)
{
	int j;
	int errs = 0;

	logger->done = 1;
	if (logger->parms.verbose)
		rtx_message ("rtx_log_close: starting");

	if (rtx_sem_post (logger->bufFull) != 0) {
		rtx_error ("rtx_log_close: error posting bufFull");
		errs++;
	}
	if (logger->parms.verbose)
		rtx_message ("rtx_log_close: waiting for file-writer thread to complete");

	if  (rtx_thread_destroy_sync (logger->writerThread) == -1) {
		rtx_error ("rtx_log_close: rtx_thread_destroy_sync() failed");
		errs++;
	}
	if (logger->parms.verbose)
		rtx_message ("rtx_log_close: file-writer thread done");

	if (rtx_mutex_destroy (logger->lock) == -1) {
		rtx_error ("rtx_log_close: rtx_mutex_destroy(lock) failed");
		errs++;
	}

	if (logger->parms.verbose)
		rtx_message ("rtx_log_close: destroying buffer mutexes");

	for (j = 0; j < logger->parms.numBufs; j++)
		if (rtx_mutex_destroy (logger->bufs[j].lock) == -1) {
			rtx_error ("rtx_log_close: rtx_mutex_destroy(bufs[].lock) failed");
			errs++;
		}
	if (logger->parms.verbose)
		rtx_message ("rtx_log_close: destroying bufFull sem");

	if (rtx_sem_destroy (logger->bufFull) == -1) {
		rtx_error ("rtx_log_close: rtx_sem_destroy(bufFull) failed");
		errs++;
	}

	rtx_log_free_logbufs(logger);

	if (logger->parms.verbose) {
		rtx_message ("rtx_log_close: %d bytes written to buffer",
				logger->stats.bytesToBuffer);
		rtx_message ("rtx_log_close: %d bytes written to file",
				logger->stats.bytesToFile);
		rtx_message ("rtx_log_close: %d files created", 
				logger->stats.filesCreated);
		rtx_message ("rtx_log_close: %d files closed", logger->stats.filesClosed);
		rtx_message ("rtx_log_close: %d open errors", logger->stats.openErrors);
		rtx_message ("rtx_log_close: %d close errors", logger->stats.closeErrors);
		rtx_message ("rtx_log_close: %d write errors", logger->stats.writeErrors);
	}
	if (stats != NULL)
		memcpy (stats, &logger->stats, sizeof (logger->stats));
	if (errs)
		return (-1);
	return (0);
}

/**************************************************************************
 * rtx_log_rollover_log - rollover log file
 */
int
rtx_log_rollover_log (
		RtxLog * logger,
		unsigned char * header,
		int headerLen,
		int newHeader
		)
{
	time_t timeNow;
	char headerComment[256];
	char timestamp[32];
	char * timeStr;
	int locErrs = 0;
	int n = 0;
	RtxLogBuf * curLogBuf;
	struct tm locTm;
	char ctimeBuf[32];

	/* Get exclusive access to the logger structure */
	if (rtx_mutex_lock (logger->lock) != 0) {
		return (rtx_error ("rtx_log_rollover_log: unable to get logger lock"));
	}
	/* Check that a new logger file can be created */
	strcpy (logger->prevPathname, logger->pathname);
	time(&timeNow);
	if (logger->parms.simpleLogFileName) {
		strcpy (logger->pathname, logger->parms.loggerDir);
		strcat (logger->pathname, logger->parms.loggerFileName);
	} else {
		memset (&locTm, 0, sizeof (locTm));
		localtime_r (&timeNow, &locTm);
		strftime (timestamp, 32, "%Y%m%d%H%M%S", &locTm);
		strcpy (logger->pathname, logger->parms.loggerDir);
		strcat (logger->pathname, logger->parms.loggerFileNamePrefix);
		strcat (logger->pathname, timestamp);
		strcat (logger->pathname, logger->parms.loggerFileNameSuffix);
	}
	if (strcmp (logger->pathname, logger->prevPathname) == 0) {
		if (rtx_mutex_unlock (logger->lock) != 0) {
			rtx_error ("rtx_log_rollover_log: unable to release logger lock");
		}
		return (rtx_error ("rtx_log_rollover_log: file exists"));
	}

	logger->curFileCreated = timeNow;
	curLogBuf = &(logger->bufs[logger->curBuf]);
	if (rtx_mutex_lock (curLogBuf->lock) != 0) {
		rtx_error ("rtx_log_rollover_log: unable to get buffer lock");
		locErrs++;
	}
	curLogBuf->rollover = 1;
	curLogBuf->full = 1;
	curLogBuf->ts = timeNow;
	if (rtx_mutex_unlock (curLogBuf->lock) != 0) {
		rtx_error ("rtx_log_rollover_log: unable to release buffer lock");
		locErrs++;
	}
	logger->curBuf++;
	if (logger->curBuf == logger->parms.numBufs)
		logger->curBuf = 0;
	curLogBuf = &(logger->bufs[logger->curBuf]);
	if (rtx_mutex_lock (curLogBuf->lock) != 0) {
		rtx_error ("rtx_log_rollover_log: unable to get buffer lock");
		locErrs++;
	}
	if (newHeader) {
		if (logger->parms.header != NULL)
			free (logger->parms.header);
		if (headerLen > 0) {
			if ((logger->parms.header = calloc (1, headerLen)) == NULL) {
				rtx_mutex_unlock (curLogBuf->lock);
				rtx_mutex_unlock (logger->lock);
				return (rtx_error_errno ("rtx_log_open: calloc"
							"(header) failed"));
			}
			memcpy (logger->parms.header, header, headerLen);
			logger->parms.headerLen = headerLen;
		} else {
			logger->parms.header = NULL;
			logger->parms.headerLen = 0;
		}
	}

	if (logger->parms.appendCreateComment) {
		timeStr = ctime_r (&timeNow, ctimeBuf);
		timeStr[24] = '\0';
		sprintf (headerComment, "%% %s: created %s by %s@%s\n", 
				logger->parms.appName, timeStr, logger->parms.myUserName,
				logger->parms.myHostName);
		n = strlen (headerComment);
		memcpy (curLogBuf->buf, headerComment, n);
		curLogBuf->curPos = n;
	}
	memcpy (&(curLogBuf->buf[n]), logger->parms.header, 
			logger->parms.headerLen);
	curLogBuf->curPos += logger->parms.headerLen;
	logger->curFileSize = logger->parms.headerLen + n;
	if (rtx_mutex_unlock (curLogBuf->lock) != 0) {
		rtx_error ("rtx_log_rollover_log: unable to release buffer lock");
		locErrs++;
	}
	logger->stats.bytesToBuffer += logger->parms.headerLen;
	if (logger->parms.verbose)
		rtx_message ("rtx_log_rollover_log: starting new log file %s", 
				logger->pathname);
	if (rtx_mutex_unlock (logger->lock) != 0) {
		rtx_error ("rtx_log_rollover_log: unable to release logger lock");
		locErrs++;
	}
	if (rtx_sem_post (logger->bufFull) != 0) {
		rtx_error ("rtx_log_rollover_log: unable to post sem");
		locErrs++;
	}
	if (locErrs)
		return (-1);
	return (0);
}

/**************************************************************************
 * rtx_log_alloc_logbufs - allocate memory for the log buffers
 */
int
rtx_log_alloc_logbufs (
		RtxLog * logger
		)
{
	int i, j;

	if (logger->parms.verbose)
		rtx_message ("rtx_log_alloc_logbufs: creating log buffers");
	/* Create log buffers headers */
	if ((logger->bufs = (RtxLogBuf *) calloc (logger->parms.numBufs, 
					sizeof (RtxLogBuf))) == NULL) {
		return (rtx_error_errno ("rtx_log_alloc_logbufs: calloc failed"));
	}
	/* Create log buffers */
	for (i = 0; i < logger->parms.numBufs; i++) {
		if ((logger->bufs[i].buf = (unsigned char *) calloc (1,
						logger->parms.bufSize)) == NULL) {
			for (j = 0; j < i; j++)
				free(logger->bufs[j].buf);
			free(logger->bufs);
			return (rtx_error ("rtx_log_alloc_logbufs: calloc failed"));
		}
	}
	return (0);
}

/**************************************************************************
 * rtx_log_create_logfile - returns a new file opened for writing
 *
 * The rtx_message returns -1 on error, 0 on success.
 */

static int
rtx_log_create_logfile (
		RtxLog * logger
		)
{
	int locErrs = 0;

	if (rtx_mutex_lock (logger->lock) != 0) {
		rtx_error ("rtx_log_create_logfile: unable to get logger lock");
		locErrs++;
	}
	if (logger->parms.verbose)
		rtx_message ("rtx_log_create_logfile: creating log file %s", 
				logger->pathname);
#ifdef USE_GZIP_COMPRESSION
	if (logger->parms.verbose) {
		rtx_message("Logger: %d zip",logger->parms.gzip);
	}
	if (logger->parms.gzip) {
		char command[256];
		snprintf(command,255,"gzip -%d -c > %s.gz",
				logger->parms.gzip, logger->pathname);
		command[255] = 0;
		logger->gzPipe = popen(command,"w");
		if (logger->gzPipe) {
			logger->fd = fileno(logger->gzPipe);
			if (logger->parms.verbose) {
				rtx_message("GZip pipe successfully open");
			}
		} else {
			rtx_error_errno ("rtx_log_create_logfile: failed to open gzip pipe %s",
					logger->pathname);
			rtx_error_flush("Using uncompressed log files");
			if ((logger->fd = open (logger->pathname, 
							O_CREAT | O_WRONLY, 0664)) == -1) {
				rtx_error_errno ("rtx_log_create_logfile: open failed on %s",
						logger->pathname);
				logger->stats.openErrors++;
				locErrs++;
			}
		}
	} else {
		logger->gzPipe = NULL;
#endif
		if ((logger->fd = open (logger->pathname, 
						O_CREAT | O_WRONLY, 0664)) == -1) {
			rtx_error_errno ("rtx_log_create_logfile: open failed on %s",
					logger->pathname);
			logger->stats.openErrors++;
			locErrs++;
		}
#ifdef USE_GZIP_COMPRESSION
	}
#endif
	if (logger->parms.indexFileName[0]) {
		FILE * index = fopen(logger->parms.indexFileName,"a");
		if (index) {
			fprintf(index,"\t%s\n",logger->pathname);
			fclose(index);
			printf("Added '%s' to index\n",logger->pathname);
		}
	}
	logger->stats.filesCreated++;
	if (rtx_mutex_unlock (logger->lock) != 0) {
		rtx_error ("rtx_log_create_logfile: unable to get logger lock");
		locErrs++;
	}
	if (locErrs)
		return (-1);
	return (0);
}

/**************************************************************************
 * rtx_log_free_logbufs - free allocated memory
 */
static void
rtx_log_free_logbufs (
		RtxLog * logger
		)
{
	int i;

	if (logger->bufs != NULL) {
		for (i = 0; i < logger->parms.numBufs; i++)
			if (logger->bufs[i].buf != NULL)
				free (logger->bufs[i].buf);
		free(logger->bufs);
	}
}

/**************************************************************************
 * rtx_log_write_bufs - write multiple buffers to logbuf
 *
 * Note: These write's need to be atomic as a whole. Therefore, cant use
 * the rtx_log_write() function.
 *
 */
int
rtx_log_write_bufs (
		RtxLog * logger,
		RtxLogWriteBufs * bufs
		)
{
	int i, locErrs = 0;
	int bufSpace = 0, numFreeBufs;
	int totalBytesToWrite = 0;
	RtxLogBuf * curLogBuf;
	unsigned char * p;
	int n, bytesToWrite;

	for (i=0; ((i<MAX_WRITE_BUFS) && (bufs->buf[i] != NULL) && 
				(bufs->bufLen[i] >= 0)); i++)
		totalBytesToWrite += bufs->bufLen[i];

	/* Get exclusive access to the logger structure */
	if (rtx_mutex_lock (logger->lock) != 0) {
		return (rtx_error ("rtx_log_write_bufs: unable to get logger lock"));
	}
	curLogBuf = &(logger->bufs[logger->curBuf]);
	if (rtx_mutex_lock (curLogBuf->lock) != 0) {
		rtx_error ("rtx_log_write_bufs: unable to get buffer lock");
		locErrs++;
	}
	if (logger->curBuf >= logger->bufToWrite)
		numFreeBufs = logger->parms.numBufs - (logger->curBuf -
				logger->bufToWrite) - 1;
	else
		numFreeBufs = logger->bufToWrite - logger->curBuf - 1;
	/*
	   numFreeBufs = logger->curBuf - logger->bufToWrite - 1;
	   if (numFreeBufs < 0)
	   numFreeBufs += logger->parms.numBufs;
	   */
	if (logger->bufWriteSuspended) {
		if (numFreeBufs == (logger->parms.numBufs - 1)) {
			logger->bufWriteSuspended = 0;
			rtx_message ("rtx_log_buf_write: all bufs available");
		}
	}
	bufSpace = numFreeBufs * logger->parms.bufSize + logger->parms.bufSize - 
		curLogBuf->curPos;
	if (logger->bufWriteSuspended) {
		logger->stats.bufFullErrors++;
	} else if (totalBytesToWrite > bufSpace) {
		logger->stats.bufOverflow++;
		logger->stats.bufFullErrors++;
		logger->bufWriteSuspended = 1;
		rtx_message ("rtx_log_buf_write: all bufs full");
	} else {
		for (i=0; ((i<MAX_WRITE_BUFS) && (bufs->buf[i] != NULL) && 
					(bufs->bufLen[i] >= 0)); i++) {
			n = bufs->bufLen[i];
			p = bufs->buf[i];
			while (n > 0) {
				bufSpace = logger->parms.bufSize - curLogBuf->curPos;
				if (bufSpace >= n) {
					bytesToWrite = n;
				} else {
					bytesToWrite = bufSpace;
				}
				memcpy (curLogBuf->buf + curLogBuf->curPos, p, bytesToWrite);
				n -= bytesToWrite;
				p += bytesToWrite;
				curLogBuf->curPos += bytesToWrite;
				if (curLogBuf->curPos >= logger->parms.bufSize) {
					curLogBuf->full = 1;
					if (rtx_mutex_unlock (curLogBuf->lock) != 0) {
						rtx_error ("rtx_log_write: unable to release buffer lock");
						locErrs++;
					}
					if (rtx_sem_post (logger->bufFull) != 0) {
						rtx_error ("rtx_log_write: unable to post sem");
						locErrs++;
					}
					logger->curBuf++;
					if (logger->curBuf == logger->parms.numBufs)
						logger->curBuf = 0;
					curLogBuf = &(logger->bufs[logger->curBuf]);
					if (rtx_mutex_lock (curLogBuf->lock) != 0) {
						rtx_error ("rtx_log_write: unable to get buffer lock");
						locErrs++;
					}
				}
			}
		}
		logger->stats.bytesToBuffer += totalBytesToWrite;
		logger->curFileSize += totalBytesToWrite;
	}

	if (rtx_mutex_unlock (curLogBuf->lock) != 0) {
		rtx_error ("rtx_log_write: unable to release buffer lock");
		locErrs++;
	}
	if (rtx_mutex_unlock (logger->lock) != 0) {
		rtx_error ("rtx_log_write: unable to release logger lock");
		locErrs++;
	}
	if (locErrs)
		return (-1);
	return (0);
}

/**************************************************************************
 * rtx_log_write - write to buffer and signal full buffer to be written
 *
 */
int
rtx_log_write (
		RtxLog * logger,
		unsigned char *p,
		int n
		)
{
	int locErrs = 0;
	int bufSpace = 0, numFreeBufs;
	RtxLogBuf * curLogBuf;
	int bytesToWrite;

	/* Get exclusive access to the logger structure */
	if (rtx_mutex_lock (logger->lock) != 0) {
		return (rtx_error ("rtx_log_write: unable to get logger lock"));
	}
	curLogBuf = &(logger->bufs[logger->curBuf]);
	if (rtx_mutex_lock (curLogBuf->lock) != 0) {
		rtx_error ("rtx_log_write: unable to get buffer lock");
		locErrs++;
	}
	numFreeBufs = logger->curBuf - logger->bufToWrite - 1;
	if (numFreeBufs < 0)
		numFreeBufs += logger->parms.numBufs;
	bufSpace = numFreeBufs * logger->parms.bufSize + logger->parms.bufSize - 
		curLogBuf->curPos;
	if (n > bufSpace) {
		rtx_error ("rtx_log_write: not enough free memory (%d needed, "
				"%d available)", n, bufSpace);
		locErrs++;
	} else {
		while (n > 0) {
			bufSpace = logger->parms.bufSize - curLogBuf->curPos;
			if (bufSpace >= n) {
				bytesToWrite = n;
			} else {
				bytesToWrite = bufSpace;
			}
			memcpy (curLogBuf->buf + curLogBuf->curPos, p, bytesToWrite);
			n -= bytesToWrite;
			p += bytesToWrite;
			curLogBuf->curPos += bytesToWrite;
			if (curLogBuf->curPos >= logger->parms.bufSize) {
				curLogBuf->full = 1;
				if (rtx_mutex_unlock (curLogBuf->lock) != 0) {
					rtx_error ("rtx_log_write: unable to release buffer lock");
					locErrs++;
				}
				if (rtx_sem_post (logger->bufFull) != 0) {
					rtx_error ("rtx_log_write: unable to post sem");
					locErrs++;
				}
				logger->curBuf++;
				if (logger->curBuf == logger->parms.numBufs)
					logger->curBuf = 0;
				curLogBuf = &(logger->bufs[logger->curBuf]);
				if (rtx_mutex_lock (curLogBuf->lock) != 0) {
					rtx_error ("rtx_log_write: unable to get buffer lock");
					locErrs++;
				}
			}
		}
		logger->curFileSize += n;
		logger->stats.bytesToBuffer += n;
	}
	if (rtx_mutex_unlock (curLogBuf->lock) != 0) {
		rtx_error ("rtx_log_write: unable to release buffer lock");
		locErrs++;
	}
	if (rtx_mutex_unlock (logger->lock) != 0) {
		rtx_error ("rtx_log_write: unable to release logger lock");
		locErrs++;
	}
	if (locErrs)
		return (-1);
	return (0);
}

int 
rtx_log_write_item (
		RtxLog * logger, 
		int id, 
		RtxTime * tp,
		unsigned char * buf,
		int n
		)
{
	RtxLogItemHeader hdr;
	RtxLogWriteBufs bufs;
	RtxTime ts;

	memset (&hdr, 0, sizeof (hdr));
	bufs.buf[0] = (unsigned char *) &hdr;
	bufs.bufLen[0] = sizeof (hdr);
	bufs.buf[1] = buf;
	bufs.bufLen[1] = n;
	bufs.buf[2] = NULL;
	bufs.bufLen[2] = 0;

	hdr.id = id;
	if (tp == NULL) {
		rtx_time_get (&ts);
		hdr.ts = ts;
	} else {
		hdr.ts = * tp;
	}

	if (rtx_log_write_bufs (logger, &bufs) == -1)
		return (rtx_error ("rtx_log_write_item: rtx_log_write_bufs() failed"));
	return (0);
}

unsigned char * 
rtx_log_build_header (
		char * headerBody,
		int * n
		)
{
	int hdrLen, hdrLen1, endianInt = RTX_LOG_HEADER_MAGIC_NUMBER;
	char * hdr = NULL, * hdr1 = NULL;
	int needLinefeed = 0;

	if ((hdr1 = calloc (1, 1024)) == NULL)
		return (rtx_error_null ("rtx_log_build_header: calloc(1024) failed"));
	sprintf (hdr1,
#ifdef	DARWIN
			"%%%%primitiveDataSize: %lu %lu %lu %lu %lu %lu\n"
			"%%%%primitiveDataAlignment: %lu %lu %lu %lu %lu %lu\n",
#else
			"%%%%primitiveDataSize: %d %d %d %d %d %d\n"
			"%%%%primitiveDataAlignment: %d %d %d %d %d %d\n",
#endif
			sizeof (char), sizeof (short), sizeof (int), sizeof (long),
			sizeof (float), sizeof (double), __alignof__ (char),
			__alignof__ (short), __alignof__ (int), __alignof__ (long),
			__alignof__ (float), __alignof__ (double));
	hdrLen1 = strlen (hdr1) + strlen (headerBody) + 
		strlen (RTX_LOG_HEADER_EOH);
	if (headerBody[strlen(headerBody)-1] != '\n')
		needLinefeed = 1;
	if (needLinefeed)
		hdrLen1++;
	hdrLen = hdrLen1 + sizeof (int);
	if ((hdr = calloc (1, hdrLen)) == NULL)
		return (rtx_error_null ("rtx_log_build_header: calloc(%d) failed",
					hdrLen));

	strcat (hdr, hdr1);
	strcat (hdr, headerBody);
	if (needLinefeed)
		strcat (hdr, "\n");
	strcat (hdr, RTX_LOG_HEADER_EOH);
	memcpy (&(hdr[hdrLen1]), &endianInt, sizeof (int));
	* n = hdrLen;
	return (unsigned char*)(hdr);

}

/**************************************************************************
 * rtx_log_set_header - change the logfile header
 *
 * Note: Causes an immediate rollover
 *
 */
int 
rtx_log_set_header (
		RtxLog * logger, 
		unsigned char * header, 
		int n
		)
{
	return (rtx_log_rollover_log (logger, header, n, 1));
}

#ifdef USE_GZIP_COMPRESSION
/**************************************************************************
 * rtx_log_set_gzip - Set the gzip compression level
 *
 */
int 
rtx_log_set_gzip (
		RtxLog * logger, 
		int gzip_compression
		)
{
	if (gzip_compression < 0) {
		gzip_compression = 0;
	}
	if (gzip_compression > 9) {
		gzip_compression = 9;
	}
	logger->parms.gzip = gzip_compression;
	rtx_message("Gzip compression set to %d",logger->parms.gzip);
	return 0;
}
#endif

/**************************************************************************
 * rtx_log_write_logbuf_to_file - writes a buffer to file
 */
static int
rtx_log_write_logbuf_to_file (
		RtxLog * logger
		)
{
	int charsToWrite;
	RtxLogBuf * logbufToWrite;
	int locErrs = 0;

	if (logger->parms.verbose)
		rtx_message ("rtx_log_write_logbuf_to_file: writing full buffer # %d",
				logger->bufToWrite);
	logbufToWrite = &(logger->bufs[logger->bufToWrite]);
	if (rtx_mutex_lock (logbufToWrite->lock) != 0) {
		rtx_error ("write_buf_to_file: unable to get buffer lock");
		locErrs++;
	}
	charsToWrite = logger->bufs[logger->bufToWrite].curPos;
	logger->stats.bytesToFile += charsToWrite;
	if (logbufToWrite->full) {
		if (write (logger->fd, (char *) logbufToWrite->buf, charsToWrite)
				== -1) {
			rtx_error_errno ("rtx_log_write_logbuf_to_file: write() failed");
			locErrs++;
			logger->stats.writeErrors++;
		} else {
			/*
			 * Buffer successfully written. Mark it as
			 * empty and move on to the next buffer.
			 */
			logbufToWrite->full = 0;
			logbufToWrite->curPos = 0;
			if (logbufToWrite->rollover) {
				logbufToWrite->rollover = 0;
#ifdef USE_GZIP_COMPRESSION
				if (logger->parms.gzip && logger->gzPipe) {
					if (logger->parms.verbose) {
						rtx_message("Closing gzip pipe");
					}
					if (pclose (logger->gzPipe) == -1) {
						rtx_error_errno ("rtx_log_write_logbuf_to_file: pclose failed");
						locErrs++;
						logger->stats.closeErrors++;
					} else {
						logger->stats.filesClosed++;
					}
				} else {
#endif
					if (close (logger->fd) == -1) {
						rtx_error_errno ("rtx_log_write_logbuf_to_file: close failed");
						locErrs++;
						logger->stats.closeErrors++;
					} else {
						logger->stats.filesClosed++;
					}
#ifdef USE_GZIP_COMPRESSION
				}
#endif
				if (rtx_log_create_logfile (logger) == -1) {
					rtx_error ("rtx_log_write_logbuf_to_file: creat() failed");
					locErrs++;
				}
			}
			logger->bufToWrite++;
			if (logger->bufToWrite == logger->parms.numBufs)
				logger->bufToWrite = 0;
		}
	} else {
		rtx_error ("rtx_log_write_logbuf_to_file: writing empty buffer %d",
				logger->bufToWrite);
		locErrs++;
	}
	if (rtx_mutex_unlock (logbufToWrite->lock) != 0) {
		rtx_error ("write_buf_to_file: unable to get buffer lock");
		locErrs++;
	}
	if (locErrs)
		return (-1);
	return (0);
}

/**************************************************************************
 * rtx_log_write_logfile_thread - writes a full buffer to file
 *
 * This thread generates no fatal errors, only rtx_message_warnings. It waits in a loop
 * for full buffers to be written to file. When the logger shutdowns, it
 * signals the thread to terminate. The thread then writes the remaining
 * buffers and exits.
 *
 * It always returns 0 on termination.
 */

static int
rtx_log_write_logfile_thread (
		RtxLog * logger
		)
{
	int errs;

	if (logger->parms.verbose)
		rtx_message ("rtx_log_write_logfile_thread: starting");

	/* Loop until the logger issues a terminate signal */
	while (logger->done == 0) {
		errs = 0;
		rtx_thread_testcancel ();
		if (logger->done != 0) {
			rtx_timer_sleep (0.01);
			continue;
		}
		if (logger->parms.verbose)
			rtx_message ("rtx_log_write_logfile_thread: waiting for semaphore");
		/* Wait for a full buffer */
		if (rtx_sem_wait_ignore_eintr (logger->bufFull) != 0) {
			rtx_error ("rtx_log_write_logfile_thread: sem_wait error");
			errs++;
		}
		rtx_thread_testcancel ();
		if (logger->done != 0) {
			rtx_timer_sleep (0.01);
			continue;
		}
		rtx_thread_setcancel (RTX_THREAD_CANCEL_DISABLE);
		rtx_log_write_logbuf_to_file (logger);
		/* If the current file exceeds the time or size limits, open a
		 * new log file.
		 */
		if (((!logger->parms.noTimeLimit) &&
					(difftime(time(NULL), logger->curFileCreated) 
					 >= logger->parms.timeLimit)) ||
				((!logger->parms.noSizeLimit) && 
				 (logger->curFileSize >= logger->parms.sizeLimit))) {
			if (logger->parms.verbose) {
				rtx_message("Logger rollover: size %d age %d",logger->curFileSize,
						difftime(time(NULL), logger->curFileCreated));
			}
			rtx_log_rollover_log (logger, NULL, 0, 0);
		}
		rtx_thread_setcancel (RTX_THREAD_CANCEL_DEFERRED);
	}
	rtx_thread_testcancel ();

	return (0);
}

static void
rtx_log_write_logfile_thread_cleanup (
		RtxLog * logger
		)
{
	int i;
	int nBufsToWrite;
	int errs = 0;

	/*
	 * The verbose output is commented out as it causes a segfault
	 * under Linux. Need to maintain a watching brief as we move
	 * up the NPTL versions.
	 */

	/* write the remaining buffers to file before terminating */
	/*
	   if (logger->parms.verbose)
	   rtx_message ("rtx_log_write_logfile_thread: exiting, "
	   "writing remaining buffers");
	   */
	nBufsToWrite = logger->curBuf - logger->bufToWrite;
	if (nBufsToWrite < 0)
		nBufsToWrite += logger->parms.numBufs;
	for (i = 0; i < nBufsToWrite; i++)
		rtx_log_write_logbuf_to_file (logger);
	logger->stats.bytesToFile += logger->bufs[logger->curBuf].curPos;
	/*
	   if (logger->parms.verbose)
	   rtx_message ("rtx_log_write_logfile_thread: writing %d bytes in "
	   "current buffer", logger->bufs[logger->curBuf].curPos);
	   */
	if (write (logger->fd, (char *) logger->bufs[logger->curBuf].buf,
				logger->bufs[logger->curBuf].curPos) == -1) {
		/*
		   rtx_error ("rtx_log_write_logfile_thread: write() failed");
		   */
		errs++;
	}
#ifdef USE_GZIP_COMPRESSION
	if (logger->parms.gzip && logger->gzPipe) {
		if (logger->parms.verbose) {
			rtx_message("Closing gzip pipe");
		}
		if (pclose (logger->gzPipe) == -1) {
			if (logger->parms.verbose) {
				rtx_error_errno ("rtx_log_write_logbuf_to_file: pclose failed");
			}
			errs++;
			logger->stats.closeErrors++;
		} else {
			logger->stats.filesClosed++;
		}
	} else {
#endif
		if (close (logger->fd) == -1) {
			if (logger->parms.verbose) {
			   rtx_error ("rtx_log_write_logfile_thread: could not close log file");
			}
			errs++;
			logger->stats.closeErrors++;
		} else {
			logger->stats.filesClosed++;
		}
#ifdef  USE_GZIP_COMPRESSION
	}
#endif
	/*
	   if (errs)
	   rtx_error_flush ("errors in thread");
	   */
}


