/***********************************************************************
 * 
 * CSIRO Autonomous Systems Laboratory
 * Queensland Centre for Advanced Technologies
 * PO Box 883, Kenmore, QLD 4069, Australia
 * http://www.ict.csiro.au/
 *  
 * Copyright (c) CSIRO 
 ***********************************************************************/

/**
 * \file lock.c
 * \brief File locking functions
 * \author Pavan Sikka
 *
 * This module provides file-locking functions using the fcntl ()
 * function.
 */

#include        <sys/file.h>
#include	<sys/types.h>
#include	<sys/stat.h>
#include        <unistd.h>
#include	<fcntl.h>
#include        <errno.h>
#include        <string.h>

#include "rtx/defines.h"
#include "rtx/error.h"

static char rcsid[] RTX_UNUSED = "$Id: lock.c 2274 2007-12-23 05:37:32Z roy029 $";

/**
 * lock a file.
 *
 * @return fd if lock obtained, else -1.
 * @see fcntl
 */

int
rtx_lock_get (
	 char *name       /**< filename of file to be locked */
	 )
{
	int	lockFile;
	struct flock lock;
	int     result;

	if ((lockFile = open(name, O_RDWR|O_CREAT, 0666)) < 0) {
  	     return (rtx_error_errno ("rtx_lock_get (): open () failed: "));
        }

	lock.l_type = F_WRLCK;
	lock.l_start = 0;
	lock.l_whence = SEEK_SET;
	lock.l_len = 0;

	/* if lock cannot be obtained, fcntl returns with errno set to
	 * EACCESS or EAGAIN
	 */
	if ((result = fcntl (lockFile, F_SETLK, &lock)) == -1)
	    return (rtx_error_errno ("rtx_lock_get (): fcntl () failed: "));

        return (lockFile);
}

/**
 *  unlock a file.
 *
 * @return 0 if lock released, else -1.
 * @see fcntl
 */

int
rtx_lock_return (
		 char *name       /**< filename of file to be unlocked */
		 )
{
	int	lockFile;
	struct flock lock;

	if ((lockFile = open(name, O_RDWR|O_CREAT, 0666)) < 0) {
	    return (rtx_error_errno ("rtx_lock_return (): fcntl () failed: "));
        }

	lock.l_type = F_UNLCK;
	lock.l_start = 0;
	lock.l_whence = SEEK_SET;
	lock.l_len = 0;

	if (fcntl (lockFile, F_SETLK, &lock) == -1)
	    return (rtx_error_errno ("rtx_lock_return (): fcntl () failed: "));

        return (0);
}

/**
 * Return lock status.
 *
 * @return 0 if file is locked, 1 if file is unlocked, -1 on error
 */

int
rtx_lock_test (
	       char *name       /**< filename of file to test */
	       )
{
	int     lockFile;
	struct flock lock;

	if ((lockFile = open(name, O_RDWR|O_CREAT, 0666)) < 0) {
	    return (rtx_error_errno ("rtx_lock_test (): open () failed: "));
        }

	lock.l_type = F_WRLCK;
	lock.l_start = 0;
	lock.l_whence = SEEK_SET;
	lock.l_len = 0;

	if (fcntl (lockFile, F_GETLK, &lock) == -1)
	    return (rtx_error_errno ("rtx_lock_test (): fcntl () failed: "));

	if (lock.l_type == F_UNLCK)
	        return (0);

	return (1);
}

