/***********************************************************************
 * 
 * CSIRO Autonomous Systems Laboratory
 * Queensland Centre for Advanced Technologies
 * PO Box 883, Kenmore, QLD 4069, Australia
 * http://www.ict.csiro.au/
 *  
 * Copyright (c) CSIRO 
 ***********************************************************************/

#include <unistd.h>
#include <stdlib.h>
#include <string.h>


#include "rtx/thread.h"
#include "rtx/error.h"
#include "rtx/mutex.h"
#include "rtx/message.h"
#include "rtx/list.h"

static char rcsid[] RTX_UNUSED = "$Id: list.c 3076 2008-05-15 05:18:25Z roy029 $";

/**
 * \file list.c
 * \brief Doubly-linked lists
 * \author Pavan Sikka
 *
 * Thread-safe (and thread-cancellation-safe) doubly-linked lists
 *
 */

/**
 * initialize a thread-safe doubly-linked list
 *
 * @return pointer to list, NULL on error
 */
RtxList * 
rtx_list_init (void)
{
	RtxList * list = NULL;

	if ((list = calloc (1, sizeof (RtxList))) == NULL)
		return (rtx_error_null ("rtx_list_init: calloc() failed"));
	list->head = NULL;
	list->tail = NULL;
	if (rtx_mutex_init (&(list->mutex), RTX_MUTEX_DEFAULT, 0) == NULL) {
		free (list);
		return (rtx_error_null ("rtx_list_init: rtx_mutex_init() "
					"failed"));
	}
	return (list);
}

/**
 * add a node to the front of the list
 *
 * @return 0 if OK, -1 on error
 */
int 
rtx_list_add (
	      RtxList * list,  /**< list */
	      const char * name,     /**< node name */
	      void * d         /**< node data */
	      )
{
    RtxListNode * node;
    RtxThreadCancel oldCancelState;

    if ((node = calloc (1, sizeof (RtxListNode))) == NULL)
        return (rtx_error ("rtx_list_add: calloc() failed"));
    node->d = d;
    if ((node->name = strdup (name)) == NULL) {
        free (node);
        return (rtx_error ("rtx_list_add: strdup() failed"));
    }
    oldCancelState = rtx_thread_setcancel (RTX_THREAD_CANCEL_DISABLE);
    if (rtx_mutex_lock (&(list->mutex)) == -1) {
        free (node);
        return (rtx_error ("rtx_list_add: rtx_mutex_lock() failed"));
    }
    if (list->head == NULL) { /* empty list */
        list->head = node;
	list->tail = node;
	node->next = NULL;
	node->prev = NULL;
    } else {
        node->next = list->head;
	node->prev = NULL;
	list->head->prev = node;
        list->head = node;
    }
    (list->n)++;
    if (rtx_mutex_unlock (&(list->mutex)) == -1) {
        return (rtx_error ("rtx_list_add: rtx_mutex_lock() failed"));
    }
    rtx_thread_setcancel (oldCancelState);
    return (0);
}

/**
 * lookup a node from the list
 *
 * @return pointer to node data, NULL if not found
 */
void * 
rtx_list_lookup (
		 RtxList * list,  /**< list */
		 const char * name      /**< node name */
		 )
{
    RtxListNode * node;
    RtxThreadCancel oldCancelState;
    int found = 0;

    if (list->head == NULL)
        return (NULL);
    oldCancelState = rtx_thread_setcancel (RTX_THREAD_CANCEL_DISABLE);
    if (rtx_mutex_lock (&(list->mutex)) == -1) {
        return (rtx_error_null ("rtx_list_lookup: rtx_mutex_lock()"
				" failed"));
    }
    for (node = list->head; node != NULL; node = node->next)
        if (strcmp (name, node->name) == 0) {
	    found = 1;
	    break;
	}
    if (rtx_mutex_unlock (&(list->mutex)) == -1) {
        return (rtx_error_null ("rtx_list_lookup: rtx_mutex_lock()"
				" failed"));
    }
    rtx_thread_setcancel (oldCancelState);
    if (! found)
        return (NULL);
    else
        return (node->d);
}

/**
 * iterate over all nodes in the list
 *
 * @return pointer to node data, NULL if at end
 */
void * 
rtx_list_iterate (
		  RtxList * list
		  )
{
	RtxListNode * node;
	static RtxThreadCancel oldCancelState = -1;

	if (list->head == NULL)
		return (NULL);
	if (list->iterate) {
		if (list->cur == NULL) {
			list->iterate = 0;
			list->cur = list->head;
			if (rtx_mutex_unlock (&(list->mutex)) == -1) {
				return (rtx_error_null ("rtx_list_iterate: "
							"rtx_mutex_lock() failed"));
			}
			rtx_thread_setcancel (oldCancelState);
			return (NULL);
		} else {
			node = list->cur;
			list->cur = list->cur->next;
			return (node->d);
		}
	} else {
		oldCancelState = rtx_thread_setcancel 
			(RTX_THREAD_CANCEL_DISABLE);
		if (rtx_mutex_lock (&(list->mutex)) == -1) {
			return (rtx_error_null ("rtx_list_iterate: rtx_mutex_lock()"
						" failed"));
		}
		list->iterate = 1;
		list->cur = list->head->next;
		return (list->head->d);
	}
}

/**
 * delete a node from the list
 *
 * @return pointer to node data
 */
void *
rtx_list_del (
	      RtxList * list,  /**< list */
	      const char * name,     /**< node name */
	      int freeNodeData /**< free node data flag */
	      )
{
	RtxListNode * node;
	RtxThreadCancel oldCancelState;
	int found = 0;
	void * ret = NULL;

	if (list->head == NULL)
		return (NULL);
	oldCancelState = rtx_thread_setcancel (RTX_THREAD_CANCEL_DISABLE);
	if (rtx_mutex_lock (&(list->mutex)) == -1) {
		return (rtx_error_null ("rtx_list_del: rtx_mutex_lock() "
					"failed"));
	}
	for (node = list->head; node != NULL; node = node->next)
		if (strcmp (name, node->name) == 0) {
			found = 1;
			break;
		}
	if (found) {
		if (node->prev == NULL) {
			if (node->next == NULL) {  /* only node in list */
				list->head = NULL;
				list->tail = NULL;
				free (node->name);
				if (freeNodeData)
					free (node->d);
				else
					ret = node->d;
				free (node);
			} else {   /* head node in list */
				list->head = node->next;
				list->tail = NULL;
				list->head->prev = NULL;
				free (node->name);
				if (freeNodeData)
					free (node->d);
				else
					ret = node->d;
				free (node);
			}
		} else {
			if (node->next == NULL) {    /* last node in list */
				list->tail = node->prev;
				list->tail->next = NULL;
				free (node->name);
				if (freeNodeData)
					free (node->d);
				else
					ret = node->d;
				free (node);
			} else {
				node->prev->next = node->next;
				node->next->prev = node->prev;
				free (node->name);
				if (freeNodeData)
					free (node->d);
				else
					ret = node->d;
				free (node);
			}
		}
		(list->n)--;
	} else {
		rtx_message("Could not find '%s'",name);
	}
	if (rtx_mutex_unlock (&(list->mutex)) == -1) {
		return (rtx_error_null ("rtx_list_del: rtx_mutex_lock() "
					"failed"));
	}
	rtx_thread_setcancel (oldCancelState);
	return (ret);
}

/**
 * delete a node from the list
 *
 * @return pointer to node data
 */
void *
rtx_list_del_node (
		   RtxList * list,  /**< list */
		   void * ndData,   /**< pointer to node data */
		   int freeNodeData /**< free node data flag */
	      )
{
    RtxListNode * node;
    RtxThreadCancel oldCancelState;
    int found = 0;
    void * ret = NULL;

    if (list->head == NULL)
        return (NULL);
    oldCancelState = rtx_thread_setcancel (RTX_THREAD_CANCEL_DISABLE);
    if (rtx_mutex_lock (&(list->mutex)) == -1) {
        return (rtx_error_null ("rtx_list_del: rtx_mutex_lock() "
				"failed"));
    }
    for (node = list->head; node != NULL; node = node->next)
        if (node->d == ndData) {
	    found = 1;
	    break;
	}
    if (found) {
        if (node->prev == NULL) {
	    if (node->next == NULL) {  /* only node in list */
	        list->head = NULL;
		list->tail = NULL;
		free (node->name);
		if (freeNodeData)
		    free (node->d);
		else
		    ret = node->d;
		free (node);
	    } else {   /* head node in list */
	        list->head = node->next;
		list->tail = NULL;
		list->head->prev = NULL;
		free (node->name);
		if (freeNodeData)
		    free (node->d);
		else
		    ret = node->d;
		free (node);
	    }
	} else {
	    if (node->next == NULL) {    /* last node in list */
		list->tail = node->prev;
		list->tail->next = NULL;
		free (node->name);
		if (freeNodeData)
		    free (node->d);
		else
		    ret = node->d;
		free (node);
	    } else {
	        node->prev->next = node->next;
	        node->next->prev = node->prev;
		free (node->name);
		if (freeNodeData)
		    free (node->d);
		else
		    ret = node->d;
		free (node);
	    }
	}
	(list->n)--;
    }
    if (rtx_mutex_unlock (&(list->mutex)) == -1) {
        return (rtx_error_null ("rtx_list_del: rtx_mutex_lock() "
				"failed"));
    }
    rtx_thread_setcancel (oldCancelState);
    return (ret);
}

/**
 * add a node to the front of the list - use it as a stack
 *
 * @return 0 if OK, -1 on error
 */
int 
rtx_list_push (
	      RtxList * list,  /**< list */
	      void * d         /**< node data */
	      )
{
    RtxListNode * node;
    RtxThreadCancel oldCancelState;

    if ((node = calloc (1, sizeof (RtxListNode))) == NULL)
        return (rtx_error ("rtx_list_add: calloc() failed"));
    node->d = d;
    /* its a stack; the name is not used */
    node->name = NULL;
    oldCancelState = rtx_thread_setcancel (RTX_THREAD_CANCEL_DISABLE);
    if (rtx_mutex_lock (&(list->mutex)) == -1) {
        free (node);
        return (rtx_error ("rtx_list_add: rtx_mutex_lock() failed"));
    }
    if (list->head == NULL) { /* empty list */
        list->head = node;
	list->tail = node;
	node->next = NULL;
	node->prev = NULL;
    } else {
        node->next = list->head;
	node->prev = NULL;
	list->head->prev = node;
        list->head = node;
    }
    (list->n)++;
    if (rtx_mutex_unlock (&(list->mutex)) == -1) {
        return (rtx_error ("rtx_list_add: rtx_mutex_lock() failed"));
    }
    rtx_thread_setcancel (oldCancelState);
    return (0);
}

/**
 * remove a node from the front of the list - use it as a stack
 *
 * @return 0 if OK, -1 on error
 */
void *
rtx_list_pop (
	      RtxList * list  /**< list */
	      )
{
    RtxListNode * node = NULL;
    RtxThreadCancel oldCancelState;
    void * ret = NULL;

    oldCancelState = rtx_thread_setcancel (RTX_THREAD_CANCEL_DISABLE);
    if (rtx_mutex_lock (&(list->mutex)) == -1) {
        free (node);
        return (rtx_error_null ("rtx_list_add: rtx_mutex_lock() failed"));
    }
    if (list->head != NULL) {
	node = list->head;
        ret = node->d;
	list->head = node->next;
	if (list->tail == node)
	    list->tail = node->next;
	free (node);
	(list->n)--;
    }
    if (rtx_mutex_unlock (&(list->mutex)) == -1) {
        return (rtx_error_null ("rtx_list_add: rtx_mutex_lock() failed"));
    }
    rtx_thread_setcancel (oldCancelState);
    return (ret);
}

/**
 * add a node to the back of the list - use it as a queue
 *
 * @return 0 if OK, -1 on error
 */
int 
rtx_list_enqueue (
		  RtxList * list,  /**< list */
		  void * d         /**< node data */
		  )
{
    RtxListNode * node;
    RtxThreadCancel oldCancelState;

    if ((node = calloc (1, sizeof (RtxListNode))) == NULL)
        return (rtx_error ("rtx_list_add: calloc() failed"));
    node->d = d;
    /* its a queue; the name is not used */
    node->name = NULL;
    oldCancelState = rtx_thread_setcancel (RTX_THREAD_CANCEL_DISABLE);
    if (rtx_mutex_lock (&(list->mutex)) == -1) {
        free (node);
        return (rtx_error ("rtx_list_add: rtx_mutex_lock() failed"));
    }
    if (list->tail == NULL) { /* empty list */
        list->head = node;
	list->tail = node;
	node->next = NULL;
	node->prev = NULL;
    } else {
        list->tail->next = node;
	node->next = NULL;
	node->prev = list->tail;
	list->tail = node;
    }
    (list->n)++;
    if (rtx_mutex_unlock (&(list->mutex)) == -1) {
        return (rtx_error ("rtx_list_add: rtx_mutex_lock() failed"));
    }
    rtx_thread_setcancel (oldCancelState);
    return (0);
}

/**
 * remove a node from the front of the list - use it as a queue
 *
 * @return 0 if OK, -1 on error
 */
void *
rtx_list_dequeue (
		  RtxList * list  /**< list */
		  )
{
    RtxListNode * node = NULL;
    RtxThreadCancel oldCancelState;
    void * ret = NULL;

    oldCancelState = rtx_thread_setcancel (RTX_THREAD_CANCEL_DISABLE);
    if (rtx_mutex_lock (&(list->mutex)) == -1) {
        free (node);
        return (rtx_error_null ("rtx_list_add: rtx_mutex_lock() failed"));
    }
    if (list->head != NULL) {
	node = list->head;
        ret = node->d;
	list->head = node->next;
	if (list->tail == node)
	    list->tail = node->next;
	free (node);
	(list->n)--;
    }
    if (rtx_mutex_unlock (&(list->mutex)) == -1) {
        return (rtx_error_null ("rtx_list_add: rtx_mutex_lock() failed"));
    }
    rtx_thread_setcancel (oldCancelState);
    return (ret);
}

/**
 * Get the data from the node at the front of the list without removing it from the list or queue.
 * Returns the next item that would be returned by pop or dequeue.
 *
 * @return The data if OK, -1 on error.
 */
void *
rtx_list_peek_head (
		  RtxList * list  /**< list */
		  )
{
    RtxThreadCancel oldCancelState;
    void * ret = NULL;

    oldCancelState = rtx_thread_setcancel (RTX_THREAD_CANCEL_DISABLE);
    if (rtx_mutex_lock (&(list->mutex)) == -1) {
        return (rtx_error_null ("rtx_list_peek_head: rtx_mutex_lock() failed"));
    }
    if (list->head != NULL) {
        ret = list->head->d;
    }
    if (rtx_mutex_unlock (&(list->mutex)) == -1) {
        return (rtx_error_null ("rtx_list_peek_head: rtx_mutex_unlock() failed"));
    }
    rtx_thread_setcancel (oldCancelState);
    return (ret);
}

/**
 * Get the data from the node at the back of the list without removing it from the list or queue.
 * Returns the last item that was enqueue or the first item that was pushed.
 *
 * @return The data if OK, -1 on error.
 */
void *
rtx_list_peek_tail (
		  RtxList * list  /**< list */
		  )
{
    RtxThreadCancel oldCancelState;
    void * ret = NULL;

    oldCancelState = rtx_thread_setcancel (RTX_THREAD_CANCEL_DISABLE);
    if (rtx_mutex_lock (&(list->mutex)) == -1) {
        return (rtx_error_null ("rtx_list_peek_tail: rtx_mutex_lock() failed"));
    }
    if (list->tail != NULL) {
        ret = list->tail->d;
    }
    if (rtx_mutex_unlock (&(list->mutex)) == -1) {
        return (rtx_error_null ("rtx_list_peek_tail: rtx_mutex_unlock() failed"));
    }
    rtx_thread_setcancel (oldCancelState);
    return (ret);
}

/**
 * destroy the list
 *
 */
void
rtx_list_destroy (
		  RtxList * list,     /**< list */
		  int freeNodeData    /**< free node data flag */
		  )
{
	RtxListNode * node, * n;
	rtx_mutex_destroy(&(list->mutex));

	if (list->head == NULL) {
		free(list);
		return;
	}
	node = list->head;
	while (node != NULL) {
		n = node->next;
		free (node->name);
		if (freeNodeData)
			free (node->d);
		free (node);
		node = n;
	}
	free(list);
}

/**
 * print the nodes in the list
 *
 */
void
rtx_list_print (
		RtxList * list  /**< list */
		)
{
    RtxListNode * node;
    RtxThreadCancel oldCancelState;
    int n = 1;

    if (list->head == NULL) {
        rtx_message ("rtx_print_list: empty");
        return;
    }
    oldCancelState = rtx_thread_setcancel (RTX_THREAD_CANCEL_DISABLE);
    if (rtx_mutex_lock (&(list->mutex)) == -1) {
        rtx_error_null ("rtx_list_print: rtx_mutex_lock()"
			" failed");
    }
    for (node = list->head; node != NULL; node = node->next)
        rtx_message ("rtx_print_list [%d]: %s, 0x%08X", n++, 
		     node->name, (unsigned int) node->d);
    if (rtx_mutex_unlock (&(list->mutex)) == -1) {
        rtx_error_null ("rtx_list_print: rtx_mutex_lock()"
			" failed");
    }
    rtx_thread_setcancel (oldCancelState);
}

