/***********************************************************************
 * 
 * CSIRO Autonomous Systems Laboratory
 * Queensland Centre for Advanced Technologies
 * PO Box 883, Kenmore, QLD 4069, Australia
 * http://www.ict.csiro.au/
 *  
 * Copyright (c) CSIRO 
 ***********************************************************************/

/**
 * \file inet.c
 * \brief a simple interface to the network
 * \author Pavan Sikka
 *
 * A simple interface to the BSD-style networking facilities
 * provided by most UNIX-like systems.
 *
 */

#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netinet/tcp.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <pthread.h>
#include <string.h>
#include <errno.h>

#include "rtx/defines.h"
#include "rtx/error.h"
#include "rtx/thread.h"
#include "rtx/list.h"
#include "rtx/message.h"
#include "rtx/inet.h"
#include "rtx/timer.h"

static char rcsid[] RTX_UNUSED = "$Id: inet.c 2733 2008-03-05 04:36:03Z ale077 $";

/* forward declarations */

#define VERBOSE_THREAD 0
#define VERBOSE_NET 0

int rtx_inet_get_net_addr (const char * hostname, int portnum,
		RtxInetEndpoint * p);
char * rtx_inet_print_net_addr (RtxInetEndpoint * p);
int rtx_inet_print_net_portnum (RtxInetEndpoint * p);
RtxInetConn * rtx_inet_init_socket (const char * hostname,
		int portnum,
		RtxInetConnType conntype);
int rtx_inet_close_socket (RtxInetConn * p);
int rtx_inet_connect (RtxInetConn * p, const char * hostname, int portnum);
int rtx_inet_listen (RtxInetConn * p);
RtxInetConn * rtx_inet_accept (RtxInetConn * p);
void * rtx_inet_tcp_server_user_function (void * arg);
void * rtx_inet_tcp_server_listener (void * arg);
int rtx_inet_start_tcp_server (RtxInet * handle, int portnum,
		void * (* fn) (void *, void *),
		void * arg);
int rtx_inet_kill_tcp_server (RtxInet * handle);
int rtx_inet_join_multicast (RtxInetConn * p, const char * multiAddr,
		int portNum);
int rtx_inet_leave_multicast (RtxInetConn * p, const char * multiAddr);
int rtx_inet_set_multicast_ttl (RtxInet * p, int ttl);

/* inet module */

/**
 * initialize a handle for the specified service/connection
 *
 * This function is heavily over-loaded and the results vary greatly
 * depending on connType:
 * 
 * - RTX_INET_UDP: The function returns an initialized UDP socket. The
 *   only arguments used are localHost and localPort. If localHost is NULL,
 *   then INADDR_ANY is used to bind the socket locally. If localPort is 0,
 *   a port number is assigned by the system.
 * - RTX_INET_UDP_CONNECTED: The function returns an initialized, connected
 *   UDP socket. Its similar to RTX_INET_UDP except that the created socket
 *   is bound to the remote address specified by remoteHost and remotePort.
 * - RTX_INET_UDP_MULTICAST: The function returns an initialized UDP socket
 *   that can be used for multicasting. The multicast address is specified
 *   by remoteHost and remotePort.
 * - RTX_INET_TCP_CLIENT: The function returns an initialized, connected
 *   TCP socket. The arguments used are the same as for RTX_INET_UDP_CONNECTED.
 * - RTX_INET_TCP_SERVER: The function does a lot of work in this case. It
 *   creates a TCP socket bound to the address specified by localHost and
 *   localPort. The arguments remoteHost and remotePort are not used.
 *   The function also creates a thread that listens for connections on the
 *   socket. When a connection request is received, it is accepted, resulting
 *   in the creation of a new socket. The listener thread then launches a new
 *   thread which uses the newly created socket for communicating with the
 *   client. This thread runs the function specified by the server argument. The
 *   pointer specified by arg is provided to this thread as an argument. When
 *   the client completes, the function specified by the cleanup argument is
 *   run. Whew !
 *
 * @return handle, NULL if error
 */
RtxInet * 
rtx_inet_init (
		RtxInetConnType connType,   /**< connection type */
		const char * localHost,  /**< local host name (NULL for INADDR_ANY) */
		int localPort,     /**< local port (0 for any) */
		const char * remoteHost, /**< remote host name */
		int remotePort,    /**< remote port number */
		void * (* server) (void *, void *),   /**< user-defined TCP/IP 
											   ** server function */
		void * (* cleanup) (void *, void *),  /**< user-defined TCP/IP
											   ** server cleanup function */
		void * arg  /**< user-defined argument */
		)
{
	RtxInet * handle = NULL;

	if ((handle = (RtxInet *) calloc (1, sizeof (RtxInet))) == NULL)
		return (rtx_error_null ("rtx_inet_init: calloc() failed"));
	handle->connType = connType;
	handle->cleanupfn = cleanup;
	switch (connType) {
		case RTX_INET_UDP :
			if ((handle->sock = rtx_inet_init_socket (localHost, localPort,
							connType)) == NULL) {
				free (handle);
				return (rtx_error_null ("rtx_inet_init: rtx_inet_init_socket"));
			}
			return (handle);
		case RTX_INET_UDP_CONNECTED :
			if ((handle->sock = rtx_inet_init_socket (localHost, localPort,
							connType)) == NULL) {
				free (handle);
				return (rtx_error_null ("rtx_inet_init: rtx_inet_init_socket"));
			}
			if (rtx_inet_connect (handle->sock, remoteHost, remotePort) == -1) {
				free (handle);
				return (rtx_error_null ("rtx_inet_init: rtx_inet_init_socket"));
			}
			return (handle);
		case RTX_INET_UDP_MULTICAST :
			if ((handle->sock = rtx_inet_init_socket (localHost, remotePort,
							connType)) == NULL) {
				free (handle);
				return (rtx_error_null ("rtx_inet_init: rtx_inet_init_socket"));
			}
			handle->sock->multicastTtl = RTX_INET_DEFAULT_MULTICAST_TTL;
			if (rtx_inet_join_multicast (handle->sock, remoteHost, 
						remotePort) == -1) {
				free (handle);
				return (rtx_error_null ("rtx_inet_init: rtx_inet_join_multicast"));
			}
			return (handle);
		case RTX_INET_TCP_CLIENT :
			if ((handle->sock = rtx_inet_init_socket (localHost, localPort,
							connType)) == NULL) {
				free (handle);
				return (rtx_error_null ("rtx_inet_init: rtx_inet_init_socket"));
			}
			if (rtx_inet_connect (handle->sock, remoteHost, remotePort) == -1) {
				free (handle);
				return (rtx_error_null ("rtx_inet_init: rtx_inet_init_socket [%s:%d]",
							remoteHost, remotePort));
			}
			return (handle);
		case RTX_INET_TCP_SERVER :
			if ((handle->sock = rtx_inet_init_socket (localHost, localPort,
							connType)) == NULL) {
				free (handle);
				return (rtx_error_null ("rtx_inet_init: rtx_inet_init_socket"));
			}
			if (rtx_inet_start_tcp_server (handle, localPort, server,
						arg) == -1) {
				free (handle);
				return (rtx_error_null ("rtx_inet_init: rtx_inet_init_socket"));
			}
			return (handle);
		default :
			free (handle);
			return (rtx_error_null ("invalid conn type [%d]",
						connType));
	}

	free (handle);
	return (rtx_error_null ("rtx_inet_init: reached end of function [BAD]"));
}

/**
 * cleanup/shuttdown specified connection/service
 *
 * @return 0, -1 on error
 */
int 
rtx_inet_done (
		RtxInet * p    /**< handle */
		)
{
	int errs = 0;
	RtxInetTcpClient * clnt = NULL;

	switch (p->connType) {
		case RTX_INET_UDP :
		case RTX_INET_UDP_CONNECTED :
			if (rtx_inet_close_socket (p->sock) == -1)
				errs++;
			free (p);
			break;
		case RTX_INET_UDP_MULTICAST :
			if (rtx_inet_leave_multicast 
					(p->sock, p->sock->remote.ifname) == -1)
				errs++;
			if (rtx_inet_close_socket (p->sock) == -1)
				errs++;
			free (p);
			break;
		case RTX_INET_TCP_CLIENT :
			if (rtx_inet_close_socket (p->sock) == -1)
				errs++;
			free (p);
			break;
		case RTX_INET_TCP_SERVER :
#if VERBOSE_NET
			rtx_message("destroying TCP server");
#endif
			while ((clnt = (RtxInetTcpClient *) rtx_list_iterate (p->clients)) != NULL) {
#if VERBOSE_NET
				rtx_message("destroying client %08x",clnt->thread->id);
#endif
				rtx_thread_destroy(clnt->thread);
			}
			if (rtx_thread_destroy_sync(p->listenerThread) == -1) {
				errs++;
				rtx_error ("rtx_inet_done: rtx_thread_destroy_sync");
			}
			if (rtx_inet_close_socket (p->sock) == -1)
				errs++;
			rtx_inet_kill_tcp_server (p);
			free (p);
			break;
		default :
			if (rtx_inet_close_socket (p->sock) == -1)
				errs++;
			free (p);
			return (rtx_error ("invalid conn type [%d]",
						p->connType));
	}
	if (errs)
		return (-1);
	return (0);
}

/**
 * write
 */
int
rtx_inet_write (
		RtxInetConn * s,       /**< handle */
		const char * buf,            /**< data */
		int len,               /**< number of bytes */
		RtxInetEndpoint * p    /**< address to write to, if not NULL */
		)
{
	int n;

	if (p != NULL) {
		if ((n = sendto (s->sockfd, buf, len, 0,
						(struct sockaddr *) p->addr,
						p->addrLen)) == -1) {
			return (rtx_error_errno ("sendto()"));
		}   
		return (n);
	} else {
		if ((n = send (s->sockfd, buf, len, 0)) == -1) {
			return (rtx_error_errno ("send()"));
		}   
		return (n);
	}
	return (rtx_error ("rtx_inet_write: should never get here"));
}

/**
 * read
 */
int
rtx_inet_read (
		RtxInetConn * s,       /**< handle */
		char * buf,            /**< data */
		int len,               /**< number of bytes */
		RtxInetEndpoint * p    /**< source of the data, if not NULL */
		)
{
	int numChars;

	struct sockaddr_in mySock; int sokSize = sizeof(mySock);

	if (p != NULL) {
		p->addrLen = RTX_INET_MAX_ADDR_LEN; 
		numChars = recvfrom (s->sockfd, buf, len, 0,
				(struct sockaddr *) &mySock, (socklen_t*)&sokSize);
		p->portNum = ntohs(mySock.sin_port); 
		p->addrLen = sokSize; 
		sprintf(p->addr, "%s", inet_ntoa(mySock.sin_addr)); 

		if (numChars == -1) {
			return (rtx_error_errno ("recvfrom()"));
		}
		return (numChars);
	} else {

		numChars = recv (s->sockfd, buf, len, 0); 
		
		if (numChars == -1) {
			return (rtx_error_errno ("recvfrom()"));
		}
		return (numChars);

	} return (rtx_error ("rtx_inet_read: should never get here"));
}

/**
 * read a line (max n chars) from a connected socket
 */
int
rtx_inet_readline (
		RtxInetConn * s,       /**< handle */
		char * buf,            /**< data */
		int len,               /**< number of bytes */
		RtxInetEndpoint * p    /**< source of the data, if not NULL */
		)
{
	int i = 0, n = 0, m = len - 1;
	char ch;

	while (i < m) {
		n = read (s->sockfd, &ch, 1);
		if (n == -1) /* read error */
			return (rtx_error_errno ("rtx_inet_readline: "
						"read() failed"));
		if (n == 0)    /* end-of-file */
			break;
		buf[i++] = ch;
		if (ch == '\n')
			break;
	}
	buf[i] = '\0';
	return (i);
}

/**
 * write specified number of bytes to a connected socket
 */
int
rtx_inet_writen (
		RtxInetConn * s,
		const char * buf,
		int len
		)
{
	int n = len, m;
	const char * p = buf;

	while (n) {
		if ((m = write (s->sockfd, p, n)) <= 0) {
			return (rtx_error_errno ("write()"));
		}
		n -= m;
		p += m;
	}
	return (len);
}

/**
 * initialize a network connection end-point
 */
RtxInetConn *
rtx_inet_init_socket (
		const char * hostname,
		int portnum,
		RtxInetConnType conntype
		)
{
	int sockfd;
	struct sockaddr_in * addr = NULL;
	RtxInetConn * p = NULL;
	int tmp = 0;
	struct linger optval;

	if ((p = (RtxInetConn *) calloc (1, sizeof (RtxInetConn))) == NULL)
		return (rtx_error_null ("calloc() failed"));

	switch (conntype) {
		case RTX_INET_UDP :
		case RTX_INET_UDP_CONNECTED :
			/* create an Internet UDP socket */
			if ((sockfd = socket (AF_INET, SOCK_DGRAM, 
							IPPROTO_UDP)) == -1) {
				free (p);
				return (rtx_error_errno_null ("socket() failed"));
			}
			break;
		case RTX_INET_UDP_MULTICAST :
			/* create an Internet UDP socket */
			if ((sockfd = socket (AF_INET, SOCK_DGRAM, 
							IPPROTO_UDP)) == -1) {
				free (p);
				return (rtx_error_errno_null ("socket() failed"));
			}
#ifdef SO_REUSEPORT
			/* set option to allow multiple clients */
			tmp = 1;
			if (setsockopt (sockfd, SOL_SOCKET, SO_REUSEPORT, &tmp, 
						sizeof (tmp)) == -1)
				return (rtx_error_errno_null ("rtx_inet_join_multicast_"
							"group: setsockopt("
							"SO_REUSEPORT)"));
#endif
#ifdef SO_REUSEADDR
			/* set option to allow multiple clients */
			tmp = 1;
			if (setsockopt (sockfd, SOL_SOCKET, SO_REUSEADDR, &tmp, 
						sizeof (tmp)) == -1)
				return (rtx_error_errno_null ("rtx_inet_join_multicast_"
							"group: setsockopt("
							"SO_REUSEPORT)"));
#endif
			break;
		case RTX_INET_TCP_CLIENT :
		case RTX_INET_TCP_SERVER :
			/* create an Internet TCP socket */
			if ((sockfd = socket (AF_INET, SOCK_STREAM, 
							IPPROTO_TCP)) == -1) {
				free (p);
				return (rtx_error_errno_null ("socket() failed"));
			}
			tmp = 1;
			if (setsockopt (sockfd, IPPROTO_TCP, TCP_NODELAY, 
						&tmp, sizeof (tmp)) == -1) {
				free (p);
				close (sockfd);
				return (rtx_error_errno_null
						("setsockopt(TCP_NODELAY)"));
			}
			tmp = 1;
			if (setsockopt (sockfd, SOL_SOCKET, SO_KEEPALIVE, 
						&tmp, sizeof (tmp)) == -1) {
				free (p);
				close (sockfd);
				return (rtx_error_errno_null 
						("setsockopt(TCP_KEEPALIVE)"));
			}
			optval.l_onoff = 1;
			optval.l_linger = 0;
			if (setsockopt (sockfd, SOL_SOCKET, SO_LINGER, 
						&optval, sizeof (optval)) == -1) {
				free (p);
				close (sockfd);
				return (rtx_error_errno_null 
						("setsockopt(SO_LINGER)"));
			}
			break;
		default :
			free (p);
			return (rtx_error_null ("invalid conn type [%d]",
						conntype));
	}
	p->hostname[RTX_INET_MAX_HOST_NAME_LEN-1] = '\0';
	if (gethostname (p->hostname, RTX_INET_MAX_HOST_NAME_LEN-1)
			== -1) {
		free (p);
		close (sockfd);
		return (rtx_error_errno_null ("gethostname()"));
	}
	p->sockfd = sockfd;
	p->local.portNum = portnum;
	/* bind it */
	if (hostname != NULL) {
		strncpy (p->local.ifname, hostname, RTX_INET_MAX_HOST_NAME_LEN-1);
		if (rtx_inet_get_net_addr (hostname, portnum, 
					&p->local) == -1) {
			free (p);
			close (sockfd);
			return (rtx_error_null ("rtx_inet_get_net_addr()"));
		}
	} else {
		strcpy (p->local.ifname, "INADDR_ANY");
		addr = (struct sockaddr_in *) p->local.addr;
		addr->sin_family = AF_INET;
		addr->sin_addr.s_addr = htonl (INADDR_ANY);
		addr->sin_port = htons (portnum);
		p->local.addrLen = sizeof (struct sockaddr_in);
	}
	if (bind (sockfd, (struct sockaddr *) addr, p->local.addrLen)
			== -1)
		return (rtx_error_errno_null ("bind(%s:%d)", 
					p->local.ifname,
					p->local.portNum));

	return (p);
}


/**
 * close a socket for communication
 */
int
rtx_inet_close_socket (
		RtxInetConn * p
		)
{
	int errs = 0;

	if (shutdown (p->sockfd, 2) == -1) {
		if (errno != ENOTCONN) {
			rtx_error_errno ("rtx_inet_close_socket: shutdown (%d)",
					p->sockfd);
			errs++;
		}
	}
	if (close (p->sockfd) == -1) {
		errs++;
		rtx_error_errno ("rtx_inet_close_socket: close (%d)", 
				p->sockfd);
	}
	free (p);
	if (errs)
		return (-1);
	return (0);
}

/**
 * connect to the specified address
 */
int
rtx_inet_connect (
		RtxInetConn * p,
		const char * hostname,
		int portnum
		)
{
	struct sockaddr_in * addr = NULL;

	addr = (struct sockaddr_in *) p->remote.addr;
	if (rtx_inet_get_net_addr (hostname, portnum, &p->remote) == -1)
		return (rtx_error ("rtx_inet_get_net_addr()"));

	if (connect (p->sockfd, (struct sockaddr *) addr,
				p->remote.addrLen) == -1) {
		return (rtx_error_errno ("connect()"));
	}
	return (0);
}

/**
 * listen for a connection on the specified socket
 */
int
rtx_inet_listen (
		RtxInetConn * p
		)
{
	if (listen (p->sockfd, RTX_INET_MAX_NUM_CONNS) == -1)
		return (rtx_error_errno ("listen()"));
	return (0);
}

/**
 * accept a connection on the specified socket
 */
RtxInetConn *
rtx_inet_accept (
		RtxInetConn * p
		)
{
	RtxInetConn * t = NULL;

	if ((t = (RtxInetConn *) calloc (1, sizeof (RtxInetConn))) == NULL)
		return (rtx_error_null ("calloc() failed"));
	rtx_thread_cleanup_push(free,t);
	memcpy (t, p, sizeof (RtxInetConn));
	t->remote.addrLen = RTX_INET_MAX_ADDR_LEN;
	memset (t->remote.addr, 0, RTX_INET_MAX_ADDR_LEN);
	if ((t->sockfd = accept (p->sockfd, 
					(struct sockaddr *) t->remote.addr,
					(socklen_t*)&(t->remote.addrLen))) == -1) {
		free(t);
		return (rtx_error_errno_null ("accept()"));
	}
	strncpy (t->remote.ifname, 
			rtx_inet_print_net_addr (&t->remote),
			RTX_INET_MAX_HOST_NAME_LEN-1);
	t->remote.portNum = rtx_inet_print_net_portnum (&t->remote);
	rtx_thread_cleanup_pop(0);
	return (t);
}


void *
rtx_inet_tcp_server_user_function (
		void * arg
		)
{
	RtxInetTcpClient * clnt;

#if VERBOSE_NET
	rtx_message("user function");
#endif
	clnt = (RtxInetTcpClient *) arg;
	if (clnt->inetServer->userfn != NULL)
		(* clnt->inetServer->userfn) (clnt,
				clnt->inetServer->userarg);
	return (NULL);
}

void
rtx_inet_tcp_server_cleanup_function (
		void * arg
		)
{
	RtxInetTcpClient * clnt;
	clnt = (RtxInetTcpClient *) arg;
#if VERBOSE_NET
	rtx_message("cleanup function");
#endif


	clnt->done = 1;
	rtx_sem_post (clnt->inetServer->cleanerSem);
}

void *
rtx_inet_tcp_server_cleaner (
		void * arg
		)
{
	RtxInet * s = NULL;
	RtxInetTcpClient * clnt = NULL, * clntFound = NULL;

	s = (RtxInet *) arg;
	while (! s->done) {
		if (rtx_sem_wait (s->cleanerSem) == -1) {
			rtx_error_flush ("rtx_sem_wait");
			/* This is just to stop flooding the display, a correct action may 
			 * be to kill the program, but we can't really do it here */
			rtx_timer_sleep (0.05);
			continue;
		}
		clntFound = NULL;
		while ((clnt = (RtxInetTcpClient *) rtx_list_iterate (s->clients)) != NULL) {
			if (clnt->done == 1)
				clntFound = clnt;
		}
#if VERBOSE_NET
		rtx_message("cleaner function");
#endif
		if (clntFound != NULL) {
			struct linger lg = {1,0};
#if VERBOSE_NET
			rtx_message("cleaner function: found client");
#endif
			/* Calling user-provided cleanup function */
			if (s->cleanupfn) {
				s->cleanupfn(clntFound,s->userarg);
			}

			/* Calling shutdown a first time, in case linger time has been
			 * set to non zero, to flush the buffer*/
#if VERBOSE_NET
			rtx_message("shutdown");
#endif
			if ((shutdown (clntFound->sock->sockfd,2) == -1) && (errno != ENOTCONN))
				rtx_error_errno_flush ("rtx_inet_tcp_server_cleaner: shutdown (%d)", 
						clntFound->sock->sockfd);
			/* Now setting the linger time to zero */
			if (setsockopt (clntFound->sock->sockfd, SOL_SOCKET, SO_LINGER, 
						&lg, sizeof (lg)) == -1)
				rtx_error_errno_flush ("rtx_inet_tcp_server_cleaner: setsockopt (%d)", 
						clntFound->sock->sockfd);
			/* And really closing the socket. This garantee that the data has
			 * been transmitted, good for connection-based transaction such as
			 * HTTP, and also really close the socket so as to make it
			 * immediately reusable */
#if VERBOSE_NET
			rtx_message("close");
#endif
			if ((close (clntFound->sock->sockfd) == -1)  && (errno != ENOTCONN))
				rtx_error_errno_flush ("rtx_inet_tcp_server_cleaner: close (%d)", 
						clntFound->sock->sockfd);
			free (clntFound->sock);
			/* We came here since the client terminated, it 
			 * is detached, so it will take care of its memory itself
			 * we don't need a destroy
			 * rtx_thread_destroy(clntFound->thread); */
			rtx_list_del_node (s->clients, clntFound, 0);
			free (clntFound);
		}
	}
	/** Normally, the list is empty at this point, so we don't need to free the
	 * data. Anyway, a simple free would not be sufficient to cleanup a
	 * RtxInetTcpClient structure. **/
	rtx_list_destroy(s->clients,0);
	return (NULL);
}

void *
rtx_inet_tcp_server_listener (
		void * arg
		)
{
	RtxInet * s = NULL;
	RtxInetTcpClient * node;
	char thName[64];
	int clientNum = 0;

	s = (RtxInet *) arg;
	if (rtx_inet_listen (s->sock) == -1) {
		rtx_error_flush ("rtx_inet_listen()");
		return (NULL);
	}
	while (! s->done) {
		if ((node = (RtxInetTcpClient *) calloc 
					(1, sizeof (RtxInetTcpClient))) == NULL) {
			rtx_error_flush ("calloc()");
			/* FIXME: find a better way of handling this failure, this one is
			 * safe at least... */
			exit(1);

		}
		node->inetServer = s;
		rtx_thread_cleanup_push(free,node);
		if ((node->sock = rtx_inet_accept (s->sock))
				== NULL) {
			free(node);
			if (! s->done)
				rtx_error_flush ("rtx_inet_accept()");
			continue;
		}
		rtx_thread_cleanup_pop(0);
#if VERBOSE_NET
		rtx_message("Received connection");
#endif
		sprintf (thName, "client_handler[%d]", clientNum);
		if ((node->thread =
					rtx_thread_create (thName, 
						VERBOSE_THREAD,
						RTX_THREAD_SCHED_OTHER, 
						RTX_THREAD_PRIO_MIN,
						0,
						RTX_THREAD_CANCEL_DEFERRED,
						rtx_inet_tcp_server_user_function,
						(void *) node,
						rtx_inet_tcp_server_cleanup_function,
						(void *) node)) == NULL)
			rtx_error_flush ("unable to launch handler thread");
		else {
			if (rtx_thread_detach (node->thread)) {
				rtx_thread_destroy (node->thread);
				rtx_error_flush("rtx_inet_tcp_server_listener: failed to detach connection thread");
			} else {
				clientNum++;
				if (rtx_list_add (s->clients, 
							node->sock->remote.ifname,
							node) == -1) {
					rtx_error_flush ("rtx_list_add() failed");
				}
			}
		}

	}
#if VERBOSE_NET
	rtx_message("listener terminated");
#endif
	return NULL;
}

/**
 * startup a TCP/IP server
 */
int
rtx_inet_start_tcp_server (
		RtxInet * s,
		int portnum,
		void * (* fn) (void *, void *),
		void * arg
		)
{
	if ((s->clients = rtx_list_init ()) == NULL)
		return (rtx_error ("rtx_list_init() failed"));
	s->userfn = fn;
	s->userarg = arg;
	if ((s->cleanerSem = rtx_sem_init (NULL, 0, 0)) == NULL)
		return (rtx_error ("rtx_sem_init()"));
	if ((s->listenerThread = 
				rtx_thread_create ("rtx_inet_tcp_server_function",
					VERBOSE_THREAD,
					RTX_THREAD_SCHED_OTHER,
					RTX_THREAD_PRIO_MIN,
					0,
					RTX_THREAD_CANCEL_DEFERRED,
					rtx_inet_tcp_server_listener,
					(void *) s,
					NULL,
					NULL)) == NULL)
		return (rtx_error ("rtx_thread_create()"));
	if ((s->cleanerThread = 
				rtx_thread_create ("rtx_inet_tcp_server_cleaner_function",
					VERBOSE_THREAD,
					RTX_THREAD_SCHED_OTHER,
					RTX_THREAD_PRIO_MIN,
					0,
					RTX_THREAD_CANCEL_DEFERRED,
					rtx_inet_tcp_server_cleaner,
					(void *) s,
					NULL,
					NULL)) == NULL)
		return (rtx_error ("rtx_thread_create()"));
	return (0);

}

/**
 * kill a TCP/IP server
 */
int
rtx_inet_kill_tcp_server (
		RtxInet * s
		)
{
	s->done = 1;
#if VERBOSE_NET
	rtx_message("kill tcp server");
#endif
	rtx_sem_post(s->cleanerSem);
	rtx_thread_join(s->cleanerThread);
	rtx_thread_destroy(s->cleanerThread);
	rtx_sem_destroy(s->cleanerSem);
	return (0);
}

/**
 * Set the multicast TTL value for the socket
 *
 * @return 0 on success, -1 on failure
 */			  
int
rtx_inet_set_multicast_ttl (
		RtxInet * p,
		int ttl
		)
{
	unsigned char lmttl;

	lmttl = (unsigned char) ttl;
	/* set option to increase packet reach */
	p->sock->multicastTtl = (unsigned char) ttl;
	if (setsockopt (p->sock->sockfd, IPPROTO_IP, IP_MULTICAST_TTL, 
				&lmttl, sizeof (lmttl)) == -1) {
		return (rtx_error_errno ("rtx_inet_join_multicast_"
					"group: setsockopt("
					"IP_MULTICAST_TTL)"));
	}
	return (0);
}

/**
 * activate loopback on multicast
 *
 * @return 0 on success
 *
 */
int
rtx_inet_set_loopback (
		RtxInet * p,   /**< socket */
		int loopback       /**< port number */
		)
{
	unsigned char multicastLoop;
#if 0
	/* get loopback option */
	size_t optlen,i;
	unsigned char state[4];
	if (getsockopt (p->sock->sockfd, IPPROTO_IP, IP_MULTICAST_LOOP, 
				state, &optlen) == -1) {
		fprintf (stderr, "join_multicast_group: getsockopt("
				"IP_MULTICAST_LOOP) failed: %s\n", 
				strerror (errno));
		return (-1);
	}
	for (i=0; i<optlen; i++)
		fprintf (stderr, "IP_MULTICAST_LOOP = 0x%02X\n", state[i]);
#endif
	/* set option to disable loopback */
	multicastLoop = loopback;
	if (setsockopt (p->sock->sockfd, IPPROTO_IP, IP_MULTICAST_LOOP, 
				&multicastLoop, sizeof (multicastLoop)) == -1) {
		return (rtx_error_errno ("rtx_inet_join_multicast_"
					"group: setsockopt("
					"IP_MULTICAST_LOOP)"));
	}
	return (0);
}


/**
 * join the multicast group specified
 *
 * @return initialzed handle on success, NULL on failure
 *
 */
int
rtx_inet_join_multicast (
		RtxInetConn * sock,   /**< socket */
		const char * multiAddr, /**< multicast address */
		int portNum       /**< port number */
		)
{
	struct ip_mreq mreq;
	unsigned char multicastLoop;

	if (rtx_inet_get_net_addr (multiAddr, portNum, &(sock->remote)) == -1)
		return (rtx_error ("rtx_inet_join_multicast_group: "
					"rtx_inet_get_net_addr()"));

	/* now, join the multicast group */
	mreq.imr_multiaddr.s_addr=inet_addr(multiAddr);
	mreq.imr_interface.s_addr=htonl(INADDR_ANY);
	if (setsockopt(sock->sockfd, IPPROTO_IP, IP_ADD_MEMBERSHIP, 
				&mreq, sizeof(mreq)) < 0)
		return (rtx_error_errno ("rtx_inet_join_multicast_"
					"group: setsockopt"
					"(IP_ADD_MEMBERSHIP)"));

	/* get loopback option */
	/*
	   if (getsockopt (sockfd, IPPROTO_IP, IP_MULTICAST_LOOP, 
	   optarg, &optlen) == -1) {
	   fprintf (stderr, "join_multicast_group: getsockopt("
	   "IP_MULTICAST_LOOP) failed: %s\n", 
	   strerror (errno));
	   return (-1);
	   }
	   for (i=0; i<optlen; i++)
	   fprintf (stderr, "IP_MULTICAST_LOOP = 0x%02X\n", optarg[i]);
	   */
	/* set option to disable loopback */
	multicastLoop = 0;
	if (setsockopt (sock->sockfd, IPPROTO_IP, IP_MULTICAST_LOOP, 
				&multicastLoop, sizeof (multicastLoop)) == -1) {
		return (rtx_error_errno ("rtx_inet_join_multicast_"
					"group: setsockopt("
					"IP_MULTICAST_LOOP)"));
	}
	/* set multicast TTL */
	if (setsockopt (sock->sockfd, IPPROTO_IP, IP_MULTICAST_TTL, 
				&(sock->multicastTtl), 
				sizeof (sock->multicastTtl)) == -1) {
		return (rtx_error_errno ("rtx_inet_join_multicast_"
					"group: setsockopt("
					"IP_MULTICAST_TTL)"));
	}

	return (0);
}

/**
 * leave the multicast group specified
 */
int
rtx_inet_leave_multicast (
		RtxInetConn * sock,
		const char * multiAddr
		)
{
	struct ip_mreq mreq;
	int errs = 0;

	/* leave the multicast group */
	mreq.imr_multiaddr.s_addr=inet_addr(multiAddr);
	mreq.imr_interface.s_addr=htonl(INADDR_ANY);
	if (setsockopt(sock->sockfd, IPPROTO_IP, IP_DROP_MEMBERSHIP,
				&mreq, sizeof(mreq)) < 0) {
		errs++;
		rtx_error_errno ("rtx_inet_leave_multicast_group: "
				"setsockopt(IP_DROP_MEMBERSHIP)");
	}
	if (errs)
		return (-1);
	return (0);
}

/**
 * get the network address for the given hostname
 */
int
rtx_inet_get_net_addr (
		const char * hostname,
		int portnum,
		RtxInetEndpoint * p
		)
{
	struct hostent * h;
	struct sockaddr_in * s = (struct sockaddr_in *) p->addr;

	memset (s, 0, sizeof (struct sockaddr_in));
	/* get the hostentry structure corresponding to hostname */
	if ((h = gethostbyname (hostname)) == NULL) {
		if ((s->sin_addr.s_addr = inet_addr (hostname)) == -1) 
			return (rtx_error ("invalid hostname [%s]", hostname));
	} else 
		s->sin_addr = * (struct in_addr *) (h->h_addr);

	s->sin_family = AF_INET;
	s->sin_port = htons (portnum);
	p->addrLen = sizeof (struct sockaddr_in);
	strncpy (p->ifname, hostname, RTX_INET_MAX_HOST_NAME_LEN-1);
	p->portNum = portnum;

	return (0);
}

/**
 * print host address
 */
char *
rtx_inet_print_net_addr (
		RtxInetEndpoint * p
		)
{
	return (inet_ntoa 
			(((struct sockaddr_in *) (p->addr))->sin_addr));
}

/**
 * print port number
 */
int
rtx_inet_print_net_portnum (
		RtxInetEndpoint * p
		)
{
	return (ntohs (((struct sockaddr_in *) (p->addr))->sin_port));
}

/**
 * wait for data to be available 
 * */
int rtx_inet_wait_data (RtxInetConn * p, double timeout)
{
	fd_set rfs;
	struct timeval to = {0,0};
	unsigned int millisec = (int)(timeout*1000);
	to.tv_sec = (millisec/1000);
	to.tv_usec = (millisec%1000) * 1000;
	FD_ZERO(&rfs);FD_SET(p->sockfd,&rfs);
	int r = select(p->sockfd+1,&rfs,NULL,NULL,&to);
	return (r >= 1)?0:1;
}


