/***********************************************************************
 * 
 * CSIRO Autonomous Systems Laboratory
 * Queensland Centre for Advanced Technologies
 * PO Box 883, Kenmore, QLD 4069, Australia
 * http://www.ict.csiro.au/
 *  
 * Copyright (c) CSIRO 
 ***********************************************************************/

static char rcsid[] RTX_UNUSED = "$Id: iio.c 2271 2007-12-23 04:41:55Z roy029 $";

/**
 * \file iio.c
 * \brief Utility routines to make IIO more RTX friendly.
 * \author Peter Corke
 *
 * Utility functions to make ease error handling with IIO.
 */

#include	<stdio.h>
#include	<stdarg.h>
#include	<stdlib.h>
#include	"iio.h"
#include	"rtx/iio.h"
#include	"rtx/error.h"

/**
 * Open an iio channel with RTX error wrapper
 *
 * @param name name of iio channel
 * @param chan IIO channel pointer
 * @param oflag IIO open flag
 * @return -1 on error.
 *
 * Provides meaningful error return string that includes name of channel.
 */
int
rtx_iio_open(char *name, IIO *chan, IIO_OFLAG oflag)
{
	if (iio_open(name, oflag, chan) < 0)
		return rtx_error_iio("rtx_iio_open: iio_open on [%s] failed for %s",
			name, iio_emessage_get());
	return 0;
}

/**
 * Open an iio channel using sprintf style channel name.
 *
 * @param chan IIO channel pointer
 * @param oflag IIO open flag
 * @param fmt sprintf() style format string
 * @param ... optional arguments to sprintf().
 * @return -1 on error.
 *
 * Typically used when opening a bunch of iio channels with similar names, for
 * example encoder0, encoder1, ...
 */
int
rtx_iio_open_sprintf(IIO *chan, IIO_OFLAG oflag, char *fmt, ...)
{
	va_list	ap;
	char	buf[BUFSIZ];

	va_start(ap, fmt);
	vsprintf(buf, fmt, ap);
	if (iio_open(buf, oflag, chan) < 0)
		return rtx_error_iio("rtx_iio_open_sprintf: iio_open on [%s] failed for %s",
			buf, iio_emessage_get());
	return 0;
}


/**
 * rtx_error() variant for use with IIO.  Appends the iio error message
 * to the end of the format string.
 *
 * @param msg	error message string.
 */
int
rtx_error_iio(char *msg, ...)
{
	char	buf[BUFSIZ];
	va_list	ap;

	va_start(ap, msg);
	vsprintf(buf, msg, ap);

	return rtx_error("%s iio_err=[%s]", buf, iio_emessage_get());
}
