/***********************************************************************
 * 
 * CSIRO Autonomous Systems Laboratory
 * Queensland Centre for Advanced Technologies
 * PO Box 883, Kenmore, QLD 4069, Australia
 * http://www.ict.csiro.au/
 *  
 * Copyright (c) CSIRO 
 ***********************************************************************/

/**
 * \file httpd.c
 * \brief http server, main functions
 */

/* The real stuff */

#include <time.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <signal.h>
#include <netdb.h>
#include <errno.h>
#include <ctype.h>

#ifdef i486_qnxrtp
#include <sys/types.h>
#define socklen_t size_t
#endif
#include <sys/socket.h>
#include <netinet/tcp.h>

#include "rtx/timer.h"
#include "rtx/thread.h"
#include "rtx/error.h"
#include "rtx/message.h"
#include "rtx/httpd.h"

static char rcsid[] RTX_UNUSED = "$Id: httpd.c 3143 2008-05-20 23:51:34Z ale077 $";

#define MAX_URL_LENGTH 4096


#if 0
#define DEBUG
#endif

#ifdef DEBUG
#define PRINTF(X...) printf(X)
#else
#define PRINTF(X...) 
#endif

const static char crlf[3] = {0x0D,0x0A,0x00};

static const char * rtx_httpd_std_error(int status) 
{
	switch (status) {
		case 100: return "Continue";
		case 101: return "Switching Protocols";
		case 200: return "OK" ;
		case 201: return "Created" ;
		case 202: return "Accepted" ;
		case 203: return "Non-Authoritative Information" ;
		case 204: return "No Content" ;
		case 205: return "Reset Content" ;
		case 206: return "Partial Content" ;
		case 300: return "Multiple Choices" ;
		case 301: return "Moved Permanently" ;
		case 302: return "Moved Temporarily" ;
		case 303: return "See Other" ;
		case 304: return "Not Modified" ;
		case 305: return "Use Proxy" ;
		case 400: return "Bad Request" ;
		case 401: return "Unauthorized" ;
		case 402: return "Payment Required" ;
		case 403: return "Forbidden" ;
		case 404: return "Not Found" ;
		case 405: return "Method Not Allowed" ;
		case 406: return "Not Acceptable" ;
		case 407: return "Proxy Authentication Required" ;
		case 408: return "Request Timeout" ;
		case 409: return "Conflict" ;
		case 410: return "Gone" ;
		case 411: return "Length Required" ;
		case 412: return "Precondition Failed" ;
		case 413: return "Request Entity Too Large" ;
		case 414: return "Request-URI Too Long" ;
		case 415: return "Unsupported Media Type" ;
		case 500: return "Internal Server Error" ;
		case 501: return "Not Implemented" ;
		case 502: return "Bad Gateway" ;
		case 503: return "Service Unavailable" ;
		case 504: return "Gateway Timeout" ;
		case 505: return "HTTP Version Not Supported" ;
		default:
			return "Unknown error status";
	}
	return NULL;
};


static void str_decode_percent(char * src) 
{
	char *sit,*dit;
	unsigned int val;
	
	sit = src; dit = src;
	while (*sit != 0) {
		if (*sit == '%') {
			val='X';
			sscanf(sit+1,"%2x",&val);
			*dit++ = val;
			sit += 2;
		} else {
			*dit++ = *sit;
		}
		sit += 1;
	}
	*dit = 0;
}

// uri modified here, in-place decoding
static int rtx_httpd_parse_uri(RtxHttpdReq *req, char * uri, unsigned int len)
{
	char * args = strchr(uri,'?');
	if (args != NULL) {
		*args = 0;
		args += 1;
		while ((args != NULL) && (*args != 0)) {
			char argname[MAX_URL_LENGTH];
			char argval[MAX_URL_LENGTH];
			char * next;
			next = strchr(args,'&');
			if (next != NULL) {*next++ = 0;}
			sscanf(args,"%[^=]=%s",argname,argval);
			str_decode_percent(argname);
			str_decode_percent(argval);
			rtx_hash_set(req->args,argname,argval, NULL);
			args = next;
		}
	}
	req->url = strdup(uri);
	str_decode_percent(req->url);

	return 0;
}
	
static int rtx_httpd_parse_post(RtxHttpdReq *req, const char * post, unsigned int len)
{
	char pattern[128];
	const char * args = post;
	sprintf(pattern,"%%%d[^=]=%%%d[^&]",MAX_URL_LENGTH-1,MAX_URL_LENGTH-1);
	while ((args != NULL) && (*args != 0)) {
		char argname[MAX_URL_LENGTH];
		char argval[MAX_URL_LENGTH];
		const char * next;
		next = strchr(args,'&');
		if (next != NULL) {next++;}
		sscanf(args,pattern,argname,argval);
		PRINTF("ARGS '%s' PATTERN '%s' NAME '%s' VAL '%s'",args,pattern,argname,argval);
		str_decode_percent(argname);
		str_decode_percent(argval);
		rtx_hash_set(req->post,argname,argval, (rtx_error_t)NULL);
		args = next;
	}
	return 0;
}
	
static int rtx_httpd_parse_method(RtxHttpdReq * req, const char * str)
{
	if (strcasecmp(str,"OPTIONS")==0){
		req->method=RTX_HTTPD_M_OPTIONS;
	} else if (strcasecmp(str,"HEAD")==0){
		req->method=RTX_HTTPD_M_HEAD;
	} else if (strcasecmp(str,"GET")==0){
		req->method=RTX_HTTPD_M_GET;
	} else if (strcasecmp(str,"PUT")==0){
		req->method=RTX_HTTPD_M_PUT;
	} else if (strcasecmp(str,"POST")==0){
		req->method=RTX_HTTPD_M_POST;
	} else if (strcasecmp(str,"DELETE")==0){
		req->method=RTX_HTTPD_M_DELETE;
	} else if (strcasecmp(str,"TRACE")==0){
		req->method=RTX_HTTPD_M_TRACE;
	} else if (strcasecmp(str,"CONNECT")==0){
		req->method=RTX_HTTPD_M_CONNECT;
	} else{
		return rtx_error("rtx_httpd_parse_method: unknown method");
	}
	return 0;
}

static int rtx_httpd_trim_right(char * s) 
{
	unsigned int n;
	signed int i;
	n = strlen(s);
	i = n-1;
	while (i>=0) {
		switch (s[i]) {
			case ' ':
			case '\t':
			case '\n':
			case '\r':
				s[i] = 0;
				break;
			default:
				return 0;
		}
		i -= 1;
	}
	return 0;
}
			

static int rtx_httpd_read_request(RtxInetConn * sock, RtxHttpdReq * req)
{
	int ret;
	char line[MAX_URL_LENGTH];
	char method[16];
	char uri[MAX_URL_LENGTH];
	line[MAX_URL_LENGTH-1] = 0;
	uri[MAX_URL_LENGTH-1] = 0;

	// Discard empty lines before beginning of request
	PRINTF("Reading header headline\n");
	do {
		if (rtx_inet_wait_data(sock,1.1)) { return +1; }
		errno = 0;
		line[0] = 0;
		ret = rtx_inet_readline(sock,line,MAX_URL_LENGTH-1,NULL);
		PRINTF("Read req header '%s' ret=%d\n",line,ret);
		if ((ret==0) || (errno==ECONNRESET)) return +1;
		if (ret<0) 
			return rtx_error("rtx_httpd_read_request: error reading URI");
		if ((ret == MAX_URL_LENGTH-1) && (line[MAX_URL_LENGTH-2]!='\n'))
			return rtx_error("rtx_httpd_read_request: header line too long");
		rtx_httpd_trim_right(line);
	} while (line[0]==0);

	ret = sscanf(line," %s %s HTTP/%d.%d",method,uri,
			&(req->http_major),&(req->http_minor));
	if (ret != 4) 
		return rtx_error("rtx_httpd_read_request: invalid header line '%s'",line);

	if (rtx_httpd_parse_method(req,method))
		return rtx_error("rtx_httpd_read_request: parse_method failed");
	
	if (rtx_httpd_parse_uri(req,uri,strlen(uri)))
		return rtx_error("rtx_httpd_read_request: parse_uri failed");

	if ((req->http_major==1) && (req->http_minor==0)) {
		// Specific to HTTP/1.0
		req->persistent = 0;
	}

	req->content_length = 0;
	req->persistent = 1;
	PRINTF("Reading next header\n");
	while (1) {
		char header[MAX_URL_LENGTH]="", value[MAX_URL_LENGTH]="";
		if (rtx_inet_wait_data(sock,1.1)) { return +1; }
		ret = rtx_inet_readline(sock,line,MAX_URL_LENGTH-1,NULL);
		if (ret<0) 
			return rtx_error("rtx_httpd_read_request: error reading header");
		if ((ret == MAX_URL_LENGTH-1) && (line[MAX_URL_LENGTH-2]!='\n'))
			return rtx_error("rtx_httpd_read_request: header too long");

		ret = sscanf(line," %[^:]: %s ", header,value);
		rtx_httpd_trim_right(header);
		PRINTF("Read header '%s' : name '%s' value '%s'\n",line,header,value);
		if (ret < 1) break;

		rtx_hash_set(req->headers,header,value, (rtx_error_t)NULL);

		/* Some specific headers are processed already here */
		if (strcasecmp(header,"Content-Length")==0) {
			sscanf(value," %ld ",&(req->content_length));
		} else if (strcasecmp(header,"Connection")==0) {
			if (strcasecmp(value,"Close")==0) 
				req->persistent = 0;
			else if (strcasecmp(value,"Keep-Alive")==0) 
				req->persistent = 1;
		}
	}
	
	if ((req->method == RTX_HTTPD_M_POST) && (req->content_length>0)) {
		char *post = (char*)malloc(req->content_length+1);
		if (!post) {
			return rtx_error("rtx_httpd_read_request: alloc post failed");
		}
		post[0] = post[req->content_length] = 0;
		rtx_inet_read(req->sock, post, req->content_length, NULL);
		PRINTF("POST line: '%s'",post);
		if (rtx_httpd_parse_post(req,post,req->content_length)) {
			free(post);
			return rtx_error("rtx_httpd_read_request: parse_post failed");
		}
		free(post);
	}
	
	return 0;
}
	


static void * connection_handler(void * p, void * arg)
{
	int r;
    RtxInetTcpClient * clnt = (RtxInetTcpClient *) p;
	RtxHttpd * s = (RtxHttpd*)arg;
	RtxHttpdReq req;

	req.sock = clnt->sock;
	req.user_data = s->user_data;
#if 1
	/* these options are set for any TCP/IP socket by the inet module
	 * We don't want them for HTTP since they brake the include function 
	 * of PHP... */
	struct linger lg = {1,1};
	int optval = 0;
	if (setsockopt (req.sock->sockfd, IPPROTO_TCP, TCP_NODELAY, 
				&optval, sizeof (optval)) == -1) {
		perror("Warning, setting option TCP_NODELAY failed:");
	}
	if (setsockopt (req.sock->sockfd, SOL_SOCKET, SO_LINGER, 
				&lg, sizeof (lg)) == -1) {
		perror("Warning, setting option SO_KEEPALIVE failed:");
	}
#endif
	req.persistent = 1;
	PRINTF("Connection initiated (%d)\n",req.sock->sockfd);
	while (!s->done) {
		if (rtx_inet_wait_data(clnt->sock,1.1)) {
			continue;
		}

		
		
		req.url = NULL;
		req.headers = rtx_hash_alloc(32, rtx_hash_deep_string_keys, rtx_hash_deep_string_values, (rtx_error_t)NULL);
		req.args = rtx_hash_alloc(32, rtx_hash_deep_string_keys, rtx_hash_deep_string_values, (rtx_error_t)NULL);
		req.post = rtx_hash_alloc(32, rtx_hash_deep_string_keys, rtx_hash_deep_string_values, (rtx_error_t)NULL);

		req.content_length = 0;
		req.outputused = 0;

		if ((r=rtx_httpd_read_request(clnt->sock,&req)) != 0) {
			rtx_hash_free(req.headers);
			rtx_hash_free(req.args);
			rtx_hash_free(req.post);
			free(req.url);
			free(req.outputbuf);
			if (r<0) return rtx_error_null("connection_handler: rtx_httpd_read_request failed.");
			return NULL;
		}

		if (s->handler) {
			req.outputbuf = (char*)malloc(MAX_URL_LENGTH);
			req.outputsize = MAX_URL_LENGTH;
			req.outputused = 0;

			if ((*s->handler)(&req)) {
				rtx_hash_free(req.headers);
				rtx_hash_free(req.args);
				rtx_hash_free(req.post);
				free(req.url);
				free(req.outputbuf);
				return rtx_error_null("connection_handler: handler failed.");
			}
			free(req.outputbuf);
			req.outputbuf = NULL;
		}

		

		
		rtx_hash_free(req.headers);
		rtx_hash_free(req.args);
		rtx_hash_free(req.post);
		free(req.url);

		if (!req.persistent) {
			break;
		}
	}
	PRINTF("Connection deleted (%d)\n",req.sock->sockfd);
	return NULL;
}


int rtx_httpd_server_set_user_data(RtxHttpd *s, void *data)
{
	if (!s) {
		return rtx_error("rtx_httpd_server_set_user_data: server is not initialised");
	}
	s->user_data = data;
	return 0;
}

RtxHttpd * rtx_httpd_server_init(const char * localhost,
		int port, int debug, int (*handler)(RtxHttpdReq *r))
{
	RtxHttpd * server = (RtxHttpd*)malloc(sizeof(RtxHttpd));
	if (!server) {
		return rtx_error_null("rtx_httpd_server_init: failed to allocate server struct");
	}

	if (!port) port = 80;
	server->done = 0;
	server->port = port;
	server->debug = debug;
	server->handler = handler;

	server->server = rtx_inet_init(RTX_INET_TCP_SERVER,
			localhost,server->port,NULL,0,connection_handler,NULL,server);
	if (!server->server) {
		free(server);
		return rtx_error_null("rtx_httpd_server_init: rtx_inet_init failed");
	}
	return server;
}

/**
 *  @return 0:ok  -1:failed
 */
int rtx_httpd_server_destroy(RtxHttpd *s)
{

	s->done = 1;
	rtx_timer_sleep(0.5);
	rtx_inet_done(s->server);
	free(s);
	
	return 0;
}

static int rtx_httpd_reply_addline(RtxHttpdReq *req,const char *line, unsigned int n)
{
	if (req->outputused+n+1 >= req->outputsize) {
		req->outputsize = req->outputused + n + 1024;
		req->outputbuf = (char*)realloc(req->outputbuf,req->outputsize);
		if (!req->outputbuf)
			return rtx_error("rtx_httpd_reply_addline: realloc failed");
	}
	memcpy(req->outputbuf+req->outputused,line,n);
	req->outputused += n;
	req->outputbuf[req->outputused] = 0;
	return 0;
}
	

int rtx_httpd_reply_send_status(RtxHttpdReq *req, 
		int status, const char * text)
{
	char line[256];
	const char * stdtext = rtx_httpd_std_error(status);
	if (text) {
		snprintf(line,255,"HTTP/%d.%d %3d %s:%s%s",
				req->http_major,req->http_minor,
				status, stdtext,text,crlf);
	} else {
		snprintf(line,255,"HTTP/%d.%d %3d %s%s",
				req->http_major,req->http_minor,
				status, stdtext,crlf);
	}
	rtx_httpd_reply_addline(req,line,strlen(line));
	return 0;

}


int rtx_httpd_reply_send_header(RtxHttpdReq *req, 
		const char * header, const char * value)
{
	char line[MAX_URL_LENGTH];
	if (!header) 
		return rtx_error("rtx_httpd_reply_send_header: NULL header");
	if (value) {
		snprintf(line,MAX_URL_LENGTH-1,"%s: %s%s",
				header,value,crlf);
	} else {
		snprintf(line,MAX_URL_LENGTH-1,"%s: %s",
				header,crlf);
	}
	rtx_httpd_reply_addline(req,line,strlen(line));
	return 0;
}

int rtx_httpd_reply_flush_headers(RtxHttpdReq *req)
{
	rtx_httpd_reply_addline(req,crlf,2);
	if (rtx_inet_write(req->sock,req->outputbuf,req->outputused,NULL) != req->outputused) 
		return rtx_error("rtx_httpd_reply_send_body: flush_buffer failed");
	return 0;
}

unsigned int rtx_httpd_reply_send_body(RtxHttpdReq *req,
		const void * body, size_t len)
{
	int r;
	char length[16];
	if (req->persistent) {
		r = rtx_httpd_reply_send_header(req,"Connection","Keep-Alive");
	} else {
		r = rtx_httpd_reply_send_header(req,"Connection","Close");
	}
	if (r) return rtx_error("rtx_httpd_reply_send_status: send_header failed");
	sprintf(length,"%d",len);
	if (rtx_httpd_reply_send_header(req,"Content-Length",length))
		return rtx_error("rtx_httpd_reply_send_body: send_header failed");
	rtx_httpd_reply_addline(req,crlf,2);
	rtx_httpd_reply_addline(req,body,len);
	if (rtx_inet_write(req->sock,req->outputbuf,req->outputused,NULL) != req->outputused) 
		return rtx_error("rtx_httpd_reply_send_body: flush_buffer failed");
	return 0;
}


int rtx_httpd_reply_send_allbody(RtxHttpdReq *req,
		const void * body, size_t len, double timeout)
{
	RtxTime ts0,ts1;
	double t0,t1;
	const char * bit;
	char length[16];
	unsigned int written = 0;
	int r;
	if (req->persistent) {
		r = rtx_httpd_reply_send_header(req,"Connection","Keep-Alive");
	} else {
		r = rtx_httpd_reply_send_header(req,"Connection","Close");
	}
	if (r) return rtx_error("rtx_httpd_reply_send_status: send_header failed");

	sprintf(length,"%d",len);
	if (rtx_httpd_reply_send_header(req,"Content-Length",length))
		return rtx_error("rtx_httpd_reply_send_body: send_header failed");
	rtx_httpd_reply_addline(req,crlf,2);
	rtx_httpd_reply_addline(req,body,len);
	bit = req->outputbuf;
	len = req->outputused;
	rtx_time_get(&ts0); t0 = rtx_time_to_double(&ts0);
	while (written != len) {
		rtx_time_get(&ts1); t1 = rtx_time_to_double(&ts1);
		if ((t1-t0) > timeout) 
			return rtx_error("rtx_httpd_reply_send_body: timeout while sending body");
		int ret = rtx_inet_write(req->sock,bit,len-written,NULL);
		if (ret < 0) 
			return rtx_error("rtx_httpd_reply_send_body: inet_write failed");
		written += ret; 
		bit += ret;
	}
	return 0;
}

#define MIN(X,Y) (((X)<(Y))?(X):(Y))
int rtx_httpd_send_file(RtxHttpdReq *r, 
		const char * path, const char * mimetype)
{
	int fd;
	unsigned char *bytes;
	struct stat st;
	unsigned int len,sent,bytesize=200000;
	r->persistent = 0;

	fd = open(path, O_RDONLY);
	if (fd < 0) {
		return rtx_httpd_send_error(r,404,"File not found");
	}

	if (stat(path,&st)) {
		close(fd);
		return rtx_httpd_send_error(r,404,"File properties cannot be found");
	}
	len = st.st_size;

	bytes = (unsigned char*)malloc(MIN(len,bytesize));
	if (!bytes) {
		close(fd);
		return rtx_httpd_send_error(r,404,"Malloc failed");
	}

	PRINTF("Sending file '%s'\n",path);

	if (rtx_httpd_reply_send_status(r,200,NULL)) {
		close(fd); free(bytes);
		return rtx_error("rtx_httpd_send_file: send_status failed");
	}
	if (mimetype) {
		rtx_httpd_reply_send_header(r, "Content-Type", mimetype);
	} else {
		rtx_httpd_reply_send_header(r, "Content-Type", "application/octet-stream" );
	}
	rtx_httpd_reply_send_header(r, "Connection", "Close");

	if (rtx_httpd_reply_flush_headers(r)) {
		close(fd); free(bytes);
		return rtx_error("rtx_httpd_send_file: flush_headers failes");
	}

	sent = 0;
	while (sent != len) {
		int readbytes,writtenbytes,bytestoread,thissend;
		bytestoread = MIN(len-sent, bytesize);
		readbytes = read(fd, bytes, bytestoread);
		PRINTF("R %d: %d/%d\n",readbytes,sent+readbytes,len);
		if (readbytes < 0) {
			close(fd); free(bytes);
			PRINTF("Read %d Expected %d\n",readbytes,bytestoread);
			return rtx_error_errno("rtx_httpd_send_file: IO error while reading file");
		}
		thissend = 0;
		while (thissend < readbytes) {
			writtenbytes = rtx_inet_write(r->sock,(char*)bytes,readbytes,NULL);
			PRINTF("W %d: %d/%d\n",writtenbytes,thissend+writtenbytes,readbytes);
			if (writtenbytes < 0 ) {
				close(fd); free(bytes);
				PRINTF("Written %d Expected %d\n",thissend,readbytes);
				return rtx_error("rtx_httpd_send_file: IO error while sending file");
			}
			thissend += writtenbytes;
		}
		sent += thissend;
	}
	close(fd); 
	free(bytes);

	return 0;
}


int rtx_httpd_send_error(RtxHttpdReq *r,
		int errorcode, const char * text)
{
	unsigned int len;
	r->persistent = 0;
	if (rtx_httpd_reply_send_status(r,errorcode,NULL)) 
		return rtx_error("rtx_httpd_send_error: send_status failed");
	if (text) {
		len = strlen(text);
		if (rtx_httpd_reply_send_body(r,text,len)!=len) 
			return rtx_error("rtx_httpd_send_error: send_status failed");
	} else {
		text = rtx_httpd_std_error(errorcode);
		len = strlen(text);
		if (rtx_httpd_reply_send_body(r,text,len)!=len) 
			return rtx_error("rtx_httpd_send_error: send_status failed");
	}

	return 0;
}


int rtx_httpd_def_handler(RtxHttpdReq * r)
{
	if(rtx_httpd_send_error(r,404,"Default Handler")!=0){
		return rtx_error("couldn't send error response");
	}
	return 0;
}

