/***********************************************************************
 * 
 * CSIRO Autonomous Systems Laboratory
 * Queensland Centre for Advanced Technologies
 * PO Box 883, Kenmore, QLD 4069, Australia
 * http://www.ict.csiro.au/
 *  
 * Copyright (c) CSIRO 
 ***********************************************************************/

#include <stdlib.h>
#include <string.h>

#include "rtx/defines.h"
#include "rtx/error.h"
#include "rtx/errors.h"
#include "rtx/hash.h"
#include "rtx/memory.h"
#include "rtx/mutex.h"

static char rcsid[] RTX_UNUSED = "$Id: hash.c 3083 2008-05-15 12:37:34Z roy029 $";

typedef struct entry *entry_t;
struct entry {
  void *key;
  void *value;
  entry_t next;
};

struct rtx_hash {
  int             capacity;
  int             used;
  RtxMutex        mutex;
  entry_t         *table;

  rtx_hash_key_hooks_t key;
  rtx_hash_value_hooks_t value;
};


/* Shallow hooks. */
static void *
shallow_copy(void *pointer, rtx_error_t error) {
  return pointer;
}

static void 
shallow_free(void *pointer) {
  return;
}

/* Deep hooks. */
static void
deep_free(void *pointer) {
  free(pointer);
}

/* String hooks. */
static unsigned int
string_key_hash(void *key) {
  char *s = (char *)key;
  unsigned int h = 0;

  while (*s) {
    h = 2 * h + *s;
    s++;
  }

  return h;
}

static void
string_key_error(void *key, rtx_error_t error) {
  RTX_ERROR_PUSH_CUSTOM(error, RTX_FORMAT_KEY_NOT_FOUND "%s", (char *)key);
}

static int
string_key_equals(void *keyA, void *keyB) {
  return strcmp(keyA, keyB);
}

static void *
deep_string_copy(void *key, rtx_error_t error) {
  void *result;

  return RTX_STRDUP(result, key, error);
}

/* Int hooks. */
static unsigned int
int_key_hash(void *key) {
  return (unsigned int)key;
}

static int
int_key_equals(void *keyA, void *keyB) {
  return ((int)keyA != (int)keyB);
}


static void
int_key_error(void *key, rtx_error_t error) {
  RTX_ERROR_PUSH_CUSTOM(error, RTX_FORMAT_KEY_NOT_FOUND "%i", (int)key);
}


rtx_hash_key_hooks_t rtx_hash_shallow_string_keys = &(struct rtx_hash_key_hooks){
  shallow_copy,
  shallow_free,
  string_key_hash,
  string_key_equals,
  string_key_error
};

rtx_hash_key_hooks_t rtx_hash_deep_string_keys = &(struct rtx_hash_key_hooks){
  deep_string_copy,
  deep_free,
  string_key_hash,
  string_key_equals,
  string_key_error
};

rtx_hash_key_hooks_t rtx_hash_int_keys = &(struct rtx_hash_key_hooks){
  shallow_copy,
  shallow_free,
  int_key_hash,
  int_key_equals,
  int_key_error
};

rtx_hash_value_hooks_t rtx_hash_shallow_values = &(struct rtx_hash_value_hooks){
  shallow_copy,
  shallow_free
};

rtx_hash_value_hooks_t rtx_hash_deep_string_values = &(struct rtx_hash_value_hooks){
  deep_string_copy,
  deep_free
};

rtx_hash_t
rtx_hash_alloc(int capacity, 
	       rtx_hash_key_hooks_t key_hooks, rtx_hash_value_hooks_t value_hooks,
	       rtx_error_t error) {
  rtx_hash_t hash;

  if (! RTX_CALLOC(hash, sizeof(struct rtx_hash), 1, error)) {
    return NULL;
  }

  hash->key = key_hooks;
  hash->value = value_hooks;

  hash->capacity = capacity ? capacity : RTX_HASH_DEFAULT_CAPACITY;
  hash->used = 0;

  if (! RTX_CALLOC(hash->table, sizeof(struct entry), hash->capacity, error)) {
    goto error;
  }

  if (rtx_mutex_init(&hash->mutex, 0, 0) == NULL) {
    RTX_ERROR_PUSH_CUSTOM(error, "failed to create mutex");
    goto error;
  }

  return hash;
  
 error:
  if (hash) free(hash->table);
  free (hash);

  return NULL;
}


int rtx_hash_used(rtx_hash_t hash)
{
  return hash->used;
}


void
rtx_hash_clear(rtx_hash_t hash)
{
  entry_t entry, next;
  int i;

  for (i = 0; i < hash->capacity; i++) {
    entry = hash->table[i];
    while (entry) {
      next = entry->next;

      hash->key->free(entry->key);
      hash->value->free(entry->value);
      free(entry);

      entry = next;
    }
    hash->table[i] = NULL;
  }

  hash->used = 0;
}


void
rtx_hash_free(rtx_hash_t hash) {
  rtx_hash_clear(hash);
  
  rtx_mutex_destroy(&hash->mutex);
  free(hash->table);
  free(hash);
}


void *
rtx_hash_find(rtx_hash_t hash, void *key, rtx_error_t error)
{
  entry_t entry;
  unsigned int h;

  h = hash->key->hash(key);
  
  entry = hash->table[h % hash->capacity];

  while (entry) {

    if (hash->key->equals(key, entry->key) == 0) {
      return entry->value;
    }

    entry = entry->next;
  }

  hash->key->error(key, error);
  return NULL;
}


int
rtx_hash_set(rtx_hash_t hash, void *key, void *value,
	     rtx_error_t error) {
  entry_t new_entry = NULL;
  entry_t entry;
  unsigned int h;

  if (rtx_mutex_lock(&hash->mutex) == -1) {
    RTX_ERROR_PUSH_CUSTOM(error, "failed to lock mutex");
    return -1;
  }

  /* See if the key already exists. */
  h = hash->key->hash(key) % hash->capacity;

  entry = hash->table[h];
  while (entry) {

    if (hash->key->equals(key, entry->key) == 0) {
      break;
    }

    entry = entry->next;
  }

  if (entry) {

    /* Free the old value. */
    hash->value->free(entry->value);

  } else {

    /* Construct the new entry. */
    if (! RTX_CALLOC(new_entry, sizeof(struct entry), 1, error)) {
      goto error;
    }

    if (! (new_entry->key = hash->key->copy(key, error))) {
      RTX_PUSH_KEY_NOT_COPIED(error);
      goto error;
    }
    
    /* Put the new entry at the start of the list. */
    new_entry->next = hash->table[h];
    hash->table[h] = new_entry;
    hash->used++;
    entry = new_entry;
  }

  if (! (entry->value = hash->value->copy(value, error))) {
    RTX_PUSH_VALUE_NOT_COPIED(error);
    goto error;
  }

  if (rtx_mutex_unlock(&hash->mutex) == -1) {
    RTX_ERROR_PUSH_SYSCALL(error, "failed to unlock mutex");
  }
  
  return 0;

 error:
  if (new_entry) hash->key->free(new_entry->key);
  if (new_entry) hash->value->free(new_entry->value);
  rtx_mutex_unlock(&hash->mutex);
  return -1;
}


int
rtx_hash_delete(rtx_hash_t hash, void *key,
		rtx_error_t error) {
  entry_t entry, previous;
  unsigned int h;
  int ret;

  if (rtx_mutex_lock(&hash->mutex) == -1) {
    RTX_ERROR_PUSH_CUSTOM(error, "failed to lock mutex");
    return -1;
  }

  h = hash->key->hash(key) % hash->capacity;

  previous = NULL;
  entry = hash->table[h];

  ret = -1;
  while (entry) {
    if (hash->key->equals(key, entry->key) == 0) {

      if (previous) {
	previous->next = entry->next;
      } else {
	hash->table[h] = entry->next;
      }

      hash->key->free(entry->key);
      hash->value->free(entry->value);
      free(entry);

      ret = 0;
      hash->used--;
      break;
    }

    previous = entry;
    entry = entry->next;
  }

  if (rtx_mutex_unlock(&hash->mutex) == -1) {
    RTX_ERROR_PUSH_CUSTOM(error, "failed to unlock mutex");
    return -1;
  }

  return ret;
}


void
rtx_hash_iter(rtx_hash_t hash,  rtx_hash_iterator_f_t iterator, void *context) {
  entry_t entry;
  int i, j, stop_iter;

  if (rtx_mutex_lock(&hash->mutex) == -1) {
    return;
  }

  j = 0;
  stop_iter = 0;
  for (i = 0; i < hash->capacity; i++) {
    entry = hash->table[i];
    while (entry) {

      iterator(entry->key,
	       entry->value,
	       ++j, hash->used,
	       &stop_iter,
	       context);

      /* Break out of the inner loop. */
      if (stop_iter) {
	break;
      }

      entry = entry->next;
    }

    /* Break out of the outer loop. */
    if (stop_iter) {
      break;
    }
  }

  if (rtx_mutex_unlock (&hash->mutex) == -1) {
  }
}


