/***********************************************************************
 * 
 * CSIRO Autonomous Systems Laboratory
 * Queensland Centre for Advanced Technologies
 * PO Box 883, Kenmore, QLD 4069, Australia
 * http://www.ict.csiro.au/
 *  
 * Copyright (c) CSIRO 
 ***********************************************************************/

/**
 * \file getopt.c
 * \brief command-line options processing module
 * \author Peter Corke and Pavan Sikka
 *
 * A module for processing command-line options.
 *
 * This module packages up command-line option processing
 * into a user-defined option list (array) and a single function 
 * call (\ref rtx_getopt_cmd).
 * Each individual option is defined by a structure (\ref RtxGetopt).
 *
 * The function used to process the options takes build date,
 * build time and version strings as arguments. The build
 * information can be obtained by the __DATE__ and __TIME__
 * macros and so a macro (\ref RTX_GETOPT_CMD) is provided as a wrapper
 * around this function.
 *
 * This module provides the following builtin options:
 * - -help\n
 *     print help and exit
 * - -config <filename>\n
 *     process options from the config file. The config file is
 *     preprocessed before the options are read.
 * - -D <definition>\n
 *     pass the definition to the pre-processor. This option can
 *     be used multiple times to specify several definitions
 * - -verbose\n
 *     specify the verbose level. This option can be used multiple
 *     times to specify greater levels of verbosity. The user program
 *     needs to use the corresponding function (\ref rtx_getopt_get_verbose)
 *     to access the value of this option
 * - -store\n
 *     specify that the store interface should be used. The user program
 *     needs to use the corresponding function (\ref rtx_getopt_get_store)
 *     to access the value of this option
 * - -debug <module list>\n
 *     specify the modules for which debugging should be turned on. The
 *     user program needs to use the corresponding function 
 *     (\ref rtx_getopt_get_debug) to access the value of this option
 * - -priority <level>\n
 *     specify the priority at which the program should run. The user program
 *     needs to use the corresponding function (\ref rtx_getopt_get_priority)
 *     to access the value of this option
 * - -export\n
 *     specify the port for the RTX export module. The user program
 *     needs to use the corresponding function (\ref rtx_getopt_get_export)
 *     to access the value of this option
 * - -sample\n
 *     specify the sampling delay (a floating-point value in seconds). The
 *     user program needs to use the corresponding function 
 *     (\ref rtx_getopt_get_sample) to access the value of this option
 *     
 */

#include <stdio.h>
#include <stdarg.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>

#include "rtx/defines.h"
#include "rtx/getopt.h"
#include "rtx/string.h"

static char rcsid[] RTX_UNUSED = "$Id: getopt.c 3067 2008-05-15 03:41:30Z roy029 $";

#ifndef DOXYGEN_SHOULD_SKIP_THIS

int rtx_getopt_debug = 0;   /**< used to turn debugging on */
static int rtxGetoptProcessConfigFile = 1; /**< process config file */

/** 
 * flag indicating that user has requested help
 */
static int rtxGetoptNeedHelp = 0;

/**
 * configuration file
 */
static char * rtxGetoptConfigFile;

/**
 * pre-processor definitions
 */
static char ** rtxGetoptCppDefs;

/**
 * verbose option
 */
static int rtxGetoptVerbose;

/**
 * debug option
 */
static char * rtxGetoptDebug;

/**
 * priority
 */
static int rtxGetoptPriority;

/**
 * sample time
 */
static double rtxGetoptSampleTime;

/**
 * store
 */
static int rtxGetoptStore;

/**
 * export program number
 */
static int rtxGetoptExportProgramNumber;

/**
 * builtin options for processing config files
 */
static RtxGetopt rtxGetoptBuiltinConfigOpts[] = {
  {"config", "configuration file", 
   {
     {RTX_GETOPT_STR, &rtxGetoptConfigFile, "file name"},
     RTX_GETOPT_END_ARG
   }
  },
  {"D", "CPP definitions", 
   {
     {RTX_GETOPT_STRCAT, &rtxGetoptCppDefs, "CPP definition"},
     RTX_GETOPT_END_ARG
   }
  },
  RTX_GETOPT_END
};

#define RTX_GETOPT_BUILTIN_CONFIG_ARG_CONFIG      0
#define RTX_GETOPT_BUILTIN_CONFIG_ARG_D           1
#define RTX_GETOPT_BUILTIN_CONFIG_ARG_END         2

#define RTX_GETOPT_NUM_BUILTIN_CONFIG_ARGS \
    (sizeof(rtxGetoptBuiltinConfigOpts)/sizeof(RtxGetopt))

/**
 * builtin options available to all programs using this module
 */
static RtxGetopt rtxGetoptBuiltinOpts[] = {
  {"help", "Print help", 
   {
     {RTX_GETOPT_SET, &rtxGetoptNeedHelp, ""},
     RTX_GETOPT_END_ARG
   }
  },
  {"verbose", "Verbosity", 
   {
     {RTX_GETOPT_INCR, &rtxGetoptVerbose, ""},
     RTX_GETOPT_END_ARG
   }
  },
  {"store", "Activate store interface", 
   {
     {RTX_GETOPT_SET, &rtxGetoptStore, ""},
     RTX_GETOPT_END_ARG
   }
  },
  {"debug", "Debug level", 
   {
     {RTX_GETOPT_STR, &rtxGetoptDebug, "module-list"},
     RTX_GETOPT_END_ARG
   }
  },
  {"priority", "Process priority", 
   {
     {RTX_GETOPT_INT, &rtxGetoptPriority, "priority"},
     RTX_GETOPT_END_ARG
   }
  },
  {"export", "Export program number", 
   {
     {RTX_GETOPT_INT, &rtxGetoptExportProgramNumber, "program number"},
     RTX_GETOPT_END_ARG
   }
  },
  {"sample", "Sampling delay", 
   {
     {RTX_GETOPT_DOUBLE, &rtxGetoptSampleTime, "sampling interval"},
     RTX_GETOPT_END_ARG
   }
  },
  RTX_GETOPT_END
};

#define RTX_GETOPT_BUILTIN_ARG_HELP      0
#define RTX_GETOPT_BUILTIN_ARG_VERBOSE   1
#define RTX_GETOPT_BUILTIN_ARG_STORE     2
#define RTX_GETOPT_BUILTIN_ARG_DEBUG     3
#define RTX_GETOPT_BUILTIN_ARG_PRIORITY  4
#define RTX_GETOPT_BUILTIN_ARG_EXPORT    5
#define RTX_GETOPT_BUILTIN_ARG_SAMPLE    6
#define RTX_GETOPT_BUILTIN_ARG_END       7

#define RTX_GETOPT_NUM_BUILTIN_ARGS \
    (sizeof(rtxGetoptBuiltinOpts)/sizeof(RtxGetopt))

/**
 * array to indicate if an corresponding built-in config arg has been specified
 * on the command-line
 */
static int rtxGetoptConfigArgFlag[RTX_GETOPT_NUM_BUILTIN_CONFIG_ARGS];

/**
 * array to indicate if an corresponding built-in arg has been specified
 * on the command-line
 */
static int rtxGetoptArgFlag[RTX_GETOPT_NUM_BUILTIN_ARGS];

/**
 * type names for option types
 */
static char * rtx_getopt_typename[] = {
    RTX_GETOPT_NONE,    
    RTX_GETOPT_CHAR,    
    RTX_GETOPT_INT,     
    RTX_GETOPT_ENUM,    
    RTX_GETOPT_DOUBLE,  
    RTX_GETOPT_STR,     
    RTX_GETOPT_SET,     
    RTX_GETOPT_INCR,    
    RTX_GETOPT_CLEAR,   
    RTX_GETOPT_STRCAT,  
    RTX_GETOPT_FUNC
};

/**
 * mark processed arguments
 */
static char argProcessed[256];

#define debug(fmt, args...) \
    if (rtx_getopt_debug) \
        fprintf (stderr, fmt, ## args)

#define RTX_GETOPT_NO_COMMAND           -1
#define RTX_GETOPT_AMBIGUOUS_COMMAND    -2

#define RTX_GETOPT_ENUM_MAX_TOKS              32
#define RTX_GETOPT_ENUM_MAX_TOK_CHARS         32

#endif

/* forward */

int rtx_getopt_get_num_strings (char ** p);
char * rtx_getopt_get_typename (RtxGetoptType type);
int rtx_getopt_get_type (const char * typename, 
			 char enumtoks[RTX_GETOPT_ENUM_MAX_TOKS][RTX_GETOPT_ENUM_MAX_TOK_CHARS],
			 int * numToks);
void rtx_getopt_exit (RtxGetopt * opt, int argc, char * argv[],
		      const char * date, const char * time, const char * version,
		      const char * desc, int exitcode, const char * fmt, ...);
int rtx_getopt_check (RtxGetopt * opt);

/** 
 * Find the minimum common length of two string
 * */
static int 
rtx_getopt_common_prefix_len(const char * s1, const char * s2)
{
	unsigned int len = 0;
	while (*s1 && (*s1 == '-')) s1++;
	while (*s2 && (*s2 == '-')) s2++;
	while (*s1 && *s2) {
		if (strncmp(s1,s2,1)) 
			break;
		len ++;	s1++; s2++;
	}
	return len;
}

/**
 * Look at all the options and find the minimum length of string to use 
 * when trying to match them. Update opt->min_cmp_len.
 *
 * @return 1 on error (ambiguous option), 0 if everything went OK
 */
static int 
rtx_getopt_compute_prefix_len(RtxGetopt ** optlist, unsigned int len)
{
	unsigned int i1, i2;
	for (i1=0;i1<len;i1++) {
		RtxGetopt * o1 = optlist[i1];
		while (o1->name != NULL) {
			o1->min_cmp_len = 1;
			for (i2=0;i2<len;i2++) {
				RtxGetopt * o2 = optlist[i2];
				while (o2->name != NULL) {
					if (o2 != o1) {
						unsigned int cmp_len = 1+rtx_getopt_common_prefix_len(o1->name,
								o2->name);
						if (cmp_len > o1->min_cmp_len) 
							o1->min_cmp_len = cmp_len;
						if (strlen(o1->name) < cmp_len) {
							fprintf(stderr,"Command line option '%s' conflicts with '%s'\n",o1->name,o2->name);
							return 1;
						}
					}
					o2++;
				}
			}
			/*printf("Option '%s', min comp len %d\n",o1->name,o1->min_cmp_len);*/
			o1++;
		}
	}
	return 0;						
}

/**
 * process command-line options
 *
 * Use the macro RTX_GETOPT_CMD (opt, argc, argv, rcsid, desc)
 *
 * @return -1 on error, 0 if a pre-defined option is specified, and index
 * of first unprocessed arg on success
 */
int 
rtx_getopt_cmd (
		RtxGetopt * opt,     /**< command-line options structure */
		int argc,            /**< number of command-line parms */
		char * argv[],       /**< command-line parms */
		const char * date,         /**< build-date */
		const char * time,         /**< build-time */
		const char * version,      /**< version string */
		const char * desc          /**< description string */
		)
{
	int i, j, numDefs = 0;
	FILE * fp;
	char b[1024];
	int configac = 128;
	char * configav[128];
	RtxGetopt * allopts[] = { rtxGetoptBuiltinConfigOpts, rtxGetoptBuiltinOpts, opt};
	int ac, newac;
	char ** av;

	debug ("rtx_getopt_cmd: built-in args = %d\n", 
			RTX_GETOPT_NUM_BUILTIN_ARGS);
	/* Donot allow user to over-ride builtin options or to provide
	 * ambiguous options.
	 */
	if (rtx_getopt_check (opt))
		rtx_getopt_exit (opt, argc, argv, date, time, version, desc, 2,
				"rtx_getopt_cmd: error in user-defined options\n");
	if (rtx_getopt_compute_prefix_len(allopts,3)) {
		rtx_getopt_exit (opt, argc, argv, date, time, version, desc, 2,
				"rtx_getopt_cmd: error while finding smallest prefix\n");
	}
	debug ("rtx_getopt_cmd: processing for config file\n");
	/* First, process the built-in config options */
	if (rtx_getopt (rtxGetoptBuiltinConfigOpts, argc, argv, date, time, 
				version, desc, rtxGetoptConfigArgFlag) == -1)
		rtx_getopt_exit (opt, argc, argv, date, time, version, desc, 2,
				"rtx_getopt_cmd: error processing built-in "
				"config options");

	if (rtxGetoptProcessConfigFile) {
		/* Process config file if specified */
		if (rtxGetoptConfigFile != NULL) {
			/* open file through CPP */
			debug ("rtx_getopt_cmd: processing config file <%s>\n", rtxGetoptConfigFile);
			numDefs = rtx_getopt_get_num_strings (rtxGetoptCppDefs);
#ifdef sparc_solaris
			sprintf (b, "/usr/local/bin/cpp -P ");
#else
			sprintf (b, "/usr/bin/cpp -P ");
#endif
			for (i=0; i<numDefs; i++) {
				strcat (b, "-D");
				strcat (b, rtxGetoptCppDefs[i]);
				strcat (b, " ");
			}
			strcat (b, rtxGetoptConfigFile);
			if ((fp = popen (b, "r")) == NULL)
				rtx_getopt_exit (opt, argc, argv, date, time, version, desc, 2,
						"rtx_getopt_cmd: failed to open config file "
						"(%s: %s)", b, strerror (errno));

			/* convert file data to argc/argv list */
			ac = newac = configac - 1;
			av = &(configav[1]);
			configav[0] = rtxGetoptConfigFile;
			while (fgets (b, 1024, fp) != NULL) {
				rtx_string_tokenize (b, &newac, av);
				for (j=0; j<newac; j++)
					if ((av[j] = strdup (av[j])) == NULL)
						rtx_getopt_exit (opt, argc, argv, date, time, 
								version, desc, 2,
								"rtx_getopt_cmd: no memory");
				av = &(av[newac]);
				ac = ac - newac;
				newac = ac;
			}
			configac = configac - ac;
			pclose (fp);

			/* use rtx_getopt() to process the argc/argv list */
			/* First, process for built-in options */
			if (rtx_getopt (rtxGetoptBuiltinOpts, configac, configav, 
						date, time, version, desc, rtxGetoptArgFlag) == -1)
				rtx_getopt_exit (opt, argc, argv, date, time, version, desc, 2,
						"rtx_getopt_cmd: error processing built-in "
						"options");
			/* Next, process for user-defined options */
			if (rtx_getopt (opt, configac, configav, date, time, version,
						desc, NULL) == -1)
				rtx_getopt_exit (opt, argc, argv, date, time, version, desc, 2,
						"rtx_getopt_cmd: error processing "
						"user-defined options");
		}
	}

	/* Process built-in options */
	debug ("rtx_getopt_cmd: processing built-in options\n");
	if (rtx_getopt (rtxGetoptBuiltinOpts, argc, argv, date, time, 
				version, desc, rtxGetoptArgFlag) == -1)
		rtx_getopt_exit (opt, argc, argv, date, time, version, desc, 2,
				"rtx_getopt_cmd: error processing "
				"built-in options");


	if (rtxGetoptNeedHelp)
		rtx_getopt_exit (opt, argc, argv, date, time, version, desc, 1,
				"");

	/* Process user-specified options */
	debug ("rtx_getopt_cmd: processing command-line\n");
	if (rtx_getopt (opt, argc, argv, date, time, version, desc, NULL) == -1)
		rtx_getopt_exit (opt, argc, argv, date, time, version, desc, 2,
				"rtx_getopt_cmd: error processing "
				"user-defined options");

	debug ("rtx_getopt_cmd: arg flags = ");
	for (i=0; i<RTX_GETOPT_NUM_BUILTIN_ARGS; i++)
		debug ("%d ", rtxGetoptArgFlag[i]);
	debug ("\n");

	j = 0;
	for (i=argc; i>0; i--)
		if (argProcessed[i] == 1) {
			j = i;
			break;
		}
	for (i=1; i<=j; i++)
		if (argProcessed[i] != 1) {
			rtx_getopt_exit (opt, argc, argv, date, time, version, desc, 2,
					"rtx_getopt_cmd: unknown option <%s>",
					argv[i]);
		}

	return (j+1);
}

/**
 * process command-line for options specified in structure provided
 *
 * @return -1 on error, 0 if OK
 */
int 
rtx_getopt (
	    RtxGetopt * opt,     /**< command-line options structure */
	    int argc,            /**< number of command-line parms */
	    char * argv[],       /**< command-line parms */
	    const char * date,         /**< build-date */
	    const char * time,         /**< build-time */
	    const char * version,      /**< version string */
	    const char * desc,         /**< description string */
	    int * argFlag        /**< flag set if arg specified */
	    )
{
	int n = 0, i, j, k, l, m, cmd;
	int numStrcatArgs = 0;
	char enumtoks[RTX_GETOPT_ENUM_MAX_TOKS][RTX_GETOPT_ENUM_MAX_TOK_CHARS];
	int numToks = 10;
	int typenum;
	const char * p, * q;
	int isPossibleOption;

	if (opt == NULL)
		return (argc);

	for (i=1; i<argc; ) {
		debug ("processing argv[%d] = %s\n", i, argv[i]);
		for (n=0; opt[n].name != NULL; n++);
		debug ("number of options = %d\n", n);
		cmd = RTX_GETOPT_NO_COMMAND;
		p = argv[i];
		isPossibleOption = 0;
		if (p[0] == '-') {
			p++;
			isPossibleOption = 1;
		}
		if (p[0] == '-') {
			p++;
			isPossibleOption = 1;
		}
		if (isPossibleOption == 0) {
			i++;
			continue;
		}
		for (j=0; j<n; j++) {
			unsigned int len = strlen(p);
			q = opt[j].name;
			if (q[0] == '-')
				q++;
			if (q[0] == '-')
				q++;
			if (strncmp (q, p, len) == 0) {
				if (len < opt[j].min_cmp_len) {
					cmd = RTX_GETOPT_AMBIGUOUS_COMMAND;
					break;
				}
				if (cmd == RTX_GETOPT_NO_COMMAND) {
					cmd = j;
				} else {
					cmd = RTX_GETOPT_AMBIGUOUS_COMMAND;
					break;
				}
			}
		}
		if (cmd == RTX_GETOPT_NO_COMMAND) {
			i++;
			continue;
		}
		if (cmd == RTX_GETOPT_AMBIGUOUS_COMMAND) {
			fprintf (stderr, "rtx_getopt: ambiguous option argv[%d] = [%s]\n",
					i, argv[i]);
			return (-1);
		}
		debug ("found option %s\n", opt[cmd].name);
		argProcessed[i] = 1;
		if (argFlag != NULL)
			argFlag[cmd] = 1;
		for (m=0; strcmp (opt[cmd].arg[m].type, RTX_GETOPT_NONE) != 0; m++);
		debug ("\tnumber of args = %d\n", m);
		for (k=i+1, j=0; j<m; j++) {
			typenum = rtx_getopt_get_type (opt[cmd].arg[j].type, enumtoks, &numToks);
			debug ("\ttype arg[%d] = <%s>[%d]\n", j, opt[cmd].arg[j].type, typenum);
			switch (typenum) {
				case RTX_GETOPT_TYPE_CHAR :
					if (k >= argc) {
						fprintf (stderr, "rtx_getopt: exhausted command-line parms "
								"while processing argv[%d] = [%s]\n", i, argv[i]);
						return (-1);
					}
					debug ("\t\targ %d %s [CHAR] = %d\n", j+1,
							opt[cmd].arg[j].name, * (char *) opt[cmd].arg[j].ptr);
					* (char *) opt[cmd].arg[j].ptr = argv[k][0];
					argProcessed[k] = 1;
					k++;
					break;
				case RTX_GETOPT_TYPE_INT :
					if (k >= argc) {
						fprintf (stderr, "rtx_getopt: exhausted command-line parms "
								"while processing argv[%d] = [%s]\n", i, argv[i]);
						return (-1);
					}
					debug ("\t\targ %d %s [INT] = %d\n", j+1,
							opt[cmd].arg[j].name, * (int *) opt[cmd].arg[j].ptr);
					* (int *) opt[cmd].arg[j].ptr = atoi (argv[k]);
					argProcessed[k] = 1;
					k++;
					break;
				case RTX_GETOPT_TYPE_ENUM :
					if (k >= argc) {
						fprintf (stderr, "rtx_getopt: exhausted command-line parms "
								"while processing argv[%d] = [%s]\n", i, argv[i]);
						return (-1);
					}
					debug ("\t\targ %d %s [INT] = %d\n", j+1,
							opt[cmd].arg[j].name, * (int *) opt[cmd].arg[j].ptr);
					* (int *) opt[cmd].arg[j].ptr = -1;
					for (l=0; l<numToks; l++)
						if (strcmp (argv[k], enumtoks[l]) == 0)
							* (int *) opt[cmd].arg[j].ptr = l;
					if (* (int *) opt[cmd].arg[j].ptr == -1) {
						fprintf (stderr, "rtx_getopt: invalid enum value [%s] "
								"while processing argv[%d] = [%s]\n", argv[k],
								i, argv[i]);
						return (-1);
					}
					argProcessed[k] = 1;
					k++;
					break;
				case RTX_GETOPT_TYPE_DOUBLE :
					if (k >= argc) {
						fprintf (stderr, "rtx_getopt: exhausted command-line parms "
								"while processing argv[%d] = [%s]\n", i, argv[i]);
						return (-1);
					}
					debug ("\t\targ %d %s [FLOAT] = %f\n", j+1,
							opt[cmd].arg[j].name, * (double *) opt[cmd].arg[j].ptr);
					* (double *) opt[cmd].arg[j].ptr = atof (argv[k]);
					argProcessed[k] = 1;
					k++;
					break;
				case RTX_GETOPT_TYPE_STR :
					if (k >= argc) {
						fprintf (stderr, "rtx_getopt: exhausted command-line parms "
								"while processing argv[%d] = [%s]\n", i, argv[i]);
						return (-1);
					}
					debug ("\t\targ %d %s [STR] = %s\n", j+1,
							opt[cmd].arg[j].name, * (char **) opt[cmd].arg[j].ptr);
					* (char **) opt[cmd].arg[j].ptr = strdup (argv[k]);
					if (* (char **) opt[cmd].arg[j].ptr == NULL) {
						fprintf (stderr, "rtx_getopt: strdup() failed "
								"while processing argv[%d] = [%s]\n", i, argv[i]);
						return (-1);
					}
					argProcessed[k] = 1;
					k++;
					break;
				case RTX_GETOPT_TYPE_STRCAT :
					if (k >= argc) {
						fprintf (stderr, "rtx_getopt: exhausted command-line parms "
								"while processing argv[%d] = [%s]\n", i, argv[i]);
						return (-1);
					}
					if ((* (char ***) opt[cmd].arg[j].ptr) == NULL) {
						debug ("\t\targ %d %s [STR] = 0 args\n", j+1,
								opt[cmd].arg[j].name);
						if (((* (char ***) opt[cmd].arg[j].ptr) = 
									calloc (2, sizeof (char *))) == NULL) {
							fprintf (stderr, "rtx_getopt: calloc() failed "
									"while processing argv[%d] = [%s]\n", i, argv[i]);
							return (-1);
						}
						numStrcatArgs = 1;
						(* (char ***) opt[cmd].arg[j].ptr) [1] = NULL;
					} else {
						numStrcatArgs = rtx_getopt_get_num_strings 
							((* (char ***) opt[cmd].arg[j].ptr));
						debug ("\t\targ %d %s [STR] = %d args\n", j+1,
								opt[cmd].arg[j].name, numStrcatArgs);
						numStrcatArgs++;
						if (((* (char ***) opt[cmd].arg[j].ptr) = 
									realloc ((* (char ***) opt[cmd].arg[j].ptr), 
										(numStrcatArgs+1) * sizeof (char *))) == NULL) {
							fprintf (stderr, "rtx_getopt: realloc() failed "
									"while processing argv[%d] = [%s]\n", i, argv[i]);
							return (-1);
						}
						(* (char ***) opt[cmd].arg[j].ptr) [numStrcatArgs] = NULL;
					}
					if (((* (char ***) opt[cmd].arg[j].ptr)[numStrcatArgs-1] = 
								strdup (argv[k])) == NULL) {
						fprintf (stderr, "rtx_getopt: strdup() failed "
								"while processing argv[%d] = [%s]\n", i, argv[i]);
						return (-1);
					}
					argProcessed[k] = 1;
					k++;
					break;
				case RTX_GETOPT_TYPE_SET :
					debug ("\t\targ %d %s [SET] = %d\n", j+1,
							opt[cmd].arg[j].name, * (int *) opt[cmd].arg[j].ptr);
					* (int *) opt[cmd].arg[j].ptr = 1;
					break;
				case RTX_GETOPT_TYPE_INCR :
					debug ("\t\targ %d %s [INCR] = %d\n", j+1,
							opt[cmd].arg[j].name, * (int *) opt[cmd].arg[j].ptr);
					(* (int *) opt[cmd].arg[j].ptr)++;
					break;
				case RTX_GETOPT_TYPE_CLEAR :
					debug ("\t\targ %d %s [CLEAR] = %d\n", j+1,
							opt[cmd].arg[j].name, * (int *) opt[cmd].arg[j].ptr);
					* (int *) opt[cmd].arg[j].ptr = 0;
					break;
				case RTX_GETOPT_TYPE_FUNC :
					debug ("\t\targ %d %s [FUNC] = %08x\n", j+1,
							opt[cmd].arg[j].name, (unsigned) opt[cmd].arg[j].ptr);
					break;
				default :
					fprintf (stderr, "rtx_getopt: argv[%d] = [%s] : option %s has"
							" unknown type %s\n", i, argv[i],
							opt[cmd].arg[j].name, opt[cmd].arg[j].type);
					break;

			}
		}
		i = k;
	}
	return (i);
}

/**
 * print an option on a line
 *
 */
void
rtx_getopt_printopt (
		     RtxGetopt * opt     /**< command-line options definition */
		     )
{
    int m, j, k, n = 0, p = 0;
    int numStrcatArgs = 0, typenum;
    char enumtoks[RTX_GETOPT_ENUM_MAX_TOKS][RTX_GETOPT_ENUM_MAX_TOK_CHARS];
    int numToks = 10;

    if (opt == NULL)
        return;

    for (m=0; strcmp (opt->arg[m].type, RTX_GETOPT_NONE) != 0; m++);
    fprintf (stderr, "    -%s%n", opt->name, &n);
    p += n;
    for (j=0; j<m; j++) {
        typenum = rtx_getopt_get_type (opt->arg[j].type, enumtoks, &numToks);
	switch (typenum) {
	    case RTX_GETOPT_TYPE_CHAR :
	    case RTX_GETOPT_TYPE_INT :
	    case RTX_GETOPT_TYPE_DOUBLE :
	    case RTX_GETOPT_TYPE_STR :
	    case RTX_GETOPT_TYPE_STRCAT :
	        fprintf (stderr, " <%s>%n", opt->arg[j].name, &n);
		p += n;
		break;
	    case RTX_GETOPT_TYPE_ENUM :
	        fprintf (stderr, " <%s>[%s]%n", opt->arg[j].name, opt->arg[j].type, &n);
		p += n;
	        break;
	    case RTX_GETOPT_TYPE_SET :
	    case RTX_GETOPT_TYPE_INCR :
	    case RTX_GETOPT_TYPE_CLEAR :
	    case RTX_GETOPT_TYPE_FUNC :
		break;
	    default :
		fprintf (stderr, "rtx_getopt_printopt: arg %d %s : unknown type %s\n", j+1,
			 opt->arg[j].name, opt->arg[j].type);
		break;
	}
    }
    n = 30 - p;
    for (j=0; j<n; j++)
        fprintf (stderr, " ");
    fprintf (stderr, " --- %s ", opt->help);
    if (m > 0)
	fprintf (stderr, "(");
    for (j=0; j<m; j++) {
	if (j > 0)
	    fprintf (stderr, " ");
        typenum = rtx_getopt_get_type (opt->arg[j].type, enumtoks, &numToks);
	switch (typenum) {
	    case RTX_GETOPT_TYPE_CHAR :
		fprintf (stderr, "%d", * (char *) opt->arg[j].ptr);
		break;
	    case RTX_GETOPT_TYPE_INT :
		fprintf (stderr, "%d", * (int *) opt->arg[j].ptr);
		break;
	    case RTX_GETOPT_TYPE_ENUM :
		fprintf (stderr, "%d", * (int *) opt->arg[j].ptr);
		break;
	    case RTX_GETOPT_TYPE_DOUBLE :
		fprintf (stderr, "%f", * (double *) opt->arg[j].ptr);
		break;
	    case RTX_GETOPT_TYPE_STR :
	        if ((* (char **) opt->arg[j].ptr) != NULL)
		    fprintf (stderr, "%s", (char *) (* (char **) opt->arg[j].ptr));
		break;
	    case RTX_GETOPT_TYPE_STRCAT :
	        if ((* (char ***) opt->arg[j].ptr) != NULL) {
		    numStrcatArgs = rtx_getopt_get_num_strings 
			                ((* (char ***) opt->arg[j].ptr));
		    for (k=0; k<numStrcatArgs-1; k++)
		        fprintf (stderr, "%s,", (char *) ((* (char ***) opt->arg[j].ptr)[k]));
		    fprintf (stderr, "%s", (char *) ((* (char ***) opt->arg[j].ptr)[k]));
		}
		break;
	    case RTX_GETOPT_TYPE_SET :
		fprintf (stderr, "%d", * (int *) opt->arg[j].ptr);
		break;
	    case RTX_GETOPT_TYPE_INCR :
		fprintf (stderr, "%d", * (int *) opt->arg[j].ptr);
		break;
	    case RTX_GETOPT_TYPE_CLEAR :
		fprintf (stderr, "%d", * (int *) opt->arg[j].ptr);
		break;
	    case RTX_GETOPT_TYPE_FUNC :
		break;
	    default :
		fprintf (stderr, "rtx_getopt_printopt: arg %d %s : unknown type %s\n", j+1,
			 opt->arg[j].name, opt->arg[j].type);
		break;
	}
    }
    if (m > 0)
	fprintf (stderr, ")");
    fprintf (stderr, "\n");
}

/**
 * print the command-line options
 *
 * Use the macro RTX_GETOPT_PRINT (opt, prog, rcsid, desc)
 *
 */
void
rtx_getopt_print (
		  RtxGetopt * opt,     /**< command-line options definition */
		  const char * progName,     /**< program name */
		  const char * buildDate,         /**< build-date */
		  const char * buildTime,         /**< build-time */
		  const char * buildVersion,      /**< version string */
		  const char * desc          /**< description string */
		  )
{
    int n = 0, i;
    
    fprintf (stderr, "%s:\n\n", progName);

    if (desc != NULL)
        fprintf (stderr, "%s\n\n", desc);

    if (opt != NULL) {
        for (n=0; opt[n].name != NULL; n++);
	if (n)
  	    fprintf (stderr, "User-defined options:\n");
	for (i=0; i<n; i++)
	    rtx_getopt_printopt (&opt[i]);
	fprintf (stderr, "\n");
    }
    fprintf (stderr, "Built-in options:\n");
    for (n=0; rtxGetoptBuiltinConfigOpts[n].name != NULL; n++);
    for (i=0; i<n; i++)
        rtx_getopt_printopt (&rtxGetoptBuiltinConfigOpts[i]);
    fprintf (stderr, "\n");
    for (n=0; rtxGetoptBuiltinOpts[n].name != NULL; n++);
    for (i=0; i<n; i++)
        rtx_getopt_printopt (&rtxGetoptBuiltinOpts[i]);
    fprintf (stderr, "\n");

    if (buildDate != NULL) {
        fprintf (stderr, "Built: %s ", buildDate);
	if (buildTime != NULL)
	    fprintf (stderr, "%s", buildTime);
	fprintf (stderr, "\n");
    }
    if (buildVersion != NULL)
        fprintf (stderr, "Version: %s\n\n", buildVersion);
    fprintf (stderr, "CSIRO Robotics Team\nICT Centre\n");
}

/**
 * print the version information
 *
 * Use the macro RTX_GETOPT_PRINT_VERSION (opt, prog, rcsid, desc)
 *
 */
void
rtx_getopt_print_version (
			  RtxGetopt * opt,     /**< command-line options definition */
			  char * progName,     /**< program name */
			  char * buildDate,         /**< build-date */
			  char * buildTime,         /**< build-time */
			  char * buildVersion,      /**< version string */
			  char * desc          /**< description string */
			  )
{
    fprintf (stderr, "%s:\n\n", progName);
    if (buildDate != NULL) {
        fprintf (stderr, "Built: %s ", buildDate);
	if (buildTime != NULL)
	    fprintf (stderr, "%s", buildTime);
	fprintf (stderr, "\n");
    }
    if (buildVersion != NULL)
        fprintf (stderr, "Version: %s\n", buildVersion);
}

/**
 * get number of elements in a list of strings
 *
 * @return number of elements
 */
int
rtx_getopt_get_num_strings (
			    char ** p /**< NULL-terminated list of strings */
			    )
{
    int n = 0;

    if (p == NULL)
        return (0);
    while (( *p != NULL) && (n < 100)) {
        n++; p++;
    }
    return (n);
}

char * 
rtx_getopt_get_typename (
			 RtxGetoptType type
			 )
{
    if ((type < RTX_GETOPT_TYPE_CHAR) || (type > RTX_GETOPT_TYPE_FUNC))
        return (NULL);
    return (rtx_getopt_typename[type]);
}

int 
rtx_getopt_get_type (
		     const char * typename,
		     char enumtoks[RTX_GETOPT_ENUM_MAX_TOKS][RTX_GETOPT_ENUM_MAX_TOK_CHARS],
		     //char *enumtoks[],
		     int * numToks
		     )
{
    int i;
    int maxNumToks = * numToks;
    const char * p, * q;
    int n = 0;

    for (i=0; i<sizeof(rtx_getopt_typename)/sizeof (char *); i++)
        if (strcmp (typename, rtx_getopt_typename[i]) == 0)
	    return (i);
    p = typename;
    while (1) {
        q = strchr (p, '|');
	if (q == NULL) {
	    if (n < maxNumToks)
	        strcpy (enumtoks[n], p);
	    n++;
	    break;
	}
	if (n >= maxNumToks)
	    continue;
	strncpy (enumtoks[n], p, q-p);
	enumtoks[n][q-p] = '\0';
	n++;
	p = q+1;
    }
    if (n <= maxNumToks)
        * numToks = n;
    if (n <= 1)
        return (-1);
    return (RTX_GETOPT_TYPE_ENUM);
}


/**
 * return the value of the built-in verbose option
 *
 * @return verbose
 */
int
rtx_getopt_get_verbose (
			int defaultVal    /**< default value of verbose */
			)
{
    if (rtxGetoptArgFlag[RTX_GETOPT_BUILTIN_ARG_VERBOSE])
        return (rtxGetoptVerbose);
    return (defaultVal);
}

/**
 * return the value of the built-in store option
 *
 * @return store
 */
int
rtx_getopt_get_store (
		      int defaultVal     /**< default value of store */
		      )
{
    if (rtxGetoptArgFlag[RTX_GETOPT_BUILTIN_ARG_STORE])
        return (rtxGetoptStore);
    return (defaultVal);
}

/**
 * return the value of the built-in debug option
 *
 * @return debug
 */
char *
rtx_getopt_get_debug (
		      char * defaultVal     /**< default value of debug */
		      )
{
    if (rtxGetoptArgFlag[RTX_GETOPT_BUILTIN_ARG_DEBUG])
        return (rtxGetoptDebug);
    return (defaultVal);
}

/**
 * return the value of the built-in priority option
 *
 * @return priority
 */
int
rtx_getopt_get_priority (
			 int defaultVal    /**< default value of priority */
			 )
{
    if (rtxGetoptArgFlag[RTX_GETOPT_BUILTIN_ARG_PRIORITY])
        return (rtxGetoptPriority);
    return (defaultVal);
}

/**
 * return the value of the built-in export option
 *
 * @return export number
 */
int
rtx_getopt_get_export (
		       int defaultVal    /**< default value of export */
		       )
{
    if (rtxGetoptArgFlag[RTX_GETOPT_BUILTIN_ARG_EXPORT])
        return (rtxGetoptExportProgramNumber);
    return (defaultVal);
}

/**
 * return the value of the built-in sample option
 *
 * @return sample
 */
double
rtx_getopt_get_sample (
		       double defaultVal    /**< default value of sample */
		       )
{
    if (rtxGetoptArgFlag[RTX_GETOPT_BUILTIN_ARG_SAMPLE])
        return (rtxGetoptSampleTime);
    return (defaultVal);
}

/**
 * return the value of the built-in config option
 *
 * @return config
 */
char *
rtx_getopt_get_config (
		       char * defaultVal     /**< default value of config */
		       )
{
    if (rtxGetoptConfigArgFlag[RTX_GETOPT_BUILTIN_CONFIG_ARG_CONFIG])
        return (rtxGetoptConfigFile);
    return (defaultVal);
}

/**
 * return the value of the built-in define option
 *
 * @return defs
 */
char **
rtx_getopt_get_D (
		  char ** defaultVal     /**< default value of define */
		  )
{
    if (rtxGetoptConfigArgFlag[RTX_GETOPT_BUILTIN_CONFIG_ARG_D])
        return (rtxGetoptCppDefs);
    return (defaultVal);
}

int
rtx_getopt_get_int (
		    char * name
		    )
{
    char * p = name;

    if (p[0] == '-')
        p++;
    if (p[0] == '-')
        p++;
    if (strcmp ("verbose", p) == 0)
        return (rtxGetoptVerbose);
    if (strcmp ("priority", p) == 0)
        return (rtxGetoptPriority);
    if (strcmp ("store", p) == 0)
        return (rtxGetoptStore);
    if (strcmp ("export", p) == 0)
        return (rtxGetoptExportProgramNumber);
    return (-1);
}

int
rtx_getopt_set_int (
		    char * name,
		    int val
		    )
{
    char * p = name;

    if (p[0] == '-')
        p++;
    if (p[0] == '-')
        p++;
    if (strcmp ("verbose", p) == 0)
        rtxGetoptVerbose = val;
    else if (strcmp ("priority", p) == 0)
        rtxGetoptPriority = val;
    else if (strcmp ("store", p) == 0)
        rtxGetoptStore = val;
    else if (strcmp ("export", p) == 0)
        rtxGetoptExportProgramNumber = val;
    else
        return (-1);
    return (0);
}

double
rtx_getopt_get_double (
		       char * name
		       )
{
    char * p = name;

    if (p[0] == '-')
        p++;
    if (p[0] == '-')
        p++;

    if (strcmp ("sample", p) == 0)
        return (rtxGetoptSampleTime);
    return (-1.0);
}

int
rtx_getopt_set_double (
		       char * name,
		       double val
		       )
{
    char * p = name;

    if (p[0] == '-')
        p++;
    if (p[0] == '-')
        p++;
    if (strcmp ("sample", p) == 0)
        rtxGetoptSampleTime = val;
    else
        return (-1);
    return (0);
}

char *
rtx_getopt_get_str (
		    char * name
		    )
{
    char * p = name;

    if (p[0] == '-')
        p++;
    if (p[0] == '-')
        p++;
    if (strcmp ("debug", p) == 0)
        return (rtxGetoptDebug);
    if (strcmp ("config", p) == 0)
        return (rtxGetoptConfigFile);
    return (NULL);
}

int
rtx_getopt_set_str (
		    char * name,
		    char * val
		    )
{
    char * p = name;

    if (p[0] == '-')
        p++;
    if (p[0] == '-')
        p++;
    if (strcmp ("config", p) == 0)
        rtxGetoptConfigFile = val;
    else if (strcmp ("debug", p) == 0)
        rtxGetoptDebug = val;
    else
        return (-1);
    return (0);
}

char **
rtx_getopt_get_strcat (
		       char * name
		       )
{
    char * p = name;

    if (p[0] == '-')
        p++;
    if (p[0] == '-')
        p++;
    if (strcmp ("D", p) == 0)
        return (rtxGetoptCppDefs);
    return (NULL);
}

int
rtx_getopt_set_strcat (
		       char * name,
		       char ** val
		       )
{
    char * p = name;

    if (p[0] == '-')
        p++;
    if (p[0] == '-')
        p++;
    if (strcmp ("D", p) == 0)
        rtxGetoptCppDefs = val;
    else
        return (-1);
    return (0);
}

void
rtx_getopt_disable_config (void)
{
    rtxGetoptProcessConfigFile = 0;
}

void
rtx_getopt_exit (
		 RtxGetopt * opt,
		 int argc,
		 char * argv[],
		 const char * date, 
		 const char * time, 
		 const char * version, 
		 const char * desc, 
		 int exitcode, 
		 const char * fmt, 
		 ...)
{
    va_list ap;

    va_start(ap, fmt);

    vfprintf(stderr, fmt, ap);
    fprintf(stderr, "\n");
    rtx_getopt_print (opt, argv[0], date, time, version, desc);
    exit (exitcode);
}

int
rtx_getopt_check (
		  RtxGetopt * opt
		  )
{
    int i, j, l, m, n;
    const char * q, * p;

    if (opt == NULL)
        return (0);

    for (n=0; opt[n].name != NULL; n++);
    for (l=0; rtxGetoptBuiltinConfigOpts[l].name != NULL; l++);
    for (m=0; rtxGetoptBuiltinOpts[m].name != NULL; m++);
    /* check for builtin config options */
	for (i=0; i<l; i++) {
		p = rtxGetoptBuiltinConfigOpts[i].name;
		if (p[0] == '-') p++;
		if (p[0] == '-') p++;
		for (j=0; j<n; j++) {
			q = opt[j].name;
			if (q[0] == '-') q++;
			if (q[0] == '-') q++;
			if (strncmp (q, p, strlen (p)) == 0) {
				fprintf (stderr, "rtx_getopt_check: ambiguity between "
						"user-defined option <%s> and built-in option <%s>\n",
						q, p);
				return (1);
			}
			if (strncmp (q, p, strlen (q)) == 0) {
				fprintf (stderr, "rtx_getopt_check: ambiguity between "
						"user-defined option <%s> and built-in option <%s>\n",
						q, p);
				return (1);
			}
		}
	}
    /* check for builtin config options */
	for (i=0; i<m; i++) {
		p = rtxGetoptBuiltinOpts[i].name;
		if (p[0] == '-') p++;
		if (p[0] == '-') p++;
		for (j=0; j<n; j++) {
			q = opt[j].name;
			if (q[0] == '-') q++;
			if (q[0] == '-') q++;
			if (strncmp (q, p, strlen (p)) == 0) {
				fprintf (stderr, "rtx_getopt_check: ambiguity between "
						"user-defined option <%s> and built-in option <%s>\n",
						q, p);
				return (1);
			}
			if (strncmp (q, p, strlen (q)) == 0) {
				fprintf (stderr, "rtx_getopt_check: ambiguity between "
						"user-defined option <%s> and built-in option <%s>\n",
						q, p);
				return (1);
			}
		}
	}
    return (0);

}
