/***********************************************************************
 * 
 * CSIRO Autonomous Systems Laboratory
 * Queensland Centre for Advanced Technologies
 * PO Box 883, Kenmore, QLD 4069, Australia
 * http://www.ict.csiro.au/
 *  
 * Copyright (c) CSIRO 
 ***********************************************************************/

/**
 * \file filter.c
 * \brief General purpose digital filter package
 * \author Peter Corke
 */

#include	<stdio.h>
#include	<stdlib.h>
#include	<stdarg.h>
#include	<math.h>
#include	<string.h>
#include	<strings.h>
#include	<limits.h>
#include	<float.h>

#include "rtx/defines.h"
#include "rtx/filter.h"
#include "rtx/error.h"

static char rcsid[] RTX_UNUSED = "$Id: filter.c 2274 2007-12-23 05:37:32Z roy029 $";

/**
 * initialize an arbitrary SISO digital filter.
 * The form form of the filter is:
 * <pre>
 *	b0 + b1 z^-1 + b2 z^-2 + ...
 *	----------------------------
 *	a0 + a1 z^-1 + a2 z^ -2 + ...
 *
 * </pre>
 * @param	nn	number of numerator terms (>= 1).
 * @param	nd	number of denominator terms (>= 1).
 * @param	coefficients in the order b0, b1, b2, ... a0, a1, a2, ...
 * @return	pointer to a filter object, or NULL if error
 *
 */
RtxFilter *
rtx_filter_init(int nn, int nd, ...)
{
	RtxFilter	*f;
	int	i;
	va_list	ap;

	va_start(ap, nd);

	/* create storage */
	if ((f = (RtxFilter *)calloc(1, sizeof(RtxFilter))) == NULL)
		return NULL;
	if ((f->N = (double *)calloc(nn, sizeof(double))) == NULL)
		return NULL;
	if ((f->D = (double *)calloc(nd, sizeof(double))) == NULL)
		return NULL;
	if (nn > 1) {
		if ((f->u = (double *)calloc(nn-1, sizeof(double))) == NULL)
			return NULL;
	} else {
		f->u = NULL;
	}
	if (nd > 1) {
		if ((f->y = (double *)calloc(nd-1, sizeof(double))) == NULL)
			return NULL;
	} else {
		f->y = NULL;
	}

	/* stash the coefficients */
	f->type = RtxFilterLinear;
	f->nn = nn;
	f->nd = nd;
	for (i=0; i<nn; i++)
		f->N[i] = va_arg(ap , double);
	for (i=0; i<nd; i++)
		f->D[i] = va_arg(ap , double);

	/* first coefficient of denom cannot be zero */
	if (f->D[0] == 0.0)
		return NULL;
	return f;
}

/**
 * Initialize an arbitrary SISO digital filter from a file.
 * The form form of the filter is:
 * <pre>
 *	b0 + b1 z^-1 + b2 z^-2 + ...
 *	----------------------------
 *	a0 + a1 z^-1 + a2 z^ -2 + ...
 *
 * </pre>
 * The file is as exported from the Matlab filter design tool \p fdatool
 * which is a human readable file that looks like:
 * <pre>
 *	Numerator:
 * 	b0
 *	b1
 *	b2 
 *	 .
 *	 .
 *	Denominator:
 * 	a0
 *	a1
 *	a2 
 *	 .
 *	 .
 * </pre>
 * @param	filename name of file containing filter design.
 * @return	pointer to a filter object, or NULL if error.
 */
RtxFilter *
rtx_filter_init_from_file(char *filename)
{
	RtxFilter	*f;
	int	i, nn = 0, nd = 0;
	FILE	*fp;
	static char	*func = "rtx_filter_init_from_file";
	char	buf[BUFSIZ];
	double	val;

	/* create filter object */
	if ((f = (RtxFilter *)calloc(1, sizeof(RtxFilter))) == NULL)
		return NULL;
	f->type = RtxFilterLinear;

	/* open the file */
	if ((fp = fopen(filename, "r")) == NULL)
		return rtx_error_errno_null("%s: cant open file %s", filename);

	/*
	 * parse the file and count the coefficients
	 */
	fgets(buf, BUFSIZ, fp);
	if (strcmp(buf, "Numerator:\n") == 0)
		return rtx_error_null("%s: Numerator key word not found", func);
	while (fscanf(fp, "%lf\n", &val) == 1)
		nn++;
	fgets(buf, BUFSIZ, fp);
	if (strcmp(buf, "Denominator:\n") == 0)
		return rtx_error_null("%s: Denominator key word not found", func);
	while (fscanf(fp, "%lf\n", &val) == 1)
		nd++;

	/* create storage for coefficients */
	if ((f->N = (double *)calloc(nn, sizeof(double))) == NULL)
		return NULL;
	if ((f->D = (double *)calloc(nd, sizeof(double))) == NULL)
		return NULL;
	if ((f->u = (double *)calloc(nn-1, sizeof(double))) == NULL)
		return NULL;
	if ((f->y = (double *)calloc(nd-1, sizeof(double))) == NULL)
		return NULL;

	/* stash the coefficients */
	f->nn = nn;
	f->nd = nd;

	rewind(fp);
	fgets(buf, BUFSIZ, fp);
	if (strcmp(buf, "Numerator:\n") == 0)
		return rtx_error_null("%s: Numerator key word not found", func);
	for (i=0; i<nn; i++)
		fscanf(fp, "%lf\n", &f->N[i]);

	fgets(buf, BUFSIZ, fp);
	if (strcmp(buf, "Denominator:\n") == 0)
		return rtx_error_null("%s: Denominator key word not found", func);
	for (i=0; i<nd; i++)
		fscanf(fp, "%lf\n", &f->D[i]);
	/* first coefficient of denom cannot be zero */
	if (f->D[0] == 0.0)
		return NULL;

	fclose(fp);

	return f;
}

/**
 * cleanup a digital filter
 *
 * @param f	RtxFilter object.
 * @return nothing
 */
void
rtx_filter_done(RtxFilter *f)
{
	if (f->y != NULL)
		free(f->y);	
	if (f->u != NULL)
		free(f->u);	
	free(f->D);	
	free(f->N);	
	free(f);	
}

/**
 * Evaluate the filter for one time step.
 * Compute the filter output, and update states, for input data u.
 *
 * @param f	RtxFilter object.
 * @param u	filter input.
 * @return the filter output.
 */
double
rtx_filter_step(RtxFilter *f, double u)
{
	int	i;
	double	s, y;

	switch (f->type) {
	case RtxFilterLinear:
		s = u * f->N[0];
		for (i=0; i<f->nn-1; i++)
			s += f->N[i+1] * f->u[i];
		for (i=0; i<f->nd-1; i++)
			s -= f->D[i+1] * f->y[i];

		y = s / f->D[0];

		/* shuffle the old u and y data down */
		if (f->nn > 1) {
			for (i=f->nn-2; i>0; i--)
				f->u[i] = f->u[i-1];
			f->u[0] = u;
		}
		if (f->nd > 1) {
			for (i=f->nd-2; i>0; i--)
				f->y[i] = f->y[i-1];
			f->y[0] = y;
		}
		return y;

		break;

	case RtxFilterRateLimit: {
		double  delta;

		/*
		 * slew rate limit the demand
		 */
		delta = u - f->uprev;
		if (delta > f->ratelimit)
			delta = f->ratelimit;
		else if (delta < -f->ratelimit)
			  delta = -f->ratelimit;

		return f->uprev += delta;
	}
			
	case RtxFilterMaximum: {
		double	max = -DBL_MAX;

		/* shuffle the old u and y data down */
		for (i=f->nn-2; i>0; i--)
			f->u[i] = f->u[i-1];
		f->u[0] = u;

		for (i=0; i<f->nn; i++)
			if (f->u[i] < max)
				max = f->u[i];
		return max;
	}

	case RtxFilterMinimum: {
		double	min = DBL_MAX;

		/* shuffle the old u and y data down */
		for (i=f->nn-2; i>0; i--)
			f->u[i] = f->u[i-1];
		f->u[0] = u;

		for (i=0; i<f->nn; i++)
			if (f->u[i] < min)
				min = f->u[i];
		return min;
	}

	}	/* end switch */
	return 0.0;
}

/**
 * Return the last output value of the filter.
 * @param f	RtxFilter object.
 * @return the filter output after last time step.
 */
double
rtx_filter_lasty(RtxFilter *f)
{
	return f->y[0];
}

/**
 * Reset internal state to zero.
 * Useful if the filter is not used continuously and is to be restarted.
 * @param f	RtxFilter object.
 */
void
rtx_filter_reset(RtxFilter *f)
{
	int	i;

	for (i=0; i < (f->nn-1); i++)
		f->u[i] = 0.0;
	for (i=0; i < (f->nd-1); i++)
		f->y[i] = 0.0;
}

/**
 * Reset internal state
 * to a steady state condition with input and output y.
 * Useful if the filter is not used continuously and is to be restarted.
 * @param f	RtxFilter object.
 */
void
rtx_filter_reset2(RtxFilter *f, double y)
{
	int	i;

	for (i=0; i < (f->nn-1); i++)
		f->u[i] = y;
	for (i=0; i < (f->nd-1); i++)
		f->y[i] = y;
}

/**
 * Print the filter transfer function in a nice format to \c stdout.
 * @param f	RtxFilter object.
 */
void
rtx_filter_print(RtxFilter *f)
{
	int	i;
	int	wn, wd, w;
	char	bn[BUFSIZ], bd[BUFSIZ], b[BUFSIZ];

	bn[0] = '\0';
	bd[0] = '\0';

	sprintf(bn, "%.3g", f->N[0]);
	for (i=1; i<f->nn; i++) {
		sprintf(b, " %c %.3gz^-%d", 
			f->N[i] >= 0 ? '+' : '-',
			fabs(f->N[i]),
			i);
		strcat(bn, b);
	}

	sprintf(bd, "%.3g", f->D[0]);
	for (i=1; i<f->nd; i++) {
		sprintf(b, " %c %.3gz^-%d", 
			f->D[i] >= 0 ? '+' : '-',
			fabs(f->D[i]),
			i);
		strcat(bd, b);
	}

	wn = strlen(bn);
	wd = strlen(bd);
	w = (wn > wd) ? wn : wd;

	if (wn < w)
		for (i=0; i<((w-wn)/2); i++)
			putchar(' ');
	printf("%s\n", bn);
	for (i=0; i<w; i++)
		putchar('-');
	putchar('\n');
	if (wd < w)
		for (i=0; i<((w-wn)/2); i++)
			putchar(' ');
	printf("%s\n", bd);
}

/**
 * Simple first order filter.
 * A prefabricated first order smoother with a single parameter \p p in the
 * range 0 to 1.  0 is no smoothing, 1 is no change.
 * @param p smoothing parameter.
 * @return a RtxFilter object
 */
RtxFilter *
rtx_filter_smooth1(double p)
{
	return rtx_filter_init(1, 2, 1.0-p, 1.0, -p);
}

/**
 * Simple differentiator.
 * a prefabricated differentiator and first order smoother with a 
 * single parameter in the range 0 to 1.  0 is no smoothing, 1 is no change.
 *
 * @param dt sample interval.
 * @param p smoothing parameter.
 * @return a RtxFilter object
 */
RtxFilter *
rtx_filter_smoothdiff(double dt, double p)
{
	double	n;

	n = (1.0 - p) / dt;
	return rtx_filter_init(2, 2, n, -n, 1.0, -p);
}

RtxFilter *
rtx_filter_nonlinear_init(RtxFilterType type, ...)
{
	RtxFilter	*f;

	/* create filter object */
	if ((f = (RtxFilter *)calloc(1, sizeof(RtxFilter))) == NULL)
		return NULL;
	f->type = type;

	return f;
}

RtxFilter *
rtx_filter_ratelimiter_init(double ratelimit)
{
	RtxFilter	*f;

	/* create filter object */
	if ((f = (RtxFilter *)calloc(1, sizeof(RtxFilter))) == NULL)
		return NULL;
	f->type = RtxFilterRateLimit;
	f->ratelimit = ratelimit;
	f->uprev = 0.0;

	return f;
}

/* local function definitions for median filter */
static double quick_select(double a[], int n);

/**
 * Median filter initialization
 */ 
RtxFilter *
rtx_filter_median_init(int winSize)
{
	RtxFilter	*f;

	/* create storage */
	if ((f = (RtxFilter *)calloc(1, sizeof(RtxFilter))) == NULL)
		return NULL;
	if (winSize > 1) {
		if ((f->u = (double *)calloc(winSize, sizeof(double))) == NULL)
			return NULL;
	} else {
		f->u = NULL;
	}
	if (winSize > 1) {
		if ((f->y = (double *)calloc(winSize, sizeof(double))) == NULL)
			return NULL;
	} else {
		f->y = NULL;
	}
	/* stash the coefficients */
	f->nn = winSize;
	f->nd = 0;

	f->type = RtxFilterLinear;


	return f;
}


/**
 * Median filter step.
 */
double
rtx_filter_median_step(RtxFilter *f, double u)
{
	int	i;
	double y;
		
	/* add the new measurement */	
	if (f->nn > 1) {
		for (i=f->nn-1; i>0; i--){
			f->u[i] = f->u[i-1];
		}	
		f->u[0] = u;
	}
	
	/* place measurements into temporary storage */
	for (i = 0; i < f->nn; i++)
		f->y[i] = f->u[i];	
	
	/* find the median */
	y = quick_select(f->y, f->nn);
	
	/* set output history to current y */ 
	f->y[0] = y;

	return y;
}





#define PIX_SWAP(a,b) { double temp=(a);(a)=(b);(b)=temp; }

static double quick_select(double a[], int n) 
{
    int low, high ;
    int median;
    int middle, ll, hh;

    low = 0 ; high = n-1 ; median = (low + high) / 2;
    for (;;) {
        if (high <= low) /* One element only */
            return a[median] ;

        if (high == low + 1) {  /* Two elements only */
            if (a[low] > a[high])
                PIX_SWAP(a[low], a[high]) ;
            return a[median] ;
        }

    /* Find median of low, middle and high items; swap into position low */
        middle = (low + high) / 2;
        if (a[middle] > a[high])    PIX_SWAP(a[middle], a[high]) ;
        if (a[low] > a[high])       PIX_SWAP(a[low], a[high]) ;
        if (a[middle] > a[low])     PIX_SWAP(a[middle], a[low]) ;

    /* Swap low item (now in position middle) into position (low+1) */
        PIX_SWAP(a[middle], a[low+1]) ;

    /* Nibble from each end towards middle, swapping items when stuck */
        ll = low + 1;
        hh = high;
        for (;;) {
            do ll++; while (a[low] > a[ll]) ;
            do hh--; while (a[hh]  > a[low]) ;

            if (hh < ll)
            break;

            PIX_SWAP(a[ll], a[hh]) ;
        }
        
        /* Swap middle item (in position low) back into correct position */
        PIX_SWAP(a[low], a[hh]) ;
        
        /* Re-set active partition */
        if (hh <= median)
            low = ll;
        if (hh >= median)
            high = hh - 1;
    }
}   
#undef PIX_SWAP

void rtx_filter_destroy(RtxFilter * f)
{
	free(f->N);
	free(f->D);
	free(f->u);
	free(f->y);
	free(f);
}


#ifdef	TEST
main(int ac, char *av[]) 
{
	RtxFilter	*f;
	int	i;
	double	y;

	f = rtx_filter_smooth1(0.8);

	if (f == NULL) {
		fprintf(stderr, "cant initialize filter\n");
		exit(2);
	}

	rtx_filter_print(f);

	for (i=0; i<20; i++) {
		double	u;

		u = (i >= 5) ? 1.0 : 0.0;

		printf("%5d %12.3f %12.3f\n", i, u, rtx_filter_step(f, u));
	}
	rtx_filter_done(f);

	f = rtx_filter_init_from_file("filter.filt");
	rtx_filter_print(f);
}







#endif
