/***********************************************************************
 * 
 * CSIRO Autonomous Systems Laboratory
 * Queensland Centre for Advanced Technologies
 * PO Box 883, Kenmore, QLD 4069, Australia
 * http://www.ict.csiro.au/
 *  
 * Copyright (c) CSIRO 
 ***********************************************************************/

/**
 * \file export.c
 * \brief Data export package based on RTC
 * \author Peter Corke
 *
 * Allows variables inside a multi-threaded program to be read/written by
 * a remote tcl/tk client. It is currently layered over both RTC and SRPC.
 * Note that RTC support may be dropped.
 *
 * 1.
 * \code:
 *	rtx_export_init();
 *	rtx_export_var_register(char *name, void *p, int flag, ...)
 *	 .
 *	rtx_export_server_start(int prio, int rtcNumber, char *transport, char *authfile)
 *
 * \endcode
 * 
 * This code starts up both a RTC server and a SRPC server. The ev script now
 * talks to the SRPC server by default. To talk to the RTC server, the script
 * supports the -rtc option.
 *
 * 2.
 * \code
 *	rtx_export_init();
 *	rtx_export_var_register(char *name, void *p, int flag, ...)
 *	 .
 *	initialize RTC yourself
 *	rtx_export_register();
 *      rtx_export_register_srpc (int port);
 * \endcode
 *
 * - \ref exportvar_rtc.c "ExportVar Using RTC"
 * - \ref exportvar_xml.c "ExportVar Using XML"
 */

#include	<stdlib.h>
#include	<unistd.h>
#include	<stdio.h>
#include	<stdarg.h>
#include	<math.h>
#include	<signal.h>
#include        <rtc.h>

#include "rtx/auth.h"
#include "rtx/defines.h"
#include "rtx/message.h"
#include "rtx/error.h"
#include "rtx/hash.h"
#include "rtx/export.h"
#include "rtx/thread.h"
#include "rtx/mutex.h"
#include "rtx/srpc.h"

static char rcsid[] RTX_UNUSED = "$Id: export.c 3082 2008-05-15 12:34:39Z roy029 $";

char *rtxExportTypeNames[] = {
	"bad",
	"char",
	"short",
	"int",
	"long",
	"float",
	"double",
	"bad"
};

int		rtxExportDebug = 0;
rtx_hash_t 	rtxHashTablePtr = NULL;
RtxMutex	*rtxExportMutex;
RtxSrpc         *srpcHandle = NULL;
int		rtxExportInitialised = 0;
RtxExportGlobalCallback	global_callback;

/*
 * Forward defines
 */
static int rtx_export_var(char * buf, int n,
			  int argc, char *argv[]);
static int rtx_export_multi_var(char * buf, int n,
				int argc, char *argv[]);
static int rtx_export_var_list(char * buf, int n,
			       int argc, char *argv[]);
static int rtx_export_var_info(char * buf, int n,
			       int argc, char *argv[]);
static char * rtx_export_var_rtc(int argc, char *argv[]);
static char * rtx_export_multi_var_rtc(int argc, char *argv[]);
static char * rtx_export_var_list_rtc(int argc, char *argv[]);
static char * rtx_export_var_info_rtc(int argc, char *argv[]);
static RtxExportSym * rtx_export_common(char *name, void *p, int len, int flag, char *units);

/**
 * Initialize export package for use.  Creates a hash table to hold the
 * export variable symbol table and initializes it.
 *
 * @return 0 if OK, -1 on error.
 */
int
rtx_export_init(void)
{
	static char	*func = "rtx_export_init";

	/* Return OK if this function has already been called */
	if (rtxExportInitialised) 
		return 0;

	if ((rtxExportMutex = rtx_mutex_init(NULL, RTX_MUTEX_DEFAULT,0)) == NULL)
		return rtx_error("%s: rtx_mutex_init: failed", func);

	if ((rtxHashTablePtr = rtx_hash_alloc(RTX_HASH_SIZE, 
					      rtx_hash_shallow_string_keys, rtx_hash_shallow_values,	
					      (rtx_error_t)NULL)) == NULL)
		return rtx_error("%s: rtx_hash_create failed", func);

	rtxExportInitialised = 1;

	return 0;
}

/**
 * Register a global callback function to be called on any 
 * parameter change.
 *
 * @param callback pointer to callback function.  Passing a NULL pointer
 *  disables the callback.
 *
 * </pre>
 * The callback function is invoked as <pre>
 *        callback(RtxExportSym *s, int indx, char *newvalue);
 * </pre>
 *
 * where \p s is a pointer to the symbol table entry, and \p indx
 * is the array index (subscript) for array types.  For a scalar it
 * will be 0.
 * The callback is invoked after the value has been changed
 * by the client, allowing the new value to be dealt with immediately.
 *
 * The function will be called with a first argument of NULL when
 * \p rtx_export_done() is called, indicating that no more callbacks will
 * occur and callback specific actions should be cleaned up.
 *
 * A use for this is to log changes to a file.
 */
void   
rtx_export_globalcallback_set( RtxExportGlobalCallback callback)
{
	global_callback = callback;
}


/**
 * Finish with the export package.  
 *
 * @return 0 if OK, -1 on error.
 */
int
rtx_export_done(void)
{
	static char	*func = "rtx_export_done";
	int		error = 0;

	if (rtxExportInitialised) {
		/* Destroy mutex */
		if (rtx_mutex_destroy(rtxExportMutex)) {
			rtx_error("%s: rtx_mutex_destroy: failed", func);
			error--;
		}

		/* Destroy hashtable */
		rtx_hash_free(rtxHashTablePtr);

		rtxExportInitialised = 0;
	} else {
		return rtx_error("%s: export package hasn't been initialised", func);
	}

	return error;
}

/**
 * Register an integer variable for exporting.
 * The variable is entered into the the symbol table.
 *
 * @param name name that the variable will be exported under
 * @param p a pointer to the int
 * @param len length of the int array
 * @param flag options flag word, logical or of the following:
 *	- RTX_EXPORT_READONLY	variable cannot be changed via export mechanism
 *	- RTX_EXPORT_POLL	client should regularly poll this variable
 * @param units pointer to a string containing name of the variable's units ,or NULL
 *
 * \sa rtx_export_register
 */
int
rtx_export_int(char *name, int *p, int len, int flag, char *units)
{
	if ( rtx_export_common(name, (void*) p, len, 
		flag|RTX_EXPORT_INT, units) == NULL
	)
		return rtx_error("rtx_export_int: cant register");

	return 0;
}

/**
 * Register an integer variable for exporting.
 * The variable is entered into the the symbol table.
 *
 * @param name name that the variable will be exported under
 * @param p a pointer to the character array
 * @param len length of the character array
 * @param flag options flag word, logical or of the following:
 *	- RTX_EXPORT_READONLY	variable cannot be changed via export mechanism
 *	- RTX_EXPORT_POLL	client should regularly poll this variable
 * @param units pointer to a string containing name of the variable's units ,or NULL
 *
 * \sa rtx_export_register
 */
int
rtx_export_string(char *name, char *p, int len, int flag)
{
	if ( rtx_export_common(name, (void*) p, len,
		flag|RTX_EXPORT_STRING, NULL) == NULL
	)
		return rtx_error("rtx_export_string: cant register");

	return 0;
}

/**
 * Register a double variable for exporting.
 * The variable is entered into the the symbol table.
 *
 * @param name name that the variable will be exported under
 * @param p a pointer to the double
 * @param len length of the double array
 * @param flag options flag word, logical or of the following:
 *	- RTX_EXPORT_READONLY	variable cannot be changed via export mechanism
 *	- RTX_EXPORT_POLL	client should regularly poll this variable
 * @param units pointer to a string containing name of the variable's units ,or NULL
 *
 * \sa rtx_export_register rtx_export_double_linear
 */
int
rtx_export_double(char *name, double *p, int len, int flag, char *units)
{
	if ( rtx_export_common(name, (void*) p, len,
		flag|RTX_EXPORT_DOUBLE, units) == NULL
	)
		return rtx_error("rtx_export_double: cant register");

	return 0;
}

/**
 * Register a float variable for exporting.
 * The variable is entered into the the symbol table.
 *
 * @param name name that the variable will be exported under
 * @param p a pointer to the float
 * @param len length of the float array
 * @param flag options flag word, logical or of the following:
 *	- RTX_EXPORT_READONLY	variable cannot be changed via export mechanism
 *	- RTX_EXPORT_POLL	client should regularly poll this variable
 * @param units pointer to a string containing name of the variable's units ,or NULL
 *
 * \sa rtx_export_register rtx_export_float_linear
 */
int
rtx_export_float(char *name, float *p, int len, int flag, char *units)
{
	if ( rtx_export_common(name, (void*) p, len,
		flag|RTX_EXPORT_FLOAT, units) == NULL
	)
		return rtx_error("rtx_export_float: cant register");

	return 0;
}

/**
 * Register a double variable for exporting with scale and offset.
 * The variable is entered into the the symbol table.
 *
 * @param name name that the variable will be exported under
 * @param p a pointer to the double or double array
 * @param len length of the array
 * @param flag options flag word, logical or of the following:
 *	- RTX_EXPORT_READONLY	variable cannot be changed via export mechanism
 *	- RTX_EXPORT_POLL	client should regularly poll this variable
 * @param units pointer to a string containing name of the variable's units ,or NULL
 * @param scale scale factor to apply to variable when exported
 * @param offset offset to apply to variable when exported
 *
 * <pre>
 *    export = scale * value + offset
 * </pre>
 *
 * \sa rtx_export_register
 */
int
rtx_export_double_linear(char *name, double *p, int len, int flag, char *units,
	double scale, double offset)
{
	RtxExportSym	*s;

	if ( (s = rtx_export_common(name, (void*) p, len,
		flag|RTX_EXPORT_DOUBLE, units)) == NULL
	)
		return rtx_error("rtx_export_double_linear: cant register %s", name);

	s->scale =  scale;
	s->offset = offset;

	return 0;
}

/**
 * Register a float variable for exporting with scale and offset.
 * The variable is entered into the the symbol table.
 *
 * @param name name that the variable will be exported under
 * @param p a pointer to the float
 * @param len length of the float array
 * @param flag options flag word, logical or of the following:
 *	- RTX_EXPORT_READONLY	variable cannot be changed via export mechanism
 *	- RTX_EXPORT_POLL	client should regularly poll this variable
 * @param units pointer to a string containing name of the variable's units ,or NULL
 * @param scale scale factor to apply to variable when exported
 * @param offset offset to apply to variable when exported
 *
 * <pre>
 *    export = scale * value + offset
 * </pre>
 *
 *
 * \sa rtx_export_register
 */
int
rtx_export_float_linear(char *name, float *p, int len, int flag, char *units,
	double scale, double offset)
{
	RtxExportSym	*s;

	if ( (s = rtx_export_common(name, (void*) p, len,
		flag|RTX_EXPORT_FLOAT, units)) == NULL
	)
		return rtx_error("rtx_export_float_linear: cant register %s", name);

	s->scale =  scale;
	s->offset = offset;

	return 0;
}

/**
 * Register a double variable in radians for exporting as degrees.
 * The variable is entered into the the symbol table.
 *
 * @param name name that the variable will be exported under
 * @param p a pointer to the double or double array
 * @param len length of the array
 * @param flag options flag word, logical or of the following:
 *	- RTX_EXPORT_READONLY	variable cannot be changed via export mechanism
 *	- RTX_EXPORT_POLL	client should regularly poll this variable
 * <pre>
 *    export = value * pi/180;
 * </pre>
 *
 * \sa rtx_export_register
 */
int
rtx_export_double_degrees(char *name, double *p, int len, int flag)
{
	return rtx_export_double_linear(name, p, len, flag,
		"deg", 180.0/M_PI, 0.0);
}

/**
 * Register a float variable in radians for exporting as degrees.
 * The variable is entered into the the symbol table.
 *
 * @param name name that the variable will be exported under
 * @param p a pointer to the float or float array
 * @param len length of the array
 * @param flag options flag word, logical or of the following:
 *	- RTX_EXPORT_READONLY	variable cannot be changed via export mechanism
 *	- RTX_EXPORT_POLL	client should regularly poll this variable
 * <pre>
 *    export = value * pi/180;
 * </pre>
 *
 * \sa rtx_export_register
 */
int
rtx_export_float_degrees(char *name, float *p, int len, int flag)
{
	return rtx_export_float_linear(name, p, len, flag,
		"deg", 180.0/M_PI, 0.0);
}

/*
 * simplified callback interface, invokes a function to return the value 
 * for the virtual variable.
 */
static int
rtx_export_virtual_callback_proc(RtxExportSym *s, int indx, enum RtxExportOp op)
{
	*s->p.dp = (s->virtual_callback)(s, s->callback_arg);

	return 0;
}

/**
 * Register a virtual variable for export.
 * The variable is entered into the the symbol table.
 *
 * @param name name of the virtual variable
 * @param func a pointer to the function that provides the value of type RtxExportVirtualCallback
 * @param arg an argument to the function
 * @param flag options flag word, logical or of the following:
 *	- RTX_EXPORT_READONLY	variable cannot be changed via export mechanism
 *	- RTX_EXPORT_POLL	client should regularly poll this variable
 * @param units pointer to a string containing name of the variable's units ,or NULL
 *
 * The callback function is invoked as <pre>
 *	  double
 *        callback(RtxExportSym *s, void *arg)
 * </pre>
 *
 * \sa rtx_export_register
 */
int
rtx_export_virtual(char *name, RtxExportVirtualCallback func, void *arg, int flag, char *units)
{
	RtxExportSym	*s;

	if ( (s = rtx_export_common(name, NULL, 1,
		flag|RTX_EXPORT_DOUBLE, units)) == NULL
	)
		return rtx_error("rtx_export_double: cant register");

	s->p.dp = &s->value.d;		/* storage for the result */
	s->type |= RTX_EXPORT_CALLBACK;
	s->callback = (RtxExportCallback) rtx_export_virtual_callback_proc;
	s->virtual_callback = func;
	s->callback_arg = arg;
	return 0;
}



static RtxExportSym *
rtx_export_common(char *name, void *p, int len, int flag, char *units)
{
	RtxExportSym	*s;
	static char	*func = "rtx_export_common";

	if (rtxExportDebug)
		rtx_message_routine("%s: %s\n", func, name);
	if ((s = (RtxExportSym *) rtx_hash_find (rtxHashTablePtr, name, (rtx_error_t)NULL))
	        == NULL) {
	    if ((s = (RtxExportSym *) calloc (1,
			    sizeof (RtxExportSym))) == NULL)
	        return rtx_error_errno_null ("%s: calloc", func);
	    if (rtx_hash_set(rtxHashTablePtr, name, s, (rtx_error_t)NULL))
	        return rtx_error_null ("%s: rtx_hash_set_failed", func);
	}
	s->dimension = len;
	if (s->dimension == 0)
		s->dimension = (int)1;
	s->p.p = p;
	s->s_name = name;
	s->type = flag;

	if (units != NULL)
		s->units = strdup(units);
	s->scale = 1.0;
	s->offset = 0.0;

	return s;
}

/**
 * Register a variable for exporting.
 * The variable is entered into the the symbol table.
 *
 * @param name name that the variable will be exported under
 * @param p a pointer to the variable
 * @param flag options flag word, logical or of the following:
 *	- RTX_EXPORT_CHAR	variable is of char type
 *	- RTX_EXPORT_SHORT	variable is of char type
 *	- RTX_EXPORT_INT	variable is of char type
 *	- RTX_EXPORT_LONG	variable is of char type
 *	- RTX_EXPORT_FLOA	variable is of char type
 *	- RTX_EXPORT_DOUBL	variable is of char type
 *	- RTX_EXPORT_STRIN	variable is of char type
 *	- RTX_EXPORT_ARRAY(l)	variable is an array of length l
 *	- RTX_EXPORT_READONLY	variable cannot be changed via export mechanism
 *	- RTX_EXPORT_POLL	client should regularly poll this variable
 *	- RTX_EXPORT_CALLBACK	a callback function is invoked on every read or write, function is next argument
 *	- RTX_EXPORT_SPAN	a min/max value range is specified by next two arguments
 *	- RTX_EXPORT_USERDATA	arbitrary pointer can be stored in this variable's symbol table entry, pointer is next argument
 *
 * The last three options take additional function arguments.  If more than one
 * is used the arguments are taken by the options in order, ie. callback then
 * span.
 *
 * The callback function is invoked as <pre>
 *        callback(RtxExportSym *s, int indx, RTX_EXPORT_WRITE);
 *        callback(RtxExportSym *s, int indx, RTX_EXPORT_READ);
 * </pre>
 *
 * where \p s is a pointer to the symbol table entry, and \p indx
 * is the array index (subscript) for array types.  For a scalar it
 * will be 0.  The last argument indicates whether the variable is being
 * read or written by the client.  A read callback is invoked before the
 * variable is exported, allowing the function to actually compute or update
 * the value.  A write callback is invoked after the value has been changed
 * by the client, allowing the new value to be dealt with immediately.
 */
int
rtx_export_var_register(char *name, void *p, int flag, ...)
{
	RtxExportSym	*s;
	va_list		ap;

	if (rtxExportDebug)
		rtx_message_routine("rtx_export_var_register: %s\n", name);
	if ((s = (RtxExportSym *) rtx_hash_find (rtxHashTablePtr, name, (rtx_error_t)NULL))
	        == NULL) {
	    if ((s = (RtxExportSym *) calloc (1,
			    sizeof (RtxExportSym))) == NULL)
	        return (rtx_error_errno ("rtx_export_var_register: calloc"));
	    if (rtx_hash_set(rtxHashTablePtr, name, s, (rtx_error_t)NULL))
	        return (rtx_error ("rtx_export_var_register: "
				   "rtx_hash_set failed"));
	}
	s->dimension = RTX_EXPORT_ARRAY_LENGTH(flag);
	if (s->dimension == 0)
		s->dimension = (int)1;
	s->type = flag & 0xffff;
	s->p.p = p;
	s->s_name = name;

	va_start(ap, flag);

	if (flag & RTX_EXPORT_CALLBACK)
		s->callback = va_arg(ap, RtxExportCallback);
	if (flag & RTX_EXPORT_SPAN) {
		switch (s->type & RTX_EXPORT_TYPE_MASK) {
		case RTX_EXPORT_CHAR:
		case RTX_EXPORT_SHORT:
		case RTX_EXPORT_INT:
		case RTX_EXPORT_LONG:
			s->min.i = va_arg(ap, int);
			s->max.i = va_arg(ap, int);
			break;
		case RTX_EXPORT_FLOAT:
		case RTX_EXPORT_DOUBLE:
			s->min.d = va_arg(ap, double);
			s->max.d = va_arg(ap, double);
			break;
		}
	}
	if (flag & RTX_EXPORT_USERDATA) {
		s->userdata = va_arg(ap, void *);
	}
	return (0);
}

/******************************************************************
 ******************* RTC/SRPC server start/stop *******************
 ******************************************************************/

/**
 * return the SRPC handle
 * 
 * @return handle
 */
RtxSrpc * 
rtx_export_get_srpc_handle (void)
{
    return (srpcHandle);
}

/**
 * Register the export package with RTC.
 * This function must be called explicitly if you create your own RTC
 * server.
 *
 * @return 0 if OK, -1 on error.
 */
int
rtx_export_register(void)
{
	static char	*func = "rtx_export_register";

	/* register RTC handlers */
	if (rtcServerRegister("exportvar_list", rtx_export_var_list_rtc))
		return rtx_error("%s: %s", func, rtcMessageGet());
	if (rtcServerRegister("exportvar_var", rtx_export_var_rtc))
		return rtx_error("%s: %s", func, rtcMessageGet());
	if (rtcServerRegister("exportvar_multi_var", rtx_export_multi_var_rtc))
		return rtx_error("%s: %s", func, rtcMessageGet());
	if (rtcServerRegister("exportvar_info", rtx_export_var_info_rtc))
		return rtx_error("%s: %s", func, rtcMessageGet());

	return 0;
}

/**
 * Register the export package with SRPC.
 * This function must be called explicitly if you create your own RTC
 * server.
 *
 * @return 0 if OK, -1 on error.
 */
int
rtx_export_register_srpc(int port)
{
	static char	*func = "rtx_export_register_srpc";

	if ((srpcHandle = rtx_srpc_init (port)) == NULL)
	        return (rtx_error ("rtx_export_register_srpc: "
				   "rtx_srpc_init()"));

	/* register SRPC handlers */
	if (rtx_srpc_register (srpcHandle, "exportvar_list", 
			       rtx_export_var_list))
		return rtx_error("%s: %s", func, rtcMessageGet());
	if (rtx_srpc_register (srpcHandle, "exportvar_var", 
			       rtx_export_var))
		return rtx_error("%s: %s", func, rtcMessageGet());
	if (rtx_srpc_register (srpcHandle, "exportvar_multi_var", 
			       rtx_export_multi_var))
		return rtx_error("%s: %s", func, rtcMessageGet());
	if (rtx_srpc_register (srpcHandle, "exportvar_info", 
			       rtx_export_var_info    ))
		return rtx_error("%s: %s", func, rtcMessageGet());

	return 0;
}

/*
 * structure to pass to rtc handler thread
 */
struct _server {
	char	*transport;
	int	programNumber;
};

/**
 * Thread to handle exportvar requests.
 * @param s pointer to structure containing connection data.
 *
 */
static void
export_thread(void *p)
{
	struct _server *s = (struct _server *)p;
	char	hostname[BUFSIZ];

	gethostname(hostname, BUFSIZ);
	rtx_message("RTC server running on programNumber=%d, host=%s\n",
		s->programNumber, hostname);
	rtcServer(s->transport, s->programNumber, hostname);

	rtx_error("exportvarThread: rtcServer() returned");
}

/**
 * Start a server to handle requests from clients.
 * @param prio thread priority, 0 for default.
 * @param rtcNumber the RTC program number.
 * @param transport either "udp" or "tcp"
 * @param authfile the name of an authorization file, default used if NULL.
 * @return a pointer to an RTC thread on success, or NULL on error.
 */
RtxThread *
rtx_export_server_start(int prio, int rtcNumber, char *transport, char *authfile)
{
	struct _server	*s;
	static char	*func = "rtx_exprt_server_start";
	RtxThread	*thread;	

	if (rtcServerInit())
		return rtx_error_null("%s: rtcServerInit() failed", func);

	/* get the name of the authorization file */
	if (authfile == NULL)
		authfile = "/data/aa/etc/rtc.auth";

	if (rtx_auth(authfile, 0))
		return rtx_error_null("%s: authenticate failed with file <%s>",
			func, authfile);

	/* register exportvar handlers with RTC */
	rtx_export_register();
	if (rtx_export_register_srpc(rtcNumber) == -1)
	        rtx_error_flush ("rtx_export_server_start: "
				 "rtx_export_register_srpc()");

	/* now start a thread to handle RTC requests */
	s = (struct _server *)calloc(1, sizeof(struct _server));
	s->transport = transport;
	s->programNumber = rtcNumber;
	if ((thread = rtx_thread_create("export_thread", 0, 0, prio, 0,  RTX_THREAD_CANCEL_ASYNCHRONOUS, (void * (*)(void*))export_thread, (void *)s, NULL, NULL)) == NULL)
		return rtx_error_null("exportvarRun: thread_create failed");

	return thread;
}

/**
 * Stop a server to handle requests from clients.
 * @param pointer to the RTC thread as returned by rtx_export_server_start
 * @return Zero on success, -1 on error
 */
int
rtx_export_server_stop(
	RtxThread	*thread
	)
{
	if (rtx_thread_destroy(thread)){
		return rtx_error("rtx_export_server_stop: couldnt kill server thread");
	}
	if (rtx_srpc_done(srpcHandle)) {
		rtx_message("rtx_export_server_stop: couldnt finish srpc");
	}
	return 0;
}

static void
rtx_export_print_entry(void *key, void *value, int i, int n, int *stop_iter, void *p)
{
    char *name = (char *)key;
    char *chp = (char *)p;
    RtxExportSym *s = (RtxExportSym *)value;
    char lenSuffix[32];

    lenSuffix[0] = '\0';
    /*
    if (s->type & RTX_EXPORT_POLL)
        strcat(chp, "!");
	*/
    strcat (chp, name);
    if (s->dimension > 1)
        sprintf (lenSuffix, "[%d]", s->dimension);
    strcat (chp, lenSuffix);
    strcat (chp, " ");
}

/*****************************************************************
 *********************** RTC server functions ********************
 *****************************************************************/

/**
 * RTC server function returns a blank separated list of variables
 */
static char *
rtx_export_var_list_rtc(int argc, char *argv[])
{
	static char	names[BUFSIZ];

	names[0] = '\0';
	rtx_hash_iter(rtxHashTablePtr, rtx_export_print_entry, names);
	if (rtxExportDebug)
		rtx_message_routine("rtx_export_var_list: return variable "
				    "list: %s",	names);
	return names;
}

/**
 * RTC server function to get status information about an exported variable.
 *
 *	exportvar_info variable
 *
 * and it returns a list containing:
 *
 *		pollflag readonlyflag type dimension minvalue maxvalue
 * or
 *		pollflag readonlyflag type dimension
 *
 * depending on whether the variable has min/max span values set
 */
static char *
rtx_export_var_info_rtc(int argc, char *argv[])
{
	static char	result[BUFSIZ];
	RtxExportSym	*s;
	char		span[256];

	if (rtxExportDebug)
		rtx_message_routine("exportvar_info: in exportvar_info\n");
	if (argc != 2)
		return "exportvar_info: wrong number of arguments";

	s = rtx_hash_find(rtxHashTablePtr, argv[1], (rtx_error_t)NULL);
	if (s == NULL)
		return "exportvar_info: variable not found";

	sprintf(result, "%d %d %s %d %s",
		(s->type & RTX_EXPORT_POLL) != 0,
		(s->type & RTX_EXPORT_READONLY) != 0,
		rtxExportTypeNames[s->type & RTX_EXPORT_TYPE_MASK],
		s->dimension,
		s->units == NULL ? "" : s->units);

	if (s->type & RTX_EXPORT_SPAN) {
		switch (s->type  & RTX_EXPORT_TYPE_MASK) {
		case RTX_EXPORT_CHAR:
		case RTX_EXPORT_SHORT:
		case RTX_EXPORT_INT:
		case RTX_EXPORT_LONG:
			sprintf(span, " %d %d",
				s->min.i,
				s->max.i);
			break;
		case RTX_EXPORT_FLOAT:
		case RTX_EXPORT_DOUBLE:
			sprintf(span, " %f %f",
				s->min.d,
				s->max.d);

			break;
		}
		strcat(result, span);
	}
	return result;
}

/**
 * RTC server function used for reading/writing server variables.
 *
 * Has many call forms:
 *
 *	exportvar_var name		return value of variable (if vector
 *					return a list)
 *	exportvar_var name[el]		return specified element of vector
 *	exportvar_var name newval	assign new value to variable
 *	exportvar_var name[el] newval	assign new value to specified element
 *					of vector
 */
static char *
rtx_export_var_rtc(int argc, char *argv[])
{
	static char	result[BUFSIZ];
	RtxExportSym	*s;
	char		name[64];
	unsigned int	i, nels, n, indx;
	int             intNumber;
	float           floatNumber;
	double          doubleNumber;

	rtx_mutex_lock(rtxExportMutex);
	if (rtxExportDebug)
		rtx_message_routine("exportvar_var: in exportvar_var\n");
	switch (argc) {
	case 2:		/* return value */
		if (rtxExportDebug)
			rtx_message_routine("exportvar_var: reading variable %s\n", argv[1]);

		/* parse the variable name and index (optional) */
		n = sscanf(argv[1], "%[^[(]%*c%d", name, &indx);

		/* look it up, and check the index against bounds */
		s = rtx_hash_find(rtxHashTablePtr, name, (rtx_error_t)NULL);
		if (s == NULL) {
			sprintf(result, "<%s not found>", name);
			rtx_mutex_unlock(rtxExportMutex);
			return result;
		}

		if (n == 1) {
			nels = s->dimension;
			indx = 0;
		}
		else
			nels = 1;

		if ((indx < 0) || (indx >= s->dimension)){
			rtx_mutex_unlock(rtxExportMutex);
			return "<bad index>";
		}
		if (s->callback)
			(s->callback)(s, indx, RTX_EXPORT_READ);

		if (nels > 1)
			strcpy(result, "[ ");
		else
			result[0] = '\0';
		for (i=0; i<nels; i++, indx++) {
			char	tmp[BUFSIZ];

			switch (s->type & RTX_EXPORT_TYPE_MASK) {
			case RTX_EXPORT_CHAR:
				sprintf(tmp, "%d", s->p.cp[indx]);
				break;
			case RTX_EXPORT_SHORT:
				sprintf(tmp, "%d", s->p.sp[indx]);
				break;
			case RTX_EXPORT_INT:
				sprintf(tmp, "%d", s->p.ip[indx]);
				break;
			case RTX_EXPORT_LONG:
				sprintf(tmp, "%ld", s->p.lp[indx]);
				break;
			case RTX_EXPORT_FLOAT:
				sprintf(tmp, "%f",
					s->p.fp[indx]*s->scale + s->offset);
				break;
			case RTX_EXPORT_DOUBLE:
				sprintf(tmp, "%f",
					s->p.dp[indx]*s->scale + s->offset);
				break;
			case RTX_EXPORT_STRING:
				sprintf(tmp, "%s", *(s->p.s));
				break;
			}
			if ((nels > 1) && (i < (nels-1)))
				strcat(tmp, " ");
			strcat(result, tmp);
		}
		if (nels > 1)
			strcat(result, "]");
		rtx_mutex_unlock(rtxExportMutex);
		return result;
		break;

	case 3:		/* set value */
		if (rtxExportDebug)
			rtx_message_routine("exportvar_var: writing variable %s %s\n", argv[1], argv[2]);

		/* parse the variable name and index (optional) */
		n = sscanf(argv[1], "%[^[(]%*c%d", name, &indx);

		/* look it up, and check the index against bounds */
		s = rtx_hash_find(rtxHashTablePtr, name, (rtx_error_t)NULL);
		if (s == NULL){
			rtx_mutex_unlock(rtxExportMutex);
			return "<exportSym not found>";
		}
		if (n == 1) {
			nels = s->dimension;
			indx = 0;
		}
		else
			nels = 1;

		if ((indx < 0) || (indx >= s->dimension)){
			rtx_mutex_unlock(rtxExportMutex);
			return "<bad index>";
		}
		if (s->type & RTX_EXPORT_READONLY) {
			rtx_message_warning("exportvar_var: attempting to set RO variable %s\n", name);
			rtx_mutex_unlock(rtxExportMutex);
			return "RO";
		}
		switch (s->type & RTX_EXPORT_TYPE_MASK) {
		case RTX_EXPORT_CHAR:
			n = sscanf(argv[2], "%d", &intNumber);
			if (n == 1) {
				s->p.cp[indx] = intNumber;
			}
			break;
		case RTX_EXPORT_SHORT:
			n = sscanf(argv[2], "%d", &intNumber);
			if (n == 1) {
				s->p.sp[indx] = intNumber;
			}
			break;
		case RTX_EXPORT_INT:
			n = sscanf(argv[2], "%d", &intNumber);
			if (n == 1) {
				s->p.ip[indx] = intNumber;
			}
			break;
		case RTX_EXPORT_LONG:
			n = sscanf(argv[2], "%d", &intNumber);
			if (n == 1) {
				s->p.lp[indx] = intNumber;
			}
			break;
		case RTX_EXPORT_FLOAT:
			n = sscanf(argv[2], "%f", &floatNumber);
			if (n == 1) {
				s->p.fp[indx] = 
					(floatNumber - s->offset) / s->scale;
			}
			break;
		case RTX_EXPORT_DOUBLE:
			n = sscanf(argv[2], "%lf", &doubleNumber);
			if (n == 1) {
				s->p.dp[indx] =
					(doubleNumber - s->offset) / s->scale;
			}
			break;
		case RTX_EXPORT_STRING:
			strcpy(*(s->p.s), argv[2]);
			n = 1;
			break;
		}
		if (n != 1){
			rtx_mutex_unlock(rtxExportMutex);
			return "sscanf() error";
		}else {
			if (s->callback)
				(s->callback)(s, indx, RTX_EXPORT_WRITE);
			if (global_callback)
				(global_callback)(s, indx, argv[2]);
			rtx_mutex_unlock(rtxExportMutex);
			return "";
		}

		break;
	}
	rtx_mutex_unlock(rtxExportMutex);
	return "";
}

/**
 * RTC server function used for reading multiple server variables.
 *
 *	exportvar_multi_var name name ...
 */
static char *
rtx_export_multi_var_rtc(int argc, char *argv[])
{
	static char	result[BUFSIZ];
	RtxExportSym	*s;
	char		name[64];
	unsigned int	indx;
	int		i, k, nels, n;

	rtx_mutex_lock(rtxExportMutex);
	if (rtxExportDebug)
		rtx_message_routine("exportvar_multi_var: in exportvar_multi_var\n");

	result[0] = '\0';
	for (k=1; k<argc; k++) {
		if (rtxExportDebug)
			rtx_message_routine("exportvar_multi_var: reading variable %s\n", argv[k]);

		/* parse the variable name and index (optional) */
		n = sscanf(argv[k], "%[^[(]%*c%d", name, &indx);

		/* look it up, and check the index against bounds */
		s = rtx_hash_find(rtxHashTablePtr, name, (rtx_error_t)NULL);
		if (s == NULL) {
			sprintf(result, "<%s not found>", name);
			rtx_mutex_unlock(rtxExportMutex);
			return result;
		}

		if (n == 1) {
			nels = s->dimension;
			indx = 0;
		}
		else
			nels = 1;

		if ((indx < 0) || (indx >= s->dimension)){
			rtx_mutex_unlock(rtxExportMutex);
			return "<bad index>";
		}

		if (s->callback)
			(s->callback)(s, indx, RTX_EXPORT_READ);

		for (i=0; i<nels; i++, indx++) {
			char	tmp[BUFSIZ];

			switch (s->type & RTX_EXPORT_TYPE_MASK) {
			case RTX_EXPORT_CHAR:
				sprintf(tmp, "%d", s->p.cp[indx]);
				break;
			case RTX_EXPORT_SHORT:
				sprintf(tmp, "%d", s->p.sp[indx]);
				break;
			case RTX_EXPORT_INT:
				sprintf(tmp, "%d", s->p.ip[indx]);
				break;
			case RTX_EXPORT_LONG:
				sprintf(tmp, "%ld", s->p.lp[indx]);
				break;
			case RTX_EXPORT_FLOAT:
				sprintf(tmp, "%f", s->p.fp[indx]);
				break;
			case RTX_EXPORT_DOUBLE:
				sprintf(tmp, "%f", s->p.dp[indx]);
				break;
			case RTX_EXPORT_STRING:
				sprintf(tmp, "%s", *(s->p.s));
				break;
			}
			strcat(tmp, " ");
			strcat(result, tmp);
		}
	}
	rtx_mutex_unlock(rtxExportMutex);
	return result;
}

/*****************************************************************
 *********************** SRPC server functions *******************
 *****************************************************************/

/**
 * SRPC server function returns a blank separated list of variables
 */
static int
rtx_export_var_list(char * names, int n, int argc, char *argv[])
{
	names[0] = '\0';
	rtx_hash_iter(rtxHashTablePtr, rtx_export_print_entry, names);
	if (rtxExportDebug)
		rtx_message_routine("rtx_export_var_list: return variable "
				    "list: %s",	names);
	return (0);
}

/**
 * SRPC server function to get status information about an exported variable.
 *
 *	exportvar_info variable
 *
 * and it returns a list containing:
 *
 *		pollflag readonlyflag type dimension minvalue maxvalue
 * or
 *		pollflag readonlyflag type dimension
 *
 * depending on whether the variable has min/max span values set
 */
static int
rtx_export_var_info(char * result, int n, int argc, char *argv[])
{
	RtxExportSym	*s;
	char		span[256];

	if (rtxExportDebug)
		rtx_message_routine("exportvar_info: in exportvar_info\n");
	if (argc != 2) {
	        sprintf (result, "exportvar_info: wrong number of arguments");
		return (-1);
	}
	s = rtx_hash_find(rtxHashTablePtr, argv[1], (rtx_error_t)NULL);
	if (s == NULL) {
	        sprintf (result, "exportvar_info: variable not found");
		return (-1);
	}
	sprintf(result, "%d %d %s %d %s",
		(s->type & RTX_EXPORT_POLL) != 0,
		(s->type & RTX_EXPORT_READONLY) != 0,
		rtxExportTypeNames[s->type & RTX_EXPORT_TYPE_MASK],
		s->dimension,
		s->units == NULL ? "" : s->units);

	if (s->type & RTX_EXPORT_SPAN) {
		switch (s->type  & RTX_EXPORT_TYPE_MASK) {
		case RTX_EXPORT_CHAR:
		case RTX_EXPORT_SHORT:
		case RTX_EXPORT_INT:
		case RTX_EXPORT_LONG:
			sprintf(span, " %d %d",
				s->min.i,
				s->max.i);
			break;
		case RTX_EXPORT_FLOAT:
		case RTX_EXPORT_DOUBLE:
			sprintf(span, " %f %f",
				s->min.d,
				s->max.d);

			break;
		}
		strcat(result, span);
	}
	return (0);
}

/**
 * SRPC server function used for reading/writing server variables.
 *
 * Has many call forms:
 *
 *	exportvar_var name		return value of variable (if vector
 *					return a list)
 *	exportvar_var name[el]		return specified element of vector
 *	exportvar_var name newval	assign new value to variable
 *	exportvar_var name[el] newval	assign new value to specified element
 *					of vector
 */
static int
rtx_export_var(char * result, int m, int argc, char *argv[])
{
	RtxExportSym	*s;
	char		name[64];
	unsigned int	i, nels, n, indx;
	int             intNumber;
	float           floatNumber;
	double          doubleNumber;

	rtx_mutex_lock(rtxExportMutex);
	if (rtxExportDebug)
		rtx_message_routine("exportvar_var: in exportvar_var\n");
	switch (argc) {
	case 2:		/* return value */
		if (rtxExportDebug)
			rtx_message_routine("exportvar_var: reading variable %s\n", argv[1]);

		/* parse the variable name and index (optional) */
		n = sscanf(argv[1], "%[^[(]%*c%d", name, &indx);

		/* look it up, and check the index against bounds */
		s = rtx_hash_find(rtxHashTablePtr, name, (rtx_error_t)NULL);
		if (s == NULL) {
			sprintf(result, "<%s not found>", name);
			rtx_mutex_unlock(rtxExportMutex);
			return (-1);
		}

		if (n == 1) {
			nels = s->dimension;
			indx = 0;
		}
		else
			nels = 1;

		if ((indx < 0) || (indx >= s->dimension)){
			rtx_mutex_unlock(rtxExportMutex);
			sprintf (result, "<bad index>");
			return (-1);
		}
		if (s->callback)
			(s->callback)(s, indx, RTX_EXPORT_READ);

		if (nels > 1)
			strcpy(result, "[ ");
		else
			result[0] = '\0';
		for (i=0; i<nels; i++, indx++) {
			char	tmp[BUFSIZ];

			switch (s->type & RTX_EXPORT_TYPE_MASK) {
			case RTX_EXPORT_CHAR:
				sprintf(tmp, "%d", s->p.cp[indx]);
				break;
			case RTX_EXPORT_SHORT:
				sprintf(tmp, "%d", s->p.sp[indx]);
				break;
			case RTX_EXPORT_INT:
				sprintf(tmp, "%d", s->p.ip[indx]);
				break;
			case RTX_EXPORT_LONG:
				sprintf(tmp, "%ld", s->p.lp[indx]);
				break;
			case RTX_EXPORT_FLOAT:
				sprintf(tmp, "%f", 
					s->p.fp[indx]*s->scale + s->offset);
				break;
			case RTX_EXPORT_DOUBLE:
				sprintf(tmp, "%f", 
					s->p.dp[indx]*s->scale + s->offset);
				break;
			case RTX_EXPORT_STRING:
				sprintf(tmp, "%s", *(s->p.s));
				break;
			}
			if ((nels > 1) && (i < (nels-1)))
				strcat(tmp, " ");
			strcat(result, tmp);
		}
		if (nels > 1)
			strcat(result, "]");
		rtx_mutex_unlock(rtxExportMutex);
		return (0);
		break;

	case 3:		/* set value */
		if (rtxExportDebug)
			rtx_message_routine("exportvar_var: writing variable %s %s\n", argv[1], argv[2]);

		/* parse the variable name and index (optional) */
		n = sscanf(argv[1], "%[^[(]%*c%d", name, &indx);

		/* look it up, and check the index against bounds */
		s = rtx_hash_find(rtxHashTablePtr, name, (rtx_error_t)NULL);
		if (s == NULL){
			rtx_mutex_unlock(rtxExportMutex);
			sprintf (result, "<exportSym not found>");
			return (-1);
		}
		if (n == 1) {
			nels = s->dimension;
			indx = 0;
		}
		else
			nels = 1;

		if ((indx < 0) || (indx >= s->dimension)){
			rtx_mutex_unlock(rtxExportMutex);
			sprintf (result, "<bad index>");
			return (-1);
		}
		if (s->type & RTX_EXPORT_READONLY) {
			rtx_message_warning("exportvar_var: attempting to set RO variable %s\n", name);
			rtx_mutex_unlock(rtxExportMutex);
			sprintf (result, "RO");
			return (0);
		}
		switch (s->type & RTX_EXPORT_TYPE_MASK) {
		case RTX_EXPORT_CHAR:
			n = sscanf(argv[2], "%d", &intNumber);
			if (n == 1) {
				s->p.cp[indx] = intNumber;
			}
			break;
		case RTX_EXPORT_SHORT:
			n = sscanf(argv[2], "%d", &intNumber);
			if (n == 1) {
				s->p.sp[indx] = intNumber;
			}
			break;
		case RTX_EXPORT_INT:
			n = sscanf(argv[2], "%d", &intNumber);
			if (n == 1) {
				s->p.ip[indx] = intNumber;
			}
			break;
		case RTX_EXPORT_LONG:
			n = sscanf(argv[2], "%d", &intNumber);
			if (n == 1) {
				s->p.lp[indx] = intNumber;
			}
			break;
		case RTX_EXPORT_FLOAT:
			n = sscanf(argv[2], "%f", &floatNumber);
			if (n == 1) {
				s->p.fp[indx] = 
					(floatNumber - s->offset) / s->scale;
			}
			break;
		case RTX_EXPORT_DOUBLE:
			n = sscanf(argv[2], "%lf", &doubleNumber);
			if (n == 1) {
				s->p.dp[indx] =
					(doubleNumber - s->offset) / s->scale;
			}
			break;
		case RTX_EXPORT_STRING:
			strcpy(*(s->p.s), argv[2]);
			n = 1;
			break;
		}
		if (n != 1){
			rtx_mutex_unlock(rtxExportMutex);
			sprintf (result, "sscanf() error");
			return (-1);
		}else {
			if (s->callback)
				(s->callback)(s, indx, RTX_EXPORT_WRITE);
			if (global_callback)
				(global_callback)(s, indx, argv[2]);
			rtx_mutex_unlock(rtxExportMutex);
			result[0] = '\0';
			return (0);
		}

		break;
	}
	rtx_mutex_unlock(rtxExportMutex);
	result[0] = '\0';
	return (0);
}

/**
 * SRPC server function used for reading multiple server variables.
 *
 *	exportvar_multi_var name name ...
 */
static int
rtx_export_multi_var(char * result, int m, int argc, char *argv[])
{
	RtxExportSym	*s;
	char		name[64];
	unsigned int	indx;
	int		i, k, nels, n;

	rtx_mutex_lock(rtxExportMutex);
	if (rtxExportDebug)
		rtx_message_routine("exportvar_multi_var: in exportvar_multi_var\n");

	result[0] = '\0';
	for (k=1; k<argc; k++) {
		if (rtxExportDebug)
			rtx_message_routine("exportvar_multi_var: reading variable %s\n", argv[k]);

		/* parse the variable name and index (optional) */
		n = sscanf(argv[k], "%[^[(]%*c%d", name, &indx);

		/* look it up, and check the index against bounds */
		s = rtx_hash_find(rtxHashTablePtr, name, (rtx_error_t)NULL);
		if (s == NULL) {
			sprintf(result, "<%s not found>", name);
			rtx_mutex_unlock(rtxExportMutex);
			return (-1);
		}

		if (n == 1) {
			nels = s->dimension;
			indx = 0;
		}
		else
			nels = 1;

		if ((indx < 0) || (indx >= s->dimension)){
			rtx_mutex_unlock(rtxExportMutex);
			sprintf (result, "<bad index>");
			return (0);
		}

		if (s->callback)
			(s->callback)(s, indx, RTX_EXPORT_READ);

		for (i=0; i<nels; i++, indx++) {
			char	tmp[BUFSIZ];

			switch (s->type & RTX_EXPORT_TYPE_MASK) {
			case RTX_EXPORT_CHAR:
				sprintf(tmp, "%d", s->p.cp[indx]);
				break;
			case RTX_EXPORT_SHORT:
				sprintf(tmp, "%d", s->p.sp[indx]);
				break;
			case RTX_EXPORT_INT:
				sprintf(tmp, "%d", s->p.ip[indx]);
				break;
			case RTX_EXPORT_LONG:
				sprintf(tmp, "%ld", s->p.lp[indx]);
				break;
			case RTX_EXPORT_FLOAT:
				sprintf(tmp, "%f", 
					s->p.fp[indx]*s->scale + s->offset);
				break;
			case RTX_EXPORT_DOUBLE:
				sprintf(tmp, "%f", 
					s->p.dp[indx]*s->scale + s->offset);
				break;
			case RTX_EXPORT_STRING:
				sprintf(tmp, "%s", *(s->p.s));
				break;
			}
			strcat(tmp, " ");
			strcat(result, tmp);
		}
	}
	rtx_mutex_unlock(rtxExportMutex);
	return (0);
}

