/***********************************************************************
 * 
 * CSIRO Autonomous Systems Laboratory
 * Queensland Centre for Advanced Technologies
 * PO Box 883, Kenmore, QLD 4069, Australia
 * http://www.ict.csiro.au/
 *  
 * Copyright (c) CSIRO 
 ***********************************************************************/

#include <stdlib.h>
#include <stdio.h>
#include <stdarg.h>
#include <string.h>
#include <syslog.h>

#include "rtx/defines.h"
#include "rtx/error.h"
#include "rtx/message.h"

static char rcsid[] RTX_UNUSED = "$Id: error.c 3084 2008-05-15 12:42:28Z roy029 $";

#define MAX_ERROR_STR 1024

struct rtx_error {
  char str[MAX_ERROR_STR];
  int str_used;
  int depth;
};


rtx_error_t 
rtx_error_alloc() {
  rtx_error_t error;

  if (! (error = malloc(sizeof(struct rtx_error)))) {
    return NULL;
  }
  
  rtx_error_clear(error);

  return error;
}
  

void 
rtx_error_free(rtx_error_t error) {
  free(error);
}


void 
rtx_error_clear(rtx_error_t error) {
  error->depth = 0;
  error->str_used = 0;
  error->str[0] = '\0';
}


char *
rtx_error_str(rtx_error_t error) {
  return error->str;
}


void
rtx_error_log(rtx_error_t error) {
  char *destination;

  destination = getenv("RTX_ERROR");

  if (destination) { 
    if (strcmp(destination, "syslog") == 0) {
      syslog(LOG_USER, "%s", error->str);

    } else if (strcmp(destination, "message") == 0) {
      rtx_message_rtx_error(NULL, error->str);
    }

  } else {
    fprintf(stderr, "%s\n", error->str);
  }

  rtx_error_clear(error);
}


void 
rtx_error_push(rtx_error_t error, 
	       const char *function, const char *file, size_t lineno,
	       const char *format, ...) {
  va_list ap;
  int n1, n2;

  if (!error || error->str_used >= MAX_ERROR_STR) {
    return;
  }

  n1 = snprintf(error->str + error->str_used, MAX_ERROR_STR - error->str_used,
		"[%d] %s:%d %s(): ", error->depth, file, lineno, function);
    
  if (error->str_used + n1 + 1 > MAX_ERROR_STR) {
    return;
  }

  va_start(ap, format);

  n2 = vsnprintf(error->str + error->str_used + n1, MAX_ERROR_STR - error->str_used - n1,
		 format, ap);
  
  va_end(ap);

  if (error->str_used + n1 + n2 + 1 > MAX_ERROR_STR) {
    return;
  }

  /* If we get here, the full error string managed to fit; replace the
     NULL terminator of the previous level in the stack by a
     newline. */
  if (error->depth > 0) {
    error->str[error->str_used - 1] = '\n';
  }

  error->str_used += n1 + n2 + 1;
  error->depth++;
}

/**
 * \file error.c
 * \brief RTX error handler
 * \author Peter Corke
 *
 * This package provides a general purpose error handling system for multiple platforms.
 * The main principle with error handling is not to take action at the point of error since the
 * action may be inconsistant with the needs of the application.  For example taking the decision
 * to terminate, call exit(), may be inappropriate in a control system.  Similarly printing a message
 * may swamp the tty device and ruin the timing performance of the program.  Further, it makes assumptions
 * about where error messages are displayed (stderr or stdout) and if they happen very quickly they
 * may be lost, so a logging facility is also useful.
 *
 * The preferred method is to place a descriptive message onto an internal error stack and then
 * return with an error value, -1 or NULL, are often used.  The caller then checks the error status
 * and if an error was reported the caller may decide in turn to stack an error message and return with
 * error status.  Each function returns in sequence until some code decides to take specific action and 
 * print the error message stack.
 *
 * A typical way this code might be used is as follows:
 * <pre>

	if (func1() < 0)
		rtx_error_flush("func1 failed");


	int
	func1()
	{
		if (func2() < 0)
			return rtx_error("func1: func2 is bad");
		
		return 1;
	}

	int
	func2()
	{
		if (func3() < 0)
			return rtx_error("func2: func3 is bad");
		return 1;
	}

	int
	func3()
	{
		if (func4() < 0)
			return rtx_error("func3: func4 is bad");
		return 1;
	}

	int
	func4()
	{
		return rtx_error("I did it %d %d", 23, 24);
		return 1;
	}
</pre>
 * In a real-time thread that loops endlessly the code may be handled in this way
 <pre>
	thread()
	{
		while (1) {
			if (thread_action() < 0) {
				rtx_error_flush("thread action failed");
				deal with situation
			}
		}
	}

	int
	thread_action()
	{
		 .
		 .
		if (bad stuff)
			return rtx_error("bad thing 1 happened");
		 .
		 .
		if (bad stuff)
			return rtx_error("bad thing 2 happened");
		 .
		 .
		if (bad stuff)
			return rtx_error("bad thing 3 happened");

		return 0;
	}
 </pre>
 */

#include	<stdarg.h>
#include	<stdio.h>
#include    <unistd.h>
#include    <stdlib.h>
#include    <fcntl.h>
#include	<syslog.h>
#include	<errno.h>
#include	<string.h>
#include	<pthread.h>

#include "rtx/error.h"
#include "rtx/message.h"
#include "rtx/time.h"

#define RTX_ERROR_MAX_STACK_DEPTH   16
#define RTX_ERROR_MAX_LEN           256

/*
#define	DEBUG
*/
struct _rtx_err {
	char	*name;		/* thread name */
	int	flags;
	void 	(* handler)();
	char	messages[RTX_ERROR_MAX_STACK_DEPTH][RTX_ERROR_MAX_LEN];
        RtxTime ts[RTX_ERROR_MAX_STACK_DEPTH];
        char    msg[2*RTX_ERROR_MAX_LEN];
	int	depth;
};

static int rtx_err_key_created = 0;
static pthread_key_t	rtx_err_key;
static pthread_once_t   rtx_error_once_init = {PTHREAD_ONCE_INIT};

/* forward defines */
static void rtx_error_common_fn(const char *file, int line, int errnum, const char *fmt, va_list ap);
int rtx_error_error(const char *fmt, ...);
void rtx_error_write(int reporting, RtxTime * tp, char *s);
int rtx_error_traceback_stack (struct _rtx_err *e);

static void
rtx_err_key_destructor(void *p)
{
	struct _rtx_err * e = (struct _rtx_err *) p;

	if (e == NULL) {
		fprintf (stderr, "rtx_err_key_destructor: NULL pointer\n");
		return;
	}
	if (e->depth != 0)
		rtx_error_traceback_stack (e);
	free (e->name);
	free(e);
}

static void
rtx_error_init_once (void)
{
    int ret = -1;

    if ((ret = pthread_key_create((unsigned int *)&rtx_err_key, rtx_err_key_destructor)) != 0)
        rtx_error_error ("rtx_error_init_once: pthread_key_create() failed: %s",
			 strerror (errno));
    else
        rtx_err_key_created = 1;
    atexit ((void (*) (void)) rtx_error_traceback);
}


/**
 * Initialize per thread error handling.
 *
 * @param name	Name of process or thread.  Prepended to all messages.
 * @param flags Controls operation of error handler.  Flags is constructed by oring
 *	- RTX_ERROR_STDERR, send messages to processes standard error stream
 *	- RTX_ERROR_STDOUT, send messages to processes standard output stream
 *	- RTX_ERROR_MESSAGE, send messages to the RTX message stream
 *	- RTX_ERROR_SYSLOG, send message to Unix kernel syslog utility
 *	- RTX_ERROR_OCCURS, generate a report when error happens, rather than when flushed
 *	- RTX_ERROR_EXIT(v), cause process to exit with status \p v when error is flushed
 *	- RTX_ERROR_DEBUG, debug the error package
 *
 * More than one of the above can be selected, messages are sent in parallel to all
 * enabled streams.
 * @param handler A function to be called after the default error action.
 *
 * By default messages are sent to stderr and the name is \p noname.
 * @see message
 *
 * Default action with flags=0, handler=NULL, is for errors to be reported
 * on stderr.
 */
int
rtx_error_init(const char *name, int flags, void (*handler)())
{
	struct _rtx_err		*e;
	const char			*p;

	pthread_once (&rtx_error_once_init, rtx_error_init_once);

	if (rtx_err_key_created) {
	    if ((e = (struct _rtx_err *)pthread_getspecific(rtx_err_key)) != NULL) 
	        return (0);
	} else {
	    rtx_error_error ("rtx_error_init: rtx_err_key does not exist - serious error");
	}

	e = (struct _rtx_err *)calloc(1, sizeof(struct _rtx_err));
	if (e == NULL)
		return (rtx_error_error("rtx_err_init: no mem"));
	
	/* find root part of executable name */
	p = strrchr(name, '/');
	if (p == NULL)
		p = name;
	else
		p++;
	e->name = strdup(p);
	/* by default send errors to stderr */
	if (flags == 0)
		flags |= RTX_ERROR_STDERR;

	e->flags = flags;
	e->handler = handler;
	e->depth = 0;
	pthread_setspecific(rtx_err_key, (void *)e);

	return (0);
}

/**
 * Report an error to be handled according to per-thread settings.
 * The function has printf() like format and allows an arbitrary
 * number of arguments.  A newline is not required at the end of
 * the format string.
 * The errno error code string is appended to the error report.
 * This function should be used via the macro 
 * rtx_error_errno (char *fmt, ...).
 *
 * @param fmt A printf style format string.
 * @param file The source file where the error occured.
 * @param line The line number in the source file where the error occurred.
 * @return -1.
 */
int
rtx_error_errno_fn(const char *file, int line, const char *fmt, ...)
{
	va_list	ap;

	va_start(ap, fmt);

#ifdef	DEBUG
	fprintf(stderr, "rtx_error_errno[%s] ", strerror(errno));
	vfprintf(stderr, fmt, ap);
	fprintf(stderr, "\n");
#endif
	rtx_error_common_fn(file, line, errno, fmt, ap);
	return -1;
}

/**
 * Report an error to be handled according to per-thread settings.
 * The function has printf() like format and allows an arbitrary
 * number of arguments.  A newline is not required at the end of
 * the format string. This function requires the error number as an
 * explicit parameter and is useful as a wrapper around PTHREAD library
 * function which return an error number rather than -1.
 * The errno error code string is appended to the error report.
 * This function should be used via the macro 
 * rtx_error_errno2 (char *fmt, ...).
 *
 * @param fmt A printf style format string.
 * @param file The source file where the error occured.
 * @param line The line number in the source file where the error occurred.
 * @param errnum Error number.
 * @return -1.
 */
int
rtx_error_errno2_fn(const char *file, int line, int errnum, const char *fmt, ...)
{
	va_list	ap;

	va_start(ap, fmt);

#ifdef	DEBUG
	fprintf(stderr, "rtx_error_errno[%s] ", strerror(errno));
	vfprintf(stderr, fmt, ap);
	fprintf(stderr, "\n");
#endif
	rtx_error_common_fn(file, line, errnum, fmt, ap);
	return -1;
}

/**
 * Report an error to be handled according to per-thread settings.
 * The function has printf() like format and allows an arbitrary
 * number of arguments.  A newline is not required at the end of
 * the format string.
 * The errno error code string is appended to the error report.
 * This function should be used via the macro 
 * rtx_error_errno_null (char *fmt, ...).
 *
 * @param fmt A printf style format string.
 * @param file The source file where the error occured.
 * @param line The line number in the source file where the error occurred.
 * @param errnum Error number.
 * @return NULL.
 */
void *
rtx_error_errno_null_fn(const char *file, int line, const char *fmt, ...)
{
	va_list	ap;

	va_start(ap, fmt);

#ifdef	DEBUG
	fprintf(stderr, "rtx_error_errno[%s] ", strerror(errno));
	vfprintf(stderr, fmt, ap);
	fprintf(stderr, "\n");
#endif
	rtx_error_common_fn(file, line, errno, fmt, ap);
	return NULL;
}

/**
 * Report an error to be handled according to per-thread settings.
 * The function has printf() like format and allows an arbitrary
 * number of arguments.  A newline is not required at the end of
 * the format string.
 * This function should be used via the macro 
 * rtx_error (char *fmt, ...).
 *
 * @param fmt A printf style format string.
 * @param file The source file where the error occured.
 * @param line The line number in the source file where the error occurred.
 * @return -1.
 */
int
rtx_error_fn(const char *file, int line, const char *fmt, ...)
{
	va_list	ap;

	va_start(ap, fmt);

#ifdef	DEBUG
	fprintf(stderr, "rtx_error: ");
	vfprintf(stderr, fmt, ap);
	fprintf(stderr, "\n");
#endif

	rtx_error_common_fn(file, line, 0, fmt, ap);
	return -1;
}

/**
 * Report an error to be handled according to per-thread settings.
 * The function has printf() like format and allows an arbitrary
 * number of arguments.  A newline is not required at the end of
 * the format string.
 * This function should be used via the macro 
 * rtx_error_null (char *fmt, ...).
 *
 * @param fmt A printf style format string.
 * @param file The source file where the error occured.
 * @param line The line number in the source file where the error occurred.
 * @return NULL.
 */
void *
rtx_error_null_fn(const char *file, int line, const char *fmt, ...)
{
	va_list	ap;

	va_start(ap, fmt);

#ifdef	DEBUG
	fprintf(stderr, "rtx_error: ");
	vfprintf(stderr, fmt, ap);
	fprintf(stderr, "\n");
#endif

	rtx_error_common_fn(file, line, 0, fmt, ap);
	return NULL;
}

/**
 * Get error flags for this execution context.
 *
 * @return value of current error handler flags
 */
int
rtx_error_flags_get()
{
	struct _rtx_err         *e = NULL;

	if (rtx_err_key_created)
	    e = (struct _rtx_err *)pthread_getspecific(rtx_err_key);

	if (e != NULL)
	    return e->flags;
	return (0);
}

/**
 * Get internal state structure for this execution context.
 *
 * @return Pointer to a struct _rtx_err which holds error subsystem state.
 */
struct _rtx_err *
rtx_error_status_get()
{
	struct _rtx_err         *e = NULL;

	if (rtx_err_key_created)
	    e = (struct _rtx_err *)pthread_getspecific(rtx_err_key);
  
	return e;
}

/**
 * Common part of error reporting system.
 *
 * @param errnum an errno style error number
 * @param fmt the printf style format string
 * @param ap the arguments to be included in the printf string
 *
 * If \p errnum is not zero the string will be appended with \p errnum and
 * its string representation via \p strerror().
 *
 * The error message is built up and pushed onto the stack for the current
 * execution context.
 */
static void
rtx_error_common_fn(const char *file, int line, int errnum, const char *fmt, va_list ap)
{
	struct _rtx_err		*e;
	int			l = 0;
	char			*p;
        RtxTime	                ts = {0, 0};
	char                    buf[64];

	rtx_time_get (&ts);
	if (rtx_err_key_created) {
	    if ((e = (struct _rtx_err *)pthread_getspecific(rtx_err_key)) == NULL) {
	        /* 
		 * no key data, error system hasn't been initialized by this thread,
		 * do an rtx_error_init()
		 */
	        buf[63] = '\0';
	        snprintf (buf, 63, "%s (pid=%d, tid=%d)", __FILE__, (int) getpid(),
			  (int) pthread_self ());
		rtx_error_init(buf, 0, NULL);
		if ((e = (struct _rtx_err *)pthread_getspecific(rtx_err_key)) == NULL) {
		    /*
		     * we have done an init, and yet no key data. There is
		     * a serious problem.
		     */
		    rtx_error_error ("rtx_error_common: cant find key data - "
				     "serious error");
		    return;
		}
	    }
	} else {
	    /* key does not exist. No one has yet initialized the error system.
	     */
	    buf[63] = '\0';
	    snprintf (buf, 63, "%s (pid=%d, tid=%d)", __FILE__, (int) getpid(),
		      (int) pthread_self ());
	    rtx_error_init(buf, 0, NULL);
	    if (rtx_err_key_created) {
	        if ((e = (struct _rtx_err *)pthread_getspecific(rtx_err_key)) == NULL) {
		    /*
		     * we have done an init, and yet no key data. There is
		     * a serious problem.
		     */
		    rtx_error_error ("rtx_error_common: cant find key data - serious error");
		    return;
		}
	    } else {
	        rtx_error_error ("rtx_error_common: rtx_error_init() failed - serious error");
		return;
	    }
	}

	/*
	 * we have an initialized key, use it to guide what we do at this
	 * point
	 */

	if (e->depth >= RTX_ERROR_MAX_STACK_DEPTH) {
		rtx_error_error("rtx_error_common: no mem, flushing error stack");
	        rtx_error_traceback();
	}

	e->ts[e->depth] = ts;
	p = e->messages[e->depth];
	/* insert the name of the process */
	sprintf(&(p[l]), "%s: %s[%d] ",	e->name, file, line);

	/* append the printf style error message */
	l = strlen(p);
	vsnprintf(&p[l], RTX_ERROR_MAX_LEN-l, fmt, ap);

	/* remove trailing newline */
	if (p[strlen(p)-1] == '\n')
		p[strlen(p)-1] = '\0';

	/* append the errno if given */
	if (errnum != 0) {
		l = strlen(p);
		snprintf(&p[l], RTX_ERROR_MAX_LEN-l, " [%s]", strerror(errnum));
	}
#ifdef	DEBUG
	fprintf(stderr, "rtx_error_common <%s>\n", p);
#endif

	/* copy and stack it */

	e->depth++;

	if (e->flags & RTX_ERROR_OCCURS)
		rtx_error_write(e->flags, &ts, p);

	return;
}

/**
 * Report an error to be reported and acted on right now.
 */
void
rtx_error_flush_fn(const char *file, int line, const char *fmt, ...)
{
	va_list	ap;

	va_start(ap, fmt);

	rtx_error_common_fn(file, line, 0, fmt, ap);
	rtx_error_traceback();
}

/**
 * Report an error to be reported and acted on right now (to specified depth).
 */
void
rtx_error_flush_depth_fn(const char *file, int line, int depth, const char *fmt, ...)
{
	va_list	ap;

	va_start(ap, fmt);

	rtx_error_common_fn(file, line, 0, fmt, ap);
	rtx_error_traceback_depth(depth);
}

void
rtx_error_errno_flush_fn(const char *file, int line, const char *fmt, ...)
{
	va_list	ap;

	va_start(ap, fmt);

	rtx_error_common_fn(file, line, errno, fmt, ap);
	rtx_error_traceback();
}

void
rtx_error_errno_flush_depth_fn(const char *file, int line, int depth, const char *fmt, ...)
{
	va_list	ap;

	va_start(ap, fmt);

	rtx_error_common_fn(file, line, errno, fmt, ap);
	rtx_error_traceback_depth(depth);
}

/**
 * Display the given error traceback stack and terminate if required.
 * 
 * @return 0 if ok, -1 if errors to handle.
 */
int
rtx_error_traceback_stack (
			   struct _rtx_err *e
			   )
{
	int i;

	if (e == NULL)
	        return (0);
	if (e->depth == 0)
		return 0;

	sprintf(e->msg, "rtx_err_traceback: depth=%d, starting traceback:\n", e->depth);
	rtx_error_write (e->flags, NULL, e->msg);
	for (i=0; i<e->depth; i++) {
		sprintf(e->msg, " %4s %s\n", (i == 0) ? "in" : "from",
			e->messages[i]);
		rtx_error_write(e->flags, &(e->ts[i]), e->msg);
	}
	sprintf(e->msg, "end of traceback\n");
	rtx_error_write (e->flags, NULL, e->msg);

	e->depth = 0;
	if (e->flags & RTX_ERROR_EXIT(0)) {
		sprintf(e->msg, "rtx_error_traceback is exiting\n");
		rtx_error_write (e->flags, NULL, e->msg);
		exit(1);
	}
	return -1;
}

/**
 * Display the error traceback stack and terminate if required.
 * 
 * @return 0 if ok, -1 if errors to handle.
 */
int
rtx_error_traceback()
{
	struct _rtx_err		*e;

	if ((e = (struct _rtx_err *)pthread_getspecific(rtx_err_key)) == NULL)
		return (rtx_error_error("couldnt obtain key"));

	return (rtx_error_traceback_stack (e));
}

/**
 * Display the error traceback stack to specified depth and terminate if required.
 * 
 * @return 0 if ok, -1 if errors to handle.
 */
int
rtx_error_traceback_depth(
			  int depth
			  )
{
	struct _rtx_err		*e;
	int			i;
	int                     flushDepth = depth;

	if ((e = (struct _rtx_err *)pthread_getspecific(rtx_err_key)) == NULL)
		return (rtx_error_error("couldnt obtain key"));

	if (depth <= 0) {
	        e->depth = 0;
		return (0);
	}
	if (e->depth == 0)
		return 0;
	if (flushDepth > e->depth)
	        flushDepth = e->depth;

	sprintf(e->msg, "rtx_err_traceback: depth=%d, starting traceback:\n", e->depth);
	rtx_error_write (e->flags, NULL, e->msg);
	for (i=0; i<flushDepth; i++) {
		sprintf(e->msg, " %4s %s\n", (i == 0) ? "in" : "from",
			e->messages[i]);
		rtx_error_write(e->flags, &(e->ts[i]), e->msg);
	}
	sprintf(e->msg, "end of traceback\n");
	rtx_error_write (e->flags, NULL, e->msg);

	e->depth = 0;
	if (e->flags & RTX_ERROR_EXIT(0)) {
		sprintf(e->msg, "rtx_error_traceback is exiting\n");
		rtx_error_write (e->flags, NULL, e->msg);
		exit(1);
	}
	return -1;
}

/*
 * send error message to all specified destinations: stdio, syslog, messaged
 */
void
rtx_error_write(int reporting, RtxTime * tp, char *s)
{
	if (reporting & RTX_ERROR_SYSLOG)
		syslog(LOG_WARNING, s);
	if (reporting & RTX_ERROR_MESSAGE)
		rtx_message_rtx_error(tp, s);
	if (reporting & RTX_ERROR_STDERR)
		fprintf(stderr, s);
	if (reporting & RTX_ERROR_STDOUT)
		fprintf(stdout, s);
}

/*
 * deal with errors that occur within the error handler
 */
int
rtx_error_error(const char *fmt, ...)
{
	va_list	ap;

	va_start(ap, fmt);

	fprintf(stderr, "rtx_error: error handling error: ");
	vfprintf(stderr, fmt, ap);
	fprintf(stderr, "\n");
	return (-1);
}

