/***********************************************************************
 * 
 * CSIRO Autonomous Systems Laboratory
 * Queensland Centre for Advanced Technologies
 * PO Box 883, Kenmore, QLD 4069, Australia
 * http://www.ict.csiro.au/
 *  
 * Copyright (c) CSIRO 
 ***********************************************************************/

/**
 * \file delay.c
 * \brief Generalized delay line module
 * \author Peter Corke
 */

#include <stdio.h>
#include <stdlib.h>

#include "rtx/error.h"
#include "rtx/delay.h"
#include "rtx/defines.h"

static char rcsid[] RTX_UNUSED = "$Id: delay.c 2274 2007-12-23 05:37:32Z roy029 $";

#define	MODINC(x,N)	(x = ((x)+1)%N)

/**
 * Put a value into the delay line.
 * @param d a RtxDelay type.
 * @param x value to be inserted into the delayline.
 */
void
rtx_delay_put(RtxDelay *d, double x)	
{
	d->b[MODINC(d->i,d->n)] = x;
}

/**
 * Get the sample from the delay line that occurred \p t seconds in the past.
 * @param d a RtxDelay type.
 * @param t time in the past for which to retrieve value.
 * @return the delayed value from the delay line.
 */
double
rtx_delay_get(RtxDelay *d, double t)	
{
	return d->b[(d->n+d->i-(int)((t)/d->dt)) % d->n];
}

/**
 * Creates a delay line.
 * @param tmax maximum delay permissible.
 * @param dt sample interval.
 * Creates a delay line that holds samples for a maximum time of tmax
 * assuming they are sampled at dt.
 * @return a new RtxDelay type, or NULL on error.
 */
RtxDelay *
rtx_delay_create(double tmax, double dt)
{
	RtxDelay	*d;
	int		n;

	if ((d = (RtxDelay *)calloc(1, sizeof(RtxDelay))) == NULL)
		return rtx_error_errno_null("rtx_delay_create: calloc");

	n = (int)(tmax / dt);
	if ((d->b = (double *)calloc(n, sizeof(double))) == NULL)
		return rtx_error_errno_null("rtx_delay_create: calloc");

	d->dt = dt;
	d->n = n;

	return d;
}

#ifdef	MAIN
#include	<stdio.h>
#include	<stdlib.h>

main(int ac, char *av[])
{
	RtxDelay	*dl;
	double		delay;
	double		count = 1.0;
	int		i;
	const double	DT = 0.1;


	if (ac == 1) {
		fprintf(stderr, "delay lag\n\ntest utility for delayline package\n"
			"creates a delay line of length 10s with sample interval of 0.1s\n");
		exit(1);
	}

	dl = rtx_delay_create(10.0, DT);
	delay = atof(av[1]);
	printf("     t        In         Out\n");
	for (i=0; i<25; i++) {

		rtx_delay_put(dl, count);
		printf("%10f%10f %10f\n", i*DT, count, rtx_delay_get(dl, delay));
		count++;
	}
}
#endif
