/***********************************************************************
 * 
 * CSIRO Autonomous Systems Laboratory
 * Queensland Centre for Advanced Technologies
 * PO Box 883, Kenmore, QLD 4069, Australia
 * http://www.ict.csiro.au/
 *  
 * Copyright (c) CSIRO 
 ***********************************************************************/

/**
 * \file cond.c
 * \brief A simple interface to condition variables
 * \author J Roberts
 */

#include <stdlib.h>
#include <unistd.h>
#include <pthread.h>
#include <string.h>
#include <errno.h>

#include "rtx/defines.h"
#include "rtx/time.h"
#include "rtx/timer.h"
#include "rtx/error.h"
#include "rtx/mutex.h"
#include "rtx/cond.h"

static char rcsid[] RTX_UNUSED = "$Id: cond.c 2274 2007-12-23 05:37:32Z roy029 $";

/**
 * Initialize a RtxCond structure.
 *
 * @return Pointer to a condition variable on success, NULL on error
 *
 * This function provides a simple wrapper around PTHREADS condition 
 * variables. It initializes a RtxCond structure which consists of
 * a mutex, a condition variable, a function pointer and an argument
 * pointer that can be used to evaluate a condition associated with
 * the condition variable. 
 */
RtxCond *
rtx_cond_init (
        RtxCond * cp,      /**< pointer to cond struct */
	int mutexAttrs,    /**< Attributes for the mutex */
	int prio,          /**< Priority ceiling (if used) */
	int condAttrs,     /**< Attributes for the cond var */
	int (* func) (void * arg), /**< Function to evaluate condition */
	void * arg	/**< Argument to func() */
		     )
{
	RtxCond			*p = cp;
	int 			retval;
	pthread_condattr_t	attr, * attrP = NULL;

	if (p == NULL) {
  	    if ((p = (RtxCond *)calloc(1, sizeof(RtxCond))) == NULL)
		return rtx_error_null("rtx_cond_init: calloc: no memory");
	    p->memAlloc = 1;
	} else
	    p->memAlloc = 0;

	if ((rtx_mutex_init (&p->mutex, mutexAttrs, prio)) == NULL)
		return (rtx_error_null ("rtx_cond_init: rtx_mutex_init() failed"));

	if (condAttrs != RTX_MUTEX_DEFAULT) {
	    attrP = &attr;
	    if ((retval = pthread_condattr_init (attrP))) {
	        rtx_error_errno2 (retval, "rtx_cond_init: pthread_condattr_init: ");
		return NULL;
	    }
	    if (condAttrs & RTX_MUTEX_PROCESS_SHARED) {
#ifdef _POSIX_THREAD_PROCESS_SHARED
	        if ((retval = pthread_condattr_setpshared (attrP, 
					      PTHREAD_PROCESS_SHARED))) {
		    rtx_error_errno2 (retval, "rtx_cond_init: pthread_condattr_setpshared: ");
		    return NULL;
		}
#else
		return (rtx_error_null ("rtx_cond_init: RTX_MUTEX_PROCESS_SHARED "
					"not supported for condition variables"));
#endif
	    }
	}
	if ((retval = pthread_cond_init (&(p->cond), attrP))) {
	    rtx_error_errno2 (retval, "rtx_cond_init: pthread_cond_init: ");
	    return NULL;
	}
    
	p->func = func;
	p->arg = arg;
    
	return p;
}

/**
 * Destroy a RtxCond structure.
 *
 * @return 0 if OK, -1 on error
 *
 * This function provides a simple wrapper around PTHREADS condition 
 * variables. It destroys the mutex and the condition variable associated
 * with the RtxCond structure.
 */
int 
rtx_cond_destroy (
	RtxCond	*p    /**< Pointer to RtxCond struct to be destroyed */
	)
{
    int	retval = 0, errs = 0;  

    if (rtx_mutex_destroy (&p->mutex) == -1) {
        rtx_error_errno2 (retval, "rtx_cond_destroy: "
			  "rtx_mutex_destroy() failed");
	errs++;
    }
    if ((retval = pthread_cond_destroy (&(p->cond)))) {
        rtx_error_errno2 (retval, "rtx_cond_destroy: "
				  "pthread_cond_destroy() failed");
	errs++;
    }
    if (p->memAlloc)
        free (p);
    if (errs)
        return (-1);
    return (0);
}

/**
 * Wait for a condition variable to be signalled.
 *
 * @return 0 if OK, -1 on error.
 *
 * This function waits on the condition variable until the associated
 * condition is satisfied. The condition is considered to be satisfied
 * when the function p->func() returns 0.
 */
int
rtx_cond_wait (
	RtxCond	*p    /**< Pointer to RtxCond struct */
	)
{
	int retval, errs = 0;

	if ((retval = pthread_mutex_lock (&(p->mutex.mutex))))
        	return (rtx_error_errno2 (retval, "rtx_cond_wait: "
				  "pthread_mutex_lock failed"));

	if (p->func == NULL) {
       		if ((retval = pthread_cond_wait (&(p->cond), &(p->mutex.mutex))))
			rtx_error_errno2 (retval, "rtx_cond_wait: pthread_cond_wait: ");
	} else {
		while ((p->func (p->arg)) && (! errs)) {
        		if ((retval = pthread_cond_wait (&(p->cond), &(p->mutex.mutex)))) {
				rtx_error_errno2 (retval, "rtx_cond_wait: pthread_cond_wait: ");
		    		rtx_timer_sleep (0.001);
		    		errs++;
			}
    		}
	}

	if ((retval = pthread_mutex_unlock (&(p->mutex.mutex))))
		return (rtx_error_errno2 (retval, "rtx_cond_wait: pthread_mutex_unlock: "));
	if (errs)
        	return (-1);
	
	return (0);
}

/**
 * Wait (with timeout) for a condition variable to be signalled.
 *
 * @return 0 if OK, 1 if timed out, and -1 on error.
 *
 * This function waits on the condition variable until the associated
 * condition is satisfied. The condition is considered to be satisfied
 * when the function p->func() returns 0. It is different from the normal
 * wait in that a timeout for the wait is specified.
 */
int
rtx_cond_timed_wait (
	RtxCond 	*p,	/**< Pointer to RtxCond struct */
	double		timeout /**< Timeout */
    )
{
	int		retval = 0, timedout = 0, errs = 0;
	RtxTime		elapsed, remaining, tsEntry, tsNow;
	struct timespec	tsRemaining;

	rtx_time_get (&tsEntry);
	rtx_time_from_double (&remaining, timeout);
	tsRemaining.tv_sec = remaining.seconds;
	tsRemaining.tv_nsec = remaining.nanoSeconds;
	if ((retval = pthread_mutex_lock (&(p->mutex.mutex))))
		return (rtx_error_errno2 (retval, "rtx_cond_timed_wait: "
				  "pthread_mutex_lock failed"));

	if (p->func == NULL) {
        	if ((retval = pthread_cond_timedwait (&(p->cond), &(p->mutex.mutex), &tsRemaining)))
			if (retval && (retval!=ETIMEDOUT))
					rtx_error_errno2 (retval, "rtx_cond_timed_wait: pthread_cond_wait: ");
	} else {
		while ((p->func (p->arg)) && (! errs)) {
        		if ((retval = pthread_cond_timedwait (&(p->cond), &(p->mutex.mutex), &tsRemaining))) {
					if (retval && (retval!=ETIMEDOUT))
						rtx_error_errno2 (retval, "rtx_cond_timed_wait: pthread_cond_wait: ");
				rtx_timer_sleep (0.001);
				errs++;
			} else {
  	    			rtx_time_get (&tsNow);
	    			rtx_time_subtract (&elapsed, &tsNow, &tsEntry);
	    			if (rtx_time_cmp (&elapsed, &remaining) >= 0) {
	       	 		errs++;
	    			} else {
	       	 		rtx_time_subtract (&remaining, &remaining, &elapsed);
					tsRemaining.tv_sec = remaining.seconds;
					tsRemaining.tv_nsec = remaining.nanoSeconds;
	    			}
			}
		}
    	}

	if (retval == ETIMEDOUT)
		timedout = 1;

	if ((retval = pthread_mutex_unlock (&(p->mutex.mutex))))
		return (rtx_error_errno2 (retval, "rtx_cond_timed_wait: pthread_mutex_unlock: "));

	if (timedout)
		return (1);

	return (0);
}

/**
 * Signal a condition variable.
 *
 * @return 0 if OK, -1 on error
 *
 * This function provides a wrapper around the PTHREADS pthread_cond_broadcast()
 * function which returns an error number if an error occurs. This function
 * returns -1 on error and pushes the error number on the error stack using
 * rtx_error_errno2().
 */
int 
rtx_cond_signal (
	RtxCond	*p    /**< Pointer to RtxCond struct */
	)
{
    int retval;
    
    if ((retval = pthread_cond_signal (&(p->cond))))
        return (rtx_error_errno2 (retval, "rtx_cond_signal: "
				  "pthread_cond_signal failed"));
    
    return (0);
}

/**
 * Broadcast a condition variable.
 *
 * @return 0 if OK, -1 on error
 *
 * This function provides a wrapper around the PTHREADS pthread_cond_broadcast()
 * function which returns an error number if an error occurs. This function
 * returns -1 on error and pushes the error number on the error stack using
 * rtx_error_errno2().
 */
int 
rtx_cond_broadcast (
	RtxCond	*p    /**< Pointer to RtxCond struct */
	)
{
    int retval;
    
    if ((retval = pthread_cond_broadcast (&(p->cond))))
        return (rtx_error_errno2 (retval, "rtx_cond_broadcast: "
				  "pthread_cond_broadcast: "));
    
    return (0);
}


